/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *	Memory access.
 *-----------------------------------------------------------------------------
 * $Id: c64.c,v 1.18 1996/07/04 02:18:22 johns Exp root $
 * $Log: c64.c,v $
 * Revision 1.18  1996/07/04 02:18:22  johns
 * Timers now cycle syncron with 6510. Bug in rasterline irq removed.
 * Serial IEC functions moved in own file (iec.c), now traps in port read/write.
 * New emulator traps for own serial functions. Keyboard input commandline
 * option added. Modified library code for disk image support.
 *
 * Revision 1.17  1996/06/16 00:38:44  johns
 * Turn sound off only for speaker and opl3 without IO-privilege.
 *
 * Revision 1.16  1996/06/12 23:54:14  ari
 * Work around for svgalib et4000w32x bug. Removed toupper.
 *
 * Revision 1.15  1996/06/11 02:05:29  johns
 * OPL3 and better DSP support. RMW bug of 6510 implemented.
 * Dynamic X scrolling improved. New interrupt handling.
 * Videosync commandline option added. Extensions off command line option.
 * Changed behavior if getiopriv fails. Serial traps now do CLI.
 * Returns better value for SID Envelope register.
 *
 * Revision 1.14  1996/05/08 00:49:23  johns
 * New interrupt routines and handling.
 *
 * Revision 1.13  1996/05/03 17:41:02  johns
 * Many bugs fixed, TOD irq, CIA-SP, VIC X and Y scrolling.
 * Register argument parsing added. Started new memory interface and cleanup.
 *
 * Revision 1.12  1994/07/16  13:59:03  root
 * update year
 *
 * Revision 1.11  1994/06/02  15:46:04  johns
 * getopt :u fix ( was forgotten )
 *
 * Revision 1.10  1994/06/02  14:54:24  johns
 * Use demosum.h, always revert meaning of '-j'.
 *
 * Revision 1.9  1993/12/30  11:48:12  johns
 * go32 changes
 *
 * Revision 1.8  1993/10/01  18:53:28  johns
 * GO32 has reverse meaning of '-j'
 *
 * Revision 1.7  1993/08/31  20:30:03  johns
 * go32 initial support
 *
 * Revision 1.6  1993/06/13  12:24:03  johns
 * wuebbel patches, cleanup
 *
 * Revision 1.5  1993/01/05  12:41:24  johns
 * Checkin before applying wuebbel patches
 *
 * Revision 1.4  1992/07/28  19:47:42  johns
 * For debugging VicFetch and VicFetchAdd variables.
 * Testing global argument parsing for config routines.
 *
 * Revision 1.3  1992/07/20  04:15:46  johns
 * Argument parsing written. Better sound chip emulation.
 * Quicker sprite handling.
 *
 * Revision 1.2  1992/07/13  04:36:26  johns
 * Rasterline Interrupt now a action, iec-read bug fixed.
 *
 * Revision 1.1  1992/07/11  21:52:24  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

/*
**	4 cycles for store, 1 cycle to start and an unknown.
**	4 cycles for read.
*/
#define TIMER_ADD	6		/* shift timer cycles <-> 6510 */
#define TIMER_SUB	4		/* shift timer cycles <-> 6510 */

#include "c64.h"
#include "vic.h"
#include "sid.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/signal.h>

#if defined(__linux__) || defined(M_UNIX)
#include <termios.h>
#include <sys/ioctl.h>
#endif
























static char* ProgramName;

/*---------------------------------------------------------------------------*/

static void EmulCia1TimerA(void);
static void EmulCia1TimerB(void);
static void EmulCia2TimerA(void);
static void EmulCia2TimerB(void);

/*---------------------------------------------------------------------------*/

/* Currently ratio 1541-CPU%C64-CPU*1000 */
int DoMystic=1015;				/* To support mystic effects */

/*---------------------------------------------------------------------------*/

/* 0: -LORAM	A000-AFFF ROM/RAM */
/* 1: -HIRAM	E000-FFFF ROM/RAM */
/* 2: -CHAREN	D000-DFFF IO/CHAR-ROM */

#define CONFIG_SIZE (64*1024+255)

static unsigned char Config0[CONFIG_SIZE];
static unsigned char Config1[CONFIG_SIZE];
static unsigned char Config2[CONFIG_SIZE];
static unsigned char Config3[CONFIG_SIZE];
static unsigned char Config5[CONFIG_SIZE];
static unsigned char Config6[CONFIG_SIZE];
static unsigned char Config7[CONFIG_SIZE];

unsigned char* CONST ConfigTable[8] = {
    Config0,				/* Ram Ram   Ram	Ram */
    Config1,				/* Ram Ram   Character	Ram */
    Config2,				/* Ram Ram   Character	Kernel */
    Config3,				/* Ram Basic Character	Kernel */
    Config0,				/* Ram Ram   Ram	Ram */
    Config5,				/* Ram Ram   I/O	Ram */
    Config6,				/* Ram Ram   I/O	Kernel */
    Config7				/* Ram Basic I/O	Kernel */
};

unsigned char* Config;			/* Current memory configuration */

unsigned char Ram[64*1024];		/* 64k C64	 RAM */
unsigned char ExternalRom[16*1024];	/* 16k External	 ROM */
unsigned char BasicRom[8*1024];		/*  8k Basic	 ROM */
unsigned char CharacterRom[4*1024];	/*  4k Character ROM */
unsigned char KernelRom[8*1024];	/*  8k Kernel	 ROM */

unsigned char VIC[47];			/* Vic chip register */
unsigned char SID[29];			/* Sid chip register */
unsigned char ColorRam[1024+80];	/* 1k Color RAM */
unsigned char CIA1[16];			/* Cia1 chip register */
unsigned char CIA2[16];			/* Cia2 chip register */

#ifdef ALE_RAM
unsigned char AleRam1[512];		/* 512 Byte additional RAM */
unsigned char AleRam2[512];		/* 512 Byte additional RAM */
#endif
int ExtensionOff;			/* extension off */

/*
 *	Emulation
 */
int AutoFire;				/* Autofire on */
int JoystickGiven;			/* Flag joystick given on commandline */
unsigned JoyStick1=0xFF;		/* Joystick on port 1 */
unsigned JoyStick2=0xFF;		/* Joystick on port 2 */
unsigned* JoyStick=&JoyStick2;		/* Current emulated joystick */
unsigned* OtherJoyStick=&JoyStick1;	/* Optional hardware joystick */

/*
**	CIA Emulation!
*/
int Cia1SpCount;			/* serial port count */
int Cia1SpLatch;			/* serial port latch */
int Cia1TimerA;				/* Loaded count */
unsigned Cia1TimerA_Action;		/* Next action */
int Cia1TimerB;
unsigned Cia1TimerB_Action;
int Cia1IrqMask;			/* Interrupt mask */
int Cia1TodBuffered;			/* TOD read buffered */
int Cia1TodStopped;			/* TOD halted */
unsigned char Cia1Time[4];		/* Time buffer */ 
unsigned char Cia1Alarm[4];		/* Alarm time */ 

int Cia2SpCount;			/* serial port count */
int Cia2SpLatch;			/* serial port latch */
int Cia2TimerA;				/* Loaded count */
unsigned Cia2TimerA_Action;		/* Next action */
int Cia2TimerB;
unsigned Cia2TimerB_Action;
int Cia2IrqMask;			/* Interrupt mask */
int Cia2TodBuffered;			/* TOD read buffered */
int Cia2TodStopped;			/* TOD halted */
unsigned char Cia2Time[4];		/* Time buffer */ 
unsigned char Cia2Alarm[4];		/* Alarm time */ 

/*---------------------------------------------------------------------------*/

/*
**	quadruple a nibble.
*/
CONST unsigned long QuadNibble[256] = {
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F,
    0x00000000, 0x01010101, 0x02020202, 0x03030303,
    0x04040404, 0x05050505, 0x06060606, 0x07070707,
    0x08080808, 0x09090909, 0x0A0A0A0A, 0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D, 0x0E0E0E0E, 0x0F0F0F0F
};

/*
**	quadruple a byte.
*/
CONST unsigned long QuadByte[256] = {
    0x00000000, 0x01010101,0x02020202,0x03030303,
    0x04040404, 0x05050505,0x06060606,0x07070707,
    0x08080808, 0x09090909,0x0A0A0A0A,0x0B0B0B0B,
    0x0C0C0C0C, 0x0D0D0D0D,0x0E0E0E0E,0x0F0F0F0F,
    0x10101010, 0x11111111,0x12121212,0x13131313,
    0x14141414, 0x15151515,0x16161616,0x17171717,
    0x18181818, 0x19191919,0x1A1A1A1A,0x1B1B1B1B,
    0x1C1C1C1C, 0x1D1D1D1D,0x1E1E1E1E,0x1F1F1F1F,
    0x20202020, 0x21212121,0x22222222,0x23232323,
    0x24242424, 0x25252525,0x26262626,0x27272727,
    0x28282828, 0x29292929,0x2A2A2A2A,0x2B2B2B2B,
    0x2C2C2C2C, 0x2D2D2D2D,0x2E2E2E2E,0x2F2F2F2F,
    0x30303030, 0x31313131,0x32323232,0x33333333,
    0x34343434, 0x35353535,0x36363636,0x37373737,
    0x38383838, 0x39393939,0x3A3A3A3A,0x3B3B3B3B,
    0x3C3C3C3C, 0x3D3D3D3D,0x3E3E3E3E,0x3F3F3F3F,
    0x40404040, 0x41414141,0x42424242,0x43434343,
    0x44444444, 0x45454545,0x46464646,0x47474747,
    0x48484848, 0x49494949,0x4A4A4A4A,0x4B4B4B4B,
    0x4C4C4C4C, 0x4D4D4D4D,0x4E4E4E4E,0x4F4F4F4F,
    0x50505050, 0x51515151,0x52525252,0x53535353,
    0x54545454, 0x55555555,0x56565656,0x57575757,
    0x58585858, 0x59595959,0x5A5A5A5A,0x5B5B5B5B,
    0x5C5C5C5C, 0x5D5D5D5D,0x5E5E5E5E,0x5F5F5F5F,
    0x60606060, 0x61616161,0x62626262,0x63636363,
    0x64646464, 0x65656565,0x66666666,0x67676767,
    0x68686868, 0x69696969,0x6A6A6A6A,0x6B6B6B6B,
    0x6C6C6C6C, 0x6D6D6D6D,0x6E6E6E6E,0x6F6F6F6F,
    0x70707070, 0x71717171,0x72727272,0x73737373,
    0x74747474, 0x75757575,0x76767676,0x77777777,
    0x78787878, 0x79797979,0x7A7A7A7A,0x7B7B7B7B,
    0x7C7C7C7C, 0x7D7D7D7D,0x7E7E7E7E,0x7F7F7F7F,
    0x80808080, 0x81818181,0x82828282,0x83838383,
    0x84848484, 0x85858585,0x86868686,0x87878787,
    0x88888888, 0x89898989,0x8A8A8A8A,0x8B8B8B8B,
    0x8C8C8C8C, 0x8D8D8D8D,0x8E8E8E8E,0x8F8F8F8F,
    0x90909090, 0x91919191,0x92929292,0x93939393,
    0x94949494, 0x95959595,0x96969696,0x97979797,
    0x98989898, 0x99999999,0x9A9A9A9A,0x9B9B9B9B,
    0x9C9C9C9C, 0x9D9D9D9D,0x9E9E9E9E,0x9F9F9F9F,
    0xA0A0A0A0, 0xA1A1A1A1,0xA2A2A2A2,0xA3A3A3A3,
    0xA4A4A4A4, 0xA5A5A5A5,0xA6A6A6A6,0xA7A7A7A7,
    0xA8A8A8A8, 0xA9A9A9A9,0xAAAAAAAA,0xABABABAB,
    0xACACACAC, 0xADADADAD,0xAEAEAEAE,0xAFAFAFAF,
    0xB0B0B0B0, 0xB1B1B1B1,0xB2B2B2B2,0xB3B3B3B3,
    0xB4B4B4B4, 0xB5B5B5B5,0xB6B6B6B6,0xB7B7B7B7,
    0xB8B8B8B8, 0xB9B9B9B9,0xBABABABA,0xBBBBBBBB,
    0xBCBCBCBC, 0xBDBDBDBD,0xBEBEBEBE,0xBFBFBFBF,
    0xC0C0C0C0, 0xC1C1C1C1,0xC2C2C2C2,0xC3C3C3C3,
    0xC4C4C4C4, 0xC5C5C5C5,0xC6C6C6C6,0xC7C7C7C7,
    0xC8C8C8C8, 0xC9C9C9C9,0xCACACACA,0xCBCBCBCB,
    0xCCCCCCCC, 0xCDCDCDCD,0xCECECECE,0xCFCFCFCF,
    0xD0D0D0D0, 0xD1D1D1D1,0xD2D2D2D2,0xD3D3D3D3,
    0xD4D4D4D4, 0xD5D5D5D5,0xD6D6D6D6,0xD7D7D7D7,
    0xD8D8D8D8, 0xD9D9D9D9,0xDADADADA,0xDBDBDBDB,
    0xDCDCDCDC, 0xDDDDDDDD,0xDEDEDEDE,0xDFDFDFDF,
    0xE0E0E0E0, 0xE1E1E1E1,0xE2E2E2E2,0xE3E3E3E3,
    0xE4E4E4E4, 0xE5E5E5E5,0xE6E6E6E6,0xE7E7E7E7,
    0xE8E8E8E8, 0xE9E9E9E9,0xEAEAEAEA,0xEBEBEBEB,
    0xECECECEC, 0xEDEDEDED,0xEEEEEEEE,0xEFEFEFEF,
    0xF0F0F0F0, 0xF1F1F1F1,0xF2F2F2F2,0xF3F3F3F3,
    0xF4F4F4F4, 0xF5F5F5F5,0xF6F6F6F6,0xF7F7F7F7,
    0xF8F8F8F8, 0xF9F9F9F9,0xFAFAFAFA,0xFBFBFBFB,
    0xFCFCFCFC, 0xFDFDFDFD,0xFEFEFEFE,0xFFFFFFFF
};

/*
**	Table of all (I hope) read-modify-write opcodes of 65xx.
*/
static CONST unsigned char RMW[256] = {
	0,	0,	0,	ILL03,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	ASL_A,	ILL0F,
	0,	0,	0,	ILL13,	0,	0,	0,	0,
	0,	0,	0,	ILL1B,	0,	0,	ASL_AX,	ILL1F,
	0,	0,	0,	ILL23,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	ROL_A,	ILL2F,
	0,	0,	0,	ILL33,	0,	0,	0,	0,
	0,	0,	0,	ILL3B,	0,	0,	ROL_AX,	ILL3F,
	0,	0,	0,	ILL43,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	LSR_A,	ILL4F,
	0,	0,	0,	ILL53,	0,	0,	0,	0,
	0,	0,	0,	ILL5B,	0,	0,	LSR_AX,	ILL5F,
	0,	0,	0,	ILL63,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	ROR_A,	ILL6F,
	0,	0,	0,	ILL73,	0,	0,	0,	0,
	0,	0,	0,	ILL7B,	0,	0,	ROR_AX,	ILL7F,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	0,	0,
	0,	0,	0,	ILLC3,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	DEC_A,	ILLCF,
	0,	0,	0,	ILLD3,	0,	0,	0,	0,
	0,	0,	0,	ILLDB,	0,	0,	DEC_AX,	ILLDF,
	0,	0,	0,	ILLE3,	0,	0,	0,	0,
	0,	0,	0,	0,	0,	0,	INC_A,	ILLEF,
	0,	0,	0,	ILLF3,	0,	0,	0,	0,
	0,	0,	0,	ILLFB,	0,	0,	INC_AX,	ILLFF
};

/*---------------------------------------------------------------------------*/

#ifdef NEW_ACTION

/*
**	Update processor action.
*/
void UpdateAction(void)
{
    unsigned a;
    void (*f)(void);

    a=VicIrqAction;
    f=EmulRasterLineIrq;

    if( a>Cia1TimerA_Action ) {
	a=Cia1TimerA_Action;
	f=EmulCia1TimerA;
    }
    if( a>Cia1TimerB_Action ) {
	a=Cia1TimerB_Action;
	f=EmulCia1TimerB;
    }
    if( a>Cia2TimerA_Action ) {
	a=Cia2TimerA_Action;
	f=EmulCia2TimerA;
    }
    if( a>Cia2TimerB_Action ) {
	a=Cia2TimerB_Action;
	f=EmulCia2TimerB;
    }
#ifdef MONITOR
    if( a>MonAction ) {
	a=MonAction;
	f=MonFunction;
    }
#endif
    if( a>VicAction ) {			/* vic before hardware */
	HwAction=a;
	Action=VicAction;
	ActionFunction=EmulVic;
    } else {				/* vic after hardware */
	Action=a;
	ActionFunction=f;
	HwAction=0;
    }
}

#else

/*
**	Update processor action.
*/
void UpdateAction(void)
{
    unsigned a;
    void (*f)(void);

    a=VicAction;
    f=EmulVic;

    if( a>VicIrqAction ) {
	a=VicIrqAction;
	f=EmulRasterLineIrq;
    }
    if( a>Cia1TimerA_Action ) {
	a=Cia1TimerA_Action;
	f=EmulCia1TimerA;
    }
    if( a>Cia1TimerB_Action ) {
	a=Cia1TimerB_Action;
	f=EmulCia1TimerB;
    }
    if( a>Cia2TimerA_Action ) {
	a=Cia2TimerA_Action;
	f=EmulCia2TimerA;
    }
    if( a>Cia2TimerB_Action ) {
	a=Cia2TimerB_Action;
	f=EmulCia2TimerB;
    }
#ifdef MONITOR
    if( a>MonAction ) {
	a=MonAction;
	f=MonFunction;
    }
#endif
    Action=a;
    ActionFunction=f;
}

#endif

/*---------------------------------------------------------------------------*/

#ifdef __GO32__	/* { */

#define GetIoPriv() 1

#else	/* }{ __GO32__ */

/*
**	FIXME: move this to md.c
*/
static int GetIoPriv(void)
{
    int success=0;

    if( 
#if defined(PCSOLARIS)
	1
#else
	!iopl(3)
#endif
#if defined(SID_SPEAKER) || defined(SID_AUTO)
	    && !ioperm(0x40,4,1)
	    && !ioperm(0x61,1,1)	/* speaker ports */
#endif
#if defined(SID_OPL3) || defined(SID_AUTO)
	    && !ioperm(FM_PORT,4,1)	/* opl3 ports */
#endif
#if defined(SVGALIB)
	    && !ioperm(0x3B0,0x30,1)
	    && !ioperm(0x200,2,1)	/* vga ports */
#endif
	    ) {
	success=1;
    }
#if !defined(SVGALIB)
    setuid(getuid());			/* svgalib need this for some driver */
#else
    putenv("IOPERM=1");			/* help et4000w32x driver to keep iopl() */
#endif
    return success;
}

#endif	/* } !__GO32__ */


#ifndef M_UNIX	/* { */
/*
**	FIXME: move this to md.c
*/

static FILE *deffp = NULL;

int defopen(CONST char *n)
{
    if (deffp) fclose(deffp);
    if (n) {
	deffp = fopen(n,"r");
	return deffp == NULL;
    }
    deffp = NULL;
    return 1;
}

char *defread(CONST char *n)
{
    int l;
    char* p;
    static char defbuf[256];

    if (n && (l = strlen(n))) {
	if (deffp) {
	    rewind(deffp);
	    while (fgets(defbuf,256,deffp)) {
		if (strncmp(defbuf,n,l) == 0) {
		    p = strchr(defbuf,0);
		    if (p) {	/* never trust libaries */
			p--;
			if (*p == '\n')
			    *p = 0;
		    }
		    return defbuf+l;
		}
	    }
	}
    }
    return NULL;
}

#endif /* } M_UNIX */

/*---------------------------------------------------------------------------*/

#ifdef MONITOR	/* { */

/*
**	Display head for register dumps.
*/
void DisplayHead(void)
{
    printf("....: 0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F\n");
}

/*
**	Display contents of CIA1 registers.
*/
void DisplayCia1(void)
{
    int i;

    printf("CIA1:");
    for( i=0; i<16; ++i ) {
	printf("%02X ",CIA1[i]);
    }
    printf("\n");
}

/*
**	Display contents of CIA2 registers.
*/
void DisplayCia2(void)
{
    int i;

    printf("CIA2:");
    for( i=0; i<16; ++i ) {
	printf("%02X ",CIA2[i]);
    }
    printf("\n");
}

/*
**	Display contents of VIC registers.
*/
void DisplayVic(void)
{
    int i;

    printf("VIC0:");
    for( i=0; i<16; ++i ) {
	printf("%02X ",VIC[i]);
    }
    printf("\n");
    printf("VIC1:");
    for( i=0; i<16; ++i ) {
	printf("%02X ",VIC[16+i]);
    }
    printf("\n");
    printf("VIC2:");
    for( i=0; i<15; ++i ) {
	printf("%02X ",VIC[32+i]);
    }
    printf("\n");
}

/*
**	Display contents of SID registers.
*/
void DisplaySid(void)
{
    int i;

    printf("SID1:");
    for( i=0; i<16; ++i ) {
	printf("%02X ",SID[i]);
    }
    printf("\n");
    printf("SID2:");
    for( i=0; i<16; ++i ) {
	printf("%02X ",SID[16+i]);
    }
    printf("\n");
}

#endif	/* } MONITOR */

/*---------------------------------------------------------------------------*/

#ifdef NEW_MEMORY

#define MEM_BANK_SIZE	4096		/* size of memory segments */
#define MEM_BANKS	32		/* number of segments */

regparm1	*RdConfig[8][MEM_BANKS];	/* Read  memory */
regparm2	*WrConfig[8][MEM_BANKS];	/* Write memory */

regparm1	(*(*RdIt)[MEM_BANKS]);		/* Read  memory */
regparm2	(*(*WrIt)[MEM_BANKS]);		/* Write memory */

#endif

#ifdef GLOBAL_ARGS

int GlobalAdr;
int GlobalVal;

int	(*R_Config[C_MAX])(void);		/* Read  memory */
void	(*W_Config[C_MAX])(void);		/* Write memory */

#define R_FUN(n)	static int n(void)
#define W_FUN(n)	static void n(void)
#define VAL		GlobalVal
#define ADR		GlobalAdr

#else

//int ((*R_Config[C_MAX] )(int)REGPARM1)	;	/* Read  memory */
//void ((*W_Config[C_MAX] )(int,int)REGPARM2)	;	/* Write memory */
regparm1	*R_Config[C_MAX];	/* Read  memory */
regparm2	*W_Config[C_MAX];	/* Write memory */

/*
**	Macro to define memory read functions.
*/
#define R_FUN(n) \
	static int REGPARM1 n (int x) \

/*
**	Macro to define memory write functions.
*/
#define W_FUN(n) \
	static void REGPARM2 n (int x,int v) \

#define VAL     v
#define ADR     x

#endif

/*** FIXME: { START OF CLEANED UP CODE ****/

/******************************************************************************
 *	Memory emulation
 *****************************************************************************/

/* 
**	Read/Write-functions
*/

/*=============================================================================
 *	Processor port ($0/$1)
 *===========================================================================*/

#if 0
Processor port:
	0	Direction port	0=Input 1=Output
	1	Data port
		Bit0	!LOWRAM
		Bit1	!HIRAM
		Bit2	!CHAREN
		Bit3	Write data cassette
		Bit4	Cass sense cassette play key
		Bit5	Motor on cassette
		Bit6	Not available
		Bit7	Not available
#endif

W_FUN( W_PPDir )			/* $0 Processor port Direction */
{
    Ram[0]=VAL;
    Config=ConfigTable[(Ram[1]|(~VAL))&7];
#ifdef NEW_MEMORY
    RdIt=&RdConfig[(Ram[1]|(~VAL))&7];
    WrIt=&WrConfig[(Ram[1]|(~VAL))&7];
#endif
}

W_FUN( W_PPData )			/* $1 Processor port Data */
{
    Ram[1]=VAL|0x10;			/* Play key */
    Config=ConfigTable[(VAL|(~Ram[0]))&7];
#ifdef NEW_MEMORY
    RdIt=&RdConfig[(Ram[1]|(~VAL))&7];
    WrIt=&WrConfig[(Ram[1]|(~VAL))&7];
#endif
}

/*=============================================================================
 *	64K RAM
 *===========================================================================*/

R_FUN( R_Zero )				/* Read RAM for 64k round about */
{
    return Ram[ADR-0x10000];
}

W_FUN( W_Zero )				/* Write RAM for 64k round about */
{
    Ram[ADR-0x10000]=VAL;
}

R_FUN( R_Ram )				/* Read RAM */
{
    return Ram[ADR];
}

W_FUN( W_Ram )				/* Write RAM */
{
    Ram[ADR]=VAL;
}

/*=============================================================================
 *	1K ALE Ram
 *===========================================================================*/

#ifdef ALE_RAM	/* { */

R_FUN( R_AleRam1 )			/* Read RAM at D200-D3FF */
{
    return AleRam1[ADR-ADR_ALE_RAM1];
}

W_FUN( W_AleRam1 )			/* Write RAM at D200-D3FF */
{
    AleRam1[ADR-ADR_ALE_RAM1]=VAL;
}

R_FUN( R_AleRam2 )			/* Read RAM at D600-D7FF */
{
    return AleRam2[ADR-ADR_ALE_RAM2];
}

W_FUN( W_AleRam2 )			/* Write RAM at D600-D7FF */
{
    AleRam2[ADR-ADR_ALE_RAM2]=VAL;
}

#endif	/* } ALE_RAM */

/*=============================================================================
 *	8K/16K External ROM
 *===========================================================================*/

R_FUN( R_ExternalRom )			/* Read ROM at 8000-9FFF or 8000-BFFF */
{
    return ExternalRom[ADR-ADR_EXTERNAL];
}

/*=============================================================================
 *	8K Basic ROM
 *===========================================================================*/

R_FUN( R_BasicRom )			/* Read ROM at A000-BFFF */
{
    return BasicRom[ADR-ADR_BASIC];
}

/*=============================================================================
 *	8K Kernel ROM
 *===========================================================================*/

R_FUN( R_KernelRom )			/* Read ROM at E000-FFFF */
{
    return KernelRom[ADR-ADR_KERNEL];
}

/*=============================================================================
 *	4K Character ROM
 *===========================================================================*/

R_FUN( R_CharacterRom )			/* Read ROM at D000-DFFF */
{
    return CharacterRom[ADR-ADR_VIC];
}

/*** FIXME: } END OF CLEANED UP CODE ****/

/*=============================================================================
 *	VIC II 6566/6567	Memory emulation
 *===========================================================================*/

R_FUN( R_VIC00 )			/* MOB 0 X - position */
{
    return VIC[0x00];
}

W_FUN( W_VIC00 )			/* MOB 0 X - position */
{
    VIC[0x00]=VAL;
    VicSprite[0].X=(VicSprite[0].X&0x100)|VAL;
}

R_FUN( R_VIC01 )			/* MOB 0 Y - position */
{
    return VIC[0x01];
}

W_FUN( W_VIC01 )			/* MOB 0 Y - position */
{
    VIC[0x01]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[0].Y]&=~0x01;
    VicSprite[0].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x01 )
	VicSpriteDMA[VicSprite[0].Y]|=0x01;
#else
    VicSprite[0].Y=VAL+1;
#endif
}

R_FUN( R_VIC02 )			/* MOB 1 X - position */
{
    return VIC[0x02];
}

W_FUN( W_VIC02 )			/* MOB 1 X - position */
{
    VIC[0x02]=VAL;
    VicSprite[1].X=(VicSprite[1].X&0x100)|VAL;
}

R_FUN( R_VIC03 )			/* MOB 1 Y - position */
{
    return VIC[0x03];
}

W_FUN( W_VIC03 )			/* MOB 1 Y - position */
{
    VIC[0x03]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[1].Y]&=~0x02;
    VicSprite[1].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x02 )
	VicSpriteDMA[VicSprite[1].Y]|=0x02;
#else
    VicSprite[1].Y=VAL+1;
#endif
}

R_FUN( R_VIC04 )			/* MOB 2 X - position */
{
    return VIC[0x04];
}

W_FUN( W_VIC04 )			/* MOB 2 X - position */
{
    VIC[0x04]=VAL;
    VicSprite[2].X=(VicSprite[2].X&0x100)|VAL;
}

R_FUN( R_VIC05 )			/* MOB 2 Y - position */
{
    return VIC[0x05];
}

W_FUN( W_VIC05 )			/* MOB 2 Y - position */
{
    VIC[0x05]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[2].Y]&=~0x04;
    VicSprite[2].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x04 )
	VicSpriteDMA[VicSprite[2].Y]|=0x04;
#else
    VicSprite[2].Y=VAL+1;
#endif
}

R_FUN( R_VIC06 )			/* MOB 3 X - position */
{
    return VIC[0x06];
}

W_FUN( W_VIC06 )			/* MOB 3 X - position */
{
    VIC[0x06]=VAL;
    VicSprite[3].X=(VicSprite[3].X&0x100)|VAL;
}

R_FUN( R_VIC07 )			/* MOB 3 Y - position */
{
    return VIC[0x07];
}

W_FUN( W_VIC07 )			/* MOB 3 Y - position */
{
    VIC[0x07]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[3].Y]&=~0x08;
    VicSprite[3].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x08 )
	VicSpriteDMA[VicSprite[3].Y]|=0x08;
#else
    VicSprite[3].Y=VAL+1;
#endif
}

R_FUN( R_VIC08 )			/* MOB 4 X - position */
{
    return VIC[0x08];
}

W_FUN( W_VIC08 )			/* MOB 4 X - position */
{
    VIC[0x08]=VAL;
    VicSprite[4].X=(VicSprite[4].X&0x100)|VAL;
}

R_FUN( R_VIC09 )			/* MOB 4 Y - position */
{
    return VIC[0x09];
}

W_FUN( W_VIC09 )			/* MOB 4 Y - position */
{
    VIC[0x09]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[4].Y]&=~0x10;
    VicSprite[4].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x10 )
	VicSpriteDMA[VicSprite[4].Y]|=0x10;
#else
    VicSprite[4].Y=VAL+1;
#endif
}

R_FUN( R_VIC0A )			/* MOB 5 X - position */
{
    return VIC[0x0A];
}

W_FUN( W_VIC0A )			/* MOB 5 X - position */
{
    VIC[0x0A]=VAL;
    VicSprite[5].X=(VicSprite[5].X&0x100)|VAL;
}

R_FUN( R_VIC0B )			/* MOB 5 Y - position */
{
    return VIC[0x0B];
}

W_FUN( W_VIC0B )			/* MOB 5 Y - position */
{
    VIC[0x0B]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[5].Y]&=~0x20;
    VicSprite[5].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x20 )
	VicSpriteDMA[VicSprite[5].Y]|=0x20;
#else
    VicSprite[5].Y=VAL+1;
#endif
}

R_FUN( R_VIC0C )			/* MOB 6 X - position */
{
    return VIC[0x0C];
}

W_FUN( W_VIC0C )			/* MOB 6 X - position */
{
    VIC[0x0C]=VAL;
    VicSprite[6].X=(VicSprite[6].X&0x100)|VAL;
}

R_FUN( R_VIC0D )			/* MOB 6 Y - position */
{
    return VIC[0x0D];
}

W_FUN( W_VIC0D )			/* MOB 6 Y - position */
{
    VIC[0x0D]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[6].Y]&=~0x40;
    VicSprite[6].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x40 )
	VicSpriteDMA[VicSprite[6].Y]|=0x40;
#else
    VicSprite[6].Y=VAL+1;
#endif
}

R_FUN( R_VIC0E )			/* MOB 7 X - position */
{
    return VIC[0x0E];
}

W_FUN( W_VIC0E )			/* MOB 7 X - position */
{
    VIC[0x0E]=VAL;
    VicSprite[7].X=(VicSprite[7].X&0x100)|VAL;
}

R_FUN( R_VIC0F )			/* MOB 7 Y - position */
{
    return VIC[0x0F];
}

W_FUN( W_VIC0F )			/* MOB 7 Y - position */
{
    VIC[0x0F]=VAL;
#ifdef NEW_SPRITES
    VicSpriteDMA[VicSprite[7].Y]&=~0x80;
    VicSprite[7].Y=(VAL+1)&0xFF;
    if( VIC[0x15]&0x80 )
	VicSpriteDMA[VicSprite[7].Y]|=0x80;
#else
    VicSprite[7].Y=VAL+1;
#endif
}

R_FUN( R_VIC10 )			/* MOB MSB of X - position */
{
    return VIC[0x10];
}

W_FUN( W_VIC10 )			/* MOB MSB of X - position */
{
    VIC[0x10]=VAL;
    VAL<<=1;
    VicSprite[7].X=(VicSprite[7].X&0xFF)|(VAL&0x100);
    VAL<<=1;
    VicSprite[6].X=(VicSprite[6].X&0xFF)|(VAL&0x100);
    VAL<<=1;
    VicSprite[5].X=(VicSprite[5].X&0xFF)|(VAL&0x100);
    VAL<<=1;
    VicSprite[4].X=(VicSprite[4].X&0xFF)|(VAL&0x100);
    VAL<<=1;
    VicSprite[3].X=(VicSprite[3].X&0xFF)|(VAL&0x100);
    VAL<<=1;
    VicSprite[2].X=(VicSprite[2].X&0xFF)|(VAL&0x100);
    VAL<<=1;
    VicSprite[1].X=(VicSprite[1].X&0xFF)|(VAL&0x100);
    VAL<<=1;
    VicSprite[0].X=(VicSprite[0].X&0xFF)|(VAL&0x100);
}

R_FUN( R_VIC11 )			/* Control register 1 */
{
    /*
    **	ca 11us before line fetch.
    */
    if( Cycle>=VicAction-(VIC_FETCH+VicFetchAdd) ) {
	return ((VicRasterLine>>1)&0x80)|(VIC[0x11]&0x7F);
    } else if( VicRasterLine ) {
	return (((VicRasterLine-1)>>1)&0x80)|(VIC[0x11]&0x7F);
    } else {
	return (((LINES_PER_FRAME-1)>>1)&0x80)|(VIC[0x11]&0x7F);
    }
}

/*
 *	VIC Modes:
 *
 *	D011:	-XX- ----
 *		 ||
 *		 |`- Bitmap mode
 *		 `-  Extended mode
 *
 *	D016:	---X ----
 *		   |
 *		   `- Multicolor mode
 */
static void (* CONST VicModes[8])(void) = {
/* --- */	TextDL,
/* --M */	MultiColorTextDL,
/* -B- */	BitmapDL,
/* -BM */	MultiColorBitmapDL,
/* E-- */	ExtendedColorTextDL,
/* E-M */	IllegalDL,
/* EB- */	IllegalDL,
/* EBM */	IllegalDL,
};

/*
**	VIC 11 Tricks:
**		Cycle 0:	Irq and increment rasterline.
**		Cycle 11:	BA -> LOW for bad lines.
**		Cycle 14:	Line index -> 0
**		Cycle 15:	Fetch of first byte of video matrix
**		Cycle 54:	Fetch of last byte of video matrix
**		Cycle 55:	BA -> Low for first sprite (55,56,57)
**		Cycle 58:	Increment VicOffset if VicScanLine=7
**				Start Sprite DMA
**		Cycle 62:	Last cycle of line
**		Cycle 63:	Start of new line
*/
W_FUN( W_VIC11 )			/* Control register 1 */
{
    int i;

    /*
    **	New try to get VicYOffset manipulation correct.
    **	Badline.
    */

    /*
    **	Generating a bad line: (bug in vic, used for x-scrolling!)
    **		Don't know if for every line ?
    **		needed for creature and lickpipe demo of triad.
    */
    if( VicRasterLine==0x33 && 2==(VAL&7) ) {
	i=VicAction-Cycle;
	if( i>27 ) {			/* VicFetch+15 cycles */
#ifdef DEBUG
	    if( 1 ) { char buf[80];
	    sprintf(buf,"X+SCR:%03X:%2d %3d %X>%X",
		VicRasterLine,VicScanLine,i,VIC[0x11],VAL);
	    VicMessage(buf,FRAMES_PER_SECOND);
	    }
#endif
	    VicScanLine=7;
	    VicOffset+=i-28+40;
	}
    } else if( VicRasterLine==0x32 && 2==(VAL&7) ) {
	i=VicAction-Cycle;
	if( i>0 && i<5 ) {
#ifdef DEBUG
	    if( 1 ) { char buf[80];
	    sprintf(buf,"X-SCR:%03X:%2d %3d %X>%X",
		VicRasterLine,VicScanLine,i,VIC[0x11],VAL);
	    VicMessage(buf,FRAMES_PER_SECOND);
	    }
#endif
	    /* FIXME: here we must display a line !!!!
	    **	Creature change from 24 -> 25 lines later!
	    **	Bug fixed in vic.c moved 24/25 checks later.
	    */
	    VicOffset+=i-5+40;
	} else if( i==5 ) {
#ifdef DEBUG
	    if( 1 ) { char buf[80];
	    sprintf(buf,"X=SCR:%03X:%2d %3d %X>%X",
		VicRasterLine,VicScanLine,i,VIC[0x11],VAL);
	    VicMessage(buf,FRAMES_PER_SECOND);
	    }
#endif
	    VicOffset+=i-5+40;
	}
    }

    if( VicRasterLine>=0x30 ) {
    if( VicYOffset!=(VAL&7) && ((VicRasterLine)&7)==VicYOffset ) {
	/*
	** 	New line and before fetch, badline removed!!
	*/
	i=VicAction-Cycle;
#ifdef DEBUG
	if( 0 ) { char buf[80];
	sprintf(buf,"Y-SCR:%03X:%2d %3d %X>%X",
	    VicRasterLine,VicScanLine,i,VIC[0x11],VAL);
	VicMessage(buf,FRAMES_PER_SECOND);
	}
#endif
	if( i>5 && i<VIC_FETCH ) {	/* ca 12 cycles */
	    /*
	    printf("Cycle %d VicAction %d diff %d scanline %d add %d\n",
		    Cycle,VicAction,VicAction-Cycle,VicScanLine,
		    i-27);
	    */
	    VicScanLine=7;
	    VicOffset+=40;
	}
    }
    }

    //VIC[0x11]=VAL&0x7F;
    VIC[0x11]=VAL;
    VicYOffset=VAL&7;

    /*
    **	Calculate new irq point.
    */
    /* FIXME: */
#if 0
    if( VicIrqRasterLine!=((VicIrqRasterLine&0xFF)|((VAL&0x80)<<1)) ) {
	VicIrqRasterLine=(VicIrqRasterLine&0xFF)|((VAL&0x80)<<1);
#endif
    i=VIC[0x12]|((VAL&0x80)<<1);
    if( VicIrqRasterLine!=i ) {
	VicIrqRasterLine=i;
	VicIrqAction=VicAction-VIC_FETCH+
		(VicIrqRasterLine-VicRasterLine)*CYCLES_PER_LINE;
	if( VicIrqAction<VicAction-VIC_FETCH-CYCLES_PER_LINE )
	    VicIrqAction+=LINES_PER_FRAME*CYCLES_PER_LINE;
	if( VicIrqAction<Cycle ) {
	    /* FIXME: this is needed but why ????? PITSTOP ][ */
	    // printf("WHY: %d %d = %d\n",Cycle,VicIrqAction,Cycle-VicIrqAction);
	    // EmulRasterLineIrq();
	    // VicIrqAction+=LINES_PER_FRAME*CYCLES_PER_LINE;
	}
	ChangeAction(VicIrqAction,EmulRasterLineIrq);
    }

    /*
    **	VIC[0x11] 0x40 Extendedcolor 0x20 Bitmap
    **	VIC[0x16] 0x10 Multicolor
    */
    VicDisplayLine=VicModes[((VAL&0x60)|(VIC[0x16]&0x10))>>4];
}

R_FUN( R_VIC12 )			/* Rasterline register */
{
    /*
    **	ca 11us before line fetch.
    */
    if( Cycle>=VicAction-(VIC_FETCH+VicFetchAdd) ) {
	return VicRasterLine&0xFF;
    } else if( VicRasterLine ) {
	return (VicRasterLine-1)&0xFF;
    } else {
	return (LINES_PER_FRAME-1)&0xFF;
    }
}

W_FUN( W_VIC12 )			/* Irq Rasterline register */
{
    int i;

    VIC[0x12]=VAL;
    // FIXME: ?? i=VAL|(VicIrqRasterLine&0x100);
    i=VAL|((VIC[0x11]&0x80)<<1);
    if( i!=VicIrqRasterLine ) {
	VicIrqRasterLine=i;

	VicIrqAction=VicAction-VIC_FETCH+
		(VicIrqRasterLine-VicRasterLine)*CYCLES_PER_LINE;
	if( VicIrqAction<VicAction-VIC_FETCH-CYCLES_PER_LINE )
	    VicIrqAction+=LINES_PER_FRAME*CYCLES_PER_LINE;
#if 0
	if( VicIrqAction<Cycle ) {
	    /* FIXME: this is needed but why ????? PITSTOP ][ */
	    // printf("WHY: %d %d = %d\n",Cycle,VicIrqAction,Cycle-VicIrqAction);
	    // EmulRasterLineIrq();
	    // VicIrqAction+=LINES_PER_FRAME*CYCLES_PER_LINE;
	}
#endif
	ChangeAction(VicIrqAction,EmulRasterLineIrq);
    }
}

R_FUN( R_VIC13 )			/* Light pen X */
{
    return 0xE0;
}

W_FUN( W_VIC13 )			/* Light pen X */
{
}

R_FUN( R_VIC14 )			/* Light pen Y */
{
    return 0x01;
}

W_FUN( W_VIC14 )			/* Light pen Y */
{
}

R_FUN( R_VIC15 )			/* Enable MOB */
{
    return VIC[0x15];
}

W_FUN( W_VIC15 )			/* Enable MOB */
{
    VIC[0x15]=VAL;
    if( VIC[0x15]&0x80 )
	VicSpriteDMA[VicSprite[7].Y]|=0x80;
    else
	VicSpriteDMA[VicSprite[7].Y]&=~0x80;
    if( VIC[0x15]&0x40 )
	VicSpriteDMA[VicSprite[6].Y]|=0x40;
    else
	VicSpriteDMA[VicSprite[6].Y]&=~0x40;
    if( VIC[0x15]&0x20 )
	VicSpriteDMA[VicSprite[5].Y]|=0x20;
    else
	VicSpriteDMA[VicSprite[5].Y]&=~0x20;
    if( VIC[0x15]&0x10 )
	VicSpriteDMA[VicSprite[4].Y]|=0x10;
    else
	VicSpriteDMA[VicSprite[4].Y]&=~0x10;
    if( VIC[0x15]&0x08 )
	VicSpriteDMA[VicSprite[3].Y]|=0x08;
    else
	VicSpriteDMA[VicSprite[3].Y]&=~0x08;
    if( VIC[0x15]&0x04 )
	VicSpriteDMA[VicSprite[2].Y]|=0x04;
    else
	VicSpriteDMA[VicSprite[2].Y]&=~0x04;
    if( VIC[0x15]&0x02 )
	VicSpriteDMA[VicSprite[1].Y]|=0x02;
    else
	VicSpriteDMA[VicSprite[1].Y]&=~0x02;
    if( VIC[0x15]&0x01 )
	VicSpriteDMA[VicSprite[0].Y]|=0x01;
    else
	VicSpriteDMA[VicSprite[0].Y]&=~0x01;
}

R_FUN( R_VIC16 )			/* Control register 2 */
{
    return VIC[0x16]|0xC0;
}

W_FUN( W_VIC16 )			/* Control register 2 */
{
    /*
    ** FIXME: 0x20 Reset Bit 
    */
    VIC[0x16]=VAL;
    /*
     *	VIC[0x11] 0x40 Extendedcolor 0x20 Bitmap
     *	VIC[0x16] 0x10 Multicolor
     */
    VicDisplayLine=VicModes[((VIC[0x11]&0x60)|(VAL&0x10))>>4];
    /* DEBUG
    if( VicXOffset!=(VAL&7) )
	printf("X-OFFSET %d $%04X\n",VAL&7,GlobalRegPC);
    */
    VicXOffset=VAL&7;
}

R_FUN( R_VIC17 )			/* MOB Y-expand */
{
    return VIC[0x17];
}

W_FUN( W_VIC17 )			/* MOB Y-expand */
{
    VIC[0x17]=VAL;
#ifdef JOHNS
    // printf("VicAction %d Cycle %d\n",VicAction,63-VicAction);
#endif
}

R_FUN( R_VIC18 )			/* Memory Pointers */
{
    return VIC[0x18]|0x01;
}

W_FUN( W_VIC18 )			/* Memory Pointers */
{
    int a;

    VIC[0x18]=VAL;
    /* Video memory: */
    /* 15	14	 13	  12	   11	    10		*/
    /* CIA2-0-1 CIA2-0-0 VIC-18-7 VIC-18-6 VIC-18-5 VIC-18-4	*/
    VicVRam=VicBank+((VAL&0xF0)<<6);
    /* Bitmap memory */
    /* 15	14	 13	  12	   11	    10		*/
    /* CIA2-0-1 CIA2-0-0 VIC-18-3 --	   --	    --		*/
    VicBRam=VicBank+((VAL&0x08)<<10);
    VicCRam=VicBank+((VAL&0x0E)<<10);
    a=VicCRam-Ram;
    /*
    **	Map character RAM addresses 0x1000-0x1FFF 
    **	and 0x9000-0x9FFF to character ROM
    */
    if( (a&0x7000)==0x1000 ) {
	VicCRam=CharacterRom+(a&0x0FFF);
    }
}

R_FUN( R_VIC19 )			/* Read IRR Interrupt register */
{
    return VIC[0x19]|0x70;
}

W_FUN( W_VIC19 )			/* Write IRR Interrupt register */
{
    VIC[0x19]&=~(0x80|VAL);		/* Clear interrupts */

    /*
    **	This is to implement the RMW bug of 65xx.
    **		ASL $D019, INC $D019, ... writes first the old value back.
    */
    if( VIC[0x19]&VIC[0x1A] ) {		 /* Still irq on */
	if( RMW[RBMem(GlobalRegPC)] ) {
#if defined(X11) && defined(DEBUG) && 0
	    printf("RMW $%04X\t",GlobalRegPC);
	    Dis(GlobalRegPC);
	    printf("\n");
#endif
	    VIC[0x19]=0x00;
	} else {			/* keep the IRQ */
	    /* FIXME: Perfect? extra internal irq register? */
#if defined(X11) && defined(DEBUG)
	    printf("VIC IRQ be hold: $%04X\t",GlobalRegPC);
	    Dis(GlobalRegPC);
	    printf("\n");
#endif
	    VIC[0x19]|=0x80;
	}
    }
    IrqLine=VIC[0x19]&CIA1[CIA_ICR]&0x80;/* irq line update */
}

R_FUN( R_VIC1A )			/* Read IMR Enable interrupt */
{
    return VIC[0x1A]|0xF0;
}

W_FUN( W_VIC1A )			/* Write IMR Enable interrupt */
{
    VIC[0x1A]=(VAL&=7);

    /* Remove active interrupt source */
    if( (VIC[0x19]&0x80) && !(VIC[0x19]&VAL) ) {
#if defined(X11) && defined(DEBUG)
	printf("Removing VIC IRQ: $%04X\n",GlobalRegPC);
#endif
	VIC[0x19]&=0x7F;
	IrqLine=CIA1[CIA_ICR]&0x80;	/* irq line update */
    }
    /* adding an interrupt here isn't possible */
}

R_FUN( R_VIC1B )			/* MOB-DATA Priority */
{
    return VIC[0x1B];
}

W_FUN( W_VIC1B )			/* MOB-DATA Priority */
{
    VIC[0x1B]=VAL;
}

R_FUN( R_VIC1C )			/* MOB Multicolor selector */
{
    return VIC[0x1C];
}

W_FUN( W_VIC1C )			/* MOB Multicolor selector */
{
    VIC[0x1C]=VAL;
}

R_FUN( R_VIC1D )			/* MOB X-expand */
{
    return VIC[0x1D];
}

W_FUN( W_VIC1D )			/* MOB X-expand */
{
    VIC[0x1D]=VAL;
}

R_FUN( R_VIC1E )			/* MOB-MOB collision */
{
    int a;

    a=VIC[0x1E];
    VIC[0x1E]=0;
    return a;
}

W_FUN( W_VIC1E )			/* MOB-MOB collision */
{
}

R_FUN( R_VIC1F )			/* MOB-Background collision */
{
    int a;

    a=VIC[0x1F];
    VIC[0x1F]=0;
    return a;
}

W_FUN( W_VIC1F )			/* MOB-DATA collision */
{
}

R_FUN( R_VIC20 )			/* Exterior color (Border) */
{
    return VIC[0x20]|0xF0;
}

W_FUN( W_VIC20 )			/* Exterior color (Border) */
{
    VIC[0x20]=VAL&=0xF;
    VicExterior=QuadNibble[VAL]|0x10101010;
}

R_FUN( R_VIC21 )			/* Background #0 color */
{
    return VIC[0x21]|0xF0;
}

W_FUN( W_VIC21 )			/* Background #0 color */
{
    VIC[0x21]=VAL&=0xF;
    VicBackground0=QuadNibble[VAL]|0x10101010;
}

R_FUN( R_VIC22 )			/* Background #1 color */
{
    return VIC[0x22]|0xF0;
}

W_FUN( W_VIC22 )			/* Background #1 color */
{
    VIC[0x22]=VAL&0xF;
}

R_FUN( R_VIC23 )			/* Background #2 color */
{
    return VIC[0x23]|0xF0;
}

W_FUN( W_VIC23 )			/* Background #2 color */
{
    VIC[0x23]=VAL&0xF;
}

R_FUN( R_VIC24 )			/* Background #3 color */
{
    return VIC[0x24]|0xF0;
}

W_FUN( W_VIC24 )			/* Background #3 color */
{
    VIC[0x24]=VAL&0xF;
}

R_FUN( R_VIC25 )			/* MOB Multicolor #0 */
{
    return VIC[0x25]|0xF0;
}

W_FUN( W_VIC25 )			/* MOB Multicolor #0 */
{
    VIC[0x25]=VAL&0xF;
}

R_FUN( R_VIC26 )			/* MOB Multicolor #1 */
{
    return VIC[0x26]|0xF0;
}

W_FUN( W_VIC26 )			/* MOB Multicolor #1 */
{
    VIC[0x26]=VAL&0xF;
}

R_FUN( R_VIC27 )			/* MOB 0 color */
{
    return VIC[0x27]|0xF0;
}

W_FUN( W_VIC27 )			/* MOB 0 color */
{
    VIC[0x27]=VAL&=0xF;
    VicSprite[0].Color=QuadNibble[VAL]|0x00000000;
}

R_FUN( R_VIC28 )			/* MOB 1 color */
{
    return VIC[0x28]|0xF0;
}

W_FUN( W_VIC28 )			/* MOB 1 color */
{
    VIC[0x28]=VAL&=0xF;
    VicSprite[1].Color=QuadNibble[VAL]|0x20202020;
}

R_FUN( R_VIC29 )			/* MOB 2 color */
{
    return VIC[0x29]|0xF0;
}

W_FUN( W_VIC29 )			/* MOB 2 color */
{
    VIC[0x29]=VAL&=0xF;
    VicSprite[2].Color=QuadNibble[VAL]|0x40404040;
}

R_FUN( R_VIC2A )			/* MOB 3 color */
{
    return VIC[0x2A]|0xF0;
}

W_FUN( W_VIC2A )			/* MOB 3 color */
{
    VIC[0x2A]=VAL&=0xF;
    VicSprite[3].Color=QuadNibble[VAL]|0x60606060;
}

R_FUN( R_VIC2B )			/* MOB 4 color */
{
    return VIC[0x2B]|0xF0;
}

W_FUN( W_VIC2B )			/* MOB 4 color */
{
    VIC[0x2B]=VAL&=0xF;
    VicSprite[4].Color=QuadNibble[VAL]|0x80808080;
}

R_FUN( R_VIC2C )			/* MOB 5 color */
{
    return VIC[0x2C]|0xF0;
}

W_FUN( W_VIC2C )			/* MOB 5 color */
{
    VIC[0x2C]=VAL&=0xF;
    VicSprite[5].Color=QuadNibble[VAL]|0xA0A0A0A0;
}

R_FUN( R_VIC2D )			/* MOB 6 color */
{
    return VIC[0x2D]|0xF0;
}

W_FUN( W_VIC2D )			/* MOB 6 color */
{
    VIC[0x2D]=VAL&=0xF;
    VicSprite[6].Color=QuadNibble[VAL]|0xC0C0C0C0;
}

R_FUN( R_VIC2E )			/* MOB 7 color */
{
    return VIC[0x2E]|0xF0;
}

W_FUN( W_VIC2E )			/* MOB 7 color */
{
    VIC[0x2E]=VAL&=0xF;
    VicSprite[7].Color=QuadNibble[VAL]|0xE0E0E0E0;
}

/*===========================================================================*
 *	SID 6581	Memory emulation
 *===========================================================================*/

R_FUN( R_SID )				/* Most SID registers are write only */
{
    return 0x00;
}

/*
**	Registers of voice 1.
**		All writes are hooked for the sound driver.
*/
W_FUN( W_SID00 ) { SidFrqLo(0,VAL); SID[0x00]=VAL; }
W_FUN( W_SID01 ) { SidFrqHi(0,VAL); SID[0x01]=VAL; }
W_FUN( W_SID02 ) { SidPulLo(0,VAL); SID[0x02]=VAL; }
W_FUN( W_SID03 ) { SidPulHi(0,VAL); SID[0x03]=VAL; }
W_FUN( W_SID04 )			/* Voice 0 Control register */
{
    int old;

    old=SID[0x04];
    SidCtrl(0,VAL);			/* Here for DSP */
    /* FIXME: This should be moved into SidCtrl */
    if( (VAL&1)!=(old&1) ) {		/* Gate switched */
	SidVoice[0].Cycle=Cycle;
	if( VAL&1 ) {
	    SidVoice[0].State=SID_ATTACK;
	} else {
	    SidVoice[0].State=SID_RELEASE;
	}
    }
    if( !(VAL&0xF0) || (VAL&0x08) ) {	/* No waveforms selected or test */
	SidVoice[0].On=0;
    } else {
	SidVoice[0].On=1;
    }
    SID[0x04]=VAL;
}
W_FUN( W_SID05 ) { SidAtkDcy(0,VAL); SID[0x05]=VAL; }
W_FUN( W_SID06 ) { SidStnRls(0,VAL); SID[0x06]=VAL; }

/*
**	Registers of voice 2.
*/
W_FUN( W_SID07 ) { SidFrqLo(1,VAL); SID[0x07]=VAL; }
W_FUN( W_SID08 ) { SidFrqHi(1,VAL); SID[0x08]=VAL; }
W_FUN( W_SID09 ) { SidPulLo(1,VAL); SID[0x09]=VAL; }
W_FUN( W_SID0A ) { SidPulHi(1,VAL); SID[0x0A]=VAL; }
W_FUN( W_SID0B )			/* Voice 1 Control register */
{
    int old;

    old=SID[0x0B];
    SidCtrl(1,VAL);
    /* FIXME: This should be moved into SidCtrl */
    if( (VAL&1)!=(old&1) ) {		/* Gate switched */
	SidVoice[1].Cycle=Cycle;
	if( VAL&1 ) {
	    SidVoice[1].State=SID_ATTACK;
	} else {
	    SidVoice[1].State=SID_RELEASE;
	}
    }
    if( !(VAL&0xF0) || (VAL&0x08) ) {   /* No waveforms selected or test */
	SidVoice[1].On=0;
    } else {
	SidVoice[1].On=1;
    }
    SID[0x0B]=VAL;
}
W_FUN( W_SID0C ) { SidAtkDcy(1,VAL); SID[0x0C]=VAL; }
W_FUN( W_SID0D ) { SidStnRls(1,VAL); SID[0x0D]=VAL; }

/*
**	Registers of voice 3.
*/
W_FUN( W_SID0E ) { SidFrqLo(2,VAL); SID[0x0E]=VAL; }
W_FUN( W_SID0F ) { SidFrqHi(2,VAL); SID[0x0F]=VAL; }
W_FUN( W_SID10 ) { SidPulLo(2,VAL); SID[0x10]=VAL; }
W_FUN( W_SID11 ) { SidPulHi(2,VAL); SID[0x11]=VAL; }
W_FUN( W_SID12 )			/* Voice 2 Control register */
{
    int old;

    old=SID[0x12];
    SidCtrl(2,VAL);
    /* FIXME: This should be moved into SidCtrl */
    if( (VAL&1)!=(old&1) ) {		/* Gate switched */
	SidVoice[2].Cycle=Cycle;
	if( VAL&1 ) {
	    SidVoice[2].State=SID_ATTACK;
	} else {
	    SidVoice[2].State=SID_RELEASE;
	}
    }
    if( !(VAL&0xF0) || (VAL&0x08)
	    || (SID[0x18]&0x80) ) {	/* No waveforms selected or test */
	SidVoice[2].On=0;
    } else {
	SidVoice[2].On=1;
    }
    SID[0x12]=VAL;
}
W_FUN( W_SID13 ) { SidAtkDcy(2,VAL); SID[0x13]=VAL; }
W_FUN( W_SID14 ) { SidStnRls(2,VAL); SID[0x14]=VAL; }

/*
**	Filter register (FIXME: not supported!)
*/
W_FUN( W_SID15 ) { SID[0x15]=VAL; }
W_FUN( W_SID16 ) { SID[0x16]=VAL; }
W_FUN( W_SID17 ) { SID[0x17]=VAL; }

W_FUN( W_SID18 )			/* Mode volume */
{
    SidVolume(VAL);
    
    if( SidSoundOff )
	SID[0x18]=VAL&0xF0;
    else {
	SID[0x18]=VAL;
	/*
	DigitalSound(VAL&0xF);
	*/
    }
    if( !(SID[0x12]&0xF0) || (SID[0x12]&0x08)
	    || (VAL&0x80) ) {		/* No waveforms selected or test */
	SidVoice[2].On=0;
    } else {
	SidVoice[2].On=1;
    }
}

R_FUN( R_SID19 )                        /* A/D wandler 1 */
{
    return 0xFF;                        /* Not emulated */
}

W_FUN( W_SID19 )
{
}

R_FUN( R_SID1A )                        /* A/D wandler 2 */
{
    return 0xFF;                        /* Not emulated */
}

W_FUN( W_SID1A )
{
}

R_FUN( R_SID1B )                        /* Rauschgenerator of channel 2 */
{
    static int NoiseRegister=0x7ffff8;
    int temp;
    int output;

    SidWavRd();
    /*
    **	FIXME: must depend on voice 3.
    */
    temp=NoiseRegister;
    output=  ((temp>>(22-7))&0x80)
	    |((temp>>(20-6))&0x40)
	    |((temp>>(16-5))&0x20)
	    |((temp>>(13-4))&0x10)
	    |((temp>>(11-3))&0x08)
	    |((temp>>( 7-2))&0x04)
	    |((temp>>( 4-1))&0x02)
	    |((temp>>( 2-0))&0x01);
    temp<<=1;
    temp|=((temp>>23)^(temp>>18))&1;
    NoiseRegister=temp;
    return output;
    /* return rand()&0xFF;	malfunction on DOS */
}

W_FUN( W_SID1B )
{
}

R_FUN( R_SID1C )                        /* Huellkurvengenerator of channel 2 */
{
#if defined(X11) && defined(DEBUG)
    printf("Read SID register 0x1C at $%04X\n",GlobalRegPC);
#endif
    SidEnvRd();
    return SidEnvelope(2);
}

W_FUN( W_SID1C )
{
}

/*-----------------------------------------------------------------------------
 *      1K Color RAM low nibble
 *---------------------------------------------------------------------------*/

R_FUN( R_ColorRam )                     /* Read color RAM */
{
    return (ColorRam[ADR-ADR_CRAM]>>4)|0xF0;
}

W_FUN( W_ColorRam )                     /* Write color RAM */
{
    ColorRam[ADR-ADR_CRAM]=(VAL<<4)&0xF0;
}

/*-----------------------------------------------------------------------------
 *	CIA-1 6526
 *---------------------------------------------------------------------------*/

R_FUN( R_CIA1_0 )			/* PRA: keyboard matrix / Joystick 2 */
{
    unsigned o;
    unsigned m;

    m=0xFF;
    o=(CIA1[CIA_PRB]^0xFF)&CIA1[CIA_DDRB]; /* Keyboard matrix line from PRB */
    if( ~KeyMatrix[0xFE]&o ) m&=~0x01;
    if( ~KeyMatrix[0xFD]&o ) m&=~0x02;
    if( ~KeyMatrix[0xFB]&o ) m&=~0x04;
    if( ~KeyMatrix[0xF7]&o ) m&=~0x08;
    if( ~KeyMatrix[0xEF]&o ) m&=~0x10;
    if( ~KeyMatrix[0xDF]&o ) m&=~0x20;
    if( ~KeyMatrix[0xBF]&o ) m&=~0x40;
    if( ~KeyMatrix[0x7F]&o ) m&=~0x80;

    return (CIA1[CIA_PRA]|(CIA1[CIA_DDRA]^0xFF))&JoyStick2&m;
}

W_FUN( W_CIA1_0 )			/* PRA: select matrix -> PRB */
{
    CIA1[CIA_PRA]=VAL;
}

R_FUN( R_CIA1_1 )                       /* PRB: keyboard matrix / Joystick 1 */
{
    unsigned char m;

    /* FIXME: PB6 + PB7 Timer controlled ?? */

    /*
     *  Port A: Lows & Output -> Keyboard Matrix -> Port B: Lows
     */
    m=KeyMatrix[ CIA1[CIA_PRA]|(CIA1[CIA_DDRA]^0xFF) ];
    /*
     *  + Joystick 1: Lows
     */
    m&=JoyStick1;
    /*
     *  Input: High low port
     * Output: Data in port Highs -> Low if port low 
     */
    return m&(CIA1[CIA_PRB]|(CIA1[CIA_DDRB]^0xFF));
}

W_FUN( W_CIA1_1 )                       /* PRB: select matrix -> PRA */
{
    CIA1[CIA_PRB]=VAL;
}

R_FUN( R_CIA1_2 )                       /* DDRA */
{
    return CIA1[CIA_DDRA];
}

W_FUN( W_CIA1_2 )                       /* DDRA */
{
    CIA1[CIA_DDRA]=VAL;
}

R_FUN( R_CIA1_3 )                       /* DDRB */
{
    return CIA1[CIA_DDRB];
}

W_FUN( W_CIA1_3 )                       /* DDRB */
{
    CIA1[CIA_DDRB]=VAL;
}

R_FUN( R_CIA1_4 )                       /* TimerA Low */
{
    if( Cia1TimerA_Action!=-1 ) {       /* TimerA running */
	return (Cia1TimerA_Action-Cycle-TIMER_SUB)&0xFF;
    } else {
	return Cia1TimerA&0xFF;         /* TimerA stopped */
    }
}

W_FUN( W_CIA1_4 )                       /* TimerA Low */
{
    CIA1[CIA_TALO]=VAL;
}

R_FUN( R_CIA1_5 )                       /* TimerA High */
{
    if( Cia1TimerA_Action!=-1 ) {       /* TimerA running */
	return ((Cia1TimerA_Action-Cycle-TIMER_SUB)>>8)&0xFF;
    } else {                            /* TimerA stopped */
	return Cia1TimerA>>8;
    }
}

W_FUN( W_CIA1_5 )                       /* TimerA High */
{
    CIA1[CIA_TAHI]=VAL;
    if( !(CIA1[CIA_CRA]&1) )            /* TimerA stopped */
	Cia1TimerA=WORD_FETCH(CIA1+CIA_TALO);
}

R_FUN( R_CIA1_6 )                       /* TimerB Low */
{
    if( Cia1TimerB_Action!=-1 ) {       /* TimerB running */
	return (Cia1TimerB_Action-Cycle-TIMER_SUB)&0xFF;
    } else {
	return Cia1TimerB&0xFF;         /* TimerB stopped */
    }
}

W_FUN( W_CIA1_6 )                       /* TimerB Low */
{
    CIA1[CIA_TBLO]=VAL;
}

R_FUN( R_CIA1_7 )                       /* TimerB High */
{
    if( Cia1TimerB_Action!=-1 ) {       /* TimerB running */
	return ((Cia1TimerB_Action-Cycle-TIMER_SUB)>>8)&0xFF;
    } else {                            /* TimerB stopped */
	return Cia1TimerB>>8;
    }
}

W_FUN( W_CIA1_7 )                       /* TimerB High */
{
    CIA1[CIA_TBHI]=VAL;
    if( !(CIA1[CIA_CRB]&1) )            /* TimerB stopped */
	Cia1TimerB=WORD_FETCH(CIA1+CIA_TBLO);
}

R_FUN( R_CIA1_8 )                       /* TOD 10THS */
{
    if( Cia1TodBuffered ) {             /* Buffered data */
	Cia1TodBuffered=0;
	return Cia1Time[0];
    }
    return CIA1[CIA_TOD10THS];
}

W_FUN( W_CIA1_8 )                       /* TOD 10THS Alarm/Time */
{
    if( CIA1[CIA_CRB]&0x80 ) {          /* Write alarm */
	Cia1Alarm[0]=VAL&0xF;
    } else {
	CIA1[CIA_TOD10THS]=VAL&0xF;
	/*
	**	Alarm reached!
	*/
	if( CIA1[CIA_TOD10THS]==Cia1Alarm[0]
		&& CIA1[CIA_TODSEC]==Cia1Alarm[1]
		&& CIA1[CIA_TODMIN]==Cia1Alarm[2]
		&& CIA1[CIA_TODHR]==Cia1Alarm[3] ) {
	    CIA1[CIA_ICR]|=0x04;
	    if( Cia1IrqMask&0x04 ) {	/* Irq TOD */
		CIA1[CIA_ICR]|=0x80;
		Irq();			/* FIXME: CAN'T DO IRQ HERE */
	    }
	}
	Cia1TodStopped=0;               /* Restart tod */
    }
}

R_FUN( R_CIA1_9 )                       /* TOD SEC */
{
    if( Cia1TodBuffered ) {             /* Read buffer */
	return Cia1Time[1];
    }
    return CIA1[CIA_TODSEC];
}

W_FUN( W_CIA1_9 )                       /* TOD SEC */
{
    if( CIA1[CIA_CRB]&0x80 ) {          /* Write alarm */
	Cia1Alarm[1]=VAL&0x7F;
    } else {
	CIA1[CIA_TODSEC]=VAL&0x7F;
    }
}

R_FUN( R_CIA1_A )                       /* TOD MIN */
{
    if( Cia1TodBuffered ) {             /* Read buffer */
	return Cia1Time[2];
    }
    return CIA1[CIA_TODMIN];
}

W_FUN( W_CIA1_A )                       /* TOD MIN */
{
    if( CIA1[CIA_CRB]&0x80 ) {          /* Write alarm */
	Cia1Alarm[2]=VAL&0x7F;
    } else {
	CIA1[CIA_TODMIN]=VAL&0x7F;
    }
}

R_FUN( R_CIA1_B )                       /* TOD HR */
{
    Cia1TodBuffered=1;
    Cia1Time[0]=CIA1[CIA_TOD10THS];
    Cia1Time[1]=CIA1[CIA_TODSEC];
    Cia1Time[2]=CIA1[CIA_TODMIN];
    return Cia1Time[3]=CIA1[CIA_TODHR];
}

W_FUN( W_CIA1_B )                       /* TOD HR */
{
    if( CIA1[CIA_CRB]&0x80 )            /* Write alarm */
	Cia1Alarm[3]=VAL&0x9F;
    else {
	CIA1[CIA_TODHR]=VAL&0x9F;
	Cia1TodStopped=1;
    }
}

R_FUN( R_CIA1_C )                       /* SDR Serial port */
{
#if defined(X11) && defined(DEBUG)
    printf("Serial 1 read $%04X\n",GlobalRegPC);
#endif
    return CIA1[CIA_SDR];
}

W_FUN( W_CIA1_C )                       /* SDR Serial port */
{
#if defined(X11) && defined(DEBUG)
    printf("Serial 1 write $%04X\n",GlobalRegPC);
#endif
    CIA1[CIA_SDR]=VAL;
    if( Cia1SpCount ) {			/* output running latch it */
	Cia1SpLatch=1;
    } else {				/* start output */
	Cia1SpCount=16;
    }
}

R_FUN( R_CIA1_D )                       /* ICR Read Irq Request */      
{
    int o;

    o=CIA1[CIA_ICR];
    CIA1[CIA_ICR]=0;			/* clear interrupts */
    IrqLine=VIC[0x19]&0x80;		/* irq line update */
    return o;
}

W_FUN( W_CIA1_D )                       /* ICR Set Irq Mask */
{
    /* ICR: S/C 0 0 SDR ALR TB TA */
    if( VAL&0x80 ) {                    /* Set Mask */
	Cia1IrqMask|=VAL&0x7F;
	/* adding an interrupt here isn't possible */
    } else {                            /* Clear Mask */
	Cia1IrqMask&=~VAL;
	/*
	**	Removing an active interrupt source.
	**	FIXME: not perfect must only clear real active interrupts.
	*/
	if( !(CIA1[CIA_ICR]&Cia1IrqMask) ) {
	    CIA1[CIA_ICR]&=0x7F;
	    IrqLine=VIC[0x19]&0x80;	/* irq line update */
	}
    }
}

R_FUN( R_CIA1_E )                       /* Timer A Control */
{
    return CIA1[CIA_CRA]&0xEF;
}

W_FUN( W_CIA1_E )                       /* Timer A Control */
{
    /*
     *  | TOD   | SP      | IN   | LOAD   | RUN     | OUT     | PB-6 | START  |
     *  | IN    | MODE    | MODE |        | MODE    | MODE    | ON   |        |
     *  |-------|---------|------|--------|---------|---------|------|--------|
     *  | 0=60Hz| 0=Input | 0=Q2 | 1=FORCE| 0=cont  | 0=PULSE | 0=OFF| 0=STOP |
     *  | 1=50Hz| 1=Output| 1=CNT|   LOAD | 1=one S.| 1=TOGGLE| 1=ON | 1=START|
     */
    CIA1[CIA_CRA]=VAL;
    if( VAL&0x10 ) {                    /* 1=force load */
	Cia1TimerA=WORD_FETCH(CIA1+CIA_TALO);
    }
    if( VAL&0x01 ) {                    /* 1=Start/0=Stop */
	if( Cia1TimerA ) {
	    Cia1TimerA_Action=Cycle+Cia1TimerA+TIMER_ADD;
	} else {
#if defined(X11) && defined(DEBUG)
	    printf("Cia1TimerA ZERO\n");
#endif
	    Cia1TimerA_Action=Cycle+65536+TIMER_ADD;
	}
	if( VAL&0x10 ) {
	    ++Cia1TimerA_Action;
	}
	InsertAction(Cia1TimerA_Action,EmulCia1TimerA);
    } else {
	if( Cia1TimerA_Action!=-1 ) {
	    if( !(VAL&0x10) )           /* !force load */
		Cia1TimerA=Cia1TimerA_Action-Cycle;
	    Cia1TimerA_Action=-1;
	    DeleteAction(Cia1TimerA_Action,EmulCia1TimerA);
	}
    }
    /* FIXME: counts CNT */
}

R_FUN( R_CIA1_F )                       /* Timer B Control */
{
    return CIA1[CIA_CRB]&0xEF;
}

W_FUN( W_CIA1_F )                       /* Timer B Control */
{
    /*
     *  | ALARM  | IN | MODE    | LOAD   | RUN     | OUT     | PB-7 | START  |
     *  |        |    |         |        | MODE    | MODE    | ON   |        |
     *  |--------|----|---------|--------|---------|---------|------|--------|
     *  | 0=TOD  | 0  | 0=Q2    | 1=FORCE| 0=cont  | 0=PULSE | 0=OFF| 0=STOP |
     *  | 1=ALARM| 0  | 1=CNT   |   LOAD | 1=one S.| 1=TOGGLE| 1=ON | 1=START|
     *  |        | 1  | 0=TA    |        |         |         |      |        |
     *  |        | 1  | 1=CNT-TA|        |         |         |      |        |
     */
    CIA1[CIA_CRB]=VAL;

    if( VAL&0x10 ) {                    /* 1=force load */
	Cia1TimerB=WORD_FETCH(CIA1+CIA_TBLO);
    }
    if( VAL&0x01 ) {                    /* 1=Start/0=Stop */
	if( Cia1TimerB ) {
	    Cia1TimerB_Action=Cycle+Cia1TimerB+TIMER_ADD;
	} else {
#if defined(X11) && defined(DEBUG)
	    printf("Cia1TimerB ZERO\n");
#endif
	    Cia1TimerB_Action=Cycle+65536+TIMER_ADD;
	}
	if( VAL&0x10 ) {
	    ++Cia1TimerB_Action;
	}
	/* FIXME: we need action only for interrupts!! */
	if( (CIA1[CIA_CRB]&0x61)!=0x41 ) {
	    InsertAction(Cia1TimerB_Action,EmulCia1TimerB);
	} else {
	    Cia1TimerB_Action=-1;
	    DeleteAction(Cia1TimerB_Action,EmulCia1TimerB);
	}
    } else {
	if( Cia1TimerB_Action!=-1 ) {
	    if( !(VAL&0x10) )           /* !force load */
		Cia1TimerB=Cia1TimerB_Action-Cycle;
	    Cia1TimerB_Action=-1;
	    DeleteAction(Cia1TimerB_Action,EmulCia1TimerB);
	}
    }
    /* FIXME:   count timer A + CNT */
}

static void EmulCia1TimerA(void)
{
    Cia1TimerA=WORD_FETCH(CIA1+CIA_TALO);
    if( CIA1[CIA_CRA]&0x08 ) {          /* One shot */
	CIA1[CIA_CRA]&=~1;
	Cia1TimerA_Action=-1;
	DeleteAction(Cia1TimerA_Action,EmulCia1TimerA);
    } else if( Cia1TimerA ) {           /* Continuous */
	Cia1TimerA_Action+=Cia1TimerA+1;
	ChangeAction(Cia1TimerA_Action,EmulCia1TimerA);
    } else {
	Cia1TimerA_Action+=65536+1;
	ChangeAction(Cia1TimerA_Action,EmulCia1TimerA);
    }

    /*
    **	Serial port interrupt?
    **		FIXME: not correct implemented!!
    */
    if( Cia1SpCount && !--Cia1SpCount ) {
	if( Cia1SpLatch ) {
	    Cia1SpLatch=0;
	    Cia1SpCount=16;
	} else {
	    CIA1[CIA_ICR]|=0x08;
	    if( Cia1IrqMask&0x08 ) {	/* Irq Serial port empty */
		CIA1[CIA_ICR]|=0x80;
		Irq();
	    }
	}
    }

#if defined(X11) && defined(DEBUG)
    /* FIXME: TimerB Counts TimerA */
    if( (CIA1[CIA_CRB]&0x61)==0x41 ) {
	printf("COUNTING 1 TIMER A\n");
    }
#endif

    CIA1[CIA_ICR]|=0x01;
    if( Cia1IrqMask&0x01 ) {		/* Irq TA */
	CIA1[CIA_ICR]|=0x80;
	Irq();
    }
}

static void EmulCia1TimerB(void)
{
    Cia1TimerB=WORD_FETCH(CIA1+CIA_TBLO);
    if( CIA1[CIA_CRB]&0x08 ) {          /* One shot */
	CIA1[CIA_CRB]&=~1;
	Cia1TimerB_Action=-1;
	DeleteAction(Cia1TimerB_Action,EmulCia1TimerB);
    } else if( Cia1TimerB ) {           /* Continuous */
	Cia1TimerB_Action+=Cia1TimerB+1;
	ChangeAction(Cia1TimerB_Action,EmulCia1TimerB);
    } else {
	Cia1TimerB_Action+=65536+1;
	ChangeAction(Cia1TimerB_Action,EmulCia1TimerB);
    }
    CIA1[CIA_ICR]|=0x02;
    if( Cia1IrqMask&0x02 ) {		/* Irq TB */
	CIA1[CIA_ICR]|=0x80;
	Irq();
    }
}

/*
**	Called in the video vertical blank to update TOD.	
*/
void EmulCia1Tod(void)
{
    static int Cia1TodCounter;

    if( !Cia1TodStopped ) {		/* TOD running */
	if( ++Cia1TodCounter==5+(CIA1[CIA_CRA]>>7) ) {
	    Cia1TodCounter=0;
	    ++CIA1[CIA_TOD10THS];
	    if( CIA1[CIA_TOD10THS]>=0x0A ) {
		CIA1[CIA_TOD10THS]=0;
		++CIA1[CIA_TODSEC];
		if( (CIA1[CIA_TODSEC]&0xF)>=0x0A ) {
		    CIA1[CIA_TODSEC]&=0xF0;
		    CIA1[CIA_TODSEC]+=0x10;
		    if( (CIA1[CIA_TODSEC]&0xF0)>=0x70 ) {
			++CIA1[CIA_TODMIN];
			if( (CIA1[CIA_TODMIN]&0xF)>=0x0A ) {
			    CIA1[CIA_TODMIN]&=0xF0;
			    CIA1[CIA_TODMIN]+=0x10;
			    if( (CIA1[CIA_TODMIN]&0xF0)>=0x70 ) {
				++CIA1[CIA_TODHR];
				if( (CIA1[CIA_TODHR]&0xF)>=0x0A ) {
				    CIA1[CIA_TODHR]&=0xF0;
				    CIA1[CIA_TODHR]+=0x10;
				}
				if( (CIA1[CIA_TODHR]&0x1F)>=12 ) {
				    CIA1[CIA_TODHR]&=0x80;
				    CIA1[CIA_TODHR]^=0x80;
				}
			    }
			}
		    }
		}
	    }
	    /*
	    **	Alarm reached!
	    */
	    if( CIA1[CIA_TOD10THS]==Cia1Alarm[0]
		    && CIA1[CIA_TODSEC]==Cia1Alarm[1]
		    && CIA1[CIA_TODMIN]==Cia1Alarm[2]
		    && CIA1[CIA_TODHR]==Cia1Alarm[3] ) {
		CIA1[CIA_ICR]|=0x04;
		if( Cia1IrqMask&0x04 ) {	/* Irq TOD */
		    CIA1[CIA_ICR]|=0x80;
		    Irq();			/* FIXME: CAN'T DO IRQ HERE */
		}
	    }
	}
    }
}

/*-----------------------------------------------------------------------------
 *	CIA-2 6526
 *---------------------------------------------------------------------------*/

/*
**	PRA: Description
**	7:	Serial bus data input
**	6:	Serial bus clock pulse input
**	5:	Serial bus data output
**	4:	Serial bus clock pulse output
**	3:	Serial bus ATN signal output
**	2:	RS-232 Data output
**	1-0:	VIC Bank select
*/
R_FUN( R_CIA2_0 )			/* PRA: Serial lines */
{
    int i;
    int o;

    i=CIA2[CIA_PRA]|(CIA2[CIA_DDRA]^0xFF);
    o=~CIA2[CIA_PRA]&CIA2[CIA_DDRA];

    i&=((o&0x30)<<2)|0x3F;		/* Clock+Data out -> Clock+Data in */
    return i&~IEC_ReadPortTrap();	/* IEC-SERIAL-BUS emulation */
}

W_FUN( W_CIA2_0 )			/* PRA: Serial lines/Vic bank */
{
    int a;
    int o;

    CIA2[CIA_PRA]=VAL;
					/* PRA: 01 VRAM 14/15 */
    /* 15	14	 13	  12	   11	    10	      */
    /* CIA2-0-1 CIA2-0-0 VIC-18-7 VIC-18-6 VIC-18-5 VIC-18-4  */
    o=~VAL&CIA2[CIA_DDRA];
    VicBank=Ram+((o&3)<<14);
    VicVRam=VicBank+((VIC[0x18]&0xF0)<<6);
    VicBRam=VicBank+((VIC[0x18]&0x08)<<10);
    VicCRam=VicBank+((VIC[0x18]&0x0E)<<10);
    a=VicCRam-Ram;
    /*
    **	Map character ROM addresses 0x1000-0x1FFF 
    **	and 0x9000-0x9FFF to character ROM
    */
    if( (a&0x7000)==0x1000 ) {
	VicCRam=CharacterRom+(a&0x0FFF);
    }

    IEC_WritePortTrap(o);		/* IEC-SERIAL-BUS Emulation */
}

R_FUN( R_CIA2_1 )			/* PRB RS-232 */
{
    return CIA2[CIA_PRB]|(CIA2[CIA_DDRB]^0xFF);
}

W_FUN( W_CIA2_1 )			/* PRB RS-232 */
{
    CIA2[CIA_PRB]=VAL;
}

R_FUN( R_CIA2_2 )			/* DDRA */
{
    return CIA2[CIA_DDRA];
}

W_FUN( W_CIA2_2 )			/* DDRA */
{
    int o;
    int a;

    CIA2[CIA_DDRA]=VAL;
    /*
    **	Changing the output to input: changes the outputs (vic bank)!
    */
					/* PRA: 01 VRAM 14/15 */
    /* 15	14	 13	  12	   11	    10	      */
    /* CIA2-0-1 CIA2-0-0 VIC-18-7 VIC-18-6 VIC-18-5 VIC-18-4  */
    o=~CIA2[CIA_PRA]&VAL;
    VicBank=Ram+((o&3)<<14);
    VicVRam=VicBank+((VIC[0x18]&0xF0)<<6);
    VicBRam=VicBank+((VIC[0x18]&0x08)<<10);
    VicCRam=VicBank+((VIC[0x18]&0x0E)<<10);
    a=VicCRam-Ram;
    /*
    **	Map character ROM addresses 0x1000-0x1FFF 
    **	and 0x9000-0x9FFF to character ROM
    */
    if( (a&0x7000)==0x1000 ) {
	VicCRam=CharacterRom+(a&0x0FFF);
    }

    IEC_WritePortTrap(o);		/* IEC-SERIAL-BUS Emulation */
}

R_FUN( R_CIA2_3 )			/* DDRB */
{
    return CIA2[CIA_DDRB];
}

W_FUN( W_CIA2_3 )			/* DDRB */
{
    CIA2[CIA_DDRB]=VAL;
}

R_FUN( R_CIA2_4 )			/* TimerA Low */
{
    if( Cia2TimerA_Action!=-1 ) {       /* TimerA running */
	return (Cia2TimerA_Action-Cycle-TIMER_SUB)&0xFF;
    } else {
	return Cia2TimerA&0xFF;         /* TimerA stopped */
    }
}

W_FUN( W_CIA2_4 )                       /* TimerA Low */
{
    CIA2[CIA_TALO]=VAL;
}

R_FUN( R_CIA2_5 )                       /* TimerA High */
{
    if( Cia2TimerA_Action!=-1 ) {       /* TimerA running */
	return ((Cia2TimerA_Action-Cycle-TIMER_SUB)>>8)&0xFF;
    } else {                            /* TimerA stopped */
	return Cia2TimerA>>8;
    }
}

W_FUN( W_CIA2_5 )                       /* TimerA High */
{
    CIA2[CIA_TAHI]=VAL;
    if( !(CIA2[CIA_CRA]&1) )            /* TimerA stopped */
	Cia2TimerA=WORD_FETCH(CIA2+CIA_TALO);
}

R_FUN( R_CIA2_6 )                       /* TimerB Low */
{
    if( Cia2TimerB_Action!=-1 ) {       /* TimerB running */
	return (Cia2TimerB_Action-Cycle-TIMER_SUB)&0xFF;
    } else {
	return Cia2TimerB&0xFF;         /* TimerB stopped */
    }
}

W_FUN( W_CIA2_6 )                       /* TimerB Low */
{
    CIA2[CIA_TBLO]=VAL;
}

R_FUN( R_CIA2_7 )                       /* TimerB High */
{
    if( Cia2TimerB_Action!=-1 ) {       /* TimerB running */
	return ((Cia2TimerB_Action-Cycle-TIMER_SUB)>>8)&0xFF;
    } else {                            /* TimerB stopped */
	return Cia2TimerB>>8;
    }
}

W_FUN( W_CIA2_7 )                       /* TimerB High */
{
    CIA2[CIA_TBHI]=VAL;
    if( !(CIA2[CIA_CRB]&1) )            /* TimerB stopped */
	Cia2TimerB=WORD_FETCH(CIA2+CIA_TBLO);
}

R_FUN( R_CIA2_8 )                       /* TOD 10THS */
{
    if( Cia2TodBuffered ) {             /* Buffered data */
	Cia2TodBuffered=0;
	return Cia2Time[0];
    }
    return CIA2[CIA_TOD10THS];
}

W_FUN( W_CIA2_8 )                       /* TOD 10THS Alarm/Time */
{
    if( CIA2[CIA_CRB]&0x80 ) {		/* Write alarm */
	Cia2Alarm[0]=VAL&0xF;
    } else {
	CIA2[CIA_TOD10THS]=VAL&0xF;
	/*
	**	Alarm reached!
	*/
	if( CIA2[CIA_TOD10THS]==Cia2Alarm[0]
		&& CIA2[CIA_TODSEC]==Cia2Alarm[1]
		&& CIA2[CIA_TODMIN]==Cia2Alarm[2]
		&& CIA2[CIA_TODHR]==Cia2Alarm[3] ) {

	    CIA2[CIA_ICR]|=0x04;
	    if( Cia2IrqMask&0x04 ) {		/* Nmi TOD */
		if( !(CIA2[CIA_ICR]&0x80) )	/* Not already nmi */
		    Nmi();		/* FIXME: CAN'T DO NMI HERE */
		CIA2[CIA_ICR]|=0x80;
	    }
	}
	Cia2TodStopped=0;		/* Restart tod */
    }
}

R_FUN( R_CIA2_9 )                       /* TOD SEC */
{
    if( Cia2TodBuffered ) {             /* Read buffer */
	return Cia2Time[1];
    }
    return CIA2[CIA_TODSEC];
}

W_FUN( W_CIA2_9 )                       /* TOD SEC */
{
    if( CIA2[CIA_CRB]&0x80 ) {          /* Write alarm */
	Cia2Alarm[1]=VAL&0x7F;
    } else {
	CIA2[CIA_TODSEC]=VAL&0x7F;
    }
}

R_FUN( R_CIA2_A )                       /* TOD MIN */
{
    if( Cia2TodBuffered ) {             /* Read buffer */
	return Cia2Time[2];
    }
    return CIA2[CIA_TODMIN];
}

W_FUN( W_CIA2_A )                       /* TOD MIN */
{
    if( CIA2[CIA_CRB]&0x80 ) {          /* Write alarm */
	Cia2Alarm[2]=VAL&0x7F;
    } else {
	CIA2[CIA_TODMIN]=VAL&0x7F;
    }
}

R_FUN( R_CIA2_B )                       /* TOD HR */
{
    Cia2TodBuffered=1;
    Cia2Time[0]=CIA2[CIA_TOD10THS];
    Cia2Time[1]=CIA2[CIA_TODSEC];
    Cia2Time[2]=CIA2[CIA_TODMIN];
    return Cia2Time[3]=CIA2[CIA_TODHR];
}

W_FUN( W_CIA2_B )			/* TOD HR */
{
    if( CIA2[CIA_CRB]&0x80 )            /* Write alarm */
	Cia2Alarm[3]=VAL&0x9F;
    else {
	CIA2[CIA_TODHR]=VAL&0x9F;
	Cia2TodStopped=1;
    }
}

R_FUN( R_CIA2_C )			/* SDR Serial port */
{
#if defined(X11) && defined(DEBUG)
    printf("Serial 2 read at $%04X\n",GlobalRegPC);
#endif
    return CIA2[CIA_SDR];
}

W_FUN( W_CIA2_C )			/* SDR Serial port */
{
#if defined(X11) && defined(DEBUG)
    printf("Serial 2 write at $%04X\n",GlobalRegPC);
#endif
    CIA2[CIA_SDR]=VAL;
    if( Cia2SpCount ) {			/* output running latch it */
	Cia2SpLatch=1;
    } else {				/* start output */
	Cia2SpCount=16;
    }
}

R_FUN( R_CIA2_D )			/* ICR Read Irq Request */
{
    int o;

    o=CIA2[CIA_ICR];
    CIA2[CIA_ICR]=0;
#if defined(DEBUG)
    if( (RBMem(GlobalRegPC)&0x1F)>0x19 ) {
	printf("Index bug: $%04X\n",GlobalRegPC);
    }
#endif

#if 0
    /* FIXME: BIG BUG IN 6502 
     *	STA	$DC0D,Y		Reads port
     *	STA	$DC0D,X		Reads port
     *	or any xxxx,y = DC0D
     *	or any xxxx,X = DC0D
     */
    CIA1[CIA_ICR]=0;
#endif

#if 0
    /* FIXME: if we need this */
    if( Cia2IrqMask&CIA2[CIA_ICR] ) {	/* Irq TA or TB */
	if( !(CIA2[CIA_ICR]&0x80) )	/* Not already nmi */
	    Nmi();			/* Can't do nmi here */
	CIA2[CIA_ICR]|=0x80;
    }
#endif
    return o;
}

W_FUN( W_CIA2_D )			/* ICR: Set Irq Mask */
{
    /* ICR: S/C 0 0 SDR ALR TB TA */
    if( VAL&0x80 ) {                    /* Set Mask */
	Cia2IrqMask|=VAL&0x7F;
	/* adding an interrupt here isn't possible */
    } else {                            /* Clear Mask */
	Cia2IrqMask&=~VAL;
	/*
	**	Removing an active interrupt source.
	**	FIXME: not perfect must only clear real active interrupts.
	*/
	if( !(CIA2[CIA_ICR]&Cia2IrqMask) ) {
	    CIA2[CIA_ICR]&=0x7F;
	    IrqLine=VIC[0x19]&0x80;	/* irq line update */
	}
    }
#if 0
    /* FIXME: if we need this */
    if( CIA2[CIA_ICR]&Cia2IrqMask ) {	/* Irq TA or TB */
	if( !(CIA2[CIA_ICR]&0x80) ) {	/* Not already nmi */
	    Nmi();			/* Can't do nmi here */
	}
	CIA2[CIA_ICR]|=0x80;
    }
#endif
}

R_FUN( R_CIA2_E )			/* Timer A Control */
{
    return CIA2[CIA_CRA]&0xEF;
}

W_FUN( W_CIA2_E )			/* Timer A Control */
{
    /*
     *  | TOD   | SP      | IN   | LOAD   | RUN     | OUT     | PB-6 | START  |
     *  | IN    | MODE    | MODE |        | MODE    | MODE    | ON   |        |
     *  |-------|---------|------|--------|---------|---------|------|--------|
     *  | 0=60Hz| 0=Input | 0=Q2 | 1=FORCE| 0=cont  | 0=PULSE | 0=OFF| 0=STOP |
     *  | 1=50Hz| 1=Output| 1=CNT|   LOAD | 1=one S.| 1=TOGGLE| 1=ON | 1=START|
     */
    CIA2[CIA_CRA]=VAL;
    if( VAL&0x10 ) {                    /* 1=force load */
	Cia2TimerA=WORD_FETCH(CIA2+CIA_TALO);
    }
    if( VAL&0x01 ) {                    /* 1=Start/0=Stop */
	if( Cia2TimerA ) {
	    // printf("searching... %d %02X [%d]\n",Cycle,VAL,DoMystic);
	    Cia2TimerA_Action=Cycle+Cia2TimerA+TIMER_ADD;
	} else {
#if defined(X11) && defined(DEBUG)
	    printf("Cia2TimerA ZERO\n");
#endif
	    Cia2TimerA_Action=Cycle+65536+TIMER_ADD;
	}
	if( VAL&0x10 ) {
	    ++Cia2TimerA_Action;
	}
	InsertAction(Cia2TimerA_Action,EmulCia2TimerA);
    } else {
	if( Cia2TimerA_Action!=-1 ) {
	    if( !(VAL&0x10) )           /* !force load */
		Cia2TimerA=Cia2TimerA_Action-Cycle;
	    Cia2TimerA_Action=-1;
	    DeleteAction(Cia2TimerA_Action,EmulCia2TimerA);
	}
    }
    /* FIXME: counts CNT */
}

R_FUN( R_CIA2_F )                       /* Timer B Control */
{
    return CIA2[CIA_CRB]&0xEF;
}

W_FUN( W_CIA2_F )                       /* Timer B Control */
{
    /*
     *  | ALARM  | IN | MODE    | LOAD   | RUN     | OUT     | PB-7 | START  |
     *  |        |    |         |        | MODE    | MODE    | ON   |        |
     *  |--------|----|---------|--------|---------|---------|------|--------|
     *  | 0=TOD  | 0  | 0=Q2    | 1=FORCE| 0=cont  | 0=PULSE | 0=OFF| 0=STOP |
     *  | 1=ALARM| 0  | 1=CNT   |   LOAD | 1=one S.| 1=TOGGLE| 1=ON | 1=START|
     *  |        | 1  | 0=TA    |        |         |         |      |        |
     *  |        | 1  | 1=CNT-TA|        |         |         |      |        |
     */
    CIA2[CIA_CRB]=VAL;

    if( VAL&0x10 ) {                    /* 1=force load */
	Cia2TimerB=WORD_FETCH(CIA2+CIA_TBLO);
    }
    if( VAL&0x01 ) {                    /* 1=Start/0=Stop */
	if( Cia2TimerB ) {
	    Cia2TimerB_Action=Cycle+Cia2TimerB+TIMER_ADD;
	} else {
#if defined(X11) && defined(DEBUG)
	    printf("Cia2TimerA ZERO\n");
#endif
	    Cia2TimerB_Action=Cycle+65536+TIMER_ADD;
	}
	if( VAL&0x10 ) {
	    ++Cia2TimerB_Action;
	}
	/* FIXME: we need action only for interrupts!! */
	if( (CIA2[CIA_CRB]&0x61)!=0x41 ) {
	    InsertAction(Cia2TimerB_Action,EmulCia2TimerB);
	} else {
	    Cia2TimerB_Action=-1;
	    DeleteAction(Cia2TimerB_Action,EmulCia2TimerB);
	}
    } else {
	if( Cia2TimerB_Action!=-1 ) {
	    if( !(VAL&0x10) )           /* !force load */
		Cia2TimerB=Cia2TimerB_Action-Cycle;
	    Cia2TimerB_Action=-1;
	    DeleteAction(Cia2TimerB_Action,EmulCia2TimerB);
	}
    }
    /* FIXME:   count timer A + CNT */
}

static void EmulCia2TimerA(void)
{
    Cia2TimerA=WORD_FETCH(CIA2+CIA_TALO);
    if( CIA2[CIA_CRA]&0x08 ) {          /* One shot */
	CIA2[CIA_CRA]&=~1;
	Cia2TimerA_Action=-1;
	DeleteAction(Cia2TimerA_Action,EmulCia2TimerA);
    } else if( Cia2TimerA ) {           /* Continuous */
	Cia2TimerA_Action+=Cia2TimerA+1;
	ChangeAction(Cia2TimerA_Action,EmulCia2TimerA);
    } else {
	Cia2TimerA_Action+=65536+1;
	ChangeAction(Cia2TimerA_Action,EmulCia2TimerA);
    }

    /*
    **	Serial port interrupt?
    **		FIXME: not correct implemented!!
    */
    if( Cia2SpCount && !--Cia2SpCount ) {
	if( Cia2SpLatch ) {
	    Cia2SpLatch=0;
	    Cia2SpCount=16;
	} else {
	    CIA2[CIA_ICR]|=0x08;
	    if( Cia2IrqMask&0x08 ) {	/* Nmi Serial port empty */
		if( !(CIA2[CIA_ICR]&0x80) )
		    Nmi();
		CIA2[CIA_ICR]|=0x80;
	    }
	}
    }

#if defined(X11) && defined(DEBUG)
    /* FIXME: TimerB Counts TimerA */
    if( (CIA2[CIA_CRB]&0x61)==0x41 ) {
	printf("COUNTING 2 TIMER A\n");
    }
#endif

    CIA2[CIA_ICR]|=0x01;
    if( Cia2IrqMask&0x01 ) {		/* Nmi TA */
	if( !(CIA2[CIA_ICR]&0x80) )
	    Nmi();
	CIA2[CIA_ICR]|=0x80;
    }
}

static void EmulCia2TimerB(void)
{
    Cia2TimerB=WORD_FETCH(CIA2+CIA_TBLO);
    if( CIA2[CIA_CRB]&0x08 ) {          /* One shot */
	CIA2[CIA_CRB]&=~1;
	Cia2TimerB_Action=-1;
	DeleteAction(Cia2TimerB_Action,EmulCia2TimerB);
    } else if( Cia2TimerB ) {           /* Continuous */
	Cia2TimerB_Action+=Cia2TimerB+1;
	ChangeAction(Cia2TimerB_Action,EmulCia2TimerB);
    } else {
	Cia2TimerB_Action+=65536+1;
	ChangeAction(Cia2TimerB_Action,EmulCia2TimerB);
    }
    CIA2[CIA_ICR]|=0x02;
    if( Cia2IrqMask&0x02 ) {            /* Nmi TB */
	if( !(CIA2[CIA_ICR]&0x80) )
	    Nmi();
	CIA2[CIA_ICR]|=0x80;
    }
}

/*
**	Called in the video vertical blank to update TOD.	
*/
void EmulCia2Tod(void)
{
    static int Cia2TodCounter;

    if( !Cia2TodStopped ) {		/* TOD running */
	if( ++Cia2TodCounter==5+(CIA2[CIA_CRA]>>7) ) {
	    Cia2TodCounter=0;
	    ++CIA2[CIA_TOD10THS];
	    if( CIA2[CIA_TOD10THS]>=0x0A ) {
		CIA2[CIA_TOD10THS]=0;
		++CIA2[CIA_TODSEC];
		if( (CIA2[CIA_TODSEC]&0xF)>=0x0A ) {
		    CIA2[CIA_TODSEC]&=0xF0;
		    CIA2[CIA_TODSEC]+=0x10;
		    if( (CIA2[CIA_TODSEC]&0xF0)>=0x70 ) {
			++CIA2[CIA_TODMIN];
			if( (CIA2[CIA_TODMIN]&0xF)>=0x0A ) {
			    CIA2[CIA_TODMIN]&=0xF0;
			    CIA2[CIA_TODMIN]+=0x10;
			    if( (CIA2[CIA_TODMIN]&0xF0)>=0x70 ) {
				++CIA2[CIA_TODHR];
				if( (CIA2[CIA_TODHR]&0xF)>=0x0A ) {
				    CIA2[CIA_TODHR]&=0xF0;
				    CIA2[CIA_TODHR]+=0x10;
				}
				if( (CIA2[CIA_TODHR]&0x1F)>=12 ) {
				    CIA2[CIA_TODHR]&=0x80;
				    CIA2[CIA_TODHR]^=0x80;
				}
			    }
			}
		    }
		}
	    }
	    /*
	    **	Alarm reached!
	    */
	    if( CIA2[CIA_TOD10THS]==Cia2Alarm[0]
		    && CIA2[CIA_TODSEC]==Cia2Alarm[1]
		    && CIA2[CIA_TODMIN]==Cia2Alarm[2]
		    && CIA2[CIA_TODHR]==Cia2Alarm[3] ) {
		CIA2[CIA_ICR]|=0x04;
		if( Cia2IrqMask&0x04 ) {	/* Nmi TOD */
		    if( !(CIA2[CIA_ICR]&0x80) )	/* Not already nmi */
			Nmi();
		    CIA2[CIA_ICR]|=0x80;
		}
	    }
	}
    }
}

/*-----------------------------------------------------------------------------
 *	Unconnected adress space
 *---------------------------------------------------------------------------*/

R_FUN( R_Free )
{
    return 0xFF;
}

W_FUN( W_Free )
{
}

#ifdef NEW_MEMORY

/*=============================================================================
 *	Bank dispatcher
 *===========================================================================*/

W_FUN( WrBank0 )
{
    switch( ADR ) {
	case 0:
	    W_PPDir(ADR,VAL);
	    return;
	case 1:
	    W_PPData(ADR,VAL);
	    return;
	default:
	    W_Ram(ADR,VAL);
	    return;
    }
}

R_FUN( RdBankD )
{
    return R_Config[Config[ADR]](ADR);
}


W_FUN( WrBankD )
{
    W_Config[Config[ADR]](ADR,VAL);
}

/*=============================================================================
 *	Init memory
 *===========================================================================*/

int R_YYY(int adr)
{
    return R_Config[Config[adr]](adr);
}

int R_XXX(int adr)
{
    return ((*RdIt)[(adr&0xF000)>>12]((adr)));
}

void W_XXX(int adr,int val)
{
    WBMem(adr,val);
}

/*
**	Build and initialise tables for c64 memory map emulation.
*/
void MemInit(void)
{
    int bank;
    int i;

    /*
    **	0000-FFFF:	Init all for RAM read/write.
    */
    for( bank=0x0; bank<0x10; ++bank ) {
	for( i=0; i<8; ++i ) {
	    RdConfig[i][bank]=R_Ram;
	    WrConfig[i][bank]=W_Ram;
	}
    }

    /*
    **	0000-0FFF:	Processor port change
    */
    for( i=0; i<8; ++i ) {
	WrConfig[i][0]=WrBank0;
    }

    /*
    **	8000-9FFF:	External ROM Area
    **		FIXME: exrom = module support !!
    */
    for( bank=0x8; bank<0xA; ++bank ) {
	for( i=0; i<8; ++i ) {
	    WrConfig[i][bank]=W_Ram;
	}
    }

    /*
    **	A000-BFFF:	Basic ROM or External ROM Area
    **		FIXME: exrom = module support !!
    */
    for( bank=0xA; bank<0xC; ++bank ) {
	RdConfig[3][bank]=RdConfig[7][bank]=R_BasicRom;
    }

    /*
    **	D000-DFFF:	Ram/IO/Character ROM Area
    */
    for( bank=0xD; bank<0xE; ++bank ) {
	RdConfig[1][bank]=RdConfig[2][bank]=RdConfig[3][bank]=R_CharacterRom;
	RdConfig[5][bank]=RdConfig[6][bank]=RdConfig[7][bank]=RdBankD;
	WrConfig[5][bank]=WrConfig[6][bank]=WrConfig[7][bank]=WrBankD;
    }

    /*
    **	E000-FFFF:	Kernal ROM
    */
    for( bank=0xE; bank<0x10; ++bank ) {
	RdConfig[2][bank]=RdConfig[3][bank]=
	    RdConfig[6][bank]=RdConfig[7][bank]=R_KernelRom;
    }

    /*
    **	10000-10FFF
    */
    for( i=0; i<8; ++i ) {
	RdConfig[i][16]=RdConfig[i][0];
	WrConfig[i][16]=WrConfig[i][0];
    }
}

#else

void MemInit(void) {}

#endif

/*
**	Initialise memory access config tables.
*/
void InitConfig(void)
{
    int i;

    MemInit();
    Config0[0]=Config1[0]=Config2[0]=Config3[0]=
	Config5[0]=Config6[0]=Config7[0]=C_PDIR;
    Config0[1]=Config1[1]=Config2[1]=Config3[1]=
	Config5[1]=Config6[1]=Config7[1]=C_PDATA;

    for( i=2; i<ADR_BASIC; ++i ) {				/* 0000-9FFF */
	Config0[i]=Config1[i]=Config2[i]=Config3[i]=
	Config5[i]=Config6[i]=Config7[i]=C_RW_RAM;
    }

    for( i=ADR_BASIC; i<ADR_BASIC+8192; ++i ) {			/* A000-BFFF */
	Config0[i]=Config1[i]=Config2[i]=
	Config5[i]=Config6[i]=C_RW_RAM;
	Config3[i]=Config7[i]=C_BROM;
    }

    for( i=0xC000; i<ADR_VIC; ++i ) {				/* C000-CFFF */
	Config0[i]=Config1[i]=Config2[i]=Config3[i]=
	Config5[i]=Config6[i]=Config7[i]=C_RW_RAM;
    }

    for( i=ADR_VIC; i<0xE000; ++i ) {				/* D000-DFFF */
	Config0[i]=C_RW_RAM;
	Config1[i]=Config2[i]=Config3[i]=C_CROM;
	Config5[i]=Config6[i]=Config7[i]=C_FREE;
    }

    /* D000-D3FF VIC | D000-D1FF if Ale Ram */
    for( i=ADR_VIC; i<(ADR_VIC+0x400); ++i ) {
	if( (i&0x3F)<0x2F ) {
	    Config5[i]=Config6[i]=Config7[i]=C_VIC00+(i&0x3F);
	} else {
	    Config5[i]=Config6[i]=Config7[i]=C_FREE;
	}
    }
#ifdef ALE_RAM
    if( !ExtensionOff ) {
	/* D200-D3FF Ale RAM 512 bytes 1 Bank */
	for( i=ADR_ALE_RAM1; i<(ADR_ALE_RAM1+0x200); ++i ) {
	    Config5[i]=Config6[i]=Config7[i]=C_ALE_RAM1;
	}
    }
#endif
    /* D400-D7FF SID | D400-D5FF if Ale Ram */
    for( i=ADR_SID; i<(ADR_SID+0x400); ++i ) {
	if( (i&0x1F)<0x1D ) {
	    Config5[i]=Config6[i]=Config7[i]=C_SID00+(i&0x1F);
	} else {
	    Config5[i]=Config6[i]=Config7[i]=C_FREE;
	}
    }
#ifdef ALE_RAM
    if( !ExtensionOff ) {
	/* D600-D7FF Ale RAM 512 bytes 2 Bank */
	for( i=ADR_ALE_RAM2; i<(ADR_ALE_RAM2+0x200); ++i ) {
	    Config5[i]=Config6[i]=Config7[i]=C_ALE_RAM2;
	}
    }
#endif
    /* D800-DBFF Color RAM */
    for( i=ADR_CRAM; i<(ADR_CRAM+0x400); ++i ) {
	Config5[i]=Config6[i]=Config7[i]=C_CRAM;
    }
    /* DC00-DCFF CIA 1 */
    for( i=ADR_CIA1; i<(ADR_CIA1+0x100); ++i ) {
	Config5[i]=Config6[i]=Config7[i]=C_CIA1_0+(i&0xF);
    }
    /* DD00-DDFF CIA 2 */
    for( i=ADR_CIA2; i<(ADR_CIA2+0x100); ++i ) {
	Config5[i]=Config6[i]=Config7[i]=C_CIA2_0+(i&0xF);
    }
    /* DE00-DFFF Free I/O Port area */
    for( i=0xDE00; i<ADR_KERNEL; ++i ) {
	Config5[i]=Config6[i]=Config7[i]=C_FREE;
    }
    for( i=ADR_KERNEL; i<ADR_KERNEL+8192; ++i ) {
	Config0[i]=Config1[i]=Config5[i]=C_RW_RAM;
	Config2[i]=Config3[i]=Config6[i]=Config7[i]=C_KROM;
    }

    /*
    **	64K wrap around:
    */
    for( i=0; i<0xFF; ++i ) {
	Config0[0x10000+i]=Config1[0x10000+i]=Config2[0x10000+i]=
	    Config3[0x10001+i]=Config5[0x10000+i]=
		Config6[0x10000+i]=Config7[0x10000+i]=C_ZERO;
    }

    Config0[0x10000]=Config1[0x10000]=Config2[0x10000]=Config3[0x10001]=
	Config5[0x10000]=Config6[0x10000]=Config7[0x10000]=C_PDIR;
    Config0[0x10001]=Config1[0x10001]=Config2[0x10001]=Config3[0x10001]=
	Config5[0x10001]=Config6[0x10001]=Config7[0x10001]=C_PDATA;

    /*
    **	Build function table.
    */
    for( i=0; i<C_MAX; ++i ) {
	switch( i ) {
	    case C_PDIR:  R_Config[i]=R_Ram;         W_Config[i]=W_PPDir; break;
	    case C_PDATA: R_Config[i]=R_Ram;         W_Config[i]=W_PPData;break;
	    case C_ZERO:  R_Config[i]=R_Zero;        W_Config[i]=W_Zero;  break;
	    case C_RW_RAM:R_Config[i]=R_Ram;         W_Config[i]=W_Ram; break;
	    case C_BROM:  R_Config[i]=R_BasicRom;    W_Config[i]=W_Ram; break;
	    case C_KROM:  R_Config[i]=R_KernelRom;   W_Config[i]=W_Ram; break;
	    case C_CROM:  R_Config[i]=R_CharacterRom;W_Config[i]=W_Ram; break;

#ifdef ALE_RAM
	    case C_ALE_RAM1:R_Config[i]=R_AleRam1; W_Config[i]=W_AleRam1; break;
	    case C_ALE_RAM2:R_Config[i]=R_AleRam2; W_Config[i]=W_AleRam2; break;
#endif

	    case C_VIC00: R_Config[i]=R_VIC00; W_Config[i]=W_VIC00; break;
	    case C_VIC01: R_Config[i]=R_VIC01; W_Config[i]=W_VIC01; break;
	    case C_VIC02: R_Config[i]=R_VIC02; W_Config[i]=W_VIC02; break;
	    case C_VIC03: R_Config[i]=R_VIC03; W_Config[i]=W_VIC03; break;
	    case C_VIC04: R_Config[i]=R_VIC04; W_Config[i]=W_VIC04; break;
	    case C_VIC05: R_Config[i]=R_VIC05; W_Config[i]=W_VIC05; break;
	    case C_VIC06: R_Config[i]=R_VIC06; W_Config[i]=W_VIC06; break;
	    case C_VIC07: R_Config[i]=R_VIC07; W_Config[i]=W_VIC07; break;
	    case C_VIC08: R_Config[i]=R_VIC08; W_Config[i]=W_VIC08; break;
	    case C_VIC09: R_Config[i]=R_VIC09; W_Config[i]=W_VIC09; break;
	    case C_VIC0A: R_Config[i]=R_VIC0A; W_Config[i]=W_VIC0A; break;
	    case C_VIC0B: R_Config[i]=R_VIC0B; W_Config[i]=W_VIC0B; break;
	    case C_VIC0C: R_Config[i]=R_VIC0C; W_Config[i]=W_VIC0C; break;
	    case C_VIC0D: R_Config[i]=R_VIC0D; W_Config[i]=W_VIC0D; break;
	    case C_VIC0E: R_Config[i]=R_VIC0E; W_Config[i]=W_VIC0E; break;
	    case C_VIC0F: R_Config[i]=R_VIC0F; W_Config[i]=W_VIC0F; break;
	    case C_VIC10: R_Config[i]=R_VIC10; W_Config[i]=W_VIC10; break;
	    case C_VIC11: R_Config[i]=R_VIC11; W_Config[i]=W_VIC11; break;
	    case C_VIC12: R_Config[i]=R_VIC12; W_Config[i]=W_VIC12; break;
	    case C_VIC13: R_Config[i]=R_VIC13; W_Config[i]=W_VIC13; break;
	    case C_VIC14: R_Config[i]=R_VIC14; W_Config[i]=W_VIC14; break;
	    case C_VIC15: R_Config[i]=R_VIC15; W_Config[i]=W_VIC15; break;
	    case C_VIC16: R_Config[i]=R_VIC16; W_Config[i]=W_VIC16; break;
	    case C_VIC17: R_Config[i]=R_VIC17; W_Config[i]=W_VIC17; break;
	    case C_VIC18: R_Config[i]=R_VIC18; W_Config[i]=W_VIC18; break;
	    case C_VIC19: R_Config[i]=R_VIC19; W_Config[i]=W_VIC19; break;
	    case C_VIC1A: R_Config[i]=R_VIC1A; W_Config[i]=W_VIC1A; break;
	    case C_VIC1B: R_Config[i]=R_VIC1B; W_Config[i]=W_VIC1B; break;
	    case C_VIC1C: R_Config[i]=R_VIC1C; W_Config[i]=W_VIC1C; break;
	    case C_VIC1D: R_Config[i]=R_VIC1D; W_Config[i]=W_VIC1D; break;
	    case C_VIC1E: R_Config[i]=R_VIC1E; W_Config[i]=W_VIC1E; break;
	    case C_VIC1F: R_Config[i]=R_VIC1F; W_Config[i]=W_VIC1F; break;
	    case C_VIC20: R_Config[i]=R_VIC20; W_Config[i]=W_VIC20; break;
	    case C_VIC21: R_Config[i]=R_VIC21; W_Config[i]=W_VIC21; break;
	    case C_VIC22: R_Config[i]=R_VIC22; W_Config[i]=W_VIC22; break;
	    case C_VIC23: R_Config[i]=R_VIC23; W_Config[i]=W_VIC23; break;
	    case C_VIC24: R_Config[i]=R_VIC24; W_Config[i]=W_VIC24; break;
	    case C_VIC25: R_Config[i]=R_VIC25; W_Config[i]=W_VIC25; break;
	    case C_VIC26: R_Config[i]=R_VIC26; W_Config[i]=W_VIC26; break;
	    case C_VIC27: R_Config[i]=R_VIC27; W_Config[i]=W_VIC27; break;
	    case C_VIC28: R_Config[i]=R_VIC28; W_Config[i]=W_VIC28; break;
	    case C_VIC29: R_Config[i]=R_VIC29; W_Config[i]=W_VIC29; break;
	    case C_VIC2A: R_Config[i]=R_VIC2A; W_Config[i]=W_VIC2A; break;
	    case C_VIC2B: R_Config[i]=R_VIC2B; W_Config[i]=W_VIC2B; break;
	    case C_VIC2C: R_Config[i]=R_VIC2C; W_Config[i]=W_VIC2C; break;
	    case C_VIC2D: R_Config[i]=R_VIC2D; W_Config[i]=W_VIC2D; break;
	    case C_VIC2E: R_Config[i]=R_VIC2E; W_Config[i]=W_VIC2E; break;

	    case C_SID00: R_Config[i]=R_SID  ; W_Config[i]=W_SID00; break;
	    case C_SID01: R_Config[i]=R_SID  ; W_Config[i]=W_SID01; break;
	    case C_SID02: R_Config[i]=R_SID  ; W_Config[i]=W_SID02; break;
	    case C_SID03: R_Config[i]=R_SID  ; W_Config[i]=W_SID03; break;
	    case C_SID04: R_Config[i]=R_SID  ; W_Config[i]=W_SID04; break;
	    case C_SID05: R_Config[i]=R_SID  ; W_Config[i]=W_SID05; break;
	    case C_SID06: R_Config[i]=R_SID  ; W_Config[i]=W_SID06; break;
	    case C_SID07: R_Config[i]=R_SID  ; W_Config[i]=W_SID07; break;
	    case C_SID08: R_Config[i]=R_SID  ; W_Config[i]=W_SID08; break;
	    case C_SID09: R_Config[i]=R_SID  ; W_Config[i]=W_SID09; break;
	    case C_SID0A: R_Config[i]=R_SID  ; W_Config[i]=W_SID0A; break;
	    case C_SID0B: R_Config[i]=R_SID  ; W_Config[i]=W_SID0B; break;
	    case C_SID0C: R_Config[i]=R_SID  ; W_Config[i]=W_SID0C; break;
	    case C_SID0D: R_Config[i]=R_SID  ; W_Config[i]=W_SID0D; break;
	    case C_SID0E: R_Config[i]=R_SID  ; W_Config[i]=W_SID0E; break;
	    case C_SID0F: R_Config[i]=R_SID  ; W_Config[i]=W_SID0F; break;
	    case C_SID10: R_Config[i]=R_SID  ; W_Config[i]=W_SID10; break;
	    case C_SID11: R_Config[i]=R_SID  ; W_Config[i]=W_SID11; break;
	    case C_SID12: R_Config[i]=R_SID  ; W_Config[i]=W_SID12; break;
	    case C_SID13: R_Config[i]=R_SID  ; W_Config[i]=W_SID13; break;
	    case C_SID14: R_Config[i]=R_SID  ; W_Config[i]=W_SID14; break;
	    case C_SID15: R_Config[i]=R_SID  ; W_Config[i]=W_SID15; break;
	    case C_SID16: R_Config[i]=R_SID  ; W_Config[i]=W_SID16; break;
	    case C_SID17: R_Config[i]=R_SID  ; W_Config[i]=W_SID17; break;
	    case C_SID18: R_Config[i]=R_SID  ; W_Config[i]=W_SID18; break;
	    case C_SID19: R_Config[i]=R_SID19; W_Config[i]=W_SID19; break;
	    case C_SID1A: R_Config[i]=R_SID1A; W_Config[i]=W_SID1A; break;
	    case C_SID1B: R_Config[i]=R_SID1B; W_Config[i]=W_SID1B; break;
	    case C_SID1C: R_Config[i]=R_SID1C; W_Config[i]=W_SID1C; break;

	    case C_CRAM: R_Config[i]=R_ColorRam; W_Config[i]=W_ColorRam; break;

	    case C_CIA1_0: R_Config[i]=R_CIA1_0; W_Config[i]=W_CIA1_0; break;
	    case C_CIA1_1: R_Config[i]=R_CIA1_1; W_Config[i]=W_CIA1_1; break;
	    case C_CIA1_2: R_Config[i]=R_CIA1_2; W_Config[i]=W_CIA1_2; break;
	    case C_CIA1_3: R_Config[i]=R_CIA1_3; W_Config[i]=W_CIA1_3; break;
	    case C_CIA1_4: R_Config[i]=R_CIA1_4; W_Config[i]=W_CIA1_4; break;
	    case C_CIA1_5: R_Config[i]=R_CIA1_5; W_Config[i]=W_CIA1_5; break;
	    case C_CIA1_6: R_Config[i]=R_CIA1_6; W_Config[i]=W_CIA1_6; break;
	    case C_CIA1_7: R_Config[i]=R_CIA1_7; W_Config[i]=W_CIA1_7; break;
	    case C_CIA1_8: R_Config[i]=R_CIA1_8; W_Config[i]=W_CIA1_8; break;
	    case C_CIA1_9: R_Config[i]=R_CIA1_9; W_Config[i]=W_CIA1_9; break;
	    case C_CIA1_A: R_Config[i]=R_CIA1_A; W_Config[i]=W_CIA1_A; break;
	    case C_CIA1_B: R_Config[i]=R_CIA1_B; W_Config[i]=W_CIA1_B; break;
	    case C_CIA1_C: R_Config[i]=R_CIA1_C; W_Config[i]=W_CIA1_C; break;
	    case C_CIA1_D: R_Config[i]=R_CIA1_D; W_Config[i]=W_CIA1_D; break;
	    case C_CIA1_E: R_Config[i]=R_CIA1_E; W_Config[i]=W_CIA1_E; break;
	    case C_CIA1_F: R_Config[i]=R_CIA1_F; W_Config[i]=W_CIA1_F; break;

	    case C_CIA2_0: R_Config[i]=R_CIA2_0; W_Config[i]=W_CIA2_0; break;
	    case C_CIA2_1: R_Config[i]=R_CIA2_1; W_Config[i]=W_CIA2_1; break;
	    case C_CIA2_2: R_Config[i]=R_CIA2_2; W_Config[i]=W_CIA2_2; break;
	    case C_CIA2_3: R_Config[i]=R_CIA2_3; W_Config[i]=W_CIA2_3; break;
	    case C_CIA2_4: R_Config[i]=R_CIA2_4; W_Config[i]=W_CIA2_4; break;
	    case C_CIA2_5: R_Config[i]=R_CIA2_5; W_Config[i]=W_CIA2_5; break;
	    case C_CIA2_6: R_Config[i]=R_CIA2_6; W_Config[i]=W_CIA2_6; break;
	    case C_CIA2_7: R_Config[i]=R_CIA2_7; W_Config[i]=W_CIA2_7; break;
	    case C_CIA2_8: R_Config[i]=R_CIA2_8; W_Config[i]=W_CIA2_8; break;
	    case C_CIA2_9: R_Config[i]=R_CIA2_9; W_Config[i]=W_CIA2_9; break;
	    case C_CIA2_A: R_Config[i]=R_CIA2_A; W_Config[i]=W_CIA2_A; break;
	    case C_CIA2_B: R_Config[i]=R_CIA2_B; W_Config[i]=W_CIA2_B; break;
	    case C_CIA2_C: R_Config[i]=R_CIA2_C; W_Config[i]=W_CIA2_C; break;
	    case C_CIA2_D: R_Config[i]=R_CIA2_D; W_Config[i]=W_CIA2_D; break;
	    case C_CIA2_E: R_Config[i]=R_CIA2_E; W_Config[i]=W_CIA2_E; break;
	    case C_CIA2_F: R_Config[i]=R_CIA2_F; W_Config[i]=W_CIA2_F; break;

	    case C_FREE:   R_Config[i]=R_Free;   W_Config[i]=W_Free;   break;

	    default:
		Cleanup();
		printf("Illegal config token %d\n",i);
		abort();
		break;
	}
    }
}

/*---------------------------------------------------------------------------*/

/*
**	Ram initial state.
*/
void RamReset(void)
{
    int p,i;

    for( p=0; p<256; ++p ) {		/* 256 Pages */
	for( i=0; i<256; i+=4 ) {	/* Emulate RAM initial states */
	    Ram[p*256+i+0]=0xFF;
	    Ram[p*256+i+1]=0xFF;
	    Ram[p*256+i+2]=0x00;
	    Ram[p*256+i+3]=0x00;
	}
	++p;
	for( i=0; i<256; i+=4 ) {
	    Ram[p*256+i+0]=0x00;
	    Ram[p*256+i+1]=0x00;
	    Ram[p*256+i+2]=0xFF;
	    Ram[p*256+i+3]=0xFF;
	}
    }
}

/*
**      Do hardware reset.
**      Called from processor reset.
*/
void HardwareReset(void)
{
    W_PPDir(0,0); W_PPData(1,0);	/* Reset processor port */

    Ram[0x8004]=0x00;			/* No CBM80 */

    VicReset();
    SidReset();

    memset(KeyMatrix,0xFF,256);		/* Clear keyboard matrix */

    memset(CIA1,0,16);			/* CIA 1 DC00 */
    memset(Cia1Alarm,0,4);
    Cia1TimerA=Cia1TimerB=0;
    Cia1IrqMask=0;
    Cia1TodBuffered=0;
    Cia1TodStopped=0;
    Cia1TimerA_Action=-1;
    Cia1TimerB_Action=-1;

    memset(CIA2,0,16);			/* CIA 2 DD00 */
    memset(Cia2Alarm,0,4);
    Cia2TimerA=Cia2TimerB=0;
    Cia2IrqMask=0;
    Cia2TodBuffered=0;
    Cia2TodStopped=0;
    Cia2TimerA_Action=-1;
    Cia2TimerB_Action=-1;

#ifdef IEC_HARD
    ResetSIEC();
#endif
    Init_IECDos();
#ifdef REAL1541
    ResetReal1541();
#endif

    ShowLeds(0111);			/* Show led status */

    Cycle=0;
    Action=0;
    ActionFunction=EmulVic;

#ifdef MONITOR
    MonAction=-1;
    MonFunction=0;
#endif
}

/*-----------------------------------------------------------------------------
 *      1541 Emulation
 *---------------------------------------------------------------------------*/

#define TRAP(x,v)	\
    { (x)+0, ILL02 },	\
    { (x)+1, (v) },

static struct patches {
    CONST unsigned short	Addr;
    CONST unsigned char		Data;
    unsigned char		Save;
} PatchTable[] = {
#if 0
    { 0xE1DA,8		},		/* Default load/save device */
    { 0xE1DC,1		}, 		/* Default load/save secondary */
    { 0xECED,0xD5	},
    { 0xECEE,0x3A	},		/* load+run -> load+rU: */
#endif
    TRAP(0xED09,PIEC_TALK)		/* Trap IEC-Routines */
    TRAP(0xEDC7,PIEC_SECTALK)
    TRAP(0xEDEF,PIEC_UNTALK)
    TRAP(0xEE13,PIEC_READ)

    TRAP(0xED0C,PIEC_LISTEN)
    TRAP(0xEDB9,PIEC_SECLISTEN)
    TRAP(0xEDFE,PIEC_UNLISTEN)
    TRAP(0xEDDD,PIEC_WRITE)
    { 0xEDEA, 0x58	}		/* CLI: for kernal trap return */
};

/*
**	Patch kernel.
*/
void PatchKernel(void)
{
    int i;

    if( FastLoaderOff ) {
	for( i=0; i<sizeof(PatchTable)/sizeof(*PatchTable); ++i ) {
	    KernelRom[PatchTable[i].Addr-ADR_KERNEL]=PatchTable[i].Data;
	}
    } else {
	for( i=0; i<sizeof(PatchTable)/sizeof(*PatchTable); ++i ) {
	    KernelRom[PatchTable[i].Addr-ADR_KERNEL]=PatchTable[i].Save;
	}
    }
}


/*
**	Initialise C64 1541 level.
*/
void Enter1541(void)
{
    int i;

    Init_IECDos();

#ifdef REAL1541
    EnterReal1541();			/* Complete 1541 emulation */
#endif

    /*
    **	Save patched kernel data.
    */
    for( i=0; i<sizeof(PatchTable)/sizeof(*PatchTable); ++i ) {
	PatchTable[i].Save=KernelRom[PatchTable[i].Addr-ADR_KERNEL];
    }

    PatchKernel();
}

/*
**	Called to set iec-status for kernal.
*/
static void ReallySetIECStatus(int st)
{
    st|=RBDP(0x90);
    WBDP(0x90,st);
}

/*
**       Emulator special instructions.
*/
void EmulatorTrap(int cmd)
{
    switch( cmd ) {
	case PIEC_TALK:
	    IEC_Talk(Integer(GlobalRegA));
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEE82;		/* CLI CLC RTS */
	    break;
	case PIEC_SECTALK:
	    WBDP(0x95,Integer(GlobalRegA));
	    IEC_SEC_Talk(Integer(GlobalRegA));
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEE82;		/* CLI */
	    break;
	case PIEC_UNTALK:
	    IEC_Untalk();
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEE82;		/* CLI CLC RTS */
	    break;
	case PIEC_READ:
	    WBDP(0xA4,IEC_Read());
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEE80;		/* LDA CLI CLC RTS */
	    break;
	case PIEC_LISTEN:
	    IEC_Listen(Integer(GlobalRegA));
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEE82;		/* CLI CLC RTS */
	    break;
	case PIEC_SECLISTEN:
	    WBDP(0x95,Integer(GlobalRegA));
	    IEC_SEC_Listen(Integer(GlobalRegA));
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEE82;		/* CLI CLC RTS */
	    break;
	case PIEC_UNLISTEN:
	    IEC_Unlisten();
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEE82;		/* CLI CLC RTS */
	    break;
	case PIEC_WRITE:
	    IEC_Write(Integer(GlobalRegA));
	    ReallySetIECStatus(IEC_Status);
	    GlobalRegPC=0xEDEA;		/* CLI STA CLC RTS */
	    break;

	case IEC_TALK:
	    IEC_Talk(Integer(GlobalRegA));	/* IEC TALK */
	    GlobalRegPC+=2;
	    break;
	case IEC_SECTALK:
	    IEC_SEC_Talk(Integer(GlobalRegA));
	    GlobalRegPC+=2;
	    break;
	case IEC_UNTALK:
	    IEC_Untalk();
	    GlobalRegPC+=2;
	    break;
	case IEC_READ:
	    LowByte(GlobalRegA)=IEC_Read();
	    GlobalRegPC+=2;
	    break;
	case IEC_LISTEN:
	    IEC_Listen(Integer(GlobalRegA));
	    GlobalRegPC+=2;
	    break;
	case IEC_SECLISTEN:
	    IEC_SEC_Listen(Integer(GlobalRegA));
	    GlobalRegPC+=2;
	    break;
	case IEC_UNLISTEN:
	    IEC_Unlisten();
	    GlobalRegPC+=2;
	    break;
	case IEC_WRITE:
	    IEC_Write(Integer(GlobalRegA));
	    GlobalRegPC+=2;
	    break;

	case IEC_STATUS:
	    LowByte(GlobalRegA)=IEC_Status;
	    GlobalRegPC+=2;
	    break;

	default:
	    printf("Illegal emulator trap %d at $%04X\n",cmd,GlobalRegPC);
#ifdef MONITOR
	    MonitorOn();
#else
	    Reset();
#endif
	    break;
    }
}

/*---------------------------------------------------------------------------*/

#ifdef PRINTER_SUPPORT
extern char* printer_printcmd;
#endif
static char* TheWhatVersion="@(#) " C64Version ;
static char* ExternalName;
static char* ImageName;
static char* AutoLoadName;
static char* C64Dir;
static char* C64Lib;
int DoEmulJoy = 0;
extern char* R1541ImageName;		/* 1541 ROM Image file name */

//extern	char*	getenv();
extern	int	defopen(CONST char*);
extern	char*	defread(CONST char*);

/*
**	Get default for options
**
**	1) Try command line option.
**	2) Try environment variable.
**	3) Try setting in defaults.
**	4) Try compiled setting.
*/
char* GetDefault(CONST char* name,char* arg,char* def)
{
    char* cp;

    if( arg ) {				/* Command line argument */
	cp=arg;
    } else if( (cp=getenv(name)) ) {	/* Environment */
	;
    } else if( !defopen(C64DEFAULT) 
	    || !defopen("alec64rc") ) {	/* /etc/default/file */
	if( !(cp=defread(name)) || !*cp || !cp[1] ) {
	    cp=def;			/* Compiled default */
	} else {
	    ++cp;
	}
	defopen(0L);
    } else {
	cp=def;				/* Compiled default */
    }
    return cp;
}

/*---------------------------------------------------------------------------*/

static void
fatal(char *str, ...)
{
    if (ProgramName)
	fprintf(stderr, "%s: ", ProgramName);
    vfprintf(stderr, str, &str + 1);
    exit(1);
}

int streq(char *a, char *b)
{
    while ((*((unsigned char *)a) & (unsigned char)0xDF) == ((*((unsigned char *)b)++) & (unsigned char)0xDF))
	if (*a++ == (unsigned char)0)
	    return 1;
    return 0;
}

FILE *
OpenLib(void)
{
    char *x;
    FILE *fp;
    
    if ( (x = GetDefault("C64LIB", C64Lib, C64LIB)) )
	if ( (fp = fopen(x, "r")) )
	    return fp;
    fatal("library '%s' not present.\n", x);
    return 0; /* not reached */
}

void
CloseLib(FILE *fp)
{
    fclose(fp);
}

char **
ReadLib(FILE *fp, char *buf, int size)
{
    extern char *strtok();
    static char *x[3];

    while (fgets(buf, size, fp))
    {
	if ((x[0] = strtok(buf, " \t\n")) && **x && **x != '#')
	    if ( (x[1] = strtok(0, "\t\n")) )
		if ( (x[2] = strtok(0, "\t\n")) )
		    if (strtok(0, "\t\n") == 0)
			return x;
    }
    return 0;
}

int
ListLib(void)
{
    int col = 0, cols = 0;
#ifdef TIOCWINSZ
    struct winsize ws;
#endif
    char **x, *y, buf[256];
    FILE *fp;

    if ( (y = getenv("COLUMNS")) )
	cols = atoi(y);

#ifdef TIOCWINSZ
    if (cols < 1 && ioctl(0, TIOCGWINSZ, &ws) == 0)
	cols = ws.ws_col;
#endif

    if (cols < 1)
	cols = 80;

    fp = OpenLib();
    while ( (x = ReadLib(fp, buf, sizeof(buf))) )
    {
	printf("%-14s", x[0]);
	col += 15;
	if (col + 15 < cols)
	    putchar(' ');
	else
	    putchar('\n'), col = 0;
    }
    CloseLib(fp);
    if (col)
	putchar('\n');
    exit(0);
    return 0; /* not reached */
}

/* Modifies C64Dir and sets AutoFeedStr */
int
RunLib(char *name)
{
    FILE *fp;
    char **x, buf[256];

    if (streq(name, "LIST"))
	return ListLib();

    fp = OpenLib();
    while ( (x = ReadLib(fp, buf, sizeof(buf))) )
	if (streq(x[0], name))
	{
	    C64Dir = strdup(x[1]);
	    AutoFeedStr = strdup(x[2]);
	    CloseLib(fp);
	    return 0;
	}
    CloseLib(fp);
    return -1;
}

/*** FIXME: { START OF CLEANED UP CODE ****/
/*---------------------------------------------------------------------------*/

/*
**	Format of the ROM image file:
**
**		8K Basic ROM
**		8K Kernel ROM
**		4K Character ROM
*/

/*
**	Read images.
*/
static void ReadImages(void)
{
    char* cp;
    int f;
    int i;
    unsigned char buf[2];
    unsigned l;

    /*
    **	Read Basic,Kernal,Character ROM.
    */
    cp=GetDefault("C64IMAGE",ImageName,C64IMAGE);
    f=open(cp,O_RDONLY|BINARY_FILE);
    if( f==-1 ) {
	f=open(C64IMAGE,O_RDONLY|BINARY_FILE);
	if( f==-1 ) {
	    printf("Can't find C64 ROM images\n");
	    Exit(1);
	}
    }

    if( read(f,BasicRom,8192)!=8192 ) {
	printf("Read error: Basic ROM\n");
	Exit(1);
    }
    if( read(f,KernelRom,8192)!=8192 ) {
	printf("Read error: Kernel ROM\n");
	Exit(1);
    }
    if( read(f,CharacterRom,4096)!=4096 ) {
	printf("Read error: Character ROM\n");
	Exit(1);
    }
    close(f);

    /*
    **	Read External ROM.
    */
    f=open(C64EXROM,O_RDONLY|BINARY_FILE);
    if( f==-1 ) {
	cp=GetDefault("C64EXROM",ExternalName,C64EXROM);
	f=open(cp,O_RDONLY|BINARY_FILE);
    }
    if( f!=-1 ) {
	i=read(f,ExternalRom,16348);
	if( i!=8192 && i!=16348 ) {
	    printf("Read error: External ROM\n");
	    Exit(1);
	}
	close(f);
    }

    /*
    **	Read autoload file.
    */
    f=open(C64AUTOLOAD,O_RDONLY|BINARY_FILE);
    if( f==-1 ) {
	cp=GetDefault("C64AUTOLOAD",AutoLoadName,C64AUTOLOAD);
	f=open(cp,O_RDONLY|BINARY_FILE);
	if( f==-1 ) {
	    return;
	}
    }
    read(f,buf,2);                      /* 2 Bytes address */
    l=buf[0]|(buf[1]<<8);
    read(f,Ram+l,64*1024-l);            /* Read until eof or end of Ram */
    close(f);
}

/*
**	Cleanup for program exit.
*/
void Cleanup(void)
{
    LeaveSound();
    LeaveJoy();
    LeaveKey();
    LeaveVideo();
}

/*
**	Called to exit the emulator.
*/
void Exit(int err)
{
    Cleanup();
    exit(err);
}

/*
**	Called on error.
*/
void Error(int err)
{
    Cleanup();
    printf("%d\n",err);
}

/*** FIXME: } END OF CLEANED UP CODE ****/

/*
**	Display usage of the c64 emulator.
*/
static void Usage(FILE *fp)
{
    fprintf(fp,"%s as of %s, Copyright 1992-96 by ALE\n",TheWhatVersion+5,__DATE__);
    fprintf(fp,"\tC64 Emulator written by Lutz Sammer and others.\n");
    fprintf(fp,"Usage: %s [-1|-2] [-a n] [-c n] [-d n] [-e n] [-f n] [-F n] [-g g] [-h] [-i n]\n\t[-I n] [-j] [-l lib] [-q] [-r n] [-s|-S] [-t n] [-u n] [-v n] [progid]\n",ProgramName);
    fprintf(fp,"\t-1 -2\tUse keypad for joystick 1 or 2.\n");
    fprintf(fp,"\t-a name\tPath and name of the C64 autoload file.\n");
    fprintf(fp,"\t-c dir\tChange directory to dir for the emulator.\n");
    fprintf(fp,"\t-d file\tUse file as diskette image.\n");
    fprintf(fp,"\t-e name\tPath and name of the C64 external ROM image.\n");
    fprintf(fp,"\t-f n\tVIC fetch to irq position.\n");
    fprintf(fp,"\t-F n\tAdd to VIC fetch to irq position for read.\n");
    fprintf(fp,"\t-g g\tGeometry of the X11 window.\n");
    fprintf(fp,"\t-h\tPrint this help page.\n");
    fprintf(fp,"\t-i name\tPath and name of the C64 ROM image.\n");
#ifdef REAL1541
    fprintf(fp,"\t-I name\tPath and name of the 1541 ROM image.\n");
#endif
    fprintf(fp,"\t-j\tUse digital hardware joystick.\n");
    fprintf(fp,"\t-l lib\tSpecify name of library database.\n");
    fprintf(fp,"\t-k keys\tC64 Keyboard input.\n");
    fprintf(fp,"\t-q\tQuiet, no sound emulation\n");
#ifdef REAL1541
    fprintf(fp,"\t-s -S\tSlow, PAL/NTSC  fast-loader 1541 emulation\n");
    fprintf(fp,"\t-t N\tTimeout for fast-loader 1541 emulation in N/50s\n");
#endif
    fprintf(fp,"\t-r n\tDisplay emulation recalculation (every n frames).\n");
    fprintf(fp,"\t-u n\tUpdate rate of X-display (see -r).\n");
    fprintf(fp,"\t-v n\tVideosync emulator to n%% of the original speed.\n");

    if( isatty(fileno(stdin)) && isatty(fileno(fp)) ) {
	fprintf(fp,"[PRESS RETURN TO CONTINUE]");
	fflush(fp);
	getchar();
    }

    fprintf(fp,"\tprogid\tProgram identifier to lookup in library file.\n\t\t\t(use list to show the contents).\n\n");

#ifdef PRINTER_SUPPORT
    fprintf(fp,"Environment:\t  C64IMAGE, C64AUTOLOAD, C64DIR, C64LIB, C64PRINTCMD, C64EXROM.\n");
    fprintf(fp,"/etc/default/c64: C64IMAGE, C64AUTOLOAD, C64DIR, C64LIB, C64PRINTCMD, C64EXROM.\n");
#else
    fprintf(fp,"Environment:\t  C64IMAGE, C64AUTOLOAD, C64DIR, C64LIB, C64EXROM.\n");
    fprintf(fp,"/etc/default/c64: C64IMAGE, C64AUTOLOAD, C64DIR, C64LIB, C64EXROM.\n");
#endif
    fprintf(fp,"Current Values:\n");
#ifdef REAL1541
    fprintf(fp,"R1541IMAGE=%s\n",
	GetDefault("R1541IMAGE",R1541ImageName,R1541IMAGE));
#endif
    fprintf(fp,"C64IMAGE=%s\n",GetDefault("C64IMAGE",ImageName,C64IMAGE));
    fprintf(fp,"C64AUTOLOAD=%s\n",
	GetDefault("C64AUTOLOAD",AutoLoadName,C64AUTOLOAD));
    fprintf(fp,"C64DIR=%s\n",GetDefault("C64DIR",C64Dir,"."));
    fprintf(fp,"C64LIB=%s\n",GetDefault("C64LIB",C64Lib,C64LIB));
#ifdef PRINTER_SUPPORT
    fprintf(fp,"C64PRINTCMD=%s\n",GetDefault("C64PRINTCMD",NULL,"lp -s -R %s"));
#endif
    fprintf(fp,"C64EXROM=%s\n",GetDefault("C64EXROM",ExternalName,C64EXROM));
    exit(fp == stderr ? 1 : 0);
}

/*
**	The C64 Emulator:
*/
int main(int argc,char** argv)
{
    int i;
    char *cp, *run, *geometry=0;
    extern char* optarg;
    extern int optind;
    extern int getopt(int,char* CONST*,CONST char*);
    char* timeout;

    // FIXME: change all to new option layout: timeout
    timeout=(char*)0;

    if( *argv && **argv ) {		/* Find out program name */
	if( (cp=strrchr(*argv,'/')) ) {	/* Remove leading path */
	    ++cp;
	} else {
	    cp=*argv;
	}
#ifdef __GO32__
	/*
	**	Remove ".exe" for dos version.
	*/
	if( (run=strrchr(cp,'.')) && !strcmp(run,".exe") ) {
	    *run='\0';
	}
#endif
    } else {
	cp="c64";
    }
    ProgramName=cp;

    for( ;; ) {				/* parse the arguments */
	switch( i=getopt(argc,argv,"12a:c:d:e:l:f:F:g:hi:I:jk:m:qr:sSt:u:v:x") ) {
	    case '1':			/* joystick 1 */
		JoystickGiven=1;
		JoyStick=&JoyStick1;
		OtherJoyStick=&JoyStick2;
		continue;
	    case '2':			/* joystick 2 */
		JoystickGiven=1;
		JoyStick=&JoyStick2;
		OtherJoyStick=&JoyStick1;
		continue;
	    case 'a':			/* autoload filename */
		AutoLoadName=optarg;
		continue;
	    case 'c':			/* change directory */
		C64Dir=optarg;
		continue;
	    case 'd':			/* diskette image */
		FloppyImage=optarg;
		continue;
	    case 'e':			/* external ROM filename */
		ExternalName=optarg;
		continue;
	    case 'f':			/* vic fetch position */
		VicFetch=atol(optarg);
		continue;
	    case 'F':			/* vic fetch read position */
		VicFetchAdd=atol(optarg);
		continue;
	    case 'g':			/* geometry */
	    	geometry=optarg;
		continue;
	    case 'h':			/* help */
		Usage(stdout);
	    case 'i':			/* image */
		ImageName=optarg;
		continue;
#ifdef REAL1541
	    case 'I':			/* 1541 image */
		R1541ImageName=optarg;
		continue;
#endif
	    case 'j':			/* joystick emulation */
		DoEmulJoy=1;
		continue;
	    case 'k':			/* keyboard input */
		AutoFeedStr=strdup(optarg);
		continue;
	    case 'l':			/* library */
		C64Lib=optarg;
		continue;
	    case 'm':			/* mystic */
		DoMystic=atol(optarg);
		continue;
	    case 'q':			/* quiet */
		SidSoundOff|=3;
		continue;
	    case 'r':			/* recalculation rate */
		VicEmulateRate=atol(optarg);
		continue;
#ifdef REAL1541
	    case 's':			/* slow: PAL fast-loader support */
		FastLoaderOff=0;
		continue;
	    case 'S':			/* slow: NTSC fast-loader support */
		FastLoaderOff=0;
		DoMystic=978;
		continue;
	    case 't':			/* serial timeout */
		timeout=optarg;
		continue;
#endif
	    case 'u': 			/* update display rate */
	    	VicRefreshRate=atol(optarg);
		continue;
	    case 'v': 			/* video sync rate */
	    	VideoSyncSpeed=atol(optarg);
		if( VideoSyncSpeed<1 ) {
		    VideoSyncSpeed=100;
		}
		continue;
	    case 'x':			/* disable eXtensions */
		ExtensionOff=1;
		continue;
	    case '?':
		Usage(stderr);
	    default:
		fatal("Please correct the code\n");
	    case -1:
		break;
	}
	break;
    }

#ifdef PRINTER_SUPPORT
    if( (cp=GetDefault("C64PRINTCMD",NULL,"lp -s -R %s")) ) {
	printer_printcmd=strdup(cp);
    }
#endif

    /*
    **	Parse the options.
    */
#ifdef REAL1541
    SerialTimeout=atoi(GetDefault("C64TIMEOUT",timeout,SERIAL_TIMEOUT));
#endif

    if (optind < argc)
	run = argv[optind++];
    else if (strcmp(ProgramName, "c64") != 0)
	run = ProgramName;
    else
	run = 0;

    if (optind < argc)
	Usage(stderr);

    /* first change to C64Dir (everything may be relative to it) */
    if ((cp = GetDefault("C64DIR", C64Dir, 0)) && chdir(cp) == -1)
	fatal("Can't chdir to '%s'\n", cp);

    if (run && RunLib(run) == -1)	/* will modify C64Dir! */
	fatal("cannot locate '%s'\n", run);

    if( !GetIoPriv() ) {
	// fatal("Cannot get IO-Privilege!\n");
	fprintf(stderr,"Cannot get IO-Privilege!\n");
	fprintf(stderr,"Sound and SVGALIB may not work correct\n");
#if defined(SID_SPEAKER) || defined(SID_OPL3)
	SidSoundOff=3;
#endif
    }

/*#ifndef DEBUG*/
/*
    for( i=0; i<NSIG; ++i )
	signal(i,Error);
*/    
    signal(SIGSEGV,Error);
#ifdef SIGIOT
    signal(SIGIOT,Error);
#endif

    signal(SIGINT,Exit);
    signal(SIGQUIT,Exit);
    signal(SIGHUP,Exit);
    signal(SIGTERM,Exit);
/*#endif*/

    InitConfig();			/* C64 RAM configurations */ 

    RamReset();				/* Fill RAM with 0xff 0x00 */
    ReadImages();			/* Read ROM images */
    Enter1541();			/* Patch kernel */
    EnterVideo(argc,argv,geometry);	/* Hardware dependend video init */
    EnterVic();				/* Video emulation */
    EnterKey();				/* Keyboard emulation */
    EnterJoy();				/* Joystick emulation */
    EnterSound();			/* Sound emulation */

    /*
    **	C64Dir may have been modified by RunLib()
    */
    if (run && C64Dir ) {
	/*
	**	Modified to support images from lib.c64. (JOHNS)
	*/
	if( chdir(C64Dir)==-1 ) {
	    if( FloppyImage || access(C64Dir,R_OK) ) {
		printf("Can't chdir to '%s'\n",C64Dir);
	    } else {
		FloppyImage=C64Dir;
		C64Dir=".";
	    }
	}
    }

    Reset();				/* C64 + 6502 Reset */
    Emul6502Loop();			/* 6502 Emulator loop */
    Cleanup();				/* Cleanup */
    exit(0);
    return 0;
}
