/*
 *	The C64 emulator
 *
 *	Copyright 1992-96 by ALE.
 *	written by Lutz Sammer.
 *
 *	6502/6510 Emulation
 *-----------------------------------------------------------------------------
 * $Id: 6502.c,v 1.16 1996/07/03 02:19:32 johns Exp root $
 * $Log: 6502.c,v $
 * Revision 1.16  1996/07/03 02:19:32  johns
 * Correct cycles for branches page crossings, ADC and SBC new decimal mode.
 * JAM instruction dedection, more illegal opcodes.
 *
 * Revision 1.15  1996/06/16 00:35:43  johns
 * Fixed ADC/SBC decimal mode bug.
 *
 * Revision 1.14  1996/06/01 22:31:47  johns
 * NOCSE for linux-aout changed, new interrupt handling.
 *
 * Revision 1.13  1996/05/03 17:24:58  johns
 * Use of label references (GNU), more illegal opcodes, adc+sbc rewritten.
 * Start of cleanup.
 *
 * Revision 1.12  1995/02/13  11:27:22  ari
 * second linux port ( by ari ) integration
 *
 * Revision 1.11  1994/12/18  00:32:11  root
 * non-gcc feature macro support, icc support
 *
 * Revision 1.10  1994/06/02  14:54:24  johns
 * version 1.07 ci.
 *
 * Revision 1.9  1993/09/17  11:27:39  johns
 * support for wide ( > 80 char ) disassembly
 *
 * Revision 1.8  1993/08/31  20:30:03  johns
 * go32 initial support
 *
 * Revision 1.7  1993/06/13  12:26:30  johns
 * cleanup
 *
 * Revision 1.6  1993/01/05  12:41:24  johns
 * Checkin before applying wuebbel patches
 *
 * Revision 1.5  1992/07/28  19:43:33  johns
 * Macros for local register and Sparc written.
 *
 * Revision 1.4  1992/07/20  04:15:46  johns
 * All inline functions becomes macros. Macros for local variables inserted.
 *
 * Revision 1.3  1992/07/13  04:36:26  johns
 * FetchOpcode Macro, Read reset vector after hardware reset.
 *
 * Revision 1.2  1992/07/11  21:52:24  johns
 * asm now macro
 *
 * Revision 1.1  1992/07/11  17:45:54  johns
 * Initial revision
 *
 *-----------------------------------------------------------------------------
 */

#include "c64.h"
#include "vic.h"
#include <stdio.h>

#ifdef M_UNIX
# ifdef __ICC
#   define NOCSE(str)
# else
#   define NOCSE(str) asm("/ " str)
# endif
#endif
#ifdef __linux__
# ifdef __ELF__
#   define NOCSE(str) asm("/ " str)
# else
#   define NOCSE(str) asm("\n/ " str)
# endif
#endif
#ifdef PCSOLARIS
#   define NOCSE(str) asm("/" str)
#endif
#ifdef SPARC
#   define NOCSE(str) asm("!" str)
#endif
#ifdef __GO32__
#   define NOCSE(str) asm("/* " str " */")
#endif

/*
**	Int:	0000-0000 0000-0000 0000-000C VVVV-VVVV
**
**	C:		Carry
**	VVVV-VVVV	Last value for testing on zero
*/
static union reg6502 GlobalLastVal;	/* Delayed flags calculation */

#ifdef BIG_ENDIAN			/* sparc */
#   define LastValZero	(LastVal.Byte[3])
#   define LastValCarry	(LastVal.Byte[2])
#endif
#ifdef LITTLE_ENDIAN			/* i386/i486 */
#   define LastValZero	(LastVal.Byte[0])
#   define LastValCarry	(LastVal.Byte[1])
#endif

static union reg6502 GlobalNegativeFlag;/* Last result negative flag */

union reg6502	GlobalRegA;		/* 6502 Register A */
union reg6502	GlobalRegX;		/* 6502 Register X */
union reg6502	GlobalRegY;		/* 6502 Register Y */
union reg6502	GlobalRegP;		/* 6502 Register Flags */
union reg6502	GlobalRegS;		/* 6502 Register Stack pointer */
unsigned	GlobalRegPC;		/* 6502 Processor counter */

int		IrqLine;		/* Interupt line state */
unsigned	Cycle;			/* Current 6502 cycle */
unsigned	Action;			/* Next hardware action */
#ifdef NEW_ACTION
unsigned	HwAction;		/* Next hardware action */
#endif
void		(*ActionFunction)(void);/* Hardware action function */

/*---------------------------------------------------------------------------*/

#define RegP		GlobalRegP
#define RegS		GlobalRegS
#define RegPC		GlobalRegPC
#define LastVal		GlobalLastVal
#define NegativeFlag	GlobalNegativeFlag

/*---------------------------------------------------------------------------*/

#define noNEW_OPCODE			/* little improvements */

#ifdef NEW_OPCODE

static unsigned (*FetchFunc)(void);

static unsigned RamFetch(void)
{
    return *(unsigned long*)(Ram+RegPC);
}

static unsigned BasicFetch(void)
{
    return *(unsigned long*)(BasicRom-ADR_BASIC+RegPC);
}

static unsigned KernelFetch(void)
{
    return *(unsigned long*)(KernelRom-ADR_KERNEL+RegPC);
}

static unsigned IOFetch(void)
{
    return RBMem(RegPC+0)|(RBMem(RegPC+1)<<8)|(RBMem(RegPC+2)<<16);
}

#define NewPC()	\
do{						\
    if( Config[RegPC]==C_RW_RAM )		\
	FetchFunc=RamFetch;			\
    else if( Config[RegPC]==C_KROM )		\
	FetchFunc=KernelFetch;			\
    else if( Config[RegPC]==C_BROM )		\
	FetchFunc=BasicFetch;			\
    else					\
	FetchFunc=IOFetch;			\
}while( 0 )

#else

#define NewPC()

#endif

/*---------------------------------------------------------------------------*/


/*
**	Returns the cached flags like flag register.
*/
#define CurrentFlags() \
	(((LowByte(RegP)&~(FLAG_NEGATIVE|FLAG_ZERO|FLAG_CARRY))		\
	    | LastValCarry				/* Carry */	\
	    | (LowByte(NegativeFlag)&FLAG_NEGATIVE)	/* Negative */	\
	    | ((LastValZero==0)<<1)			/* Zero */	\
	    | (FLAG_BREAK|FLAG_UNDEFINED))&0xFF)	/* Always set */

/*
**	Called if we need the cached flags in flag register.
*/
void UpdateFlags(void)
{
    LowByte(RegP)=CurrentFlags();
}

/*-----------------------------------------------------------------------------
 *	6502/6510 Vectors
 *---------------------------------------------------------------------------*/

/*
**	Handle CPU reset line.
*/
#define DoReset() \
    do{						\
	IrqLine=0;				\
	Cycle=6;				\
	HardwareReset();			\
	/* FIXME: Is this wrong ?? */		\
	RegP.Int&=FLAG_DECIMAL;			\
	RegP.Int|=FLAG_INTERRUPT;		\
	RegPC=RWVec(RST_VECTOR);		\
	NewPC();				\
    }while( 0 )

/*
**	Handle CPU NMI line.
*/
#define DoNmi() \
    do{						\
	PushW(RegPC);				\
	PushB(CurrentFlags());			\
	RegP.Int|=FLAG_INTERRUPT;		\
	RegPC=RWVec(NMI_VECTOR);		\
	NewPC();				\
	Cycle+=7;				\
    }while( 0 )

/*
**	Handle CPU IRQ line.
*/
#define DoIrq() \
    do{\
	PushW(RegPC);				\
	PushB((CurrentFlags()&~FLAG_BREAK));	\
	RegP.Int|=FLAG_INTERRUPT;		\
	RegPC=RWVec(IRQ_VECTOR);		\
	NewPC();				\
	Cycle+=7;				\
    }while( 0 )

/*---------------------------------------------------------------------------*/

/*
**	Handle reset pin.
*/
void Reset(void)
{
    DoReset();
}

/*
**	Handle nmi pin.
*/
void Nmi(void)
{
#if defined(X11) && defined(DEBUG) && 0
    printf("NMI at $%04X\n",RegPC);
#endif
    DoNmi();
}

/*
**	Handle irq pin.
**		The IRQ pin is level triggert.
*/
void Irq(void)
{
    IrqLine=1;
    if( Integer(RegP)&FLAG_INTERRUPT ) {/* interrupts disabled */
	return;
    }
    DoIrq();
}

#undef RegP
#undef RegS
#undef RegPC
#undef LastVal
#undef NegativeFlag

/*---------------------------------------------------------------------------*/

/*
 *	Locals for 6502 registers.
 *
 *	ILL02:	Before	Move local register -> global
 *		After	Move global register -> local
 *	Main loop:
 *		Cycle and Action are used in read & write functions
 *	Hardware emulation:
 *		Cycle, Action, RegP, RegPC, RegS local register -> global
 *		If monitor is on, all local registers are needed.
 */

#ifdef SPARC	/* { */

/*
**	THIS IS NOT CORRECT: IT'S A VERY OLD VERSION!!!
**	MANY READ-WRITE FUNCTIONS CAN NOW CHANGE REGISTERS.
*/

#   define LocalRegs	\
	register union reg6502	LocalLastVal;		\
	register union reg6502	LocalNegativeFlag;	\
	register union reg6502	LocalRegA;		\
	register union reg6502	LocalRegX;		\
	register union reg6502	LocalRegY;		\
	register union reg6502	LocalRegP;		\
	register union reg6502	LocalRegS;		\
	register unsigned	LocalRegPC;		\
	register unsigned*	ActionPointer;		\
	register unsigned*	CyclePointer

#   define LoadLocalRegs()	\
	LocalRegA	=GlobalRegA;			\
	LocalRegX	=GlobalRegX;			\
	LocalRegY	=GlobalRegY;			\
	LocalRegP	=GlobalRegP;			\
	LocalRegS	=GlobalRegS;			\
	LocalRegPC	=GlobalRegPC;			\
	LocalLastVal	=GlobalLastVal;			\
	LocalNegativeFlag=GlobalNegativeFlag;		\
	ActionPointer	=&Action;			\
	CyclePointer	=&Cycle

#   define SaveLocalRegs()	\
	GlobalRegA	=LocalRegA;			\
	GlobalRegX	=LocalRegX;			\
	GlobalRegY	=LocalRegY;			\
	GlobalRegP	=LocalRegP;			\
	GlobalRegS	=LocalRegS;			\
	GlobalRegPC	=LocalRegPC;			\
	GlobalLastVal	=LocalLastVal;			\
	GlobalNegativeFlag=LocalNegativeFlag

#   define LoadLocalRegsEmulTrap()			\
	LocalRegPC=GlobalRegPC

#   define SaveLocalRegsEmulTrap()			\
	GlobalRegA=LocalRegA

#ifdef MONITOR

#   define LoadLocalRegsHardware()			\
	LoadLocalRegs()

#   define SaveLocalRegsHardware()			\
	SaveLocalRegs()

#else	/* }{ MONITOR */

#   define LoadLocalRegsHardware()			\
	LocalRegP=GlobalRegP;				\
	LocalRegS=GlobalRegS;				\
	LocalRegPC=GlobalRegPC;				\
	LocalLastVal=GlobalLastVal;			\
	LocalNegativeFlag=GlobalNegativeFlag

#   define SaveLocalRegsHardware()			\
	GlobalRegP=LocalRegP;				\
	GlobalRegS=LocalRegS;				\
	GlobalRegPC=LocalRegPC;				\
	GlobalLastVal=LocalLastVal;			\
	GlobalNegativeFlag=LocalNegativeFlag

#endif	/* } !MONITOR */

#   define LastVal	LocalLastVal
#   define NegativeFlag	LocalNegativeFlag
#   define RegA		LocalRegA
#   define RegX		LocalRegX
#   define RegY		LocalRegY
#   define RegP		LocalRegP
#   define RegS		LocalRegS
#   define RegPC	LocalRegPC
#if 0
#   define Cycle	(*CyclePointer)
#   define Action	(*ActionPointer)
#endif

#endif	/* } SPARC */

#ifndef LocalRegs
#   define LastVal	GlobalLastVal
#   define NegativeFlag	GlobalNegativeFlag
#   define RegA		GlobalRegA
#   define RegX		GlobalRegX
#   define RegY		GlobalRegY
#   define RegP		GlobalRegP
#   define RegS		GlobalRegS
#   define RegPC	GlobalRegPC

#   define LocalRegs
#   define LoadLocalRegs()
#   define SaveLocalRegs()
#   define LoadLocalRegsEmulTrap()
#   define SaveLocalRegsEmulTrap()
#   define LoadLocalRegsHardware()
#   define SaveLocalRegsHardware()
#endif

/*-----------------------------------------------------------------------------
 *	Opcode and operands fetch
 *---------------------------------------------------------------------------*/

#if 1

#ifdef BIG_ENDIAN	/* { */

static unsigned char Opcode[3];

/*
 *	Fetch 3 bytes for complete opcode and operands
 */
#define FetchOpcode()	R3Mem(RegPC,Opcode)
/*
 *	Get opcode
 */
#define RBOpc()		(Opcode[0])
/*
 *	Get byte operand
 */
#define RBIns()		(Opcode[1])
/*
 *	Get word operand
 */
#define RWIns()		(Opcode[1]|(Opcode[2]<<8))

#endif	/* } BIG_ENDIAN */

#ifdef LITTLE_ENDIAN	/* { */

//static unsigned Opcode;		/* slower on pentium */

#ifdef NEW_OPCODE

//#define FetchOpcode()	Opcode=FetchFunc()

#define FetchOpcode()	Opcode=(long*)RegPC

#else

/*
**	Fetch 3 bytes for complete opcode and operands
*/
#define FetchOpcode()	RLMem(RegPC,Opcode)

#endif

/*
**	Get opcode
*/
#define RBOpc()		(Opcode&0xFF)
/*
**	Get byte operand
*/
#define RBIns()		((Opcode>>8)&0xFF)
/*
**	Get word operand
*/
#define RWIns()		((Opcode>>8)&0xFFFF)

#endif	/* } LITTLE_ENDIAN */

#else

/*
**	Fetch 3 bytes for complete opcode and operands
*/
#define FetchOpcode()

/*
**	Get opcode
*/
#define RBOpc()		RBMem(RegPC)

/*
**	Get byte operand
*/
#define RBIns()		RBMem(RegPC+1)

/*
**	Get word operand
*/
#define RWIns()		((RBMem(RegPC+1))|(RBMem(RegPC+2)<<8))

#endif

/*-----------------------------------------------------------------------------
 * Addressing modes
 *---------------------------------------------------------------------------*/

/*
 *	FIXME:	cross page boundary AX,AY,IY +1 Cycle
 *		Cycle+=(RBIns()+RegX.Int>>8);
 *	FIXME:	error incorrect read by AX,AY by page cross.
 *	DONE:	cross 64K AX,AY,IY, Memory emulator does it for us.
 *	DONE:	read-modify-write writes first unmodified.
 *		Memory interface does it for us.
 */
#define IM() RBIns()			/* Immediate: #$00 */
#define AB() RWIns()			/* Absolute: $0000 */
#define DP() RBIns()			/* Direct Page: $00 */
#define AX() (RWIns()+RegX.Int)		/* Absolute Indexed,X: $0000,X */
#define AY() (RWIns()+RegY.Int)		/* Absolute Indexed,Y: $0000,Y */
#define DX() ((RBIns()+RegX.Int)&0xFF)	/* DP Indexed,X: $00,X */
#define DY() ((RBIns()+RegY.Int)&0xFF)	/* DP Indexed,Y: $00,Y */
#define IX() RWDP((RBIns()+RegX.Int)&0xFF) /* DP Indexed Indirect,X: ($00,X) */
#define IY() (RWDP(RBIns())+RegY.Int)	/* DP Indirect Indexed,Y: ($00),Y */

/*---------------------------------------------------------------------------*/

/*
**	Sets NEGATIVE and ZERO flags.
*/
#define FlagNZ(v) \
    (LowByte(NegativeFlag)=LastValZero=(v))

/*
**	Sets NEGATIVE and ZERO and CARRY flags.
*/
#define FlagNZC(v) \
    LowByte(NegativeFlag)=LastVal.Int=(v)+1;

#if 0
/*
**	Relative branch.
*/
#define Branch() \
    (RegPC+=2+(char)RBIns()); \
    Cycle+=3

/* FIXME: Branch Page boundary crossed cycle 1+
    if( (RegPC&0xFF00)==((RegPC+d)&0xFF00) ) {
    } else {
	Cycle+=4;
    }
*/
#endif

#if 0
/* 308%
**	Relative branch. (with correct cycles)
*/
#define Branch() \
    {					\
	int o;				\
	o=RegPC;			\
	RegPC+=2+(char)RBIns();		\
	if( (o^RegPC)&0xFF00 ) {	\
	    printf("%0x %0x\n",o,RegPC);\
	    Cycle+=4;			\
	} else {			\
	    Cycle+=3;			\
	}				\
    }
#endif

/* 312%
**	Relative branch. (with correct cycles) RISC version
*/
#define Branch() \
    {					\
	int o;				\
	o=RegPC+2;			\
	RegPC=o+(char)RBIns();		\
	NewPC();			\
	Cycle+=3+((o^RegPC)>>8);	\
    }

/*---------------------------------------------------------------------------*/

/*
**	Add with carry, decimal and normal mode.
*/
#define Adc(w) \
{									\
    unsigned v;								\
    unsigned i;								\
									\
    v=(w);								\
    i=v+Integer(RegA)+LastValCarry;	/* + CARRY */			\
									\
    LowByte(RegP)=(RegP.Int&~FLAG_OVERFLOW)				\
	| ((( (RegA.Int^~v) & (RegA.Int^i) )>>1) & FLAG_OVERFLOW);	\
									\
    if( RegP.Int&FLAG_DECIMAL ) {	/* HELPME/FIXME: Decimal mode */\
	if( (v&0xF)+(Integer(RegA)&0xF)+LastValCarry>0x9 )		\
	    i+=0x06;			/* Overflow low bcd nibble */	\
	if( i>0x99 )							\
	    i+=0x60;			/* Overflow high bcd nibble */	\
    }									\
    LastVal.Int=i;			/* ZERO/CARRY */		\
    LowByte(NegativeFlag)=LowByte(RegA)=i;				\
}

/*
**	Sub with carry, decimal and normal mode.
*/
#define Sbc(w)	\
{									\
    unsigned v;								\
    unsigned i;							 	\
									\
    v=((w)^0xFF);							\
    i=v+Integer(RegA)+LastValCarry;	/* - CARRY */			\
									\
    LowByte(RegP)=(RegP.Int&~FLAG_OVERFLOW)				\
	| ((( (RegA.Int^~v) & (RegA.Int^i) )>>1) & FLAG_OVERFLOW);	\
									\
    if( RegP.Int&FLAG_DECIMAL ) {	/* HELPME/FIXME: Decimal mode */\
	/*printf("%x ",(v&0xF)+(Integer(RegA)&0xF)+LastValCarry);*/ \
	if( (((v&0xF)+(Integer(RegA)&0xF)+LastValCarry)&0xF)>9 )	\
	    i-=0x06;			/* Underflow low bcd nibble */	\
	if( (i&0xF0)>0x90 ) {						\
	    i-=0x60;			/* Underflow high bcd nibble */	\
	}								\
	/*printf("%x\n",i);*/	\
    }									\
    LastVal.Int=i;			/* ZERO/CARRY */		\
    LowByte(NegativeFlag)=LowByte(RegA)=i;				\
}

#ifdef __GNUC__
#define Asl(w)	\
({									\
    unsigned v;								\
									\
    LastVal.Int=v=(w)<<1;		/* Carry = Bit 7 */		\
    LowByte(NegativeFlag)=v;						\
    v;									\
})
#else
static unsigned __utmpvar__;
#define Asl(w)	\
	(LastVal.Int=__utmpvar__=(w)<<1,				\
	 LowByte(NegativeFlag)=__utmpvar__,				\
	 __utmpvar__)
#endif

#ifdef __GNUC__
#define Lsr(w)	\
({									\
    unsigned v;								\
									\
    v=(w);								\
    LastValCarry=v&1;			/* Carry = Bit 0 */		\
    LowByte(NegativeFlag)=LastValZero=v>>=1;				\
    v;									\
})
#else
#define Lsr(w)	\
	(__utmpvar__=(w),						\
	LastValCarry=__utmpvar__&1,					\
	LowByte(NegativeFlag)=LastValZero=__utmpvar__>>=1,		\
	__utmpvar__)
#endif

#ifdef __GNUC__
#define Rol(w)	\
({									\
    unsigned v;								\
									\
    v=((w)<<1)|LastValCarry;		/* Carry -> Bit 0 */		\
    LastVal.Int=v;			/* Carry = Bit 7 */		\
    LowByte(NegativeFlag)=v;						\
    v;									\
})
#else
#define Rol(w)	\
	(__utmpvar__=((w)<<1)|LastValCarry,				\
	LastVal.Int=__utmpvar__,					\
	LowByte(NegativeFlag)=__utmpvar__,				\
	__utmpvar__)
#endif

#ifdef __GNUC__
#define Ror(w)	\
({									\
    unsigned v;								\
									\
    v=(w)|(LastVal.Int&0x0100);		/* Carry -> Bit 7 after shift */\
    LastValCarry=v&1;			/* Carry = Bit 0 */		\
    LowByte(NegativeFlag)=LastValZero=v>>=1;				\
    v;									\
})
#else
#define Ror(w)	\
	(__utmpvar__=(w)|(LastVal.Int&0x0100),				\
	LastValCarry=__utmpvar__&1,					\
	LowByte(NegativeFlag)=LastValZero=__utmpvar__>>=1,		\
	__utmpvar__)
#endif

#define Bit(w)	\
{									\
    unsigned v;								\
									\
    v=(w);								\
    LowByte(NegativeFlag)=v;						\
    LastValZero=v&RegA.Int;						\
    LowByte(RegP)=(RegP.Int&~FLAG_OVERFLOW)|(v&FLAG_OVERFLOW);		\
}

/*----------------------------------------------------------------------------*/

#ifdef __GNUC__	/* { */

#define JumpTable	\
    static CONST void *CONST Labels[256] = {				\
	&&_BRK,		&&_ORA_IX,	&&_ILL02,	&&_ILL03,	\
	&&_ILL04,	&&_ORA_D,	&&_ASL_D,	&&_ILL07,	\
	&&_PHP,		&&_ORA_IM,	&&_ASL,		&&_ILL0B,	\
	&&_ILL0C,	&&_ORA_A,	&&_ASL_A,	&&_ILL0F,	\
	&&_BPL,		&&_ORA_IY,	&&_ILL12,	&&_ILL13,	\
	&&_ILL14,	&&_ORA_DX,	&&_ASL_DX,	&&_ILL17,	\
	&&_CLC,		&&_ORA_AY,	&&_ILL1A,	&&_ILL1B,	\
	&&_ILL1C,	&&_ORA_AX,	&&_ASL_AX,	&&_ILL1F,	\
	&&_JSR,		&&_AND_IX,	&&_ILL22,	&&_ILL23,	\
	&&_BIT_D,	&&_AND_D,	&&_ROL_D,	&&_ILL27,	\
	&&_PLP,		&&_AND_IM,	&&_ROL,		&&_ILL2B,	\
	&&_BIT_A,	&&_AND_A,	&&_ROL_A,	&&_ILL2F,	\
	&&_BMI,		&&_AND_IY,	&&_ILL32,	&&_ILL33,	\
	&&_ILL34,	&&_AND_DX,	&&_ROL_DX,	&&_ILL37,	\
	&&_SEC,		&&_AND_AY,	&&_ILL3A,	&&_ILL3B,	\
	&&_ILL3C,	&&_AND_AX,	&&_ROL_AX,	&&_ILL3F,	\
	&&_RTI,		&&_EOR_IX,	&&_ILL42,	&&_ILL43,	\
	&&_ILL44,	&&_EOR_D,	&&_LSR_D,	&&_ILL47,	\
	&&_PHA,		&&_EOR_IM,	&&_LSR,		&&_ILL4B,	\
	&&_JMP_A,	&&_EOR_A,	&&_LSR_A,	&&_ILL4F,	\
	&&_BVC,		&&_EOR_IY,	&&_ILL52,	&&_ILL53,	\
	&&_ILL54,	&&_EOR_DX,	&&_LSR_DX,	&&_ILL57,	\
	&&_CLI,		&&_EOR_AY,	&&_ILL5A,	&&_ILL5B,	\
	&&_ILL5C,	&&_EOR_AX,	&&_LSR_AX,	&&_ILL5F,	\
	&&_RTS,		&&_ADC_IX,	&&_ILL62,	&&_ILL63,	\
	&&_ILL64,	&&_ADC_D,	&&_ROR_D,	&&_ILL67,	\
	&&_PLA,		&&_ADC_IM,	&&_ROR,		&&_ILL6B,	\
	&&_JMP_AI,	&&_ADC_A,	&&_ROR_A,	&&_ILL6F,	\
	&&_BVS,		&&_ADC_IY,	&&_ILL72,	&&_ILL73,	\
	&&_ILL74,	&&_ADC_DX,	&&_ROR_DX,	&&_ILL77,	\
	&&_SEI,		&&_ADC_AY,	&&_ILL7A,	&&_ILL7B,	\
	&&_ILL7C,	&&_ADC_AX,	&&_ROR_AX,	&&_ILL7F,	\
	&&_ILL80,	&&_STA_IX,	&&_ILL82,	&&_ILL83,	\
	&&_STY_D,	&&_STA_D,	&&_STX_D,	&&_ILL87,	\
	&&_DEY,		&&_ILL89,	&&_TXA,		&&_ILL8B,	\
	&&_STY_A,	&&_STA_A,	&&_STX_A,	&&_ILL8F,	\
	&&_BCC,		&&_STA_IY,	&&_ILL92,	&&_ILL93,	\
	&&_STY_DX,	&&_STA_DX,	&&_STX_DY,	&&_ILL97,	\
	&&_TYA,		&&_STA_AY,	&&_TXS,		&&_ILL9B,	\
	&&_ILL9C,	&&_STA_AX,	&&_ILL9E,	&&_ILL9F,	\
	&&_LDY_IM,	&&_LDA_IX,	&&_LDX_IM,	&&_ILLA3,	\
	&&_LDY_D,	&&_LDA_D,	&&_LDX_D,	&&_ILLA7,	\
	&&_TAY,		&&_LDA_IM,	&&_TAX,		&&_ILLAB,	\
	&&_LDY_A,	&&_LDA_A,	&&_LDX_A,	&&_ILLAF,	\
	&&_BCS,		&&_LDA_IY,	&&_ILLB2,	&&_ILLB3,	\
	&&_LDY_DX,	&&_LDA_DX,	&&_LDX_DY,	&&_ILLB7,	\
	&&_CLV,		&&_LDA_AY,	&&_TSX,		&&_ILLBB,	\
	&&_LDY_AX,	&&_LDA_AX,	&&_LDX_AY,	&&_ILLBF,	\
	&&_CPY_IM,	&&_CMP_IX,	&&_ILLC2,	&&_ILLC3,	\
	&&_CPY_D,	&&_CMP_D,	&&_DEC_D,	&&_ILLC7,	\
	&&_INY,		&&_CMP_IM,	&&_DEX,		&&_ILLCB,	\
	&&_CPY_A,	&&_CMP_A,	&&_DEC_A,	&&_ILLCF,	\
	&&_BNE,		&&_CMP_IY,	&&_ILLD2,	&&_ILLD3,	\
	&&_ILLD4,	&&_CMP_DX,	&&_DEC_DX,	&&_ILLD7,	\
	&&_CLD,		&&_CMP_AY,	&&_ILLDA,	&&_ILLDB,	\
	&&_ILLDC,	&&_CMP_AX,	&&_DEC_AX,	&&_ILLDF,	\
	&&_CPX_IM,	&&_SBC_IX,	&&_ILLE2,	&&_ILLE3,	\
	&&_CPX_D,	&&_SBC_D,	&&_INC_D,	&&_ILLE7,	\
	&&_INX,		&&_SBC_IM,	&&_NOP,		&&_ILLEB,	\
	&&_CPX_A,	&&_SBC_A,	&&_INC_A,	&&_ILLEF,	\
	&&_BEQ,		&&_SBC_IY,	&&_ILLF2,	&&_ILLF3,	\
	&&_ILLF4,	&&_SBC_DX,	&&_INC_DX,	&&_ILLF7,	\
	&&_SED,		&&_SBC_AY,	&&_ILLFA,	&&_ILLFB,	\
	&&_ILLFC,	&&_SBC_AX,	&&_INC_AX,	&&_ILLFF	\
    };

#define Switch(x)	goto *Labels[x];
#define Case(x)		_##x:
#define Default
#define Break		goto break_label;
#define BreakLabel	break_label:

#else	/* }{ __GNUC__ */

#define JumpTable
#define Switch(x)	switch( x )
#define Case(x)		case x:
#define Default		default:
#define Break		break;
#define BreakLabel

#endif	/* } !__GNUC__ */

/*----------------------------------------------------------------------------*/

/*
**	BREAK in emulation switch.
*/
#define BREAK(x) \
    if( Cycle<Action )	\
	continue;	\
    NOCSE(x);		\
    Break;

/*
**	6502/6510 Emulator loop
*/
void Emul6502Loop(void)
{
    unsigned int Opcode;
    JumpTable;				/* LABEL POINTERS */
    LocalRegs;				/* LOCAL REGISTERS */

    LoadLocalRegs();
    for( ;; ) {
	/*
	**  6502 Emulator
	*/
	register unsigned a;
#ifdef JOHNS
	if( LastValCarry&~1 ) abort(); 
#endif

#ifdef _1541_
	if( 0 ) {
	    static int _flg_=0;

	    if( _flg_ ) {
		if( RegPC==0xFE84 )
		    _flg_=0; //abort();
		R1541Reg();
	    } else if( RegPC==0xFE67 ) {
		_flg_=1;
		R1541Reg();
	    }
	}
	if( 0 ) {
	    if( RegPC<0x800 ) {
		R1541Reg();
	    }
	}
#endif

	FetchOpcode();
	Switch( RBOpc() ) {

	    Case( ADC_IM )		/* ADC #$xx */
		Adc(IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("ADC_IM");
	    Case( ADC_A )
		Adc(RBMem(AB()));
		RegPC+=3;
		Cycle+=4;
		BREAK("ADC_A");
	    Case( ADC_D )
		Adc(RBDP(DP()));
		RegPC+=2;
		Cycle+=3;
		BREAK("ADC_D");
	    Case( ADC_AX )
		Adc(RBMem(AX()));
		RegPC+=3;
		Cycle+=4;
		BREAK("ADC_AX");
	    Case( ADC_AY )
		Adc(RBMem(AY()));
		RegPC+=3;
		Cycle+=4;
		BREAK("ADC_AY");
	    Case( ADC_DX )
		Adc(RBDP(DX()));
		RegPC+=2;
		Cycle+=4;
		BREAK("ADC_DX");
	    Case( ADC_IX )
		Adc(RBMem(IX()));
		RegPC+=2;
		Cycle+=6;
		BREAK("ADC_IX");
	    Case( ADC_IY )
		Adc(RBMem(IY()));
		RegPC+=2;
		Cycle+=5;
		BREAK("ADC_IY");

	    Case( AND_IM )
		FlagNZ(LowByte(RegA)&=IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("AND_IM");
	    Case( AND_A )
		FlagNZ(LowByte(RegA)&=RBMem(AB()));
		RegPC+=3;
		Cycle+=4;
		BREAK("AND_A");
	    Case( AND_D )
		FlagNZ(LowByte(RegA)&=RBDP(DP()));
		RegPC+=2;
		Cycle+=3;
		BREAK("AND_D");
	    Case( AND_AX )
		FlagNZ(LowByte(RegA)&=RBMem(AX()));
		RegPC+=3;
		Cycle+=4;
		BREAK("AND_AX");
	    Case( AND_AY )
		FlagNZ(LowByte(RegA)&=RBMem(AY()));
		RegPC+=3;
		Cycle+=4;
		BREAK("AND_AY");
	    Case( AND_DX )
		FlagNZ(LowByte(RegA)&=RBDP(DX()));
		RegPC+=2;
		Cycle+=4;
		BREAK("AND_DX");
	    Case( AND_IX )
		FlagNZ(LowByte(RegA)&=RBMem(IX()));
		RegPC+=2;
		Cycle+=6;
		BREAK("AND_IX");
	    Case( AND_IY )
		FlagNZ(LowByte(RegA)&=RBMem(IY()));
		RegPC+=2;
		Cycle+=5;
		BREAK("AND_IY");

	    Case( ASL )
		LowByte(RegA)=Asl(RegA.Int);
		RegPC++;
		Cycle+=2;
		BREAK("ASL");
	    Case( ASL_A )
		a=AB();
		WBMem(a,Asl(RBMem(a)));
		RegPC+=3;
		Cycle+=6;
		BREAK("ASL_A");
	    Case( ASL_D )
		a=DP();
		WBDP(a,Asl(RBDP(a)));
		RegPC+=2;
		Cycle+=5;
		BREAK("ASL_D");
	    Case( ASL_AX )
		a=AX();
		WBMem(a,Asl(RBMem(a)));
		RegPC+=3;
		Cycle+=7;
		BREAK("ASL_AX");
	    Case( ASL_DX )
		a=DX();
		WBDP(a,Asl(RBDP(a)));
		RegPC+=2;
		Cycle+=6;
		BREAK("ASL_DX");

	    Case( BCC )
		if( LastValCarry ) {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BCC_0");
		} else {
		    Branch();
		    BREAK("BCC_1");
		}
	    Case( BCS )
		if( LastValCarry ) {
		    Branch();
		    BREAK("BCS_0");
		} else {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BCS_1");
		}
	    Case( BEQ )
		if( LastValZero ) {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BEQ_0");
		} else {
		    Branch();
		    BREAK("BEQ_1");
		}

	    Case( BIT_A )
		Bit(RBMem(AB()));
		RegPC+=3;
		Cycle+=4;
		BREAK("BIT_A");
	    Case( BIT_D )
		Bit(RBDP(DP()));
		RegPC+=2;
		Cycle+=3;
		BREAK("BIT_D");

	    Case( BMI )
		if( NegativeFlag.Int&FLAG_NEGATIVE ) {
		    Branch();
		    BREAK("BMI_0");
		} else {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BMI_1");
		}
	    Case( BNE )
		if( LastValZero ) {
		    Branch();
		    BREAK("BNE_0");
		} else {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BNE_1");
		}
	    Case( BPL )
		if( NegativeFlag.Int&FLAG_NEGATIVE ) {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BPL_0");
		} else {
		    Branch();
		    BREAK("BPL_1");
		}

	    Case( BRK )
		a=RBOpc();
		a=RegPC+2;
		PushW(a);
		PushB(CurrentFlags());
		Integer(RegP)|=FLAG_INTERRUPT;
		RegPC=RWVec(IRQ_VECTOR);
		NewPC();
		Cycle+=7;
		BREAK("BRK");

	    Case( BVC )
#ifdef _1541_
		RegPC+=2;
		Cycle+=2;
		BREAK("BVC_0");
#else
		if( RegP.Int&FLAG_OVERFLOW ) {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BVC_0");
		} else {
		    Branch();
		    BREAK("BVC_1");
		}
#endif
	    Case( BVS )
#ifdef _1541_
		Branch();
		BREAK("BVS_0");
#else
		if( RegP.Int&FLAG_OVERFLOW ) {
		    Branch();
		    BREAK("BVS_0");
		} else {
		    RegPC+=2;
		    Cycle+=2;
		    BREAK("BVS_1");
		}
#endif

	    Case( CLC )
		LastValCarry=0;
		++RegPC;
		Cycle+=2;
		BREAK("CLC");
	    Case( CLD )
		RegP.Int&=~FLAG_DECIMAL;
		++RegPC;
		Cycle+=2;
		BREAK("CLD");
	    Case( CLI )
		RegP.Int&=~FLAG_INTERRUPT;
		++RegPC;
		Cycle+=2;
		if( IrqLine ) {		/* serve pending interrupts */
		    DoIrq();
		}
		BREAK("CLI");
	    Case( CLV )
		RegP.Int&=~FLAG_OVERFLOW;
#ifdef _1541_
		RegP.Int|=FLAG_OVERFLOW;/* Connected to DATA READY! */
#endif
		++RegPC;
		Cycle+=2;
		BREAK("CLV");

	    Case( CMP_IM )
		FlagNZC(RegA.Int+(IM()^0xFF));
		RegPC+=2;
		Cycle+=2;
		BREAK("CMP_IM");
	    Case( CMP_A )
		FlagNZC(RegA.Int+(RBMem(AB())^0xFF));
		RegPC+=3;
		Cycle+=4;
		BREAK("CMP_A");
	    Case( CMP_D )
		FlagNZC(RegA.Int+(RBDP(DP())^0xFF));
		RegPC+=2;
		Cycle+=3;
		BREAK("CMP_D");
	    Case( CMP_AX )
		FlagNZC(RegA.Int+(RBMem(AX())^0xFF));
		RegPC+=3;
		Cycle+=4;
		BREAK("CMP_AX");
	    Case( CMP_AY )
		FlagNZC(RegA.Int+(RBMem(AY())^0xFF));
		RegPC+=3;
		Cycle+=4;
		BREAK("CMP_AY");
	    Case( CMP_DX )
		FlagNZC(RegA.Int+(RBDP(DX())^0xFF));
		RegPC+=2;
		Cycle+=4;
		BREAK("CMP_DX");
	    Case( CMP_IX )
		FlagNZC(RegA.Int+(RBMem(IX())^0xFF));
		RegPC+=2;
		Cycle+=6;
		BREAK("CMP_IX");
	    Case( CMP_IY )
		FlagNZC(RegA.Int+(RBMem(IY())^0xFF));
		RegPC+=2;
		Cycle+=5;
		BREAK("CMP_IY");

	    Case( CPX_IM )
		FlagNZC(RegX.Int+(IM()^0xFF));
		RegPC+=2;
		Cycle+=2;
		BREAK("CPX_IM");
	    Case( CPX_A )
		FlagNZC(RegX.Int+(RBMem(AB())^0xFF));
		RegPC+=3;
		Cycle+=4;
		BREAK("CPX_A");
	    Case( CPX_D )
		FlagNZC(RegX.Int+(RBDP(DP())^0xFF));
		RegPC+=2;
		Cycle+=3;
		BREAK("CPX_D");

	    Case( CPY_IM )
		FlagNZC(RegY.Int+(IM()^0xFF));
		RegPC+=2;
		Cycle+=2;
		BREAK("CPY_IM");
	    Case( CPY_A )
		FlagNZC(RegY.Int+(RBMem(AB())^0xFF));
		RegPC+=3;
		Cycle+=4;
		BREAK("CPY_A");
	    Case( CPY_D )
		FlagNZC(RegY.Int+(RBDP(DP())^0xFF));
		RegPC+=2;
		Cycle+=3;
		BREAK("CPY_D");

	    Case( DEC_A )
		a=AB();
		WBMem(a,FlagNZ((RBMem(a)-1)&0xFF));
		RegPC+=3;
		Cycle+=6;
		BREAK("DEC_A");
	    Case( DEC_D )
		a=DP();
		WBDP(a,FlagNZ((RBDP(a)-1)&0xFF));
		RegPC+=2;
		Cycle+=5;
		BREAK("DEC_D");
	    Case( DEC_AX )
		a=AX();
		WBMem(a,FlagNZ((RBMem(a)-1)&0xFF));
		RegPC+=3;
		Cycle+=7;
		BREAK("DEC_AX");
	    Case( DEC_DX )
		a=DX();
		WBDP(a,FlagNZ((RBDP(a)-1)&0xFF));
		RegPC+=2;
		Cycle+=6;
		BREAK("DEC_DX");

	    Case( DEX )
		FlagNZ(--LowByte(RegX));
		++RegPC;
		Cycle+=2;
		BREAK("DEX");
	    Case( DEY )
		FlagNZ(--LowByte(RegY));
		++RegPC;
		Cycle+=2;
		BREAK("DEY");

	    Case( EOR_IM )
		LowByte(RegA)^=IM();
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=2;
		BREAK("EOR_IM");
	    Case( EOR_A )
		LowByte(RegA)^=RBMem(AB());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("EOR_A");
	    Case( EOR_D )
		LowByte(RegA)^=RBDP(DP());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=3;
		BREAK("EOR_D");
	    Case( EOR_AX )
		LowByte(RegA)^=RBMem(AX());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("EOR_AX");
	    Case( EOR_AY )
		LowByte(RegA)^=RBMem(AY());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("EOR_AY");
	    Case( EOR_DX )
		LowByte(RegA)^=RBDP(DX());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("EOR_DX");
	    Case( EOR_IX )
		LowByte(RegA)^=RBMem(IX());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("EOR_IX");
	    Case( EOR_IY )
		LowByte(RegA)^=RBMem(IY());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("EOR_IY");

	    Case( INC_A )
		a=AB();
		WBMem(a,FlagNZ((RBMem(a)+1)&0xFF));
		RegPC+=3;
		Cycle+=6;
		BREAK("INC_A");
	    Case( INC_D )
		a=DP();
		WBDP(a,FlagNZ((RBDP(a)+1)&0xFF));
		RegPC+=2;
		Cycle+=5;
		BREAK("INC_D");
	    Case( INC_AX )
		a=AX();
		WBMem(a,FlagNZ((RBMem(a)+1)&0xFF));
		RegPC+=3;
		Cycle+=7;
		BREAK("INC_AX");
	    Case( INC_DX )
		a=DX();
		WBDP(a,FlagNZ((RBDP(a)+1)&0xFF));
		RegPC+=2;
		Cycle+=6;
		BREAK("INC_DX");

	    Case( INX )
		FlagNZ(++LowByte(RegX));
		++RegPC;
		Cycle+=2;
		BREAK("INX");
	    Case( INY )
		FlagNZ(++LowByte(RegY));
		++RegPC;
		Cycle+=2;
		BREAK("INY");

	    Case( JMP_A )
		RegPC=AB();
		NewPC();
		Cycle+=3;
		BREAK("JMP_A");
	    Case( JMP_AI )
		a=AB();
		/* This code for xxFF 6502 bug */
		RegPC=RBMem(a)|(RBMem((a&0xFF00)|((a+1)&0x00FF))<<8);
		NewPC();
		Cycle+=5;
		BREAK("JMP_AI");

	    Case( JSR )
		a=RegPC+2;
		PushW(a);
		RegPC=AB();
		NewPC();
		Cycle+=6;
		BREAK("JSR");

	    Case( LDA_IM )
		FlagNZ(LowByte(RegA)=IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("LDA_IM");
	    Case( LDA_A )
		LowByte(RegA)=RBMem(AB());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("LDA_A");
	    Case( LDA_D )
		FlagNZ(LowByte(RegA)=RBDP(DP()));
		RegPC+=2;
		Cycle+=3;
		BREAK("LDA_D");
	    Case( LDA_AX )
		LowByte(RegA)=RBMem(AX());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("LDA_AX");
	    Case( LDA_AY )
		LowByte(RegA)=RBMem(AY());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("LDA_AY");
	    Case( LDA_DX )
		LowByte(RegA)=RBDP(DX());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("LDA_DX");
	    Case( LDA_IX )
		LowByte(RegA)=RBMem(IX());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("LDA_IX");
	    Case( LDA_IY )
		LowByte(RegA)=RBMem(IY());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("LDA_IY");

	    Case( LDX_IM )
		FlagNZ(LowByte(RegX)=IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("LDX_IM");
	    Case( LDX_A )
		FlagNZ(LowByte(RegX)=RBMem(AB()));
		RegPC+=3;
		Cycle+=4;
		BREAK("LDX_A");
	    Case( LDX_D )
		FlagNZ(LowByte(RegX)=RBDP(DP()));
		RegPC+=2;
		Cycle+=3;
		BREAK("LDX_D");
	    Case( LDX_AY )
		FlagNZ(LowByte(RegX)=RBMem(AY()));
		RegPC+=3;
		Cycle+=4;
		BREAK("LDX_AY");
	    Case( LDX_DY )
		FlagNZ(LowByte(RegX)=RBDP(DY()));
		RegPC+=2;
		Cycle+=4;
		BREAK("LDX_DY");

	    Case( LDY_IM )
		FlagNZ(LowByte(RegY)=IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("95");
	    Case( LDY_A )
		FlagNZ(LowByte(RegY)=RBMem(AB()));
		RegPC+=3;
		Cycle+=4;
		BREAK("96");
	    Case( LDY_D )
		FlagNZ(LowByte(RegY)=RBDP(DP()));
		RegPC+=2;
		Cycle+=3;
		BREAK("97");
	    Case( LDY_AX )
		FlagNZ(LowByte(RegY)=RBMem(AX()));
		RegPC+=3;
		Cycle+=4;
		BREAK("98");
	    Case( LDY_DX )
		FlagNZ(LowByte(RegY)=RBDP(DX()));
		RegPC+=2;
		Cycle+=4;
		BREAK("99");

	    Case( LSR )
		LowByte(RegA)=Lsr(RegA.Int);
		++RegPC;
		Cycle+=2;
		BREAK("100");
	    Case( LSR_A )
		a=AB();
		WBMem(a,Lsr(RBMem(a)));
		RegPC+=3;
		Cycle+=6;
		BREAK("101");
	    Case( LSR_D )
		a=DP();
		WBDP(a,Lsr(RBDP(a)));
		RegPC+=2;
		Cycle+=5;
		BREAK("LSR_D");
	    Case( LSR_AX )
		a=AX();
		WBMem(a,Lsr(RBMem(a)));
		RegPC+=3;
		Cycle+=7;
		BREAK("103");
	    Case( LSR_DX )
		a=DX();
		WBDP(a,Lsr(RBDP(a)));
		RegPC+=2;
		Cycle+=6;
		BREAK("LSR_DX");

	    Case( NOP )
		++RegPC;
		Cycle+=2;
		BREAK("NOP");

	    Case( ORA_IM )
		LowByte(RegA)|=IM();
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=2;
		BREAK("ORA_IM");
	    Case( ORA_A )
		LowByte(RegA)|=RBMem(AB());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("107");
	    Case( ORA_D )
		LowByte(RegA)|=RBDP(DP());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=3;
		BREAK("108");
	    Case( ORA_AX )
		LowByte(RegA)|=RBMem(AX());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("109");
	    Case( ORA_AY )
		LowByte(RegA)|=RBMem(AY());
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("110");
	    Case( ORA_DX )
		LowByte(RegA)|=RBDP(DX());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("111");
	    Case( ORA_IX )
		LowByte(RegA)|=RBMem(IX());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("ORA_IX");
	    Case( ORA_IY )
		LowByte(RegA)|=RBMem(IY());
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("ORA_IY");

	    Case( PHA )
		PushB(RegA.Int);
		++RegPC;
		Cycle+=3;
		BREAK("PHA");
	    Case( PHP )
		PushB(CurrentFlags());
		++RegPC;
		Cycle+=3;
		BREAK("PHP");
	    Case( PLA )
		PopB(LowByte(RegA));
		FlagNZ(RegA.Int);
		++RegPC;
		Cycle+=4;
		BREAK("PLA");
	    Case( PLP )
		PopB(LowByte(RegP));
		LowByte(NegativeFlag)=RegP.Int;
		LastValCarry=RegP.Int&FLAG_CARRY;
		LastValZero=(RegP.Int&FLAG_ZERO)^FLAG_ZERO;
		++RegPC;
		Cycle+=4;
		if( IrqLine && !(Integer(RegP)&FLAG_INTERRUPT) ) {
		    DoIrq();
		}
		BREAK("PLP");

	    Case( ROL )
		LowByte(RegA)=Rol(RegA.Int);
		++RegPC;
		Cycle+=2;
		BREAK("ROL");
	    Case( ROL_A )
		a=AB();
		WBMem(a,Rol(RBMem(a)));
		RegPC+=3;
		Cycle+=6;
		BREAK("119");
	    Case( ROL_D )
		a=DP();
		WBDP(a,Rol(RBDP(a)));
		RegPC+=2;
		Cycle+=5;
		BREAK("120");
	    Case( ROL_AX )
		a=AX();
		WBMem(a,Rol(RBMem(a)));
		RegPC+=3;
		Cycle+=7;
		BREAK("121");
	    Case( ROL_DX )
		a=DX();
		WBDP(a,Rol(RBDP(a)));
		RegPC+=2;
		Cycle+=6;
		BREAK("122");

	    Case( ROR )
		LowByte(RegA)=Ror(RegA.Int);
		++RegPC;
		Cycle+=2;
		BREAK("ROR");
	    Case( ROR_A )
		a=AB();
		WBMem(a,Ror(RBMem(a)));
		RegPC+=3;
		Cycle+=6;
		BREAK("ROR_A");
	    Case( ROR_D )
		a=DP();
		WBDP(a,Ror(RBDP(a)));
		RegPC+=2;
		Cycle+=5;
		BREAK("ROR_D");
	    Case( ROR_AX )
		a=AX();
		WBMem(a,Ror(RBMem(a)));
		RegPC+=3;
		Cycle+=7;
		BREAK("ROR_AX");
	    Case( ROR_DX )
		a=DX();
		WBDP(a,Ror(RBDP(a)));
		RegPC+=2;
		Cycle+=6;
		BREAK("ROR_DX");

	    Case( RTI )
		a=RBOpc();		/* 6502 fetches the next byte */
		PopB(LowByte(RegP));
		LowByte(NegativeFlag)=RegP.Int;
		LastValCarry=RegP.Int&FLAG_CARRY;
		LastValZero=(RegP.Int&FLAG_ZERO)^FLAG_ZERO;
		PopW(RegPC);
		NewPC();
		Cycle+=6;
		if( IrqLine && !(RegP.Int&FLAG_INTERRUPT) ) {
		    DoIrq();
		}
		BREAK("RTI");
	    Case( RTS )
		a=RBOpc();		/* 6502 fetches the next byte */
		PopW(a);
		RegPC=a+1;
		NewPC();
		Cycle+=6;
		BREAK("RTS");

	    Case( SBC_IM )
		Sbc(IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("SBC_IM");
	    Case( SBC_A )
		Sbc(RBMem(AB()));
		RegPC+=3;
		Cycle+=4;
		BREAK("SBC_A");
	    Case( SBC_D )
		Sbc(RBDP(DP()));
		RegPC+=2;
		Cycle+=3;
		BREAK("SBC_D");
	    Case( SBC_AX )
		Sbc(RBMem(AX()));
		RegPC+=3;
		Cycle+=4;
		BREAK("SBC_AX");
	    Case( SBC_AY )
		Sbc(RBMem(AY()));
		RegPC+=3;
		Cycle+=4;
		BREAK("SBC_AY");
	    Case( SBC_DX )
		Sbc(RBDP(DX()));
		RegPC+=2;
		Cycle+=4;
		BREAK("SBC_DX");
	    Case( SBC_IX )
		Sbc(RBMem(IX()));
		RegPC+=2;
		Cycle+=6;
		BREAK("SBC_IX");
	    Case( SBC_IY )
		Sbc(RBMem(IY()));
		RegPC+=2;
		Cycle+=5;
		BREAK("SBC_IY");

	    Case( SEC )
		LastValCarry=1;
		++RegPC;
		Cycle+=2;
		BREAK("SEC");

	    Case( SED )
		RegP.Int|=FLAG_DECIMAL;
		++RegPC;
		Cycle+=2;
		BREAK("SED");

	    Case( SEI )
		RegP.Int|=FLAG_INTERRUPT;
		++RegPC;
		Cycle+=2;
		BREAK("SEI");

	    Case( STA_A )
		WBMem(AB(),RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("STA_A");
	    Case( STA_D )
		WBDP(DP(),RegA.Int);
		RegPC+=2;
		Cycle+=3;
		BREAK("STA_D");
	    Case( STA_AX )
		WBMem(AX(),RegA.Int);
		RegPC+=3;
		Cycle+=5;
		BREAK("STA_AX");
	    Case( STA_AY )
		WBMem(AY(),RegA.Int);
		RegPC+=3;
		Cycle+=5;
		BREAK("STA_AY");
	    Case( STA_DX )
		WBDP(DX(),RegA.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("STA_DX");
	    Case( STA_IX )
		WBMem(IX(),RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("STA_IX");
	    Case( STA_IY )
		WBMem(IY(),RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("STA_IY");

	    Case( STX_A )
		WBMem(AB(),RegX.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("148");
	    Case( STX_D )
		WBDP(DP(),RegX.Int);
		RegPC+=2;
		Cycle+=3;
		BREAK("149");
	    Case( STX_DY )
		WBDP(DY(),RegX.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("150");

	    Case( STY_A )
		WBMem(AB(),RegY.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("151");
	    Case( STY_D )
		WBDP(DP(),RegY.Int);
		RegPC+=2;
		Cycle+=3;
		BREAK("152");
	    Case( STY_DX )
		WBDP(DX(),RegY.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("153");

	    Case( TAX )
		FlagNZ(RegX.Int=RegA.Int);
		++RegPC;
		Cycle+=2;
		BREAK("154");
	    Case( TAY )
		FlagNZ(LowByte(RegY)=RegA.Int);
		++RegPC;
		Cycle+=2;
		BREAK("155");
	    Case( TSX )
		FlagNZ(LowByte(RegX)=Integer(RegS));
		++RegPC;
		Cycle+=2;
		BREAK("156");
	    Case( TXA )
		FlagNZ(LowByte(RegA)=RegX.Int);
		++RegPC;
		Cycle+=2;
		BREAK("157");
	    Case( TXS )
		LowByte(RegS)=RegX.Int;
		++RegPC;
		Cycle+=2;
		BREAK("158");
	    Case( TYA )
		FlagNZ(LowByte(RegA)=RegY.Int);
		++RegPC;
		Cycle+=2;
		BREAK("159");

	    /*
	    **	Illegal 6502 instructions emulation:
	    */

	    Case( ILL02 )		/* EMULATOR TRAP CODE */
		if( RBIns() ) {
		    SaveLocalRegsEmulTrap();
		    EmulatorTrap(RBIns());
		    LoadLocalRegsEmulTrap();
		} else {
		    SaveLocalRegs();
		    return;
		}
		Cycle+=3;
		BREAK("ILL02");

	    Case( ILL12 ) Case( ILL22 ) Case( ILL32 ) Case( ILL42 )
	    Case( ILL52 ) Case( ILL62 ) Case( ILL72 ) Case( ILL92 )
	    Case( ILLB2 ) Case( ILLD2 ) Case( ILLF2 )
#ifdef X11
		printf("Illegal instruction at $%04X: $%02X\n",RegPC,RBOpc());
#endif
		{ char buf[40];
#ifdef _1541_
		sprintf(buf,"1541: JAM inst at %04X",RegPC);
#else
		sprintf(buf,"JAM inst at %04X",RegPC);
#endif
		VicMessage(buf,FRAMES_PER_SECOND*3);
		}
		DoReset();
		BREAK("ILL12...");

	    Case( ILL1A )		/* NOP 1 Byte */
	    Case( ILL3A ) Case( ILL5A ) Case( ILL7A ) Case( ILLDA )
	    Case( ILLFA )
		++RegPC;
		Cycle+=2;
		BREAK("ILL1A...");
	    Case( ILL80 )		/* NOP 2 Byte 2 clocks */
	    Case( ILL82 ) Case( ILL89 ) Case( ILLC2 ) Case( ILLE2 )
		RegPC+=2;
		Cycle+=2;
		BREAK("ILL80...");
	    Case( ILL04 )		/* NOP 2 Byte 3 clocks */
	    Case( ILL44 ) Case( ILL64 )
		RegPC+=2;
		Cycle+=3;
		BREAK("ILL04...");
	    Case( ILL14 )		/* NOP 2 Byte 4 clocks */
	    Case( ILL34 ) Case( ILL54 ) Case( ILL74 ) Case( ILLD4 )
	    Case( ILLF4 )
		RegPC+=2;
		Cycle+=4;
		BREAK("ILL14...");

	    Case( ILL0C )		/* NOP 3 Byte 4 clocks */
		RegPC+=3;
		Cycle+=4;
		BREAK("ILL0C");
	    Case( ILL1C )		/* NOP 3 Byte 5 clocks */
	    Case( ILL3C ) Case( ILL5C ) Case( ILL7C ) Case( ILLDC )
	    Case( ILLFC )
		RegPC+=3;
		Cycle+=5;
		BREAK("ILL1C...");

	    Case( ILL03 )		/* ASL ORA (,X) */
		a=IX();
		WBMem(a,Asl(RBMem(a)));
		LowByte(RegA)|=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("168");
	    Case( ILL13 )		/* ASL ORA (,Y) */
		a=IY();
		WBMem(a,Asl(RBMem(a)));
		LowByte(RegA)|=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("169");
	    Case( ILL0F )		/* ASL ORA $XXXX */
		a=AB();
		WBMem(a,Asl(RBMem(a)));
		LowByte(RegA)|=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=6;
		BREAK("170");
	    Case( ILL1F )		/* ASL ORA $XXXX,X */
		a=AX();
		WBMem(a,Asl(RBMem(a)));
		LowByte(RegA)|=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("171");
	    Case( ILL1B )		/* ASL ORA $XXXX,Y */
		a=AY();
		WBMem(a,Asl(RBMem(a)));
		LowByte(RegA)|=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("172");
	    Case( ILL07 )		/* ASL ORA $XX */
		a=DP();
		WBDP(a,Asl(RBDP(a)));
		LowByte(RegA)|=RBDP(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("173");
	    Case( ILL17 )		/* ASL ORA $XX,X */
		a=DX();
		WBDP(a,Asl(RBDP(a)));
		LowByte(RegA)|=RBDP(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("174");

	    Case( ILL23 )		/* ROL AND (,X) */
		a=IX();
		WBMem(a,Rol(RBMem(a)));
		LowByte(RegA)&=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("175");
	    Case( ILL33 )		/* ROL AND (,Y) */
		a=IY();
		WBMem(a,Rol(RBMem(a)));
		LowByte(RegA)&=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("176");
	    Case( ILL2F )		/* ROL AND $XXXX */
		a=AB();
		WBMem(a,Rol(RBMem(a)));
		LowByte(RegA)&=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=6;
		BREAK("177");
	    Case( ILL3F )		/* ROL AND $XXXX,X */
		a=AX();
		WBMem(a,Rol(RBMem(a)));
		LowByte(RegA)&=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("178");
	    Case( ILL3B )		/* ROL AND $XXXX,Y */
		a=AY();
		WBMem(a,Rol(RBMem(a)));
		LowByte(RegA)&=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("179");
	    Case( ILL27 )		/* ROL AND $XX */
		a=DP();
		WBDP(a,Rol(RBDP(a)));
		LowByte(RegA)&=RBDP(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("180");
	    Case( ILL37 )		/* ROL AND $XX,X */
		a=DX();
		WBDP(a,Rol(RBDP(a)));
		LowByte(RegA)&=RBDP(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("181");

	    Case( ILL43 )		/* LSR EOR (,X) */
		a=IX();
		WBMem(a,Lsr(RBMem(a)));
		LowByte(RegA)^=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("182");
	    Case( ILL53 )		/* LSR EOR (,Y) */
		a=IY();
		WBMem(a,Lsr(RBMem(a)));
		LowByte(RegA)^=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("183");
	    Case( ILL4F )		/* LSR EOR $XXXX */
		a=AB();
		WBMem(a,Lsr(RBMem(a)));
		LowByte(RegA)^=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=6;
		BREAK("184");
	    Case( ILL5F )		/* LSR EOR $XXXX,X */
		a=AX();
		WBMem(a,Lsr(RBMem(a)));
		LowByte(RegA)^=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("185");
	    Case( ILL5B )		/* LSR EOR $XXXX,Y */
		a=AY();
		WBMem(a,Lsr(RBMem(a)));
		LowByte(RegA)^=RBMem(a);
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("186");
	    Case( ILL47 )		/* LSR EOR $XX */
		a=DP();
		WBDP(a,Lsr(RBDP(a)));
		LowByte(RegA)^=RBDP(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("ILL47");
	    Case( ILL57 )		/* LSR EOR $XX,X */
		a=DX();
		WBDP(a,Lsr(RBDP(a)));
		LowByte(RegA)^=RBDP(a);
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("188");

	    Case( ILL63 )		/* ROR ADC (,X) */
		a=IX();
		WBMem(a,Ror(RBMem(a)));
		Adc(RBMem(a));
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("189");
	    Case( ILL73 )		/* ROR ADC (,Y) */
		a=IY();
		WBMem(a,Ror(RBMem(a)));
		Adc(RBMem(a));
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=8;
		BREAK("190");
	    Case( ILL6F )		/* ROR ADC $XXXX */
		a=AB();
		WBMem(a,Ror(RBMem(a)));
		Adc(RBMem(a));
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=6;
		BREAK("191");
	    Case( ILL7F )		/* ROR ADC $XXXX,X */
		a=AX();
		WBMem(a,Ror(RBMem(a)));
		Adc(RBMem(a));
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("192");
	    Case( ILL7B )		/* ROR ADC $XXXX,Y */
		a=AY();
		WBMem(a,Ror(RBMem(a)));
		Adc(RBMem(a));
		FlagNZ(RegA.Int);
		RegPC+=3;
		Cycle+=7;
		BREAK("193");
	    Case( ILL67 )		/* ROR ADC $XX */
		a=DP();
		WBDP(a,Ror(RBDP(a)));
		Adc(RBDP(a));
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("194");
	    Case( ILL77 )		/* ROR ADC $XX,X */
		a=DX();
		WBDP(a,Ror(RBDP(a)));
		Adc(RBDP(a));
		FlagNZ(RegA.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("ILL77");

	    Case( ILLC3 )		/* DEC CMP (,X) */
		a=IX();
		WBMem(a,RBMem(a)-1);
		FlagNZC(RegA.Int+(RBMem(a)^0xFF));
		RegPC+=2;
		Cycle+=8;
		BREAK("ILLC3");
	    Case( ILLD3 )		/* DEC CMP (,Y) */
		a=IY();
		WBMem(a,RBMem(a)-1);
		FlagNZC(RegA.Int+(RBMem(a)^0xFF));
		RegPC+=2;
		Cycle+=8;
		BREAK("ILLD3");
	    Case( ILLCF )		/* DEC CMP $XXXX */
		a=AB();
		WBMem(a,RBMem(a)-1);
		FlagNZC(RegA.Int+(RBMem(a)^0xFF));
		RegPC+=3;
		Cycle+=6;
		BREAK("ILLCF");
	    Case( ILLDF )		/* DEC CMP $XXXX,X */
		a=AX();
		WBMem(a,RBMem(a)-1);
		FlagNZC(RegA.Int+(RBMem(a)^0xFF));
		RegPC+=3;
		Cycle+=7;
		BREAK("ILLDF");
	    Case( ILLDB )		/* DEC CMP $XXXX,Y */
		a=AY();
		WBMem(a,RBMem(a)-1);
		FlagNZC(RegA.Int+(RBMem(a)^0xFF));
		RegPC+=3;
		Cycle+=7;
		BREAK("200");
	    Case( ILLC7 )		/* DEC CMP $XX */
		a=DP();
		WBDP(a,RBDP(a)-1);
		FlagNZC(RegA.Int+(RBDP(a)^0xFF));
		RegPC+=2;
		Cycle+=5;
		BREAK("ILLC7");
	    Case( ILLD7 )		/* DEC CMP $XX,X */
		a=DX();
		WBDP(a,RBDP(a)-1);
		FlagNZC(RegA.Int+(RBMem(a)^0xFF));
		RegPC+=2;
		Cycle+=6;
		BREAK("ILLD7");

	    Case( ILLE3 )		/* INC SBC (,X) */
		a=IX();
		WBMem(a,RBMem(a)+1);
		Sbc(RBMem(a));
		RegPC+=2;
		Cycle+=8;
		BREAK("ILLE3");
	    Case( ILLF3 )		/* INC SBC (,Y) */
		a=IY();
		WBMem(a,RBMem(a)+1);
		Sbc(RBMem(a));
		RegPC+=2;
		Cycle+=8;
		BREAK("ILLF3");
	    Case( ILLEF )		/* INC SBC $XXXX */
		a=AB();
		WBMem(a,RBMem(a)+1);
		Sbc(RBMem(a));
		RegPC+=3;
		Cycle+=6;
		BREAK("ILLEF");
	    Case( ILLFF )		/* INC SBC $XXXX,X */
		a=AX();
		WBMem(a,RBMem(a)+1);
		Sbc(RBMem(a));
		RegPC+=3;
		Cycle+=7;
		BREAK("ILLFF");
	    Case( ILLFB )		/* INC SBC $XXXX,Y */
		a=AY();
		WBMem(a,RBMem(a)+1);
		Sbc(RBMem(a));
		RegPC+=3;
		Cycle+=7;
		BREAK("ILLFB");
	    Case( ILLE7 )		/* INC SBC $XX */
		a=DP();
		WBDP(a,RBDP(a)+1);
		Sbc(RBDP(a));
		RegPC+=2;
		Cycle+=5;
		BREAK("ILLE7");
	    Case( ILLF7 )		/* INC SBC $XX,X */
		a=DX();
		WBDP(a,RBDP(a)+1);
		Sbc(RBDP(a));
		RegPC+=2;
		Cycle+=6;
		BREAK("ILLF7");

	    Case( ILLA3 )		/* LDA TAX (,X) */
		LowByte(RegX)=LowByte(RegA)=RBMem(IX());
		FlagNZ(RegX.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("ILLA3");
	    Case( ILLB3 )		/* LDA TAX (),Y */
		LowByte(RegX)=LowByte(RegA)=RBMem(IY());
		FlagNZ(RegX.Int);
		RegPC+=2;
		Cycle+=5;
		BREAK("ILLB3");
	    Case( ILLAF )		/* LDA TAX $XXXX */
		LowByte(RegX)=LowByte(RegA)=RBMem(AB());
		FlagNZ(RegX.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("ILLAF");
	    Case( ILLBF )		/* LDA TAX $XXXX,Y */
		LowByte(RegX)=LowByte(RegA)=RBMem(AY());
		FlagNZ(RegX.Int);
		RegPC+=3;
		Cycle+=5;
		BREAK("ILLBF");
	    Case( ILLA7 )		/* LDA TAX DP */
		LowByte(RegX)=LowByte(RegA)=RBDP(DP());
		FlagNZ(RegX.Int);
		RegPC+=2;
		Cycle+=3;
		BREAK("ILLA7");
	    Case( ILLB7 )		/* LDA TAX DP,Y */
		LowByte(RegX)=LowByte(RegA)=RBDP(DY());
		FlagNZ(RegX.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("ILLB7");

	    Case( ILL0B )		/* AND #$xx N -> C */
		FlagNZ(LowByte(RegA)&=IM());
		// LowByte(NegativeFlag)=LastValCarry<<7;
		LastValCarry=Integer(NegativeFlag)>>7;
		RegPC+=2;
		Cycle+=2;
		BREAK("ILL0B");

	    Case( ILL2B )		/* AND #$xx N -> C */
		FlagNZ(LowByte(RegA)&=IM());
		// LowByte(NegativeFlag)=LastValCarry<<7;
		LastValCarry=Integer(NegativeFlag)>>7;
		RegPC+=2;
		Cycle+=2;
		BREAK("ILL2B");

	    Case( ILL4B )		/* AND #$xx LSR */
		Lsr(RegA.Int&RBMem(IM()));
		RegPC+=2;
		Cycle+=2;
		BREAK("ILL4B");

	    /* Store A/X */
	    Case( ILL83 )		/* AND X STA ($XX,X) */
		WBMem(IX(),RegA.Int&RegX.Int);
		// proved to be incorrect: FlagNZ(RegA.Int&RegX.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("ILL83");

	    Case( ILL87 )		/* AND X STA $XX */
		WBDP(DP(),RegA.Int&RegX.Int);
		// proved to be incorrect: FlagNZ(RegA.Int&RegX.Int);
		RegPC+=2;
		Cycle+=3;
		BREAK("ILL87");

	    Case( ILL8F )		/* AND X STA $XXXX */
		WBMem(AB(),RegA.Int&RegX.Int);
		// proved to be incorrect: FlagNZ(RegA.Int&RegX.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("ILL8F");

	    Case( ILL93 )		/* SHA (xx),Y */
#if defined(X11) && defined(DEBUG)
		printf("ILL93: $%04X\n",RegPC);
#endif
		WBDP(IY(),RegA.Int&RegX.Int);
		RegPC+=2;
		Cycle+=6;
		BREAK("ILL93");

	    Case( ILL97 )		/* AND X STA $XX,Y */
		WBDP(DY(),RegA.Int&RegX.Int);
		// proved to be incorrect: FlagNZ(RegA.Int&RegX.Int);
		RegPC+=2;
		Cycle+=4;
		BREAK("ILL97");

	    Case( ILL9B )		/* SHS $xxYY,y  A&X&xx+1 -> SP */
#if defined(X11) && defined(DEBUG)
		printf("ILL9B: $%04X\n",RegPC);
#endif
		// FIXME: proved to be incorrect!
		// LowByte(RegS)=Integer(RegA)&Integer(RegX)&((RWIns()>>8)+1);
		LowByte(RegS)=Integer(RegA)&Integer(RegX);
		WBMem(AY(),Integer(RegS));
		RegPC+=3;
		Cycle+=5;
		BREAK("ILL9B");

	    Case( ILL9C )		/* AND Y AND #$xx+1 $xxYY,X */
		WBMem(AX(),Integer(RegA)&Integer(RegY)&((RWIns()>>8)+1));
		RegPC+=3;
		Cycle+=5;
		BREAK("ILL9C");

	    Case( ILL9E )		/* SHX $xxYY,Y Store X&xx+1 */
#if defined(X11) && defined(DEBUG)
		printf("ILL9E: $%04X\n",RegPC);
#endif
		WBMem(AY(),RegX.Int&((RWIns()>>8)+1));
		/* FIXME:
		if( ((RWIns()&0xFF)+RegY.Int)>0xFF ) {
		    a=RegX.Int&((RWIns()>>8)+1));
		} else {
		    a=AB();
		}
		WBMem(a+RegY.Int,RegX.Int&((RWIns()>>8)+1));
		*/
		RegPC+=3;
		Cycle+=5;
		BREAK("ILL9E");

	    Case( ILL9F )		/* AND X AND #$xx+1 $xxYY,Y */
#if defined(X11) && defined(DEBUG)
		printf("ILL9F: $%04X\n",RegPC);
#endif
		WBMem(AY(),RegA.Int&RegX.Int&((RWIns()>>8)+1));
		RegPC+=3;
		Cycle+=5;
		BREAK("ILL9F");

	    Case( ILLAB )		/* AND #$XX TAX */
		/* RANDOM OPCODE */
#if defined(X11) && defined(DEBUG)
		printf("Random opcode ILLAB: $%04X\n",RegPC);
#endif
		LowByte(RegX)=LowByte(RegA)=RegA.Int&IM()&Cycle;
		FlagNZ(RegX.Int);
		RegPC+=2;
		Cycle+=2;
		BREAK("ILLAB");

	    Case( ILLCB )		/* SBX A AND X - #$XX */
		FlagNZC(LowByte(RegX)=(RegA.Int&RegX.Int)-IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("ILLCB");

	    Case( ILLEB )		/* SBC #$XX */
		Sbc(IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("ILLEB");

	    Case( ILLBB )		/* LAS $xxYY,y AND SP */
		FlagNZ(
		    LowByte(RegS)=
		    LowByte(RegX)=
		    LowByte(RegA)=RBMem(AY())&RegA.Int);
		RegPC+=3;
		Cycle+=4;
		BREAK("ILLBB");

	    Case( ILL8B )		/* LDA #xx AND A OR EE AND X */
		FlagNZ(
		    LowByte(RegA)=((Integer(RegA)|0xEE) &IM()&Integer(RegX))
		);
		RegPC+=2;
		Cycle+=2;
		BREAK("ILL8B");

	    Case( ILL6B )		/* ROR #xx AND A */
		LowByte(RegA)=Ror(Integer(RegA)|IM());
		RegPC+=2;
		Cycle+=2;
		BREAK("ILL6B");

	    Default			/* should now never happen */
		Cleanup();
		printf("Illegal instruction at $%04X: $%02X\n",
		    RegPC,RBOpc());
		Dis(RegPC);
		printf("\n");
		abort();
	}
BreakLabel

	SaveLocalRegsHardware();	/* LOCAL -> GLOBAL */

	/*
	** Hardware emulation:
	**
	**	FIXME:	6502:	Interrupts. (Should be made??)
	**	CIA1:	Timer A,Timer B,TOD,SERIAL PORT.
	**	CIA2:	Timer A,Timer B,TOD,SERIAL PORT.
	**	VIC:	Displayline, Irq rasterline.
	**		Each VIC frame:	Keyboard, Joystick, SID.
	**	Monitor: Single Step, Breakpoint.
	*/
	do {
#ifdef _1541_
	    if( !ActionFunction ) {	/* 1541: enough cycles emulated */
		SaveLocalRegs();
		return;
	    }
#endif
	    (*ActionFunction)();
	} while( Cycle>=Action );	/* until no more hardware actions */

	LoadLocalRegsHardware();	/* GLOBAL -> LOCAL */
    }
}
