/****************************************************************/
/* 8051 Simulator by Alberto Firpo                              */
/*     e-mail: firpo@educ.disi.unige.it                         */
/****************************************************************/
#include "sim8051.h"
#include "mem8051.h"
#include "int8051.h"
#include "dis_io.h"
#include "sim_io.h"

/* extern declaration useb by do_step function */
extern void LCALL(INT16);

struct def_instr_t {
  void (*dis_fun)(INT16);
  void (*sim_fun)(INT16);
  int num_of_byte;
  int num_of_cicles;
};

static struct def_instr_t def_instr[256];

INT16 do_disassembling(INT16 addr, unsigned char * str)
{
   INT8 i;
   dis_start(str);
   dis_printADDRESS(addr);
   dis_printchar(' ');
   for(i=0; i<def_instr[CMEM[addr]].num_of_byte; i++)
   {
      dis_printINT8(CMEM[addr+i],NO_H);
      dis_printchar(' ');
   }
   dis_printchar('	');
   def_instr[CMEM[addr]].dis_fun(addr);
   dis_end();
   return addr+def_instr[CMEM[addr]].num_of_byte;
}

void do_step(void)
{
   register INT16 addr;
   sim_clear();
   if( addr = checkInterrupt() )
   {
      LCALL(addr);
      CYCLES += 2;
   }
   else
   {
      if( readDirect(PCON) & 0x1 )
         CYCLES += 1;
      else
      {
         addr = PC;
         PC += def_instr[CMEM[addr]].num_of_byte;
         def_instr[CMEM[addr]].sim_fun(addr);
         CYCLES += def_instr[CMEM[addr]].num_of_cicles;
      }
   }
   if (sim_iserror())
   {
      sim_notify_at(addr);
   }
}


/* External Simulation functions */
extern void sim_ACALL_PADDR(INT16 addr);
extern INT8 ADD(INT8,INT8);
extern INT8 ADD(INT8,INT8);
extern INT8 ADD(INT8,INT8);
extern INT8 ADD(INT8,INT8);
extern INT8 ADDC(INT8,INT8);
extern INT8 ADDC(INT8,INT8);
extern INT8 ADDC(INT8,INT8);
extern INT8 ADDC(INT8,INT8);
extern void sim_AJMP_PADDR(INT16 addr);
extern INT8 ANL(INT8,INT8);
extern INT8 ANL(INT8,INT8);
extern INT8 ANL(INT8,INT8);
extern INT8 ANL(INT8,INT8);
extern INT8 ANL(INT8,INT8);
extern INT8 ANL(INT8,INT8);
extern BIT ANLC(BIT,BIT);
extern BIT ANLC(BIT,BIT);
extern void CJNE(INT8,INT8,SINT8);
extern void CJNE(INT8,INT8,SINT8);
extern void CJNE(INT8,INT8,SINT8);
extern void CJNE(INT8,INT8,SINT8);
extern void sim_CLR_A(INT16 addr);
extern void sim_CLR_C(INT16 addr);
extern void sim_CLR_BADDR(INT16 addr);
extern INT8 CPLA(INT8);
extern BIT CPLBit(BIT);
extern BIT CPLBit(BIT);
extern INT8 DA(INT8);
extern void sim_DIV_AB(INT16 addr);
extern INT8 DJNZ(INT8,SINT8);
extern INT8 DJNZ(INT8,SINT8);
extern void JB(BIT,SINT8);
extern void sim_JBC_BADDR_RADDR(INT16 addr);
extern void JC(SINT8);
extern void JMPADP(INT16);
extern void JNB(BIT,SINT8);
extern void JNC(SINT8);
extern void JNZ(SINT8);
extern void JZ(SINT8);
extern void LCALL(INT16);
extern void LJMP(INT16);
extern void sim_MOV_DIR_DIR(INT16 addr);
extern void sim_MOVX_A_INDIR(INT16 addr);
extern void sim_MOVX_INDIR_A(INT16 addr);
extern void sim_NOP(INT16 addr);
extern void sim_MUL_AB(INT16 addr);
extern INT8 ORL(INT8,INT8);
extern INT8 ORL(INT8,INT8);
extern INT8 ORL(INT8,INT8);
extern INT8 ORL(INT8,INT8);
extern INT8 ORL(INT8,INT8);
extern INT8 ORL(INT8,INT8);
extern BIT ORLC(BIT,BIT);
extern BIT ORLC(BIT,BIT);
extern void sim_POP_DIR(INT16 addr);
extern void sim_PUSH_DIR(INT16 addr);
extern void sim_RET(INT16 addr);
extern void sim_RETI(INT16 addr);
extern void sim_RL_A(INT16 addr);
extern void sim_RLC_A(INT16 addr);
extern void sim_RR_A(INT16 addr);
extern void sim_RRC_A(INT16 addr);
extern void sim_SETB_C(INT16 addr);
extern void sim_SETB_BADDR(INT16 addr);
extern void SJMP(SINT8);
extern INT8 SUBB(INT8,INT8);
extern INT8 SUBB(INT8,INT8);
extern INT8 SUBB(INT8,INT8);
extern INT8 SUBB(INT8,INT8);
extern INT8 SWAPA(INT8);
extern void sim_XCH_A_REG(INT16 addr);
extern void sim_XCH_A_DIR(INT16 addr);
extern void sim_XCH_A_INDIR(INT16 addr);
extern void sim_XCHD_A_INDIR(INT16 addr);
extern INT8 XRL(INT8,INT8);
extern INT8 XRL(INT8,INT8);
extern INT8 XRL(INT8,INT8);
extern INT8 XRL(INT8,INT8);
extern INT8 XRL(INT8,INT8);
extern INT8 XRL(INT8,INT8);


/* Disassembling functions */
static void dis_RESERVED(INT16 addr)
{
   dis_print("RESERVED");
}

static void dis_okMOV_DIR_DIR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+2]);
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_ACALL_PADDR(INT16 addr)
{
   dis_print("ACALL");
   dis_printchar(' ');
   dis_printADDRESS(getPaddr(addr));
}

static void dis_ADD_A_REG(INT16 addr)
{
   dis_print("ADD");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_ADD_A_DIR(INT16 addr)
{
   dis_print("ADD");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_ADD_A_INDIR(INT16 addr)
{
   dis_print("ADD");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_ADD_A_IMM(INT16 addr)
{
   dis_print("ADD");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_ADDC_A_REG(INT16 addr)
{
   dis_print("ADDC");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_ADDC_A_DIR(INT16 addr)
{
   dis_print("ADDC");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_ADDC_A_INDIR(INT16 addr)
{
   dis_print("ADDC");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_ADDC_A_IMM(INT16 addr)
{
   dis_print("ADDC");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_AJMP_PADDR(INT16 addr)
{
   dis_print("AJMP");
   dis_printchar(' ');
   dis_printADDRESS(getPaddr(addr));
}

static void dis_ANL_A_REG(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_ANL_A_DIR(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_ANL_A_INDIR(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_ANL_A_IMM(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_ANL_DIR_A(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_print("A");
}

static void dis_ANL_DIR_IMM(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+2],WITH_H);
}

static void dis_ANL_C_BADDR(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_print("C");
   dis_printchar(',');
   dis_printBit(CMEM[addr+1]);
}

static void dis_ANL_C_NBADDR(INT16 addr)
{
   dis_print("ANL");
   dis_printchar(' ');
   dis_print("C");
   dis_printchar(',');
   dis_printchar('/');
   dis_printBit(CMEM[addr+1]);
}

static void dis_CJNE_A_DIR_RADDR(INT16 addr)
{
   dis_print("CJNE");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_CJNE_A_IMM_RADDR(INT16 addr)
{
   dis_print("CJNE");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_CJNE_REG_IMM_RADDR(INT16 addr)
{
   dis_print("CJNE");
   dis_printchar(' ');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_CJNE_INDIR_IMM_RADDR(INT16 addr)
{
   dis_print("CJNE");
   dis_printchar(' ');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_CLR_A(INT16 addr)
{
   dis_print("CLR");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_CLR_C(INT16 addr)
{
   dis_print("CLR");
   dis_printchar(' ');
   dis_print("C");
}

static void dis_CLR_BADDR(INT16 addr)
{
   dis_print("CLR");
   dis_printchar(' ');
   dis_printBit(CMEM[addr+1]);
}

static void dis_CPL_A(INT16 addr)
{
   dis_print("CPL");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_CPL_C(INT16 addr)
{
   dis_print("CPL");
   dis_printchar(' ');
   dis_print("C");
}

static void dis_CPL_BADDR(INT16 addr)
{
   dis_print("CPL");
   dis_printchar(' ');
   dis_printBit(CMEM[addr+1]);
}

static void dis_DA_A(INT16 addr)
{
   dis_print("DA");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_DEC_A(INT16 addr)
{
   dis_print("DEC");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_DEC_REG(INT16 addr)
{
   dis_print("DEC");
   dis_printchar(' ');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_DEC_DIR(INT16 addr)
{
   dis_print("DEC");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_DEC_INDIR(INT16 addr)
{
   dis_print("DEC");
   dis_printchar(' ');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_DIV_AB(INT16 addr)
{
   dis_print("DIV");
   dis_printchar(' ');
   dis_print("AB");
}

static void dis_DJNZ_REG_RADDR(INT16 addr)
{
   dis_print("DJNZ");
   dis_printchar(' ');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
   dis_printchar(',');
   dis_printADDRESS(addr+2+(SINT8)CMEM[addr+1]);
}

static void dis_DJNZ_DIR_RADDR(INT16 addr)
{
   dis_print("DJNZ");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_INC_A(INT16 addr)
{
   dis_print("INC");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_INC_REG(INT16 addr)
{
   dis_print("INC");
   dis_printchar(' ');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_INC_DIR(INT16 addr)
{
   dis_print("INC");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_INC_INDIR(INT16 addr)
{
   dis_print("INC");
   dis_printchar(' ');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_INC_DPTR(INT16 addr)
{
   dis_print("INC");
   dis_printchar(' ');
   dis_print("DPTR");
}

static void dis_JB_BADDR_RADDR(INT16 addr)
{
   dis_print("JB");
   dis_printchar(' ');
   dis_printBit(CMEM[addr+1]);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_JBC_BADDR_RADDR(INT16 addr)
{
   dis_print("JBC");
   dis_printchar(' ');
   dis_printBit(CMEM[addr+1]);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_JC_RADDR(INT16 addr)
{
   dis_print("JC");
   dis_printchar(' ');
   dis_printADDRESS(addr+2+(SINT8)CMEM[addr+1]);
}

static void dis_JMP_ATADPTR(INT16 addr)
{
   dis_print("JMP");
   dis_printchar(' ');
   dis_print("@A+DPTR");
}

static void dis_JNB_BADDR_RADDR(INT16 addr)
{
   dis_print("JNB");
   dis_printchar(' ');
   dis_printBit(CMEM[addr+1]);
   dis_printchar(',');
   dis_printADDRESS(addr+3+(SINT8)CMEM[addr+2]);
}

static void dis_JNC_RADDR(INT16 addr)
{
   dis_print("JNC");
   dis_printchar(' ');
   dis_printADDRESS(addr+2+(SINT8)CMEM[addr+1]);
}

static void dis_JNZ_RADDR(INT16 addr)
{
   dis_print("JNZ");
   dis_printchar(' ');
   dis_printADDRESS(addr+2+(SINT8)CMEM[addr+1]);
}

static void dis_JZ_RADDR(INT16 addr)
{
   dis_print("JZ");
   dis_printchar(' ');
   dis_printADDRESS(addr+2+(SINT8)CMEM[addr+1]);
}

static void dis_LCALL_CADDR(INT16 addr)
{
   dis_print("LCALL");
   dis_printchar(' ');
   dis_printADDRESS((CMEM[addr+1]<<8)+CMEM[addr+2]);
}

static void dis_LJMP_CADDR(INT16 addr)
{
   dis_print("LJMP");
   dis_printchar(' ');
   dis_printADDRESS((CMEM[addr+1]<<8)+CMEM[addr+2]);
}

static void dis_MOV_A_REG(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_MOV_A_DIR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_MOV_A_INDIR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_MOV_A_IMM(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_MOV_REG_A(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
   dis_printchar(',');
   dis_print("A");
}

static void dis_MOV_REG_DIR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_MOV_REG_IMM(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_MOV_DIR_A(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_print("A");
}

static void dis_MOV_DIR_REG(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_MOV_DIR_DIR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printDirect(CMEM[addr+2]);
}

static void dis_MOV_DIR_INDIR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_MOV_DIR_IMM(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+2],WITH_H);
}

static void dis_MOV_INDIR_A(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
   dis_printchar(',');
   dis_print("A");
}

static void dis_MOV_INDIR_DIR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_MOV_INDIR_IMM(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_MOV_C_BADDR(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("C");
   dis_printchar(',');
   dis_printBit(CMEM[addr+1]);
}

static void dis_MOV_BADDR_C(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_printBit(CMEM[addr+1]);
   dis_printchar(',');
   dis_print("C");
}

static void dis_MOV_DPTR_IMM16(INT16 addr)
{
   dis_print("MOV");
   dis_printchar(' ');
   dis_print("DPTR");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT16((CMEM[addr+1]<<8)+CMEM[addr+2],WITH_H);
}

static void dis_MOVC_A_ATADPTR(INT16 addr)
{
   dis_print("MOVC");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@A+DPTR");
}

static void dis_MOVC_A_ATAPC(INT16 addr)
{
   dis_print("MOVC");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@A+PC");
}

static void dis_MOVX_A_INDIR(INT16 addr)
{
   dis_print("MOVX");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_MOVX_INDIR_A(INT16 addr)
{
   dis_print("MOVX");
   dis_printchar(' ');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
   dis_printchar(',');
   dis_print("A");
}

static void dis_MOVX_A_ATDPTR(INT16 addr)
{
   dis_print("MOVX");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@DPTR");
}

static void dis_MOVX_ATDPTR_A(INT16 addr)
{
   dis_print("MOVX");
   dis_printchar(' ');
   dis_print("@DPTR");
   dis_printchar(',');
   dis_print("A");
}

static void dis_NOP(INT16 addr)
{
   dis_print("NOP");
}

static void dis_MUL_AB(INT16 addr)
{
   dis_print("MUL");
   dis_printchar(' ');
   dis_print("AB");
}

static void dis_ORL_A_REG(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_ORL_A_DIR(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_ORL_A_INDIR(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_ORL_A_IMM(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_ORL_DIR_A(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_print("A");
}

static void dis_ORL_DIR_IMM(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+2],WITH_H);
}

static void dis_ORL_C_BADDR(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_print("C");
   dis_printchar(',');
   dis_printBit(CMEM[addr+1]);
}

static void dis_ORL_C_NBADDR(INT16 addr)
{
   dis_print("ORL");
   dis_printchar(' ');
   dis_print("C");
   dis_printchar(',');
   dis_printchar('/');
   dis_printBit(CMEM[addr+1]);
}

static void dis_POP_DIR(INT16 addr)
{
   dis_print("POP");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_PUSH_DIR(INT16 addr)
{
   dis_print("PUSH");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_RET(INT16 addr)
{
   dis_print("RET");
}

static void dis_RETI(INT16 addr)
{
   dis_print("RETI");
}

static void dis_RL_A(INT16 addr)
{
   dis_print("RL");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_RLC_A(INT16 addr)
{
   dis_print("RLC");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_RR_A(INT16 addr)
{
   dis_print("RR");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_RRC_A(INT16 addr)
{
   dis_print("RRC");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_SETB_C(INT16 addr)
{
   dis_print("SETB");
   dis_printchar(' ');
   dis_print("C");
}

static void dis_SETB_BADDR(INT16 addr)
{
   dis_print("SETB");
   dis_printchar(' ');
   dis_printBit(CMEM[addr+1]);
}

static void dis_SJMP_RADDR(INT16 addr)
{
   dis_print("SJMP");
   dis_printchar(' ');
   dis_printADDRESS(addr+2+(SINT8)CMEM[addr+1]);
}

static void dis_SUBB_A_REG(INT16 addr)
{
   dis_print("SUBB");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_SUBB_A_DIR(INT16 addr)
{
   dis_print("SUBB");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_SUBB_A_INDIR(INT16 addr)
{
   dis_print("SUBB");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_SUBB_A_IMM(INT16 addr)
{
   dis_print("SUBB");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_SWAP_A(INT16 addr)
{
   dis_print("SWAP");
   dis_printchar(' ');
   dis_print("A");
}

static void dis_XCH_A_REG(INT16 addr)
{
   dis_print("XCH");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_XCH_A_DIR(INT16 addr)
{
   dis_print("XCH");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_XCH_A_INDIR(INT16 addr)
{
   dis_print("XCH");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_XCHD_A_INDIR(INT16 addr)
{
   dis_print("XCHD");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_XRL_A_REG(INT16 addr)
{
   dis_print("XRL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('R');
   dis_printchar('0'+getReg(addr));
}

static void dis_XRL_A_DIR(INT16 addr)
{
   dis_print("XRL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printDirect(CMEM[addr+1]);
}

static void dis_XRL_A_INDIR(INT16 addr)
{
   dis_print("XRL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_print("@R");
   dis_printchar('0'+(CMEM[addr] & 0x1));
}

static void dis_XRL_A_IMM(INT16 addr)
{
   dis_print("XRL");
   dis_printchar(' ');
   dis_print("A");
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+1],WITH_H);
}

static void dis_XRL_DIR_A(INT16 addr)
{
   dis_print("XRL");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_print("A");
}

static void dis_XRL_DIR_IMM(INT16 addr)
{
   dis_print("XRL");
   dis_printchar(' ');
   dis_printDirect(CMEM[addr+1]);
   dis_printchar(',');
   dis_printchar('#');
   dis_printINT8(CMEM[addr+2],WITH_H);
}



/* Simulating functions */
static void sim_RESERVED(INT16 addr)
{
}

static void sim_ADD_A_REG(INT16 addr)
{
   writeDirect(ACC,   ADD(readDirect(ACC),readReg(getReg(addr))));
}

static void sim_ADD_A_DIR(INT16 addr)
{
   writeDirect(ACC,   ADD(readDirect(ACC),readDirect(CMEM[addr+1])));
}

static void sim_ADD_A_INDIR(INT16 addr)
{
   writeDirect(ACC,   ADD(readDirect(ACC),readIndirect(readReg(CMEM[addr] & 0x1))));
}

static void sim_ADD_A_IMM(INT16 addr)
{
   writeDirect(ACC,   ADD(readDirect(ACC),CMEM[addr+1]));
}

static void sim_ADDC_A_REG(INT16 addr)
{
   writeDirect(ACC,   ADDC(readDirect(ACC),readReg(getReg(addr))));
}

static void sim_ADDC_A_DIR(INT16 addr)
{
   writeDirect(ACC,   ADDC(readDirect(ACC),readDirect(CMEM[addr+1])));
}

static void sim_ADDC_A_INDIR(INT16 addr)
{
   writeDirect(ACC,   ADDC(readDirect(ACC),readIndirect(readReg(CMEM[addr] & 0x1))));
}

static void sim_ADDC_A_IMM(INT16 addr)
{
   writeDirect(ACC,   ADDC(readDirect(ACC),CMEM[addr+1]));
}

static void sim_ANL_A_REG(INT16 addr)
{
   writeDirect(ACC,   ANL(readDirect(ACC),readReg(getReg(addr))));
}

static void sim_ANL_A_DIR(INT16 addr)
{
   writeDirect(ACC,   ANL(readDirect(ACC),readDirect(CMEM[addr+1])));
}

static void sim_ANL_A_INDIR(INT16 addr)
{
   writeDirect(ACC,   ANL(readDirect(ACC),readIndirect(readReg(CMEM[addr] & 0x1))));
}

static void sim_ANL_A_IMM(INT16 addr)
{
   writeDirect(ACC,   ANL(readDirect(ACC),CMEM[addr+1]));
}

static void sim_ANL_DIR_A(INT16 addr)
{
   writeDirect(CMEM[addr+1],   ANL(readDirect(CMEM[addr+1]),readDirect(ACC)));
}

static void sim_ANL_DIR_IMM(INT16 addr)
{
   writeDirect(CMEM[addr+1],   ANL(readDirect(CMEM[addr+1]),CMEM[addr+2]));
}

static void sim_ANL_C_BADDR(INT16 addr)
{
   writeBit(CARRY,   ANLC(readBit(CARRY),readBit(CMEM[addr+1])));
}

static void sim_ANL_C_NBADDR(INT16 addr)
{
   writeBit(CARRY,   ANLC(readBit(CARRY),((readBit(CMEM[addr+1]))?0:1)));
}

static void sim_CJNE_A_DIR_RADDR(INT16 addr)
{
   CJNE(readDirect(ACC),readDirect(CMEM[addr+1]),(SINT8)CMEM[addr+2]);
}

static void sim_CJNE_A_IMM_RADDR(INT16 addr)
{
   CJNE(readDirect(ACC),CMEM[addr+1],(SINT8)CMEM[addr+2]);
}

static void sim_CJNE_REG_IMM_RADDR(INT16 addr)
{
   CJNE(readReg(getReg(addr)),CMEM[addr+1],(SINT8)CMEM[addr+2]);
}

static void sim_CJNE_INDIR_IMM_RADDR(INT16 addr)
{
   CJNE(readIndirect(readReg(CMEM[addr] & 0x1)),CMEM[addr+1],(SINT8)CMEM[addr+2]);
}

static void sim_CPL_A(INT16 addr)
{
   writeDirect(ACC,   CPLA(readDirect(ACC)));
}

static void sim_CPL_C(INT16 addr)
{
   writeBit(CARRY,   CPLBit(readBit(CARRY)));
}

static void sim_CPL_BADDR(INT16 addr)
{
   writeBit(CMEM[addr+1],   CPLBit(readBit(CMEM[addr+1])));
}

static void sim_DA_A(INT16 addr)
{
   writeDirect(ACC,   DA(readDirect(ACC)));
}

static void sim_DEC_A(INT16 addr)
{
   writeDirect(ACC,readDirect(ACC)-1);
}

static void sim_DEC_REG(INT16 addr)
{
   writeReg(getReg(addr),readReg(getReg(addr))-1);
}

static void sim_DEC_DIR(INT16 addr)
{
   writeDirect(CMEM[addr+1],readDirect(CMEM[addr+1])-1);
}

static void sim_DEC_INDIR(INT16 addr)
{
   writeIndirect(readReg(CMEM[addr] & 0x1),readIndirect(readReg(CMEM[addr] & 0x1))-1);
}

static void sim_DJNZ_REG_RADDR(INT16 addr)
{
   writeReg(getReg(addr),   DJNZ(readReg(getReg(addr)),(SINT8)CMEM[addr+1]));
}

static void sim_DJNZ_DIR_RADDR(INT16 addr)
{
   writeDirect(CMEM[addr+1],   DJNZ(readDirect(CMEM[addr+1]),(SINT8)CMEM[addr+2]));
}

static void sim_INC_A(INT16 addr)
{
   writeDirect(ACC,readDirect(ACC)+1);
}

static void sim_INC_REG(INT16 addr)
{
   writeReg(getReg(addr),readReg(getReg(addr))+1);
}

static void sim_INC_DIR(INT16 addr)
{
   writeDirect(CMEM[addr+1],readDirect(CMEM[addr+1])+1);
}

static void sim_INC_INDIR(INT16 addr)
{
   writeIndirect(readReg(CMEM[addr] & 0x1),readIndirect(readReg(CMEM[addr] & 0x1))+1);
}

static void sim_INC_DPTR(INT16 addr)
{
   writeDPTR(readDPTR()+1);
}

static void sim_JB_BADDR_RADDR(INT16 addr)
{
   JB(readBit(CMEM[addr+1]),(SINT8)CMEM[addr+2]);
}

static void sim_JC_RADDR(INT16 addr)
{
   JC((SINT8)CMEM[addr+1]);
}

static void sim_JMP_ATADPTR(INT16 addr)
{
   JMPADP(CMEM[readDirect(ACC)+buildINT16(readDirect(DPH),readDirect(DPL))]);
}

static void sim_JNB_BADDR_RADDR(INT16 addr)
{
   JNB(readBit(CMEM[addr+1]),(SINT8)CMEM[addr+2]);
}

static void sim_JNC_RADDR(INT16 addr)
{
   JNC((SINT8)CMEM[addr+1]);
}

static void sim_JNZ_RADDR(INT16 addr)
{
   JNZ((SINT8)CMEM[addr+1]);
}

static void sim_JZ_RADDR(INT16 addr)
{
   JZ((SINT8)CMEM[addr+1]);
}

static void sim_LCALL_CADDR(INT16 addr)
{
   LCALL(buildINT16(CMEM[addr+1],CMEM[addr+2]));
}

static void sim_LJMP_CADDR(INT16 addr)
{
   LJMP(buildINT16(CMEM[addr+1],CMEM[addr+2]));
}

static void sim_MOV_A_REG(INT16 addr)
{
   writeDirect(ACC,readReg(getReg(addr)));
}

static void sim_MOV_A_DIR(INT16 addr)
{
   writeDirect(ACC,readDirect(CMEM[addr+1]));
}

static void sim_MOV_A_INDIR(INT16 addr)
{
   writeDirect(ACC,readIndirect(readReg(CMEM[addr] & 0x1)));
}

static void sim_MOV_A_IMM(INT16 addr)
{
   writeDirect(ACC,CMEM[addr+1]);
}

static void sim_MOV_REG_A(INT16 addr)
{
   writeReg(getReg(addr),readDirect(ACC));
}

static void sim_MOV_REG_DIR(INT16 addr)
{
   writeReg(getReg(addr),readDirect(CMEM[addr+1]));
}

static void sim_MOV_REG_IMM(INT16 addr)
{
   writeReg(getReg(addr),CMEM[addr+1]);
}

static void sim_MOV_DIR_A(INT16 addr)
{
   writeDirect(CMEM[addr+1],readDirect(ACC));
}

static void sim_MOV_DIR_REG(INT16 addr)
{
   writeDirect(CMEM[addr+1],readReg(getReg(addr)));
}

static void sim_MOV_DIR_INDIR(INT16 addr)
{
   writeDirect(CMEM[addr+1],readIndirect(readReg(CMEM[addr] & 0x1)));
}

static void sim_MOV_DIR_IMM(INT16 addr)
{
   writeDirect(CMEM[addr+1],CMEM[addr+2]);
}

static void sim_MOV_INDIR_A(INT16 addr)
{
   writeIndirect(readReg(CMEM[addr] & 0x1),readDirect(ACC));
}

static void sim_MOV_INDIR_DIR(INT16 addr)
{
   writeIndirect(readReg(CMEM[addr] & 0x1),readDirect(CMEM[addr+1]));
}

static void sim_MOV_INDIR_IMM(INT16 addr)
{
   writeIndirect(readReg(CMEM[addr] & 0x1),CMEM[addr+1]);
}

static void sim_MOV_C_BADDR(INT16 addr)
{
   writeBit(CARRY,readBit(CMEM[addr+1]));
}

static void sim_MOV_BADDR_C(INT16 addr)
{
   writeBit(CMEM[addr+1],readBit(CARRY));
}

static void sim_MOV_DPTR_IMM16(INT16 addr)
{
   writeDPTR(buildINT16(CMEM[addr+1],CMEM[addr+2]));
}

static void sim_MOVC_A_ATADPTR(INT16 addr)
{
   writeDirect(ACC,CMEM[readDirect(ACC)+buildINT16(readDirect(DPH),readDirect(DPL))]);
}

static void sim_MOVC_A_ATAPC(INT16 addr)
{
   writeDirect(ACC,CMEM[readDirect(ACC)+PC]);
}

static void sim_MOVX_A_ATDPTR(INT16 addr)
{
   writeDirect(ACC,readExtRam(readDPTR()));
}

static void sim_MOVX_ATDPTR_A(INT16 addr)
{
   writeExtRam(readDPTR(),readDirect(ACC));
}

static void sim_ORL_A_REG(INT16 addr)
{
   writeDirect(ACC,   ORL(readDirect(ACC),readReg(getReg(addr))));
}

static void sim_ORL_A_DIR(INT16 addr)
{
   writeDirect(ACC,   ORL(readDirect(ACC),readDirect(CMEM[addr+1])));
}

static void sim_ORL_A_INDIR(INT16 addr)
{
   writeDirect(ACC,   ORL(readDirect(ACC),readIndirect(readReg(CMEM[addr] & 0x1))));
}

static void sim_ORL_A_IMM(INT16 addr)
{
   writeDirect(ACC,   ORL(readDirect(ACC),CMEM[addr+1]));
}

static void sim_ORL_DIR_A(INT16 addr)
{
   writeDirect(CMEM[addr+1],   ORL(readDirect(CMEM[addr+1]),readDirect(ACC)));
}

static void sim_ORL_DIR_IMM(INT16 addr)
{
   writeDirect(CMEM[addr+1],   ORL(readDirect(CMEM[addr+1]),CMEM[addr+2]));
}

static void sim_ORL_C_BADDR(INT16 addr)
{
   writeBit(CARRY,   ORLC(readBit(CARRY),readBit(CMEM[addr+1])));
}

static void sim_ORL_C_NBADDR(INT16 addr)
{
   writeBit(CARRY,   ORLC(readBit(CARRY),((readBit(CMEM[addr+1]))?0:1)));
}

static void sim_SJMP_RADDR(INT16 addr)
{
   SJMP((SINT8)CMEM[addr+1]);
}

static void sim_SUBB_A_REG(INT16 addr)
{
   writeDirect(ACC,   SUBB(readDirect(ACC),readReg(getReg(addr))));
}

static void sim_SUBB_A_DIR(INT16 addr)
{
   writeDirect(ACC,   SUBB(readDirect(ACC),readDirect(CMEM[addr+1])));
}

static void sim_SUBB_A_INDIR(INT16 addr)
{
   writeDirect(ACC,   SUBB(readDirect(ACC),readIndirect(readReg(CMEM[addr] & 0x1))));
}

static void sim_SUBB_A_IMM(INT16 addr)
{
   writeDirect(ACC,   SUBB(readDirect(ACC),CMEM[addr+1]));
}

static void sim_SWAP_A(INT16 addr)
{
   writeDirect(ACC,   SWAPA(readDirect(ACC)));
}

static void sim_XRL_A_REG(INT16 addr)
{
   writeDirect(ACC,   XRL(readDirect(ACC),readReg(getReg(addr))));
}

static void sim_XRL_A_DIR(INT16 addr)
{
   writeDirect(ACC,   XRL(readDirect(ACC),readDirect(CMEM[addr+1])));
}

static void sim_XRL_A_INDIR(INT16 addr)
{
   writeDirect(ACC,   XRL(readDirect(ACC),readIndirect(readReg(CMEM[addr] & 0x1))));
}

static void sim_XRL_A_IMM(INT16 addr)
{
   writeDirect(ACC,   XRL(readDirect(ACC),CMEM[addr+1]));
}

static void sim_XRL_DIR_A(INT16 addr)
{
   writeDirect(CMEM[addr+1],   XRL(readDirect(CMEM[addr+1]),readDirect(ACC)));
}

static void sim_XRL_DIR_IMM(INT16 addr)
{
   writeDirect(CMEM[addr+1],   XRL(readDirect(CMEM[addr+1]),CMEM[addr+2]));
}



/* Array initialization */
void init_instTab(void)
{
  def_instr[0xA5].dis_fun = dis_RESERVED;
  def_instr[0xA5].sim_fun = sim_RESERVED;
  def_instr[0xA5].num_of_byte = 1;
  def_instr[0xA5].num_of_cicles = 0;
  def_instr[0x11].dis_fun = dis_ACALL_PADDR;
  def_instr[0x11].sim_fun = sim_ACALL_PADDR;
  def_instr[0x11].num_of_byte = 2;
  def_instr[0x11].num_of_cicles = 2;
  def_instr[0x31].dis_fun = dis_ACALL_PADDR;
  def_instr[0x31].sim_fun = sim_ACALL_PADDR;
  def_instr[0x31].num_of_byte = 2;
  def_instr[0x31].num_of_cicles = 2;
  def_instr[0x51].dis_fun = dis_ACALL_PADDR;
  def_instr[0x51].sim_fun = sim_ACALL_PADDR;
  def_instr[0x51].num_of_byte = 2;
  def_instr[0x51].num_of_cicles = 2;
  def_instr[0x71].dis_fun = dis_ACALL_PADDR;
  def_instr[0x71].sim_fun = sim_ACALL_PADDR;
  def_instr[0x71].num_of_byte = 2;
  def_instr[0x71].num_of_cicles = 2;
  def_instr[0x91].dis_fun = dis_ACALL_PADDR;
  def_instr[0x91].sim_fun = sim_ACALL_PADDR;
  def_instr[0x91].num_of_byte = 2;
  def_instr[0x91].num_of_cicles = 2;
  def_instr[0xB1].dis_fun = dis_ACALL_PADDR;
  def_instr[0xB1].sim_fun = sim_ACALL_PADDR;
  def_instr[0xB1].num_of_byte = 2;
  def_instr[0xB1].num_of_cicles = 2;
  def_instr[0xD1].dis_fun = dis_ACALL_PADDR;
  def_instr[0xD1].sim_fun = sim_ACALL_PADDR;
  def_instr[0xD1].num_of_byte = 2;
  def_instr[0xD1].num_of_cicles = 2;
  def_instr[0xF1].dis_fun = dis_ACALL_PADDR;
  def_instr[0xF1].sim_fun = sim_ACALL_PADDR;
  def_instr[0xF1].num_of_byte = 2;
  def_instr[0xF1].num_of_cicles = 2;
  def_instr[0x28].dis_fun = dis_ADD_A_REG;
  def_instr[0x28].sim_fun = sim_ADD_A_REG;
  def_instr[0x28].num_of_byte = 1;
  def_instr[0x28].num_of_cicles = 1;
  def_instr[0x29].dis_fun = dis_ADD_A_REG;
  def_instr[0x29].sim_fun = sim_ADD_A_REG;
  def_instr[0x29].num_of_byte = 1;
  def_instr[0x29].num_of_cicles = 1;
  def_instr[0x2A].dis_fun = dis_ADD_A_REG;
  def_instr[0x2A].sim_fun = sim_ADD_A_REG;
  def_instr[0x2A].num_of_byte = 1;
  def_instr[0x2A].num_of_cicles = 1;
  def_instr[0x2B].dis_fun = dis_ADD_A_REG;
  def_instr[0x2B].sim_fun = sim_ADD_A_REG;
  def_instr[0x2B].num_of_byte = 1;
  def_instr[0x2B].num_of_cicles = 1;
  def_instr[0x2C].dis_fun = dis_ADD_A_REG;
  def_instr[0x2C].sim_fun = sim_ADD_A_REG;
  def_instr[0x2C].num_of_byte = 1;
  def_instr[0x2C].num_of_cicles = 1;
  def_instr[0x2D].dis_fun = dis_ADD_A_REG;
  def_instr[0x2D].sim_fun = sim_ADD_A_REG;
  def_instr[0x2D].num_of_byte = 1;
  def_instr[0x2D].num_of_cicles = 1;
  def_instr[0x2E].dis_fun = dis_ADD_A_REG;
  def_instr[0x2E].sim_fun = sim_ADD_A_REG;
  def_instr[0x2E].num_of_byte = 1;
  def_instr[0x2E].num_of_cicles = 1;
  def_instr[0x2F].dis_fun = dis_ADD_A_REG;
  def_instr[0x2F].sim_fun = sim_ADD_A_REG;
  def_instr[0x2F].num_of_byte = 1;
  def_instr[0x2F].num_of_cicles = 1;
  def_instr[0x25].dis_fun = dis_ADD_A_DIR;
  def_instr[0x25].sim_fun = sim_ADD_A_DIR;
  def_instr[0x25].num_of_byte = 2;
  def_instr[0x25].num_of_cicles = 1;
  def_instr[0x26].dis_fun = dis_ADD_A_INDIR;
  def_instr[0x26].sim_fun = sim_ADD_A_INDIR;
  def_instr[0x26].num_of_byte = 1;
  def_instr[0x26].num_of_cicles = 1;
  def_instr[0x27].dis_fun = dis_ADD_A_INDIR;
  def_instr[0x27].sim_fun = sim_ADD_A_INDIR;
  def_instr[0x27].num_of_byte = 1;
  def_instr[0x27].num_of_cicles = 1;
  def_instr[0x24].dis_fun = dis_ADD_A_IMM;
  def_instr[0x24].sim_fun = sim_ADD_A_IMM;
  def_instr[0x24].num_of_byte = 2;
  def_instr[0x24].num_of_cicles = 1;
  def_instr[0x38].dis_fun = dis_ADDC_A_REG;
  def_instr[0x38].sim_fun = sim_ADDC_A_REG;
  def_instr[0x38].num_of_byte = 1;
  def_instr[0x38].num_of_cicles = 1;
  def_instr[0x39].dis_fun = dis_ADDC_A_REG;
  def_instr[0x39].sim_fun = sim_ADDC_A_REG;
  def_instr[0x39].num_of_byte = 1;
  def_instr[0x39].num_of_cicles = 1;
  def_instr[0x3A].dis_fun = dis_ADDC_A_REG;
  def_instr[0x3A].sim_fun = sim_ADDC_A_REG;
  def_instr[0x3A].num_of_byte = 1;
  def_instr[0x3A].num_of_cicles = 1;
  def_instr[0x3B].dis_fun = dis_ADDC_A_REG;
  def_instr[0x3B].sim_fun = sim_ADDC_A_REG;
  def_instr[0x3B].num_of_byte = 1;
  def_instr[0x3B].num_of_cicles = 1;
  def_instr[0x3C].dis_fun = dis_ADDC_A_REG;
  def_instr[0x3C].sim_fun = sim_ADDC_A_REG;
  def_instr[0x3C].num_of_byte = 1;
  def_instr[0x3C].num_of_cicles = 1;
  def_instr[0x3D].dis_fun = dis_ADDC_A_REG;
  def_instr[0x3D].sim_fun = sim_ADDC_A_REG;
  def_instr[0x3D].num_of_byte = 1;
  def_instr[0x3D].num_of_cicles = 1;
  def_instr[0x3E].dis_fun = dis_ADDC_A_REG;
  def_instr[0x3E].sim_fun = sim_ADDC_A_REG;
  def_instr[0x3E].num_of_byte = 1;
  def_instr[0x3E].num_of_cicles = 1;
  def_instr[0x3F].dis_fun = dis_ADDC_A_REG;
  def_instr[0x3F].sim_fun = sim_ADDC_A_REG;
  def_instr[0x3F].num_of_byte = 1;
  def_instr[0x3F].num_of_cicles = 1;
  def_instr[0x35].dis_fun = dis_ADDC_A_DIR;
  def_instr[0x35].sim_fun = sim_ADDC_A_DIR;
  def_instr[0x35].num_of_byte = 2;
  def_instr[0x35].num_of_cicles = 1;
  def_instr[0x36].dis_fun = dis_ADDC_A_INDIR;
  def_instr[0x36].sim_fun = sim_ADDC_A_INDIR;
  def_instr[0x36].num_of_byte = 1;
  def_instr[0x36].num_of_cicles = 1;
  def_instr[0x37].dis_fun = dis_ADDC_A_INDIR;
  def_instr[0x37].sim_fun = sim_ADDC_A_INDIR;
  def_instr[0x37].num_of_byte = 1;
  def_instr[0x37].num_of_cicles = 1;
  def_instr[0x34].dis_fun = dis_ADDC_A_IMM;
  def_instr[0x34].sim_fun = sim_ADDC_A_IMM;
  def_instr[0x34].num_of_byte = 2;
  def_instr[0x34].num_of_cicles = 1;
  def_instr[0x1].dis_fun = dis_AJMP_PADDR;
  def_instr[0x1].sim_fun = sim_AJMP_PADDR;
  def_instr[0x1].num_of_byte = 2;
  def_instr[0x1].num_of_cicles = 2;
  def_instr[0x21].dis_fun = dis_AJMP_PADDR;
  def_instr[0x21].sim_fun = sim_AJMP_PADDR;
  def_instr[0x21].num_of_byte = 2;
  def_instr[0x21].num_of_cicles = 2;
  def_instr[0x41].dis_fun = dis_AJMP_PADDR;
  def_instr[0x41].sim_fun = sim_AJMP_PADDR;
  def_instr[0x41].num_of_byte = 2;
  def_instr[0x41].num_of_cicles = 2;
  def_instr[0x61].dis_fun = dis_AJMP_PADDR;
  def_instr[0x61].sim_fun = sim_AJMP_PADDR;
  def_instr[0x61].num_of_byte = 2;
  def_instr[0x61].num_of_cicles = 2;
  def_instr[0x81].dis_fun = dis_AJMP_PADDR;
  def_instr[0x81].sim_fun = sim_AJMP_PADDR;
  def_instr[0x81].num_of_byte = 2;
  def_instr[0x81].num_of_cicles = 2;
  def_instr[0xA1].dis_fun = dis_AJMP_PADDR;
  def_instr[0xA1].sim_fun = sim_AJMP_PADDR;
  def_instr[0xA1].num_of_byte = 2;
  def_instr[0xA1].num_of_cicles = 2;
  def_instr[0xC1].dis_fun = dis_AJMP_PADDR;
  def_instr[0xC1].sim_fun = sim_AJMP_PADDR;
  def_instr[0xC1].num_of_byte = 2;
  def_instr[0xC1].num_of_cicles = 2;
  def_instr[0xE1].dis_fun = dis_AJMP_PADDR;
  def_instr[0xE1].sim_fun = sim_AJMP_PADDR;
  def_instr[0xE1].num_of_byte = 2;
  def_instr[0xE1].num_of_cicles = 2;
  def_instr[0x58].dis_fun = dis_ANL_A_REG;
  def_instr[0x58].sim_fun = sim_ANL_A_REG;
  def_instr[0x58].num_of_byte = 1;
  def_instr[0x58].num_of_cicles = 1;
  def_instr[0x59].dis_fun = dis_ANL_A_REG;
  def_instr[0x59].sim_fun = sim_ANL_A_REG;
  def_instr[0x59].num_of_byte = 1;
  def_instr[0x59].num_of_cicles = 1;
  def_instr[0x5A].dis_fun = dis_ANL_A_REG;
  def_instr[0x5A].sim_fun = sim_ANL_A_REG;
  def_instr[0x5A].num_of_byte = 1;
  def_instr[0x5A].num_of_cicles = 1;
  def_instr[0x5B].dis_fun = dis_ANL_A_REG;
  def_instr[0x5B].sim_fun = sim_ANL_A_REG;
  def_instr[0x5B].num_of_byte = 1;
  def_instr[0x5B].num_of_cicles = 1;
  def_instr[0x5C].dis_fun = dis_ANL_A_REG;
  def_instr[0x5C].sim_fun = sim_ANL_A_REG;
  def_instr[0x5C].num_of_byte = 1;
  def_instr[0x5C].num_of_cicles = 1;
  def_instr[0x5D].dis_fun = dis_ANL_A_REG;
  def_instr[0x5D].sim_fun = sim_ANL_A_REG;
  def_instr[0x5D].num_of_byte = 1;
  def_instr[0x5D].num_of_cicles = 1;
  def_instr[0x5E].dis_fun = dis_ANL_A_REG;
  def_instr[0x5E].sim_fun = sim_ANL_A_REG;
  def_instr[0x5E].num_of_byte = 1;
  def_instr[0x5E].num_of_cicles = 1;
  def_instr[0x5F].dis_fun = dis_ANL_A_REG;
  def_instr[0x5F].sim_fun = sim_ANL_A_REG;
  def_instr[0x5F].num_of_byte = 1;
  def_instr[0x5F].num_of_cicles = 1;
  def_instr[0x55].dis_fun = dis_ANL_A_DIR;
  def_instr[0x55].sim_fun = sim_ANL_A_DIR;
  def_instr[0x55].num_of_byte = 2;
  def_instr[0x55].num_of_cicles = 1;
  def_instr[0x56].dis_fun = dis_ANL_A_INDIR;
  def_instr[0x56].sim_fun = sim_ANL_A_INDIR;
  def_instr[0x56].num_of_byte = 1;
  def_instr[0x56].num_of_cicles = 1;
  def_instr[0x57].dis_fun = dis_ANL_A_INDIR;
  def_instr[0x57].sim_fun = sim_ANL_A_INDIR;
  def_instr[0x57].num_of_byte = 1;
  def_instr[0x57].num_of_cicles = 1;
  def_instr[0x54].dis_fun = dis_ANL_A_IMM;
  def_instr[0x54].sim_fun = sim_ANL_A_IMM;
  def_instr[0x54].num_of_byte = 2;
  def_instr[0x54].num_of_cicles = 1;
  def_instr[0x52].dis_fun = dis_ANL_DIR_A;
  def_instr[0x52].sim_fun = sim_ANL_DIR_A;
  def_instr[0x52].num_of_byte = 2;
  def_instr[0x52].num_of_cicles = 1;
  def_instr[0x53].dis_fun = dis_ANL_DIR_IMM;
  def_instr[0x53].sim_fun = sim_ANL_DIR_IMM;
  def_instr[0x53].num_of_byte = 3;
  def_instr[0x53].num_of_cicles = 2;
  def_instr[0x82].dis_fun = dis_ANL_C_BADDR;
  def_instr[0x82].sim_fun = sim_ANL_C_BADDR;
  def_instr[0x82].num_of_byte = 2;
  def_instr[0x82].num_of_cicles = 2;
  def_instr[0xB0].dis_fun = dis_ANL_C_NBADDR;
  def_instr[0xB0].sim_fun = sim_ANL_C_NBADDR;
  def_instr[0xB0].num_of_byte = 2;
  def_instr[0xB0].num_of_cicles = 2;
  def_instr[0xB5].dis_fun = dis_CJNE_A_DIR_RADDR;
  def_instr[0xB5].sim_fun = sim_CJNE_A_DIR_RADDR;
  def_instr[0xB5].num_of_byte = 3;
  def_instr[0xB5].num_of_cicles = 2;
  def_instr[0xB4].dis_fun = dis_CJNE_A_IMM_RADDR;
  def_instr[0xB4].sim_fun = sim_CJNE_A_IMM_RADDR;
  def_instr[0xB4].num_of_byte = 3;
  def_instr[0xB4].num_of_cicles = 2;
  def_instr[0xB8].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xB8].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xB8].num_of_byte = 3;
  def_instr[0xB8].num_of_cicles = 2;
  def_instr[0xB9].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xB9].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xB9].num_of_byte = 3;
  def_instr[0xB9].num_of_cicles = 2;
  def_instr[0xBA].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xBA].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xBA].num_of_byte = 3;
  def_instr[0xBA].num_of_cicles = 2;
  def_instr[0xBB].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xBB].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xBB].num_of_byte = 3;
  def_instr[0xBB].num_of_cicles = 2;
  def_instr[0xBC].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xBC].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xBC].num_of_byte = 3;
  def_instr[0xBC].num_of_cicles = 2;
  def_instr[0xBD].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xBD].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xBD].num_of_byte = 3;
  def_instr[0xBD].num_of_cicles = 2;
  def_instr[0xBE].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xBE].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xBE].num_of_byte = 3;
  def_instr[0xBE].num_of_cicles = 2;
  def_instr[0xBF].dis_fun = dis_CJNE_REG_IMM_RADDR;
  def_instr[0xBF].sim_fun = sim_CJNE_REG_IMM_RADDR;
  def_instr[0xBF].num_of_byte = 3;
  def_instr[0xBF].num_of_cicles = 2;
  def_instr[0xB6].dis_fun = dis_CJNE_INDIR_IMM_RADDR;
  def_instr[0xB6].sim_fun = sim_CJNE_INDIR_IMM_RADDR;
  def_instr[0xB6].num_of_byte = 3;
  def_instr[0xB6].num_of_cicles = 2;
  def_instr[0xB7].dis_fun = dis_CJNE_INDIR_IMM_RADDR;
  def_instr[0xB7].sim_fun = sim_CJNE_INDIR_IMM_RADDR;
  def_instr[0xB7].num_of_byte = 3;
  def_instr[0xB7].num_of_cicles = 2;
  def_instr[0xE4].dis_fun = dis_CLR_A;
  def_instr[0xE4].sim_fun = sim_CLR_A;
  def_instr[0xE4].num_of_byte = 1;
  def_instr[0xE4].num_of_cicles = 1;
  def_instr[0xC3].dis_fun = dis_CLR_C;
  def_instr[0xC3].sim_fun = sim_CLR_C;
  def_instr[0xC3].num_of_byte = 1;
  def_instr[0xC3].num_of_cicles = 1;
  def_instr[0xC2].dis_fun = dis_CLR_BADDR;
  def_instr[0xC2].sim_fun = sim_CLR_BADDR;
  def_instr[0xC2].num_of_byte = 2;
  def_instr[0xC2].num_of_cicles = 1;
  def_instr[0xF4].dis_fun = dis_CPL_A;
  def_instr[0xF4].sim_fun = sim_CPL_A;
  def_instr[0xF4].num_of_byte = 1;
  def_instr[0xF4].num_of_cicles = 1;
  def_instr[0xB3].dis_fun = dis_CPL_C;
  def_instr[0xB3].sim_fun = sim_CPL_C;
  def_instr[0xB3].num_of_byte = 1;
  def_instr[0xB3].num_of_cicles = 1;
  def_instr[0xB2].dis_fun = dis_CPL_BADDR;
  def_instr[0xB2].sim_fun = sim_CPL_BADDR;
  def_instr[0xB2].num_of_byte = 2;
  def_instr[0xB2].num_of_cicles = 1;
  def_instr[0xD4].dis_fun = dis_DA_A;
  def_instr[0xD4].sim_fun = sim_DA_A;
  def_instr[0xD4].num_of_byte = 1;
  def_instr[0xD4].num_of_cicles = 1;
  def_instr[0x14].dis_fun = dis_DEC_A;
  def_instr[0x14].sim_fun = sim_DEC_A;
  def_instr[0x14].num_of_byte = 1;
  def_instr[0x14].num_of_cicles = 1;
  def_instr[0x18].dis_fun = dis_DEC_REG;
  def_instr[0x18].sim_fun = sim_DEC_REG;
  def_instr[0x18].num_of_byte = 1;
  def_instr[0x18].num_of_cicles = 1;
  def_instr[0x19].dis_fun = dis_DEC_REG;
  def_instr[0x19].sim_fun = sim_DEC_REG;
  def_instr[0x19].num_of_byte = 1;
  def_instr[0x19].num_of_cicles = 1;
  def_instr[0x1A].dis_fun = dis_DEC_REG;
  def_instr[0x1A].sim_fun = sim_DEC_REG;
  def_instr[0x1A].num_of_byte = 1;
  def_instr[0x1A].num_of_cicles = 1;
  def_instr[0x1B].dis_fun = dis_DEC_REG;
  def_instr[0x1B].sim_fun = sim_DEC_REG;
  def_instr[0x1B].num_of_byte = 1;
  def_instr[0x1B].num_of_cicles = 1;
  def_instr[0x1C].dis_fun = dis_DEC_REG;
  def_instr[0x1C].sim_fun = sim_DEC_REG;
  def_instr[0x1C].num_of_byte = 1;
  def_instr[0x1C].num_of_cicles = 1;
  def_instr[0x1D].dis_fun = dis_DEC_REG;
  def_instr[0x1D].sim_fun = sim_DEC_REG;
  def_instr[0x1D].num_of_byte = 1;
  def_instr[0x1D].num_of_cicles = 1;
  def_instr[0x1E].dis_fun = dis_DEC_REG;
  def_instr[0x1E].sim_fun = sim_DEC_REG;
  def_instr[0x1E].num_of_byte = 1;
  def_instr[0x1E].num_of_cicles = 1;
  def_instr[0x1F].dis_fun = dis_DEC_REG;
  def_instr[0x1F].sim_fun = sim_DEC_REG;
  def_instr[0x1F].num_of_byte = 1;
  def_instr[0x1F].num_of_cicles = 1;
  def_instr[0x15].dis_fun = dis_DEC_DIR;
  def_instr[0x15].sim_fun = sim_DEC_DIR;
  def_instr[0x15].num_of_byte = 2;
  def_instr[0x15].num_of_cicles = 1;
  def_instr[0x16].dis_fun = dis_DEC_INDIR;
  def_instr[0x16].sim_fun = sim_DEC_INDIR;
  def_instr[0x16].num_of_byte = 1;
  def_instr[0x16].num_of_cicles = 1;
  def_instr[0x17].dis_fun = dis_DEC_INDIR;
  def_instr[0x17].sim_fun = sim_DEC_INDIR;
  def_instr[0x17].num_of_byte = 1;
  def_instr[0x17].num_of_cicles = 1;
  def_instr[0x84].dis_fun = dis_DIV_AB;
  def_instr[0x84].sim_fun = sim_DIV_AB;
  def_instr[0x84].num_of_byte = 1;
  def_instr[0x84].num_of_cicles = 4;
  def_instr[0xD8].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xD8].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xD8].num_of_byte = 2;
  def_instr[0xD8].num_of_cicles = 2;
  def_instr[0xD9].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xD9].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xD9].num_of_byte = 2;
  def_instr[0xD9].num_of_cicles = 2;
  def_instr[0xDA].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xDA].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xDA].num_of_byte = 2;
  def_instr[0xDA].num_of_cicles = 2;
  def_instr[0xDB].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xDB].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xDB].num_of_byte = 2;
  def_instr[0xDB].num_of_cicles = 2;
  def_instr[0xDC].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xDC].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xDC].num_of_byte = 2;
  def_instr[0xDC].num_of_cicles = 2;
  def_instr[0xDD].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xDD].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xDD].num_of_byte = 2;
  def_instr[0xDD].num_of_cicles = 2;
  def_instr[0xDE].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xDE].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xDE].num_of_byte = 2;
  def_instr[0xDE].num_of_cicles = 2;
  def_instr[0xDF].dis_fun = dis_DJNZ_REG_RADDR;
  def_instr[0xDF].sim_fun = sim_DJNZ_REG_RADDR;
  def_instr[0xDF].num_of_byte = 2;
  def_instr[0xDF].num_of_cicles = 2;
  def_instr[0xD5].dis_fun = dis_DJNZ_DIR_RADDR;
  def_instr[0xD5].sim_fun = sim_DJNZ_DIR_RADDR;
  def_instr[0xD5].num_of_byte = 3;
  def_instr[0xD5].num_of_cicles = 2;
  def_instr[0x4].dis_fun = dis_INC_A;
  def_instr[0x4].sim_fun = sim_INC_A;
  def_instr[0x4].num_of_byte = 1;
  def_instr[0x4].num_of_cicles = 1;
  def_instr[0x8].dis_fun = dis_INC_REG;
  def_instr[0x8].sim_fun = sim_INC_REG;
  def_instr[0x8].num_of_byte = 1;
  def_instr[0x8].num_of_cicles = 1;
  def_instr[0x9].dis_fun = dis_INC_REG;
  def_instr[0x9].sim_fun = sim_INC_REG;
  def_instr[0x9].num_of_byte = 1;
  def_instr[0x9].num_of_cicles = 1;
  def_instr[0xA].dis_fun = dis_INC_REG;
  def_instr[0xA].sim_fun = sim_INC_REG;
  def_instr[0xA].num_of_byte = 1;
  def_instr[0xA].num_of_cicles = 1;
  def_instr[0xB].dis_fun = dis_INC_REG;
  def_instr[0xB].sim_fun = sim_INC_REG;
  def_instr[0xB].num_of_byte = 1;
  def_instr[0xB].num_of_cicles = 1;
  def_instr[0xC].dis_fun = dis_INC_REG;
  def_instr[0xC].sim_fun = sim_INC_REG;
  def_instr[0xC].num_of_byte = 1;
  def_instr[0xC].num_of_cicles = 1;
  def_instr[0xD].dis_fun = dis_INC_REG;
  def_instr[0xD].sim_fun = sim_INC_REG;
  def_instr[0xD].num_of_byte = 1;
  def_instr[0xD].num_of_cicles = 1;
  def_instr[0xE].dis_fun = dis_INC_REG;
  def_instr[0xE].sim_fun = sim_INC_REG;
  def_instr[0xE].num_of_byte = 1;
  def_instr[0xE].num_of_cicles = 1;
  def_instr[0xF].dis_fun = dis_INC_REG;
  def_instr[0xF].sim_fun = sim_INC_REG;
  def_instr[0xF].num_of_byte = 1;
  def_instr[0xF].num_of_cicles = 1;
  def_instr[0x5].dis_fun = dis_INC_DIR;
  def_instr[0x5].sim_fun = sim_INC_DIR;
  def_instr[0x5].num_of_byte = 2;
  def_instr[0x5].num_of_cicles = 1;
  def_instr[0x6].dis_fun = dis_INC_INDIR;
  def_instr[0x6].sim_fun = sim_INC_INDIR;
  def_instr[0x6].num_of_byte = 1;
  def_instr[0x6].num_of_cicles = 1;
  def_instr[0x7].dis_fun = dis_INC_INDIR;
  def_instr[0x7].sim_fun = sim_INC_INDIR;
  def_instr[0x7].num_of_byte = 1;
  def_instr[0x7].num_of_cicles = 1;
  def_instr[0xA3].dis_fun = dis_INC_DPTR;
  def_instr[0xA3].sim_fun = sim_INC_DPTR;
  def_instr[0xA3].num_of_byte = 1;
  def_instr[0xA3].num_of_cicles = 2;
  def_instr[0x20].dis_fun = dis_JB_BADDR_RADDR;
  def_instr[0x20].sim_fun = sim_JB_BADDR_RADDR;
  def_instr[0x20].num_of_byte = 3;
  def_instr[0x20].num_of_cicles = 2;
  def_instr[0x10].dis_fun = dis_JBC_BADDR_RADDR;
  def_instr[0x10].sim_fun = sim_JBC_BADDR_RADDR;
  def_instr[0x10].num_of_byte = 3;
  def_instr[0x10].num_of_cicles = 2;
  def_instr[0x40].dis_fun = dis_JC_RADDR;
  def_instr[0x40].sim_fun = sim_JC_RADDR;
  def_instr[0x40].num_of_byte = 2;
  def_instr[0x40].num_of_cicles = 2;
  def_instr[0x73].dis_fun = dis_JMP_ATADPTR;
  def_instr[0x73].sim_fun = sim_JMP_ATADPTR;
  def_instr[0x73].num_of_byte = 1;
  def_instr[0x73].num_of_cicles = 2;
  def_instr[0x30].dis_fun = dis_JNB_BADDR_RADDR;
  def_instr[0x30].sim_fun = sim_JNB_BADDR_RADDR;
  def_instr[0x30].num_of_byte = 3;
  def_instr[0x30].num_of_cicles = 2;
  def_instr[0x50].dis_fun = dis_JNC_RADDR;
  def_instr[0x50].sim_fun = sim_JNC_RADDR;
  def_instr[0x50].num_of_byte = 2;
  def_instr[0x50].num_of_cicles = 2;
  def_instr[0x70].dis_fun = dis_JNZ_RADDR;
  def_instr[0x70].sim_fun = sim_JNZ_RADDR;
  def_instr[0x70].num_of_byte = 2;
  def_instr[0x70].num_of_cicles = 2;
  def_instr[0x60].dis_fun = dis_JZ_RADDR;
  def_instr[0x60].sim_fun = sim_JZ_RADDR;
  def_instr[0x60].num_of_byte = 2;
  def_instr[0x60].num_of_cicles = 2;
  def_instr[0x12].dis_fun = dis_LCALL_CADDR;
  def_instr[0x12].sim_fun = sim_LCALL_CADDR;
  def_instr[0x12].num_of_byte = 3;
  def_instr[0x12].num_of_cicles = 2;
  def_instr[0x2].dis_fun = dis_LJMP_CADDR;
  def_instr[0x2].sim_fun = sim_LJMP_CADDR;
  def_instr[0x2].num_of_byte = 3;
  def_instr[0x2].num_of_cicles = 2;
  def_instr[0xE8].dis_fun = dis_MOV_A_REG;
  def_instr[0xE8].sim_fun = sim_MOV_A_REG;
  def_instr[0xE8].num_of_byte = 1;
  def_instr[0xE8].num_of_cicles = 1;
  def_instr[0xE9].dis_fun = dis_MOV_A_REG;
  def_instr[0xE9].sim_fun = sim_MOV_A_REG;
  def_instr[0xE9].num_of_byte = 1;
  def_instr[0xE9].num_of_cicles = 1;
  def_instr[0xEA].dis_fun = dis_MOV_A_REG;
  def_instr[0xEA].sim_fun = sim_MOV_A_REG;
  def_instr[0xEA].num_of_byte = 1;
  def_instr[0xEA].num_of_cicles = 1;
  def_instr[0xEB].dis_fun = dis_MOV_A_REG;
  def_instr[0xEB].sim_fun = sim_MOV_A_REG;
  def_instr[0xEB].num_of_byte = 1;
  def_instr[0xEB].num_of_cicles = 1;
  def_instr[0xEC].dis_fun = dis_MOV_A_REG;
  def_instr[0xEC].sim_fun = sim_MOV_A_REG;
  def_instr[0xEC].num_of_byte = 1;
  def_instr[0xEC].num_of_cicles = 1;
  def_instr[0xED].dis_fun = dis_MOV_A_REG;
  def_instr[0xED].sim_fun = sim_MOV_A_REG;
  def_instr[0xED].num_of_byte = 1;
  def_instr[0xED].num_of_cicles = 1;
  def_instr[0xEE].dis_fun = dis_MOV_A_REG;
  def_instr[0xEE].sim_fun = sim_MOV_A_REG;
  def_instr[0xEE].num_of_byte = 1;
  def_instr[0xEE].num_of_cicles = 1;
  def_instr[0xEF].dis_fun = dis_MOV_A_REG;
  def_instr[0xEF].sim_fun = sim_MOV_A_REG;
  def_instr[0xEF].num_of_byte = 1;
  def_instr[0xEF].num_of_cicles = 1;
  def_instr[0xE5].dis_fun = dis_MOV_A_DIR;
  def_instr[0xE5].sim_fun = sim_MOV_A_DIR;
  def_instr[0xE5].num_of_byte = 2;
  def_instr[0xE5].num_of_cicles = 1;
  def_instr[0xE6].dis_fun = dis_MOV_A_INDIR;
  def_instr[0xE6].sim_fun = sim_MOV_A_INDIR;
  def_instr[0xE6].num_of_byte = 1;
  def_instr[0xE6].num_of_cicles = 1;
  def_instr[0xE7].dis_fun = dis_MOV_A_INDIR;
  def_instr[0xE7].sim_fun = sim_MOV_A_INDIR;
  def_instr[0xE7].num_of_byte = 1;
  def_instr[0xE7].num_of_cicles = 1;
  def_instr[0x74].dis_fun = dis_MOV_A_IMM;
  def_instr[0x74].sim_fun = sim_MOV_A_IMM;
  def_instr[0x74].num_of_byte = 2;
  def_instr[0x74].num_of_cicles = 1;
  def_instr[0xF8].dis_fun = dis_MOV_REG_A;
  def_instr[0xF8].sim_fun = sim_MOV_REG_A;
  def_instr[0xF8].num_of_byte = 1;
  def_instr[0xF8].num_of_cicles = 1;
  def_instr[0xF9].dis_fun = dis_MOV_REG_A;
  def_instr[0xF9].sim_fun = sim_MOV_REG_A;
  def_instr[0xF9].num_of_byte = 1;
  def_instr[0xF9].num_of_cicles = 1;
  def_instr[0xFA].dis_fun = dis_MOV_REG_A;
  def_instr[0xFA].sim_fun = sim_MOV_REG_A;
  def_instr[0xFA].num_of_byte = 1;
  def_instr[0xFA].num_of_cicles = 1;
  def_instr[0xFB].dis_fun = dis_MOV_REG_A;
  def_instr[0xFB].sim_fun = sim_MOV_REG_A;
  def_instr[0xFB].num_of_byte = 1;
  def_instr[0xFB].num_of_cicles = 1;
  def_instr[0xFC].dis_fun = dis_MOV_REG_A;
  def_instr[0xFC].sim_fun = sim_MOV_REG_A;
  def_instr[0xFC].num_of_byte = 1;
  def_instr[0xFC].num_of_cicles = 1;
  def_instr[0xFD].dis_fun = dis_MOV_REG_A;
  def_instr[0xFD].sim_fun = sim_MOV_REG_A;
  def_instr[0xFD].num_of_byte = 1;
  def_instr[0xFD].num_of_cicles = 1;
  def_instr[0xFE].dis_fun = dis_MOV_REG_A;
  def_instr[0xFE].sim_fun = sim_MOV_REG_A;
  def_instr[0xFE].num_of_byte = 1;
  def_instr[0xFE].num_of_cicles = 1;
  def_instr[0xFF].dis_fun = dis_MOV_REG_A;
  def_instr[0xFF].sim_fun = sim_MOV_REG_A;
  def_instr[0xFF].num_of_byte = 1;
  def_instr[0xFF].num_of_cicles = 1;
  def_instr[0xA8].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xA8].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xA8].num_of_byte = 2;
  def_instr[0xA8].num_of_cicles = 2;
  def_instr[0xA9].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xA9].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xA9].num_of_byte = 2;
  def_instr[0xA9].num_of_cicles = 2;
  def_instr[0xAA].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xAA].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xAA].num_of_byte = 2;
  def_instr[0xAA].num_of_cicles = 2;
  def_instr[0xAB].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xAB].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xAB].num_of_byte = 2;
  def_instr[0xAB].num_of_cicles = 2;
  def_instr[0xAC].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xAC].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xAC].num_of_byte = 2;
  def_instr[0xAC].num_of_cicles = 2;
  def_instr[0xAD].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xAD].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xAD].num_of_byte = 2;
  def_instr[0xAD].num_of_cicles = 2;
  def_instr[0xAE].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xAE].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xAE].num_of_byte = 2;
  def_instr[0xAE].num_of_cicles = 2;
  def_instr[0xAF].dis_fun = dis_MOV_REG_DIR;
  def_instr[0xAF].sim_fun = sim_MOV_REG_DIR;
  def_instr[0xAF].num_of_byte = 2;
  def_instr[0xAF].num_of_cicles = 2;
  def_instr[0x78].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x78].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x78].num_of_byte = 2;
  def_instr[0x78].num_of_cicles = 1;
  def_instr[0x79].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x79].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x79].num_of_byte = 2;
  def_instr[0x79].num_of_cicles = 1;
  def_instr[0x7A].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x7A].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x7A].num_of_byte = 2;
  def_instr[0x7A].num_of_cicles = 1;
  def_instr[0x7B].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x7B].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x7B].num_of_byte = 2;
  def_instr[0x7B].num_of_cicles = 1;
  def_instr[0x7C].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x7C].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x7C].num_of_byte = 2;
  def_instr[0x7C].num_of_cicles = 1;
  def_instr[0x7D].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x7D].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x7D].num_of_byte = 2;
  def_instr[0x7D].num_of_cicles = 1;
  def_instr[0x7E].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x7E].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x7E].num_of_byte = 2;
  def_instr[0x7E].num_of_cicles = 1;
  def_instr[0x7F].dis_fun = dis_MOV_REG_IMM;
  def_instr[0x7F].sim_fun = sim_MOV_REG_IMM;
  def_instr[0x7F].num_of_byte = 2;
  def_instr[0x7F].num_of_cicles = 1;
  def_instr[0xF5].dis_fun = dis_MOV_DIR_A;
  def_instr[0xF5].sim_fun = sim_MOV_DIR_A;
  def_instr[0xF5].num_of_byte = 2;
  def_instr[0xF5].num_of_cicles = 1;
  def_instr[0x88].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x88].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x88].num_of_byte = 2;
  def_instr[0x88].num_of_cicles = 2;
  def_instr[0x89].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x89].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x89].num_of_byte = 2;
  def_instr[0x89].num_of_cicles = 2;
  def_instr[0x8A].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x8A].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x8A].num_of_byte = 2;
  def_instr[0x8A].num_of_cicles = 2;
  def_instr[0x8B].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x8B].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x8B].num_of_byte = 2;
  def_instr[0x8B].num_of_cicles = 2;
  def_instr[0x8C].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x8C].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x8C].num_of_byte = 2;
  def_instr[0x8C].num_of_cicles = 2;
  def_instr[0x8D].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x8D].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x8D].num_of_byte = 2;
  def_instr[0x8D].num_of_cicles = 2;
  def_instr[0x8E].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x8E].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x8E].num_of_byte = 2;
  def_instr[0x8E].num_of_cicles = 2;
  def_instr[0x8F].dis_fun = dis_MOV_DIR_REG;
  def_instr[0x8F].sim_fun = sim_MOV_DIR_REG;
  def_instr[0x8F].num_of_byte = 2;
  def_instr[0x8F].num_of_cicles = 2;
  def_instr[0x85].dis_fun = dis_MOV_DIR_DIR;
  def_instr[0x85].sim_fun = sim_MOV_DIR_DIR;
  def_instr[0x85].num_of_byte = 3;
  def_instr[0x85].num_of_cicles = 2;
  def_instr[0x86].dis_fun = dis_MOV_DIR_INDIR;
  def_instr[0x86].sim_fun = sim_MOV_DIR_INDIR;
  def_instr[0x86].num_of_byte = 2;
  def_instr[0x86].num_of_cicles = 2;
  def_instr[0x87].dis_fun = dis_MOV_DIR_INDIR;
  def_instr[0x87].sim_fun = sim_MOV_DIR_INDIR;
  def_instr[0x87].num_of_byte = 2;
  def_instr[0x87].num_of_cicles = 2;
  def_instr[0x75].dis_fun = dis_MOV_DIR_IMM;
  def_instr[0x75].sim_fun = sim_MOV_DIR_IMM;
  def_instr[0x75].num_of_byte = 3;
  def_instr[0x75].num_of_cicles = 2;
  def_instr[0xF6].dis_fun = dis_MOV_INDIR_A;
  def_instr[0xF6].sim_fun = sim_MOV_INDIR_A;
  def_instr[0xF6].num_of_byte = 1;
  def_instr[0xF6].num_of_cicles = 1;
  def_instr[0xF7].dis_fun = dis_MOV_INDIR_A;
  def_instr[0xF7].sim_fun = sim_MOV_INDIR_A;
  def_instr[0xF7].num_of_byte = 1;
  def_instr[0xF7].num_of_cicles = 1;
  def_instr[0xA6].dis_fun = dis_MOV_INDIR_DIR;
  def_instr[0xA6].sim_fun = sim_MOV_INDIR_DIR;
  def_instr[0xA6].num_of_byte = 2;
  def_instr[0xA6].num_of_cicles = 2;
  def_instr[0xA7].dis_fun = dis_MOV_INDIR_DIR;
  def_instr[0xA7].sim_fun = sim_MOV_INDIR_DIR;
  def_instr[0xA7].num_of_byte = 2;
  def_instr[0xA7].num_of_cicles = 2;
  def_instr[0x76].dis_fun = dis_MOV_INDIR_IMM;
  def_instr[0x76].sim_fun = sim_MOV_INDIR_IMM;
  def_instr[0x76].num_of_byte = 2;
  def_instr[0x76].num_of_cicles = 1;
  def_instr[0x77].dis_fun = dis_MOV_INDIR_IMM;
  def_instr[0x77].sim_fun = sim_MOV_INDIR_IMM;
  def_instr[0x77].num_of_byte = 2;
  def_instr[0x77].num_of_cicles = 1;
  def_instr[0xA2].dis_fun = dis_MOV_C_BADDR;
  def_instr[0xA2].sim_fun = sim_MOV_C_BADDR;
  def_instr[0xA2].num_of_byte = 2;
  def_instr[0xA2].num_of_cicles = 1;
  def_instr[0x92].dis_fun = dis_MOV_BADDR_C;
  def_instr[0x92].sim_fun = sim_MOV_BADDR_C;
  def_instr[0x92].num_of_byte = 2;
  def_instr[0x92].num_of_cicles = 2;
  def_instr[0x90].dis_fun = dis_MOV_DPTR_IMM16;
  def_instr[0x90].sim_fun = sim_MOV_DPTR_IMM16;
  def_instr[0x90].num_of_byte = 3;
  def_instr[0x90].num_of_cicles = 2;
  def_instr[0x93].dis_fun = dis_MOVC_A_ATADPTR;
  def_instr[0x93].sim_fun = sim_MOVC_A_ATADPTR;
  def_instr[0x93].num_of_byte = 1;
  def_instr[0x93].num_of_cicles = 2;
  def_instr[0x83].dis_fun = dis_MOVC_A_ATAPC;
  def_instr[0x83].sim_fun = sim_MOVC_A_ATAPC;
  def_instr[0x83].num_of_byte = 1;
  def_instr[0x83].num_of_cicles = 2;
  def_instr[0xE2].dis_fun = dis_MOVX_A_INDIR;
  def_instr[0xE2].sim_fun = sim_MOVX_A_INDIR;
  def_instr[0xE2].num_of_byte = 1;
  def_instr[0xE2].num_of_cicles = 2;
  def_instr[0xE3].dis_fun = dis_MOVX_A_INDIR;
  def_instr[0xE3].sim_fun = sim_MOVX_A_INDIR;
  def_instr[0xE3].num_of_byte = 1;
  def_instr[0xE3].num_of_cicles = 2;
  def_instr[0xF2].dis_fun = dis_MOVX_INDIR_A;
  def_instr[0xF2].sim_fun = sim_MOVX_INDIR_A;
  def_instr[0xF2].num_of_byte = 1;
  def_instr[0xF2].num_of_cicles = 2;
  def_instr[0xF3].dis_fun = dis_MOVX_INDIR_A;
  def_instr[0xF3].sim_fun = sim_MOVX_INDIR_A;
  def_instr[0xF3].num_of_byte = 1;
  def_instr[0xF3].num_of_cicles = 2;
  def_instr[0xE0].dis_fun = dis_MOVX_A_ATDPTR;
  def_instr[0xE0].sim_fun = sim_MOVX_A_ATDPTR;
  def_instr[0xE0].num_of_byte = 1;
  def_instr[0xE0].num_of_cicles = 2;
  def_instr[0xF0].dis_fun = dis_MOVX_ATDPTR_A;
  def_instr[0xF0].sim_fun = sim_MOVX_ATDPTR_A;
  def_instr[0xF0].num_of_byte = 1;
  def_instr[0xF0].num_of_cicles = 2;
  def_instr[0x0].dis_fun = dis_NOP;
  def_instr[0x0].sim_fun = sim_NOP;
  def_instr[0x0].num_of_byte = 1;
  def_instr[0x0].num_of_cicles = 1;
  def_instr[0xA4].dis_fun = dis_MUL_AB;
  def_instr[0xA4].sim_fun = sim_MUL_AB;
  def_instr[0xA4].num_of_byte = 1;
  def_instr[0xA4].num_of_cicles = 4;
  def_instr[0x48].dis_fun = dis_ORL_A_REG;
  def_instr[0x48].sim_fun = sim_ORL_A_REG;
  def_instr[0x48].num_of_byte = 1;
  def_instr[0x48].num_of_cicles = 1;
  def_instr[0x49].dis_fun = dis_ORL_A_REG;
  def_instr[0x49].sim_fun = sim_ORL_A_REG;
  def_instr[0x49].num_of_byte = 1;
  def_instr[0x49].num_of_cicles = 1;
  def_instr[0x4A].dis_fun = dis_ORL_A_REG;
  def_instr[0x4A].sim_fun = sim_ORL_A_REG;
  def_instr[0x4A].num_of_byte = 1;
  def_instr[0x4A].num_of_cicles = 1;
  def_instr[0x4B].dis_fun = dis_ORL_A_REG;
  def_instr[0x4B].sim_fun = sim_ORL_A_REG;
  def_instr[0x4B].num_of_byte = 1;
  def_instr[0x4B].num_of_cicles = 1;
  def_instr[0x4C].dis_fun = dis_ORL_A_REG;
  def_instr[0x4C].sim_fun = sim_ORL_A_REG;
  def_instr[0x4C].num_of_byte = 1;
  def_instr[0x4C].num_of_cicles = 1;
  def_instr[0x4D].dis_fun = dis_ORL_A_REG;
  def_instr[0x4D].sim_fun = sim_ORL_A_REG;
  def_instr[0x4D].num_of_byte = 1;
  def_instr[0x4D].num_of_cicles = 1;
  def_instr[0x4E].dis_fun = dis_ORL_A_REG;
  def_instr[0x4E].sim_fun = sim_ORL_A_REG;
  def_instr[0x4E].num_of_byte = 1;
  def_instr[0x4E].num_of_cicles = 1;
  def_instr[0x4F].dis_fun = dis_ORL_A_REG;
  def_instr[0x4F].sim_fun = sim_ORL_A_REG;
  def_instr[0x4F].num_of_byte = 1;
  def_instr[0x4F].num_of_cicles = 1;
  def_instr[0x45].dis_fun = dis_ORL_A_DIR;
  def_instr[0x45].sim_fun = sim_ORL_A_DIR;
  def_instr[0x45].num_of_byte = 2;
  def_instr[0x45].num_of_cicles = 1;
  def_instr[0x46].dis_fun = dis_ORL_A_INDIR;
  def_instr[0x46].sim_fun = sim_ORL_A_INDIR;
  def_instr[0x46].num_of_byte = 1;
  def_instr[0x46].num_of_cicles = 1;
  def_instr[0x47].dis_fun = dis_ORL_A_INDIR;
  def_instr[0x47].sim_fun = sim_ORL_A_INDIR;
  def_instr[0x47].num_of_byte = 1;
  def_instr[0x47].num_of_cicles = 1;
  def_instr[0x44].dis_fun = dis_ORL_A_IMM;
  def_instr[0x44].sim_fun = sim_ORL_A_IMM;
  def_instr[0x44].num_of_byte = 2;
  def_instr[0x44].num_of_cicles = 1;
  def_instr[0x42].dis_fun = dis_ORL_DIR_A;
  def_instr[0x42].sim_fun = sim_ORL_DIR_A;
  def_instr[0x42].num_of_byte = 2;
  def_instr[0x42].num_of_cicles = 1;
  def_instr[0x43].dis_fun = dis_ORL_DIR_IMM;
  def_instr[0x43].sim_fun = sim_ORL_DIR_IMM;
  def_instr[0x43].num_of_byte = 3;
  def_instr[0x43].num_of_cicles = 2;
  def_instr[0x72].dis_fun = dis_ORL_C_BADDR;
  def_instr[0x72].sim_fun = sim_ORL_C_BADDR;
  def_instr[0x72].num_of_byte = 2;
  def_instr[0x72].num_of_cicles = 2;
  def_instr[0xA0].dis_fun = dis_ORL_C_NBADDR;
  def_instr[0xA0].sim_fun = sim_ORL_C_NBADDR;
  def_instr[0xA0].num_of_byte = 2;
  def_instr[0xA0].num_of_cicles = 2;
  def_instr[0xD0].dis_fun = dis_POP_DIR;
  def_instr[0xD0].sim_fun = sim_POP_DIR;
  def_instr[0xD0].num_of_byte = 2;
  def_instr[0xD0].num_of_cicles = 2;
  def_instr[0xC0].dis_fun = dis_PUSH_DIR;
  def_instr[0xC0].sim_fun = sim_PUSH_DIR;
  def_instr[0xC0].num_of_byte = 2;
  def_instr[0xC0].num_of_cicles = 2;
  def_instr[0x22].dis_fun = dis_RET;
  def_instr[0x22].sim_fun = sim_RET;
  def_instr[0x22].num_of_byte = 1;
  def_instr[0x22].num_of_cicles = 2;
  def_instr[0x32].dis_fun = dis_RETI;
  def_instr[0x32].sim_fun = sim_RETI;
  def_instr[0x32].num_of_byte = 1;
  def_instr[0x32].num_of_cicles = 2;
  def_instr[0x23].dis_fun = dis_RL_A;
  def_instr[0x23].sim_fun = sim_RL_A;
  def_instr[0x23].num_of_byte = 1;
  def_instr[0x23].num_of_cicles = 1;
  def_instr[0x33].dis_fun = dis_RLC_A;
  def_instr[0x33].sim_fun = sim_RLC_A;
  def_instr[0x33].num_of_byte = 1;
  def_instr[0x33].num_of_cicles = 1;
  def_instr[0x3].dis_fun = dis_RR_A;
  def_instr[0x3].sim_fun = sim_RR_A;
  def_instr[0x3].num_of_byte = 1;
  def_instr[0x3].num_of_cicles = 1;
  def_instr[0x13].dis_fun = dis_RRC_A;
  def_instr[0x13].sim_fun = sim_RRC_A;
  def_instr[0x13].num_of_byte = 1;
  def_instr[0x13].num_of_cicles = 1;
  def_instr[0xD3].dis_fun = dis_SETB_C;
  def_instr[0xD3].sim_fun = sim_SETB_C;
  def_instr[0xD3].num_of_byte = 1;
  def_instr[0xD3].num_of_cicles = 1;
  def_instr[0xD2].dis_fun = dis_SETB_BADDR;
  def_instr[0xD2].sim_fun = sim_SETB_BADDR;
  def_instr[0xD2].num_of_byte = 2;
  def_instr[0xD2].num_of_cicles = 1;
  def_instr[0x80].dis_fun = dis_SJMP_RADDR;
  def_instr[0x80].sim_fun = sim_SJMP_RADDR;
  def_instr[0x80].num_of_byte = 2;
  def_instr[0x80].num_of_cicles = 2;
  def_instr[0x98].dis_fun = dis_SUBB_A_REG;
  def_instr[0x98].sim_fun = sim_SUBB_A_REG;
  def_instr[0x98].num_of_byte = 1;
  def_instr[0x98].num_of_cicles = 1;
  def_instr[0x99].dis_fun = dis_SUBB_A_REG;
  def_instr[0x99].sim_fun = sim_SUBB_A_REG;
  def_instr[0x99].num_of_byte = 1;
  def_instr[0x99].num_of_cicles = 1;
  def_instr[0x9A].dis_fun = dis_SUBB_A_REG;
  def_instr[0x9A].sim_fun = sim_SUBB_A_REG;
  def_instr[0x9A].num_of_byte = 1;
  def_instr[0x9A].num_of_cicles = 1;
  def_instr[0x9B].dis_fun = dis_SUBB_A_REG;
  def_instr[0x9B].sim_fun = sim_SUBB_A_REG;
  def_instr[0x9B].num_of_byte = 1;
  def_instr[0x9B].num_of_cicles = 1;
  def_instr[0x9C].dis_fun = dis_SUBB_A_REG;
  def_instr[0x9C].sim_fun = sim_SUBB_A_REG;
  def_instr[0x9C].num_of_byte = 1;
  def_instr[0x9C].num_of_cicles = 1;
  def_instr[0x9D].dis_fun = dis_SUBB_A_REG;
  def_instr[0x9D].sim_fun = sim_SUBB_A_REG;
  def_instr[0x9D].num_of_byte = 1;
  def_instr[0x9D].num_of_cicles = 1;
  def_instr[0x9E].dis_fun = dis_SUBB_A_REG;
  def_instr[0x9E].sim_fun = sim_SUBB_A_REG;
  def_instr[0x9E].num_of_byte = 1;
  def_instr[0x9E].num_of_cicles = 1;
  def_instr[0x9F].dis_fun = dis_SUBB_A_REG;
  def_instr[0x9F].sim_fun = sim_SUBB_A_REG;
  def_instr[0x9F].num_of_byte = 1;
  def_instr[0x9F].num_of_cicles = 1;
  def_instr[0x95].dis_fun = dis_SUBB_A_DIR;
  def_instr[0x95].sim_fun = sim_SUBB_A_DIR;
  def_instr[0x95].num_of_byte = 2;
  def_instr[0x95].num_of_cicles = 1;
  def_instr[0x96].dis_fun = dis_SUBB_A_INDIR;
  def_instr[0x96].sim_fun = sim_SUBB_A_INDIR;
  def_instr[0x96].num_of_byte = 1;
  def_instr[0x96].num_of_cicles = 1;
  def_instr[0x97].dis_fun = dis_SUBB_A_INDIR;
  def_instr[0x97].sim_fun = sim_SUBB_A_INDIR;
  def_instr[0x97].num_of_byte = 1;
  def_instr[0x97].num_of_cicles = 1;
  def_instr[0x94].dis_fun = dis_SUBB_A_IMM;
  def_instr[0x94].sim_fun = sim_SUBB_A_IMM;
  def_instr[0x94].num_of_byte = 2;
  def_instr[0x94].num_of_cicles = 1;
  def_instr[0xC4].dis_fun = dis_SWAP_A;
  def_instr[0xC4].sim_fun = sim_SWAP_A;
  def_instr[0xC4].num_of_byte = 1;
  def_instr[0xC4].num_of_cicles = 1;
  def_instr[0xC8].dis_fun = dis_XCH_A_REG;
  def_instr[0xC8].sim_fun = sim_XCH_A_REG;
  def_instr[0xC8].num_of_byte = 1;
  def_instr[0xC8].num_of_cicles = 1;
  def_instr[0xC9].dis_fun = dis_XCH_A_REG;
  def_instr[0xC9].sim_fun = sim_XCH_A_REG;
  def_instr[0xC9].num_of_byte = 1;
  def_instr[0xC9].num_of_cicles = 1;
  def_instr[0xCA].dis_fun = dis_XCH_A_REG;
  def_instr[0xCA].sim_fun = sim_XCH_A_REG;
  def_instr[0xCA].num_of_byte = 1;
  def_instr[0xCA].num_of_cicles = 1;
  def_instr[0xCB].dis_fun = dis_XCH_A_REG;
  def_instr[0xCB].sim_fun = sim_XCH_A_REG;
  def_instr[0xCB].num_of_byte = 1;
  def_instr[0xCB].num_of_cicles = 1;
  def_instr[0xCC].dis_fun = dis_XCH_A_REG;
  def_instr[0xCC].sim_fun = sim_XCH_A_REG;
  def_instr[0xCC].num_of_byte = 1;
  def_instr[0xCC].num_of_cicles = 1;
  def_instr[0xCD].dis_fun = dis_XCH_A_REG;
  def_instr[0xCD].sim_fun = sim_XCH_A_REG;
  def_instr[0xCD].num_of_byte = 1;
  def_instr[0xCD].num_of_cicles = 1;
  def_instr[0xCE].dis_fun = dis_XCH_A_REG;
  def_instr[0xCE].sim_fun = sim_XCH_A_REG;
  def_instr[0xCE].num_of_byte = 1;
  def_instr[0xCE].num_of_cicles = 1;
  def_instr[0xCF].dis_fun = dis_XCH_A_REG;
  def_instr[0xCF].sim_fun = sim_XCH_A_REG;
  def_instr[0xCF].num_of_byte = 1;
  def_instr[0xCF].num_of_cicles = 1;
  def_instr[0xC5].dis_fun = dis_XCH_A_DIR;
  def_instr[0xC5].sim_fun = sim_XCH_A_DIR;
  def_instr[0xC5].num_of_byte = 2;
  def_instr[0xC5].num_of_cicles = 1;
  def_instr[0xC6].dis_fun = dis_XCH_A_INDIR;
  def_instr[0xC6].sim_fun = sim_XCH_A_INDIR;
  def_instr[0xC6].num_of_byte = 1;
  def_instr[0xC6].num_of_cicles = 1;
  def_instr[0xC7].dis_fun = dis_XCH_A_INDIR;
  def_instr[0xC7].sim_fun = sim_XCH_A_INDIR;
  def_instr[0xC7].num_of_byte = 1;
  def_instr[0xC7].num_of_cicles = 1;
  def_instr[0xD6].dis_fun = dis_XCHD_A_INDIR;
  def_instr[0xD6].sim_fun = sim_XCHD_A_INDIR;
  def_instr[0xD6].num_of_byte = 1;
  def_instr[0xD6].num_of_cicles = 1;
  def_instr[0xD7].dis_fun = dis_XCHD_A_INDIR;
  def_instr[0xD7].sim_fun = sim_XCHD_A_INDIR;
  def_instr[0xD7].num_of_byte = 1;
  def_instr[0xD7].num_of_cicles = 1;
  def_instr[0x68].dis_fun = dis_XRL_A_REG;
  def_instr[0x68].sim_fun = sim_XRL_A_REG;
  def_instr[0x68].num_of_byte = 1;
  def_instr[0x68].num_of_cicles = 1;
  def_instr[0x69].dis_fun = dis_XRL_A_REG;
  def_instr[0x69].sim_fun = sim_XRL_A_REG;
  def_instr[0x69].num_of_byte = 1;
  def_instr[0x69].num_of_cicles = 1;
  def_instr[0x6A].dis_fun = dis_XRL_A_REG;
  def_instr[0x6A].sim_fun = sim_XRL_A_REG;
  def_instr[0x6A].num_of_byte = 1;
  def_instr[0x6A].num_of_cicles = 1;
  def_instr[0x6B].dis_fun = dis_XRL_A_REG;
  def_instr[0x6B].sim_fun = sim_XRL_A_REG;
  def_instr[0x6B].num_of_byte = 1;
  def_instr[0x6B].num_of_cicles = 1;
  def_instr[0x6C].dis_fun = dis_XRL_A_REG;
  def_instr[0x6C].sim_fun = sim_XRL_A_REG;
  def_instr[0x6C].num_of_byte = 1;
  def_instr[0x6C].num_of_cicles = 1;
  def_instr[0x6D].dis_fun = dis_XRL_A_REG;
  def_instr[0x6D].sim_fun = sim_XRL_A_REG;
  def_instr[0x6D].num_of_byte = 1;
  def_instr[0x6D].num_of_cicles = 1;
  def_instr[0x6E].dis_fun = dis_XRL_A_REG;
  def_instr[0x6E].sim_fun = sim_XRL_A_REG;
  def_instr[0x6E].num_of_byte = 1;
  def_instr[0x6E].num_of_cicles = 1;
  def_instr[0x6F].dis_fun = dis_XRL_A_REG;
  def_instr[0x6F].sim_fun = sim_XRL_A_REG;
  def_instr[0x6F].num_of_byte = 1;
  def_instr[0x6F].num_of_cicles = 1;
  def_instr[0x65].dis_fun = dis_XRL_A_DIR;
  def_instr[0x65].sim_fun = sim_XRL_A_DIR;
  def_instr[0x65].num_of_byte = 2;
  def_instr[0x65].num_of_cicles = 1;
  def_instr[0x66].dis_fun = dis_XRL_A_INDIR;
  def_instr[0x66].sim_fun = sim_XRL_A_INDIR;
  def_instr[0x66].num_of_byte = 1;
  def_instr[0x66].num_of_cicles = 1;
  def_instr[0x67].dis_fun = dis_XRL_A_INDIR;
  def_instr[0x67].sim_fun = sim_XRL_A_INDIR;
  def_instr[0x67].num_of_byte = 1;
  def_instr[0x67].num_of_cicles = 1;
  def_instr[0x64].dis_fun = dis_XRL_A_IMM;
  def_instr[0x64].sim_fun = sim_XRL_A_IMM;
  def_instr[0x64].num_of_byte = 2;
  def_instr[0x64].num_of_cicles = 1;
  def_instr[0x62].dis_fun = dis_XRL_DIR_A;
  def_instr[0x62].sim_fun = sim_XRL_DIR_A;
  def_instr[0x62].num_of_byte = 2;
  def_instr[0x62].num_of_cicles = 1;
  def_instr[0x63].dis_fun = dis_XRL_DIR_IMM;
  def_instr[0x63].sim_fun = sim_XRL_DIR_IMM;
  def_instr[0x63].num_of_byte = 3;
  def_instr[0x63].num_of_cicles = 2;
  def_instr[0xA5].dis_fun = dis_RESERVED;
  def_instr[0x85].dis_fun = dis_okMOV_DIR_DIR;
};

