/****************************************************************/
/* 8051 Simulator by Alberto Firpo                              */
/*     e-mail: firpo@educ.disi.unige.it                         */
/****************************************************************/
/* Symbol Table organization:
  A table of records each storing a symbol letter. 
*/

#include <stdio.h>
#include "sim8051.h"
#include "symbTab.h"
#include "sim_io.h"

#define MAX_SYMB_SIZE 150

#define FOUND 0
#define NF_ADD_FOLL 1
#define NF_ADD_NEXT 2

#define TERMINAL 1
#define ALTERNATIVE 2  /* elsewhere prev means the previous char in the symb */

typedef struct node {
		struct node * prev; /* previous node */
		struct node * next; /* alternative letter in table */
		struct node * foll; /* following letter in the symbol */
		INT8 type;
		unsigned char letter;
		INT16 value; /* valid only in terminal */
} Tnode;

static Tnode * table= NULL;
static Tnode * * valTab = NULL;
static int count = 0;
static int index = 0;
static unsigned char line[MAX_SYMB_SIZE];

/**********************************************************/
static Tnode * allocNode(unsigned char c,Tnode * prev)
{
	Tnode * p = (Tnode *)malloc(sizeof(Tnode));
	p->prev = prev;
	p->next = NULL;
	p->foll = NULL;
	p->type = 0;
	p->letter = c;
	return p;
}

/**********************************************************/
static void setValue(Tnode * p, INT16 val)
{
	p->value = val;
	p->type |= TERMINAL;
}

/**********************************************************/
static void setAlternative(Tnode * p)
{
	p->type |= ALTERNATIVE;
}

/**********************************************************/
/* Search in the subtable of possible alternative */
/* If not found return the point from wich start the insert */
static int findChar(unsigned char c,Tnode ** pnode)
{
    while (1)  /* recursion rapes stack! */
    {
	if ( (*pnode)->letter == c )
		return 1;  /*found!!*/
	else
		if ( (*pnode)->next == NULL )
			return 0; /* theresnt other alternative */
		else
			(*pnode) = (*pnode)->next;
    }
}

/**********************************************************/
/* If not found return the point from wich start the insert (prev value)*/
/* If found return the Terminal node */
/* *str point the first char not found (or \0  or ' ' if Found!!) */
/* NOTE: ' ' (blank) is not a valid char for Symbols */
static Tnode * findSymbol(unsigned char ** str,int * result)
{
	Tnode * pnode = table;
	if (pnode==NULL)  /* special case */
	{
		*result = NF_ADD_NEXT;
		return NULL;
	}
	while (1)  /* recursion rapes stack! */
	{
		if ( ! findChar(*(*str), &pnode))
		{
			*result = NF_ADD_NEXT;
			break; /* you have to add an alternative */
		}
		(*str)++;
		if ( ( *(*str) != '\0' ) && ( *(*str) != ' ' ) )
			if (pnode->foll != NULL)
				pnode = pnode->foll;
			else
			{
				*result = NF_ADD_FOLL;
				break; /* you have to add a foll */
			}
		else
		{
			*result = FOUND;
			break;
		}
	}
	return pnode;
}

/**********************************************************/
/* append from current p node the symbol and set the Terminal with value */
static void append(Tnode * p, unsigned char * str, INT16 val)
{
	Tnode * newNode;
	while (*str != '\0')
	{
		newNode = allocNode(*(str++),p);
		p->foll = newNode;
		p = newNode;
	}
	setValue(p, val);
}

/**********************************************************/
/* 0: OK
   1: symbol already defined
*/
static int addSymbol(unsigned char * str, INT16 val)
{
	Tnode * p;
	Tnode * newNode;
	int what2do;
	unsigned char * simb = str;
	p = findSymbol(&simb, &what2do);
	if (what2do == FOUND)
	{
		if (p->type & TERMINAL)
			return 1;
		else
		{
			setValue(p, val);
			return 0;
		}
	}
	if (what2do == NF_ADD_FOLL)
	{
		append(p,simb,val);
	}
	if (what2do == NF_ADD_NEXT)
	{
		newNode = allocNode(*(simb++),p);
		if (p != NULL) /* manage special case */
			p->next = newNode;
		else
			table = newNode;
		setAlternative(newNode);
		append(newNode,simb,val);
	}
	return 0;
}

/**********************************************************/
static void freeSubTable(Tnode * p)
{
	if (p->foll != NULL)
		freeSubTable(p->foll);
	if (p->next != NULL)
		freeSubTable(p->next);
	free(p);
	count = 0;
	index = 0;
}

/**********************************************************/
static void freeTable()
{
	if (table != NULL)
		freeSubTable(table);
	table = NULL;
	if (valTab != NULL)
		free(valTab);
	valTab = NULL;
}

/**********************************************************/
static INT16 read_value(unsigned char * s)
{
  long int val;
  s[4] = '\0';
  val = strtol(s, NULL, 16);
  s[4] = ' ';
  return val;
}

/**********************************************************
 Return values: NULL = Load Terminated Correctly
                <string> = Literaly Error Description when errno is 0
		"see errno" = use perror to get error description		
**********************************************************/
/*
  Pass 1: Building a table of letter composing simbols and count them (N)
  Pass 2: Allocate a size N array of pointer to a terminal simbol.
  Pass 3: Insert in the array every terminal found in the table
  Pass 4: array sort.
*/
static unsigned char * Pass1(FILE * sym)
{
	int i;

	while (1)
	{
		line[0] = '\0';
		if ( fgets(line,MAX_SYMB_SIZE,sym) == NULL )
		{
			break;
		}
		count++;
		if ( (line[4] != ' ') || (strlen(line) < 6) )
			return("Invalid Symbol");
		/* adjust end of line */
		for (i=strlen(line)-1; 1 ; i--)
			if ( (line[i] == '\n') || (line[i] == '\r') )
				line[i] = '\0';
			else
				break;
		if (addSymbol(&line[5], read_value(line)))
		{
			sim_print("Multiple definition of Symbol:");
			sim_print_n(&line[5]);
		}
	}
	return NULL;
}

/*  Pass 2: Allocate a size N array of pointer to a terminal simbol. */
void Pass2()
{
	int i;
	valTab = (Tnode * *)malloc(count*sizeof(Tnode *));
	for (i=0; i<count ; i++)
		valTab[i] = NULL;
	index = 0;
}

/*  Pass 3: Insert in the array every terminal found in the table */
void indexSubTable(Tnode * p)
{
	if (index == count)  /* already terminated */
		return;
	if (p->type & TERMINAL)
	{
		valTab[index++] = p;
	}
	if (p->foll != NULL)
		indexSubTable(p->foll);
	if (p->next != NULL)
		indexSubTable(p->next);
}

void Pass4()
{
	/* Using Standard Bubble Sort Algorithm with check for
	   early termination */
	int i,j;
	Tnode * p;
	int change;
	for (i=1; i<count; i++)
	{
		change = 0;
		for (j=count-1; j>=i; j--)
			if (valTab[j-1]->value > valTab[j]->value)
			{
				p = valTab[j-1];
				valTab[j-1] = valTab[j];
				valTab[j] = p;
				change = 1;
			}
		if ( ! change )
			break;
	}
}

unsigned char * loadSymbolTable(unsigned char * filename)
{
	FILE * sym;
	unsigned char * ret;

	if (table != NULL)
		freeTable();
	sym = fopen(filename, "r");
	if (sym == NULL)
		return "see errno";
	ret = Pass1(sym);
	fclose(sym);
	if (table != NULL)
	{
		Pass2();
		indexSubTable(table);  /* Pass3 */
		Pass4();
	}
	return ret;
}

int getSymbolTableDim()
{
	return count;
}

/**********************************************************/
static void new_line(int n)
{
	int i;
	sim_print_n("");
	for (i=0; i<n; i++)
		sim_putchar(' ');
}

/**********************************************************/
/* Only for debug purposes; Show the table structure with Terminals */
static void printSubTable(Tnode * p,int next_line,int curr)
{
	int n;
	n = next_line;
	sim_putchar(p->letter);
	if (p->type & TERMINAL)
		sim_putchar('*');
	if (p->next != NULL)
		n = curr;
	if (p->foll != NULL)
		printSubTable(p->foll,n,curr+1);
	else
		new_line(n);
	if (p->next != NULL)
		printSubTable(p->next,next_line,curr);
}

unsigned char * getSymbolName(Tnode * p)
{
	int alternative = 0;
	int i=MAX_SYMB_SIZE-1;
	line[i] = '\0';
	while (p != NULL)
	{
		if ( ! alternative )
			line[--i] = p->letter;
		alternative = (p->type) & ALTERNATIVE;
		p = p->prev;
	}
	return(&line[i]);
}

void printSymbolValue(INT16 v)
{
	unsigned char str[8];
	sprintf(str,"%.4X",v);
	sim_print(str);
}

void printSymbolTable()
{
	int i;
	Tnode * p;
	if (table == NULL)
	{
		sim_print_n("EMPTY Symbol Table");
		return;
	}
	#ifdef DEBUG
	printSubTable(table,0,0);
	#else
	for (i=0; i< count ; i++)
	{
		p = valTab[i];
		if (p != NULL)
		{
			sim_print(getSymbolName(p));
			sim_print(" = ");
			printSymbolValue(p->value);
			sim_print_n("");
		}
	}
	#endif
}

/* Binary search in array for val */
unsigned char * getSymbol(INT16 val)
{
	int i=count / 2;
	int size = count /2;
	while (size>=1)
	{
		size /= 2;
		if (valTab[i]->value == val)
			return getSymbolName(valTab[i]);
		else if (valTab[i]->value > val)
		{
			i -= size;
			if (i<0)
				i = 0;
		}
		else if (valTab[i]->value < val)
		{
			i += size;
			if (i>=count)
				i = count-1;
		}
	}
	return NULL;
}

int findSymbolVal(unsigned char * s, INT16 * val)
{
	Tnode * p;
	int what2do;
	unsigned char * simb = s;
	p = findSymbol(&simb, &what2do);
	if (what2do == FOUND)
	{
		*val = p->value;
		return 1;
	}
	return 0;
}
