/****************************************************************/
/* 8051 Simulator by Alberto Firpo                              */
/*     e-mail: firpo@educ.disi.unige.it                         */
/****************************************************************/
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include "sim8051.h"
#include "mem8051.h"
#include "dis_io.h"
#include "symbTab.h"

extern FILE * yyin;

unsigned long CYCLES=1;

static manual_break=-1;

static INT16 breakpoints[10];
static INT16 ignore_breakpoint=0;

INT16 disass_addr;   /* used also for code dump */
INT16 external_addr;
INT8 direct_addr;
INT8 indirect_addr;

void copyright()
{
   sim_print_n("******************************************************************");
   sim_print("*****  Alberto Firpo,   805");
#ifdef SIM8052
   sim_print("2");
#else
   sim_print("1");
#endif
   sim_print_n(" Simulator       Copyright 1996  *****");
   sim_print_n("******************************************************************");
}

void prompt()
{
#ifdef SIM8052
	sim_print("AF-8052-Sim>");
#else
	sim_print("AF-8051-Sim>");
#endif
}

void load(unsigned char * filename)
{
	unsigned char *p;
	if( (p = load_intel(filename)) == NULL)
	{
		sim_print_n("Load OK");
	}
	else
	{
		if (errno)
			p = strerror(errno);	
		sim_print("ERROR: ");	
		sim_print_n(p);	
	}
}

/********************* SYMBOLS ********************************/
/* convert string to value; string can be a Symbol reference or an
   hexadecimal value */
int str2Address(unsigned char * s, INT16 * addr)
{
	int i,j;
	while (*(s) == ' ') 
		s++;  /* skip initial blanks */
	if ( findSymbolVal(s,addr) )
		return 1;
	if ( (i = strlen(s)) == 0 )
	{
		sim_print_n("ERROR: bug in str2Address");
		return 0;
	}
	for (j=0; j<i; j++)
	{
		if ( s[j] == ' ' )
			break;
		if ( ! isxdigit(s[j]) )
		{
			sim_print_n("Symbol not Found");
			return 0;
		}
	}
	*addr = strtol(s, (char **)NULL, 16);
	return 1;
}

void loadSymb(unsigned char * filename)
{
	unsigned char *p;
	sim_print("Loading Symbol Table...");
	if( (p = loadSymbolTable(filename)) == NULL)
	{
		sim_print_n("OK");
	}
	else
	{
		if (errno)
			p = strerror(errno);	
		sim_print("ERROR: ");	
		sim_print_n(p);	
	}
}

/********************* BREAKPOINTS ********************************/
int check_breakpoint()
{
  int i;
  if ( PC == 0)
  	return 0;
  if (ignore_breakpoint == PC)
	return 0;
  for (i=0; i< 10; i++)
  	if (breakpoints[i] == PC)
  		return 1;
  return 0;
}

void set_breakpoint(unsigned char * saddr)
{
  int i;
  char s[2];
  INT16 addr;
  s[1] = 0;
  if ( ! str2Address(saddr,&addr) )
  	return;
  if (addr == 0)
  {
  	sim_print_n("0 is the Overflow/Reset address, it is a auto-breakpoint");
  	return;
  }
  for (i=0; i< 10; i++)
  	if (breakpoints[i] == 0)
  	{
  		breakpoints[i] = addr;
  		s[0] = 0x30+i;
  		sim_print("Breakpoint ");
  		sim_print(s);
  		sim_print(" ===> ");
  		sim_print_instr(breakpoints[i]);
  		return;
  	}
  sim_print_n("Breakpoint list Full");
}

void del_breakpoint(int n)
{
  int i;
  char s[2];
  s[1] = 0;
  if ( n == 0xff )
	  for (i=0; i< 10; i++)
  		breakpoints[n] = 0;
  else
	  if (breakpoints[n] != 0)
  		breakpoints[n] = 0;
	  else
	  {
  		s[0] = 0x30+n;
  		sim_print("Breakpoint ");
  		sim_print(s);
  		sim_print_n(" No set");
	  }
}

void list_breakpoint()
{
  int i;
  char s[2];
  s[1] = 0;
  for (i=0; i< 10; i++)
  	if (breakpoints[i] != 0)
  	{
  		s[0] = 0x30+i;
  		sim_print("Breakpoint ");
  		sim_print(s);
  		sim_print(" ===> ");
  		sim_print_instr(breakpoints[i]);
  	}
}

void make_steps(int n)
{
	manual_break = 0;
	if (n==0)
		n=1;
	while (n)
	{
		if (check_breakpoint())
		{
			sim_print_status();
			sim_print_n("*** Breakpoint ***");
			ignore_breakpoint = PC;
			break;
		}
		ignore_breakpoint = 0;
	 	do_step();
		if (n>0) 
		{
			sim_print_status();
			n--;
		}
		if (manual_break == 1)
		{
			if (n<0) sim_print_status();
			sim_print_n("");
			sim_print_n("*** Manual Break ***");
			break;
		}
		if ( readDirect(PCON) & 0x02 )
		{
			sim_print_n("Power Down mode Entered");
			break;
		}
		if (PC == 0)
		{
			sim_print_status();
			sim_print_n("*** 0 address reached ***");
			break;
		}
	}
	manual_break = -1;
}

void make_disassemble()
{
	int n;
	char s[80];
	for (n=0; n< LINES_FOR_DUMPS; n++)
	{
		disass_addr = do_disassembling(disass_addr, s);
		sim_print_n(s);
	}
}

int set_addr(char type, unsigned char * saddr)
{
	INT16 addr;
	if ( ! str2Address(saddr,&addr) )
  		return 0;
	switch (type)
	{
	  case 'C':
	  case 'c':
		disass_addr = addr;
		break;
	  case 'X':
	  case 'x':
		external_addr = addr;
		break;
	  case 'D':
	  case 'd':
		direct_addr = (INT8)addr;
		break;
	  case 'i':
	  case 'I':
		indirect_addr = (INT8)addr;
		break;
	}
	return 1;
}

void make_dump(char type)
{
	int n,i;
	char s[80];
	INT8 sp[BYTES_FOR_DUMP_LINE+1];
	for (n=0; n< LINES_FOR_DUMPS; n++)
	{
		dis_start(s);
		switch (type)
		{
		  case 'C':
		  case 'c':
			dis_printINT16(disass_addr,NO_H);
			dis_print(" : ");
			for(i=0; i<BYTES_FOR_DUMP_LINE; i++)
			{
				sp[i] = CMEM[disass_addr+i];
				dis_printINT8(sp[i],NO_H);
				dis_printchar(' ');
				if ( ! isprint(sp[i]) )
					sp[i] = '.';
			}
			sp[i] = 0;
			disass_addr += BYTES_FOR_DUMP_LINE;
			break;
		  case 'X':
		  case 'x':
			dis_printINT16(external_addr,NO_H);
			dis_print(" : ");
			for(i=0; i<BYTES_FOR_DUMP_LINE; i++)
			{
				sp[i] = readExtRam(external_addr+i);
				dis_printINT8(sp[i],NO_H);
				dis_printchar(' ');
				if ( ! isprint(sp[i]) )
					sp[i] = '.';
			}
			sp[i] = 0;
			external_addr += BYTES_FOR_DUMP_LINE;
			break;
		  case 'D':
		  case 'd':
			dis_printINT8(direct_addr,NO_H);
			dis_print(" : ");
			for(i=0; i<BYTES_FOR_DUMP_LINE; i++)
			{
				sp[i] = readDirect(direct_addr+i);
				dis_printINT8(sp[i],NO_H);
				dis_printchar(' ');
				if ( ! isprint(sp[i]) )
					sp[i] = '.';
			}
			sp[i] = 0;
			direct_addr += BYTES_FOR_DUMP_LINE;
			break;
		  case 'i':
		  case 'I':
			dis_printINT8(indirect_addr,NO_H);
			dis_print(" : ");
			for(i=0; i<BYTES_FOR_DUMP_LINE; i++)
			{
				sp[i] = readIndirect(indirect_addr+i);
				dis_printINT8(sp[i],NO_H);
				dis_printchar(' ');
				if ( ! isprint(sp[i]) )
					sp[i] = '.';
			}
			sp[i] = 0;
			indirect_addr += BYTES_FOR_DUMP_LINE;
			break;
		}
		/* Print printable chars */
		dis_print(" = ");
		dis_print(sp);
		/* print resulting line */
		dis_end();
		sim_print_n(s);
	}
}

void make_patch(char type,unsigned char * saddr, INT8 val)
{
  INT16 addr;
	if ( ! str2Address(saddr,&addr) )
  		return;
	switch (type)
	{
	  case 'C':
	  case 'c':
		CMEM[addr] = val;
		break;
	  case 'X':
	  case 'x':
		writeExtRam(addr, val);
		break;
	  case 'D':
	  case 'd':
		writeDirect(addr, val);
		break;
	  case 'i':
	  case 'I':
		writeIndirect(addr, val);
		break;
	}
}

void manualbreak()
{
   signal(SIGINT, manualbreak);
   if (manual_break == 0)
   	manual_break = 1;
}

main(int argc, char ** argv)
{
   int i;

   /* Initialization */
   init_instTab();

   signal(SIGINT, manualbreak);

   /* variables for dumps */
   disass_addr = 0;
   external_addr = 0;
   direct_addr = 0;
   indirect_addr = 0;

   /* breakpoints */
   del_breakpoint(0xff);

   /* Force power up conditions */
   resetSFR();

   /* Hello world !! */
   copyright();

   /* Implicit load command if filename is specified on invocation */
   if (argc >= 2)
	load(argv[1]);

   /* Implicit load command if filename is specified on invocation */
   if (argc == 3)
	loadSymb(argv[2]);

   prompt();

   while (1)
   {
       yylex();
       yyin = stdin;
   }
}
