/****************************************************************/
/* 8051 Simulator by Alberto Firpo                              */
/*     e-mail: firpo@educ.disi.unige.it                         */
/****************************************************************/
#include <stdio.h>
#include "mem8051.h"
#include "tim8051.h"
#include "int8051.h"

static unsigned long lastCYCLES;
static BIT int1pin;
static BIT int0pin;
static BIT t1pin;
static BIT t0pin;

#ifdef SIM8052
static BIT t2pin;
static BIT t2expin;
#endif

/* Define for TCON bits */
#define TF1 TCON+7
#define TR1 TCON+6
#define TF0 TCON+5
#define TR0 TCON+4
#define IE1 TCON+3
#define IT1 TCON+2
#define IE0 TCON+1
#define IT0 TCON+0

static void tim0TH0Clock(INT8 ck)
{
	INT8 th0;
	if (ck == 0)
		return;
	if (readBit(TR1))
	{
		th0 = readDirect(TH0);
		writeDirect(TH0, th0 + ck);
		if ( (INT8)(th0+ck) < th0 )  /* Overflow */
		{
			gen_int(INT_TIMER1);
			writeBit(TF1, 1);  /* TF1 is set by hardware */
		}
	}
}

static void tim0Clock(INT8 ck)
{
	INT8 val, val1;
	INT8 tmod = readDirect(TMOD);
	if (ck == 0)
		return;
	if (readBit(TR0) && ( (int0pin==1) || ((tmod & 0x8)==0) ) )
	{
		val = readDirect(TL0);
		if ((tmod & 0x3) == 0)  /* Mode0 */
		{
			writeDirect(TL0, val+ck);
			if ( (INT8)((val+ck) & 0x1f) < (val & 0x1f) ) /*Overflow*/
			{
				val1 = readDirect(TH0);
				val = (val+ck+1) & 0x1f; /*val contains Overflow*/
				writeDirect(TH0, val1+val );
				if ((INT8)(val+val1) < val1)  /*Overflow*/
				{
					gen_int(INT_TIMER0);
					writeBit(TF0, 1);  /* TF0 is set by hardware */
				}
			}
		}
		else if ((tmod & 0x3) == 0x1)  /* Mode1 */
		{
			writeDirect(TL0, val+ck);
			if ( (INT8)(val+ck) < val ) /*Overflow*/
			{
				val1 = readDirect(TH0);
				val += ck+1;   /* val contains Overflow */
				writeDirect(TH0, val1+val);
				if ((INT8)(val+val1) < val1)  /*Overflow*/
				{
					gen_int(INT_TIMER0);
					writeBit(TF0, 1);  /* TF0 is set by hardware */
				}
			}
		}
		else if ((tmod & 0x3) == 0x2)  /* Mode2 */
		{
			if ( (INT8)(val+ck) < val ) /*Overflow*/
			{
				writeDirect(TL0, readDirect(TH0)); /* Reload */
				gen_int(INT_TIMER0);
				writeBit(TF0, 1);  /* TF0 is set by hardware */
			}
			else
				writeDirect(TL0, val+ck);
		}
		else if ((tmod & 0x3) == 0x3)  /* Mode3 */
		{
			writeDirect(TL0, val+ck);
			if ( (INT8)(val+ck) < val ) /*Overflow*/
			{
				gen_int(INT_TIMER0);
				writeBit(TF0, 1);  /* TF0 is set by hardware */
			}
		}
	}
}

static void tim1Clock(INT8 ck)
{
	INT8 val, val1;
	INT8 tmod = readDirect(TMOD);
	if ( (ck == 0) || ((tmod & 0x30) == 0x30) )  /* Mode3 => Stopped */
		return;
	if (readBit(TR1) && ( (int1pin==1) || ((tmod & 0x80)==0) ) )
	{
		val = readDirect(TL1);
		if ((tmod & 0x30) == 0)  /* Mode0 */
		{
			writeDirect(TL1, val+ck);
			if ( (INT8)((val+ck) & 0x1f) < (val & 0x1f) ) /*Overflow*/
			{
				val1 = readDirect(TH1);
				val = (val+ck+1) & 0x1f; /*val contains Overflow*/
				writeDirect(TH1, val1+val );
				if ((INT8)(val+val1) < val1)  /*Overflow*/
				{
					gen_int(INT_TIMER1);
					writeBit(TF1, 1);  /* TF1 is set by hardware */
				}
			}
		}
		else if ((tmod & 0x30) == 0x10)  /* Mode1 */
		{
			writeDirect(TL1, val+ck);
			if ( (INT8)(val+ck) < val ) /*Overflow*/
			{
				val1 = readDirect(TH1);
				val += ck+1;   /* val contains Overflow */
				writeDirect(TH1, val1+val);
				if ((INT8)(val+val1) < val1)  /*Overflow*/
				{
					gen_int(INT_TIMER1);
					writeBit(TF1, 1);  /* TF1 is set by hardware */
				}
			}
		}
		else if ((tmod & 0x30) == 0x20)  /* Mode2 */
		{
			if ( (INT8)(val+ck) < val ) /*Overflow*/
			{
				writeDirect(TL1, readDirect(TH1)); /* Reload */
				gen_int(INT_TIMER1);
				writeBit(TF1, 1);  /* TF1 is set by hardware */
			}
			else
				writeDirect(TL1, val+ck);
		}
		/* in Mode 3 the timer is stopped */
	}
}

#ifdef SIM8052
static void tim2Clock(INT8 ck)
{
	INT8 val, val1;
	INT8 tmod = readDirect(TMOD);
	if (ck == 0) 
		return;
	if (readBit(T2CON+2) )  /* TR2 */
	{
		val = readDirect(TL2);
		writeDirect(TL2, val+ck);
		if ( (INT8)(val+ck) < val ) /*Overflow*/
		{
			val1 = readDirect(TH2);
			val += ck+1;   /* val contains Overflow */
			writeDirect(TH2, val1+val);
			if ((INT8)(val+val1) < val1)  /*Overflow*/
			{
				gen_int(INT_T2);
				writeBit(T2CON+7, 1);  /* TF2 is set by hardware */
			        /* Reload */
				writeDirect(TL2, readDirect(RCAP2L));
				writeDirect(TH2, readDirect(RCAP2H));
			}
		}
	}
}
#endif

void intTim1Served()
{
	writeBit(TF1, 0);  /* TF1 is cleared by hardware */
	clear_int(INT_TIMER1);
}

void intExt1Served()
{
	if ( readBit(IT1) )  /* IT1 if transition activated */
	{
		writeBit(IE1, 0);  /* IE1 is cleared by hardware */
		clear_int(INT_EXT1);
	}
}

void intTim0Served()
{
	writeBit(TF0, 0);  /* TF0 is cleared by hardware */
	clear_int(INT_TIMER0);
}

void intExt0Served()
{
	if ( readBit(IT0) )  /* IT0 if transition activated */
	{
		writeBit(IE0, 0);  /* IE0 is cleared by hardware */
		clear_int(INT_EXT0);
	}
}

void setINT0pin(BIT val)
{
	if (int0pin == val)  /* Nothing is changed */
		return;
	if ( readBit(IT0) )  /* IT0 if transition activated */
	{
		if ( val==0 )
		{
			gen_int(INT_EXT0);
			writeBit(IE0, 1);  /* IE0 is set by hardware */
		}
	}
	else  /* Level activated */
	{
		if (val==0)
			gen_int(INT_EXT0);
		else
			clear_int(INT_EXT0);
		writeBit(IE0, val?0:1);  /* IE0 is set/cleared by hardware */
	}
	int0pin = val;
}

void setINT1pin(BIT val)
{
	if (int1pin == val)  /* Nothing is changed */
		return;
	if ( readBit(IT1) )  /* IT1 if transition activated */
	{
		if ( val==0 )
		{
			gen_int(INT_EXT1);
			writeBit(IE1, 1);  /* IE1 is set by hardware */
		}
	}
	else  /* Level activated */
	{
		if (val==0)
			gen_int(INT_EXT1);
		else
			clear_int(INT_EXT1);
		writeBit(IE1, val?0:1);  /* IE1 is set/cleared by hardware */
	}
	int1pin = val;
}

void tim_poll(void)
{
	INT8 tmod = readDirect(TMOD);
	if ( (tmod & 0x40) == 0 )  /* CT1 */
		tim1Clock(CYCLES-lastCYCLES);
	if ( (tmod & 0x4) == 0 )  /* CT0 */
		tim0Clock(CYCLES-lastCYCLES);
	if ( (tmod & 0x3) == 0x3 )  /* Timer0 in Mode 3 */
		tim0TH0Clock(CYCLES-lastCYCLES);
#ifdef SIM8052
	if ( readBit(T2CON+1) == 0 )  /* CT2 */
		tim2Clock(CYCLES-lastCYCLES);
#endif
	lastCYCLES = CYCLES;
}

void tim_reset(void)
{
	int1pin = 1;
	int0pin = 1;
	t1pin = 1;
	t0pin = 1;
	lastCYCLES = CYCLES;
}

void setT0pin(BIT val)
{
	if (t0pin == val)  /* Nothing is changed */
		return;
	if ( val == 0)
		if ( readDirect(TMOD) & 0x4 )
			tim0Clock(1);
	t0pin = val;
}

void setT1pin(BIT val)
{
	if (t1pin == val)  /* Nothing is changed */
		return;
	if ( val == 0)
		if ( readDirect(TMOD) & 0x40 )
			tim1Clock(1);
	t1pin = val;
}

#ifdef SIM8052
void setT2EXpin(BIT val)
{
	if (t2expin == val)  /* Nothing is changed */
		return;
	if ( val==0 )
	{
		if (readBit(T2CON+3) )  /* EXEN2 */
		{
			gen_int(INT_T2);
			writeBit(T2CON+6, 1);  /* EXF2 is set by hardware */
			/* Test bit CP/RL2 and NOT [RT]CLK */
			if ((readDirect(T2CON) & 0x31) == 1) 
			{  /* Capture */
				writeDirect(RCAP2L, readDirect(TL2));
				writeDirect(RCAP2H, readDirect(TH2));
			}
			else
			{   /* Reload */
				writeDirect(TL2, readDirect(RCAP2L));
				writeDirect(TH2, readDirect(RCAP2H));
			}
		}
	}
	t2expin = val;
}

void setT2pin(BIT val)
{
	if (t2pin == val)  /* Nothing is changed */
		return;
	if ( val == 0)
		if ( readBit(T2CON+1) )  /* C/T2 Pin */
			tim2Clock(1);
	t2pin = val;
}

void intTim2Served()
{
	clear_int(INT_T2);
}

#endif
