/****************************************************************/
/* 8051 Simulator by Alberto Firpo                              */
/*     e-mail: firpo@educ.disi.unige.it                         */
/****************************************************************/
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <fcntl.h>
#include "mem8051.h"
#include "ser8051.h"

/* Flags to display Hex-values instead bytes on Tx */
int serShowHex = 0;

/* Flags to simulate real Tx/Rx speed */
int serRealSpeed = 0;
#define CYCLES4BYTE 10  /* cycles to Tx/Rx byte if NO Real Speed Enabled */

static FILE * serttyo = NULL;
static FILE * serttyi = NULL;
static INT8 SerialBuffer;
static long cycles4TI;
static long cycles4RI;

#define REN_BIT SCON+4  /* REN = Receive Enable */
#define RI_BIT SCON+0   /* RI = Receive Bit (Set by hardware)*/
#define TI_BIT SCON+1   /* TI = Transmit Bit (Set by hardware)*/


#define TX_DIR 0
#define RX_DIR 1
/*
 Returns the number of Cycles (= OscFreq / 12) to complete a Rx/Tx operation 
*/
static int cycles4byte(int direction)
{
	int ret;
	INT8 scon;
#ifdef SIM8052
	INT8 t2con;
#endif
	if ( ! serRealSpeed )
		return CYCLES4BYTE;
#ifdef SIM8052
	t2con = readDirect(T2CON);
#endif
	scon = readDirect(SCON);
	switch (scon & 0xc0) 
	{
	  case 0:      /* Mode 0  8 bit transmitted */
		ret = 8;  /* Baud Rate = OscFreq / 12 */
		break;
	  case 0x80:   /* Mode 2  11 bit transmitted */
		if ( readDirect(PCON) & 0x80 )
			ret = 29;  /* Baud Rate = OscFreq / 32 */
		else
			ret = 59;  /* Baud Rate = OscFreq / 64 */
		break;
	  case 0x40:   /* Mode 1  10 bit transmitted */
#ifdef SIM8052
		if ( ( (t2con & 0x10) && (direction == TX_DIR) )
		   || ( (t2con & 0x20) && (direction == RX_DIR) ) )
		{
			ret = 10*32*(65536-buildINT16(readDirect(RCAP2H), readDirect(RCAP2L)));
			ret /= 12;
		}
		else
#endif
		{
			ret = (10*32*(256-readDirect(TH1)));
			if (readDirect(PCON) & 0x80)
				ret /= 2;
		}
		break;
	  case 0xc0:   /* Mode 3  11 bit transmitted */
#ifdef SIM8052
		if ( ( (t2con & 0x10) && (direction == TX_DIR) )
		   || ( (t2con & 0x20) && (direction == RX_DIR) ) )
		{
			ret = 11*32*(65536-buildINT16(readDirect(RCAP2H), readDirect(RCAP2L)));
			ret /= 12;
		}
		else
#endif
		{
			ret = (11*32*(256-readDirect(TH1)));
			if (readDirect(PCON) & 0x80)
				ret /= 2;
		}
		break;
	}
	return ret;
}

void ser_out(INT8 c)
{
	if (serttyo)
	{
		if (serShowHex)
			fprintf(serttyo, "[%X]", c);
		else
			fputc(c, serttyo);
	}
	cycles4TI = CYCLES + cycles4byte(TX_DIR);
}

INT8 ser_in(void)
{
	return SerialBuffer;
}

void ser_poll(void)
{
	int c=-1;
	if ( cycles4TI && (cycles4TI <= CYCLES))
	{
		writeBit(TI_BIT,1);   /* TI = Transmit Bit (Set by hardware)*/
		cycles4TI = 0;
	}
	if ( serttyi && (cycles4RI <= CYCLES) )
	{
		c = fgetc(serttyi); 
		if ( c != -1 )
		{
			SerialBuffer = c;
			if (readBit(REN_BIT))
				writeBit(RI_BIT,1);   /* Set by hardware */
			/* wait before the next byte will be received */
			cycles4RI = CYCLES + cycles4byte(RX_DIR); 
		}
	}
}

void ser_reset(void)
{
	SerialBuffer = 0;
	cycles4TI = 0;
	cycles4RI = 0;
}

int ser_open(char * ttyIname, char * ttyOname)
{
	int f;
	if (serttyi)
		fclose(serttyi);
	f = open(ttyIname, O_RDONLY+O_NONBLOCK);
	if ( f != -1 )
		serttyi = fdopen(f, "r");
	if (serttyi == NULL)
		fprintf(stderr,"INPUT SERIAL DEVICE ERROR: %s\n",strerror(errno));
	else
	{
		setvbuf(serttyi, NULL, _IONBF, 1);
	}
	if (serttyo)
		fclose(serttyo);
	serttyo = fopen(ttyOname, "a");
	if (serttyo == NULL)
		fprintf(stderr,"OUTPUT SERIAL DEVICE ERROR: %s\n",strerror(errno));
	else
	{
		setvbuf(serttyo, NULL, _IONBF, 1);
	}
	ser_reset();
}

