/****************************************************************/
/* 8051 Simulator by Alberto Firpo                              */
/*     e-mail: firpo@educ.disi.unige.it                         */
/****************************************************************/
#include <stdio.h>
#include "sim8051.h"
#include "mem8051.h"
#include "int8051.h"
#include "ser8051.h"
#include "tim8051.h"
#include "sim_io.h"

/* Program Counter */
INT16 PC;

/* Program Memory 64K bytes */
INT8 CMEM[0x10000];

/* External Data Memory 64K bytes */
static INT8 XMEM[0x10000];

/* Special Function Registers (SFR) Memory */
static INT8 SFR[0x80];  /* [DIRECT ACCESS] */

/* Internal Data Memory 128 bytes (256 for 8052) */
#ifdef SIM8052
static INT8 DMEM[0x100]; /* [DIRECT & INDIRECT] + [INDIRECT] */
#else
static INT8 DMEM[0x80];	/* [DIRECT & INDIRECT] */
#endif

/* Flags for Single Memory Space ( /PSEN & /RD ) */
int singleMemSpace = 0;

/****************************************************************************/
/*                               IMPLEMENTATION                             */
/****************************************************************************/
#define READ_FUN 0
#define WRITE_FUN 1
typedef INT8 (*fun_sfr_t)(INT8,INT8,INT8);

void pushINT8(INT8 val)
{
	SFR[SP-0x80] += 1;   /* SP <- SP + 1 */
	writeIndirect(SFR[SP-0x80],val);   /* [(SP)] <- val */
}

INT8 popINT8(void)
{
	register INT8 val = readIndirect(SFR[SP-0x80]);  /* val <- [(SP)] */
	SFR[SP-0x80] -= 1;  /* SP <- SP - 1 */
	return val;
}

void pushPC(void)
{
	pushINT8(PC & 0xff);
	pushINT8(PC >> 8);
}

void popPC(void)
{
	PC = popINT8() << 8;
	PC = PC + popINT8();
}

static INT8 f_PSW(INT8 mode, INT8 Addr, INT8 val)
{
	if (mode == READ_FUN)   /* Calculate PARITY bit */
	{
		if (isParityOdd(readDirect(ACC)))
			return SFR[Addr-0x80] | 0x1;
		else
			return SFR[Addr-0x80] & 0xfe;
	}
	else
		SFR[Addr-0x80] = val;
	return 0;
}

static INT8 fSCON(INT8 mode, INT8 Addr, INT8 val)
{
	if (mode == READ_FUN)
		return SFR[Addr-0x80];
	else
	{
		SFR[Addr-0x80] = val;
		if ( val & 0x3 )
			gen_int(INT_SERIAL);
		else
			clear_int(INT_SERIAL);
	}
	return 0;
}

static INT8 fSBUF(INT8 mode, INT8 Addr, INT8 val)
{
	if (mode == READ_FUN)
		return ser_in();
	else
	{
		ser_out(val);
	}
	return 0;
}

static INT8 fIEIP(INT8 mode, INT8 Addr, INT8 val)
{
	if (mode == READ_FUN)
		return SFR[Addr-0x80];
	else
	{
		SFR[Addr-0x80] = val;
		delayInterrupt();
		/*
		if (val & 0x40)
			sim_error("1 written in reserved bit");
		*/
	}
	return 0;
}

static INT8 f_std(INT8 mode, INT8 Addr, INT8 val)
{
	if (mode == READ_FUN)
		return SFR[Addr-0x80];
	else
		SFR[Addr-0x80] = val;
	return 0;
}

/* Add here functions to manage peculiar functionality of SFR */

static fun_sfr_t fun_sfr[0x80] = {
/* 80 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* 88 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* 90 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* 98 */ fSCON,	fSBUF,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* A0 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* A8 */ fIEIP,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* B0 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* B8 */ fIEIP,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* C0 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
#ifdef SIM8052
/* C8 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std, 
#else
/* C8 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
#endif
/* D0 */ f_PSW,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* D8 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* E0 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* E8 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* F0 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,
/* F8 */ f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std,	f_std
};

void resetSFR(void)
{
	INT8 i;
	ser_reset();
	tim_reset();
	/* SFR contents after Reset */
	for (i=0; i<0x80; i++)
		SFR[i] = 0;
	SFR[SP-0x80] = 0x07;
	SFR[P0-0x80] = 0xff;
	SFR[P1-0x80] = 0xff;
	SFR[P2-0x80] = 0xff;
	SFR[P3-0x80] = 0xff;
#ifdef SIM8052
	SFR[IP-0x80] = 0xc0;
	SFR[IE-0x80] = 0x40;
#else
	SFR[IP-0x80] = 0xe0;
	SFR[IE-0x80] = 0x60;
#endif
	SFR[PCON-0x80] = 0x70;
	resetInterrupt();
	PC = 0;
}

INT8 readDirect(INT8 Daddr)
{
	if (Daddr > 0x7f)
	{
		return fun_sfr[Daddr-0x80](READ_FUN,Daddr,0);
	}
	else
		return DMEM[Daddr];
}

void writeDirect(INT8 Daddr, INT8 val)
{
	if (Daddr > 0x7f)
	{
		fun_sfr[Daddr-0x80](WRITE_FUN,Daddr,val);
	}
	else
		DMEM[Daddr] = val;
}


INT8 readIndirect(INT8 Iaddr)
{
#ifndef SIM8052
	if (Iaddr > 0x7f)
	{
		sim_error("Invalid Indirect Access ( > 0x7f )");
		return 0xff;
	}
#endif
		return DMEM[Iaddr];
}

void writeIndirect(INT8 Iaddr, INT8 val)
{
#ifndef SIM8052
	if (Iaddr > 0x7f)
	{
		sim_error("Invalid Indirect Access ( > 0x7f )");
		return;
	}
#endif
		DMEM[Iaddr] = val;
}


INT8 readExtRam(INT16 addr)
{
	return (singleMemSpace ? CMEM[addr] : XMEM[addr] );
}

void writeExtRam(INT16 addr, INT8 val)
{
	if (singleMemSpace)
		CMEM[addr] = val;
	else
		XMEM[addr] = val;
}


BIT readBit(INT8 Baddr)
{
	INT8 val;
	if (Baddr>=0x80)
		val = readDirect(Baddr & 0xf8);
	else
		val = readDirect(0x20+(Baddr>>3));
	return ( val & (0x1 << (Baddr & 0x7) ) ) ? 1 : 0;
}

void writeBit(INT8 Baddr, BIT bit)
{
	INT8 mask = (0x1 << (Baddr & 0x7) );
	if (Baddr>=0x80)
	{
		if (bit)
			writeDirect((Baddr & 0xf8), readDirect(Baddr & 0xf8) | mask);
		else
			writeDirect((Baddr & 0xf8), readDirect(Baddr & 0xf8) & ~mask);
	}
	else
	{
		if (bit)
			writeDirect(0x20+(Baddr>>3), readDirect(0x20+(Baddr>>3)) | mask);
		else
			writeDirect(0x20+(Baddr>>3), readDirect(0x20+(Baddr>>3)) & ~mask);
	}
}

INT8 readReg(INT8 reg)
{
	readDirect( reg | (readDirect(PSW) & 0x18) );
}

void writeReg(INT8 reg, INT8 val)
{
	writeDirect( reg | (readDirect(PSW) & 0x18), val );
}

INT16 readDPTR(void)
{
	return buildINT16(readDirect(DPH), readDirect(DPL));
}

void writeDPTR(INT16 val)
{
	writeDirect(DPL, val & 0xff);
	writeDirect(DPH, (val & 0xff00)>>8 );
}

char * sim_expand_PSW(void)
{
	static char s_psw[] = "caf00ofp(00)";
	INT8 psw = readDirect(PSW);
	sprintf(s_psw,"caf00ofp(%.2X)",psw);
	if (psw & 0x80)
		s_psw[0] = 'C';
	if (psw & 0x40)
		s_psw[1] = 'A';
	if (psw & 0x20)
		s_psw[2] = 'F';
	if (psw & 0x10)
		s_psw[3] = '1';
	if (psw & 0x8)
		s_psw[4] = '1';
	if (psw & 0x4)
		s_psw[5] = 'O';
	if (psw & 0x2)
		s_psw[6] = 'F';
	if (psw & 0x1)
		s_psw[7] = 'P';
	return s_psw;
}
