/****************************************************************/
/* 8051 Simulator by Alberto Firpo                              */
/*     e-mail: firpo@educ.disi.unige.it                         */
/****************************************************************/
#include <stdio.h>
#include "sim8051.h"
#include "sim_io.h"
#include "mem8051.h"
#include "int8051.h"
#include "ser8051.h"
#include "tim8051.h"

/* Flags to display when interrupts are served */
int intShowService = 0;

/* Register of Interrupt Pending */
static INT8 int_pending = 0;

/* Latch of Interrupt in Request */
static INT8 int_latch = 0;

/* Interrupt to be served */
static INT8 int2beserved = 0;

/* Delay Interrupt service */
static INT8 intdelayed = 0;

/* Interrupt Service In Progress*/
static INT8 intInProgress = 0;

void gen_int(INT8 interrupt)
{
	int_pending |= interrupt;
}

void clear_int(INT8 interrupt)
{
	int_pending &= ~interrupt;
}

char * getIntPending(void)
{
	INT8 val;
#ifndef SIM8052
	static char s_int[] = "-_txtx(00)";
	sprintf(s_int,"-_txtx(%.2X)",int_pending);
#else
	static char s_int[] = "__txtx(00)";
	sprintf(s_int,"__txtx(%.2X)",int_pending);
	if (int_pending & 0x20)
	{
		val = readDirect(T2CON) & 0xC0;
		if (val == 0xc0)
			s_int[0] = 'B';
		else if (val == 0x80)
			s_int[0] = 'T';
		else if (val == 0x40)
			s_int[0] = 'E';
		else
			s_int[0] = '?';
	}
#endif
	if (int_pending & 0x10)
	{
		val = readDirect(SCON) & 0x3;
		if (val == 0x3)
			s_int[1] = 'B';
		else if (val == 0x2)
			s_int[1] = 'T';
		else if (val == 0x1)
			s_int[1] = 'R';
		else
			s_int[1] = '?';
	}
	if (int_pending & 0x8)
		s_int[2] = 'T';
	if (int_pending & 0x4)
		s_int[3] = 'X';
	if (int_pending & 0x2)
		s_int[3] = 'T';
	if (int_pending & 0x1)
		s_int[3] = 'X';
	return s_int;
}

void resetInterrupt(void)
{
	int_pending = 0;
	int_latch = 0;
	int2beserved = 0;
	intdelayed = 0;
	intInProgress = 0;
}

/* returns 0 i NO interrupt request; otherwise the ISR begin address */
INT16 checkInterrupt(void)
{
   INT16 ret=0;
   INT8 ie = readDirect(IE);
   INT8 i,j;
   char * msg;
/*
TBD: levare la printf di debugg dalla checkInterrupt
printf("latch=0x%X int2beserved=0x%X  delay=%d  inProgress=0x%X\n", \
 int_latch, int2beserved, intdelayed, intInProgress); 
*/
/**********************************************************************
  - se devo servire un interrupt metto in ret l'indirizzo della ISR
*/
	switch (int2beserved)
	{
	  case 0:
		ret = 0;
		break;
#ifdef SIM8052
	  case INT_T2:
		ret = 0x2b;
		intTim2Served();
		msg = "Timer 2 Interrupt\n";
		break;
#endif
	  case INT_SERIAL:
		ret = 0x23;
		msg = "Serial Interrupt\n";
		break;
	  case INT_TIMER1:
		ret = 0x1b;
		intTim1Served();
		msg = "Timer 1 Interrupt\n";
		break;
	  case INT_EXT1:
		ret = 0x13;
		intExt1Served();
		msg = "External 1 Interrupt\n";
		break;
	  case INT_TIMER0:
		ret = 0xb;
		intTim0Served();
		msg = "Timer 0 Interrupt\n";
		break;
	  case INT_EXT0:
		ret = 0x3;
		intExt0Served();
		msg = "External 0 Interrupt\n";
		break;
	}
	if (ret)
	{
		intInProgress |= int2beserved;
		if (intShowService)
			sim_print(msg);
	}
/**********************************************************************
  - se posso servire un interrupt marco quale servire (secondo le priorita')
*/
	int2beserved = 0;
	if ( (ie & 0x80) && ! intdelayed )  
	{
		/* Polling for High priority */
		i = ie & readDirect(IP) & int_latch;
		for (j = 1; (j <= INT_T2) && ! int2beserved ; j <<= 1)
			if (i & j)
				int2beserved = j;
		/* Polling for Low priority */
		i = ie & int_latch;
		for (j = 1; (j <= INT_T2) && ! int2beserved ; j <<= 1)
			if (i & j)
				int2beserved = j;
		/* Avoid Interrupt if there is a"NOT minor"level in progress */
		if ( intInProgress )
			if ( (int2beserved > intInProgress) || \
			     (int2beserved & intInProgress) )
				int2beserved = 0;
	}
	intdelayed = 0;
/**********************************************************************
  - poll degli eventuali interrupt maturati
*/
	ser_poll();
	tim_poll();
/**********************************************************************
  - se ci sono interrupt attivi li marco nella lista dei richiedenti (latch)
*/
	int_latch = int_pending;
/**********************************************************************
  - se c'e' un interrupt allora esco dall'eventuale stato di Idle Mode
*/
	if (ret)
		writeDirect(PCON, readDirect(PCON) & 0xfe);
	return ret;
}

/* used by RETI and access to IE or IP to delay interrupt */
void delayInterrupt(void)
{
	intdelayed = 1;
}

int terminateISR(void)
{
	INT8 j;
	if ( intInProgress )
	{
		/* delete from pseudo-stack the terminated ISR in progress */
		for (j = 1; (j < INT_T2) && ! (intInProgress & j) ; j <<= 1) ;
		if ( ! intInProgress & j )
			return 0;  /* Interrupt NOT found BUGS in code */
		intInProgress &= ~j;
		return 1;
	}
	else
		return 0;  /* RETI without a Interrupt */
}
