/*
 *  runpipe v1.0b. This program watches a list of named pipes and
 *  executes processes to handle input or output as access is made to
 *  one of the pipes.
 */

/*
 *  Program written by Christopher Neufeld, neufeld@physics.utoronto.ca.
 *  May 18, 1995. Distributed under GPL.
 */

/*
 *  This program runs in one of three modes, USER, SYSTEM, and PARANOID.
 *  In USER mode the daemon runs as the invoking user, and only that user
 *  is allowed to manipulate the daemon.
 */

/*
 *  In SYSTEM mode the daemon runs as root, and when it is called upon to
 *  handle a pipe it forks off a child which then takes on the user ID of
 *  the person who asked that pipe to be monitored. This process then
 *  handles the request with the privileges of that user.
 */

/*
 *  In PARANOID mode the daemon immediately becomes an unprivileged ID,
 *  "nobody". It then runs all requests as if it were that unprivileged
 *  user. This means the programs run must be world-executable, and
 *  cannot write data files to the space of the invoking user.
 */

#include <ctype.h>
#include <stdio.h>
#include <stddef.h>
#include <fcntl.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/wait.h>

#include "runpipe.h"
#include "config.h"

#if ! defined(SYSTEM) && ! defined(USERMODE) && ! defined(PARANOID)
#error Configuration file "configure" is bad. Select a mode of operation.
#endif

#define VERSIONSTR "v1.0b"	    /* The version of runpiped. */

#define DEBUGSWTCH "-d"		    /* Command-line switch which gives debugging mode */



#define NE_SUCCESS 0	  /* Success in the newentry() function */
#define NE_BADFMT 1	  /* Bad format in a command received from a client */
#define NE_NOTFOUND 2	  /* Failed to resolve the path to a pipe, or the process was already being watched */
#define NE_BADMEM 3	  /* Memory allocation failure in newentry() */

#define ABORT 1		  /* Give up, something went wrong */
#define BADPERM 2	  /* Permission failure */

#ifndef NOSOCKETPERMS
#define CONNECT 3	  /* Succeeded in connecting to a socket with bad permissions */
#define NOCONNECT 4	  /* Failed to connect to a socket with bad permissions */
#endif


struct pipetable {
  char directory[MAXPATHLEN + 1];   /* Change to this directory before invoking user program */
  char filename[MAXPATHLEN + 1];    /* Name of pipe to watch. */
  int dowhat;			    /* Action. O_WRONLY, O_RDONLY, O_RDWR. See open(2) */
  char **command;		    /* A NULL-terminated list of command+arguments to run on the pipe */
  uid_t userid;			    /* The UID of the person who asked the pipe to be watched */
  int fildes;			    /* The file descriptor associated with an open pipe. */
  pid_t childpid;		    /* The PID of a running process handling a pipe */
  time_t lastt;			    /* The time the process started or stopped. */
  struct pipetable *next;	    /* A pointer to the next entry in the linked list */
};

char response[RESPONSELEN];    /* A string which might be sent back to the caller */

char *unlinkme = NULL;	      /* This will hold the socket path to kill if the daemon catches a signal to die */

struct pipetable *tableroot = NULL;   /* The root of the linked list of pipes to watch */



/* Prototype the functions in this file */

int daemon(int close_io, int cdroot);   /* No header declares this. It's in libbsd.a */

void usage(char *exename);
void cleanup(void);
void respond(int fildes, char *msg);
int ismypipe(const char *filename, uid_t myuid);
int ismysocket(const char *filename, uid_t myuid);
int strmaxcat(char *dest, const char *src, size_t maxlen);
int newentry(char *message, int sockfd);
void dropentry(struct pipetable *tableptr);
void masksigs(void);
int setupselect(void);
void lookforsock(char *socketpath, struct sockaddr_un *binding);
void checkbadperms(struct sockaddr_un *binding);
void action(int fd, struct pipetable *tableptr);
int main(int argc, char **argv);


#ifdef RESTARTFILENAME

#define OPENMODEREADONLY 1
#define OPENMODEWRITE 2


/* Open the restart file for read/write. Returns 0 on success, -1 on
 * failure */
static int open_restart_file(int mode, FILE **restartfile)
{
  char pathname[MAXPATHLEN+1];
  int oldmask;
  char *homeptr;

  if (strlen(RESTARTFILENAME) > MAXPATHLEN) {
    fprintf(stderr, "Error: Configured restart file path name is too long.\n");
    return -1;
  }

  strcpy(pathname, RESTARTFILENAME);
  if (pathname[0] != '/') {   /* Relative path, get $HOME */
    if (((homeptr = getenv(HOMEENV)) == NULL) || homeptr[0] != '/') {
      fprintf(stderr, "Error: unable to resolve your $%s for restart file path.\n", HOMEENV);
      fprintf(stderr, "Please supply a complete path name for the restart file in the configuration\n");
      fprintf(stderr, "and rebuild the package.\n");
      return -1;
    }

    if (strlen(homeptr) > MAXPATHLEN - 1) {
      fprintf(stderr, "Error: $%s is too long: %s\n", HOMEENV, homeptr);
      return -1;
    }
    strcpy(pathname, homeptr);
    strcat(pathname, "/");
  } else {
    pathname[0] = 0;
  }

  if (strlen(pathname) + strlen(RESTARTFILENAME) + 1 > MAXPATHLEN) {
    fprintf(stderr, "Error: restart file path %s%s is too long\n", pathname, RESTARTFILENAME);
    return -1;
  }

  strcat(pathname, RESTARTFILENAME);
  oldmask = umask(077);
  if (mode == OPENMODEREADONLY)
    *restartfile = fopen(pathname, "r");  /* Open for reading */
  else if (mode == OPENMODEWRITE)
    *restartfile = fopen(pathname, "w");  /* Truncate the file */
  else 
    abort();    /* What to do, what to do? */
  umask(oldmask);
  if (*restartfile != NULL) return 0;
  return -1;   /* Failed to open */
}


/* Read the restart file, and load the pipe watch tables from it. */
static int read_restart_file(void)
{
  FILE *restartfile;
  char nextline[MAXMESSLEN+1];

  if (open_restart_file(OPENMODEREADONLY, &restartfile) != 0) {
    fprintf(stderr, "Unable to read restart file.\n");
    return -1;
  }

  if (restartfile == NULL) abort();
  while (!feof(restartfile)) {
    if (fgets(nextline, MAXMESSLEN, restartfile) == NULL) { /* EOF */
      fclose(restartfile);
      return 0;
    }

    if (strlen(nextline) > 0)   /* Discard the newline character */
      nextline[strlen(nextline) - 1] = 0;

    newentry(nextline, -1);  /* Parse entry, discarding responses */
  }
  fclose(restartfile);
  return 0;
}


/* Write out the current state of the restart file */
static void write_restart_file(void)
{
  char outline[MAXMESSLEN+1];
  char scratch[MAXMESSLEN+1];
  char uidstr[20];   /* Big enough to hold the representation of a UID */
  struct pipetable *entryptr = tableroot;
  int argctr;
  char *cptr;
  FILE *restartfile;

  if (open_restart_file(OPENMODEWRITE, &restartfile) != 0) {
    fprintf(stderr, "Unable to write restart file.\n");
    return;
  }

  while (entryptr != NULL) {
    strcpy(outline, entryptr->directory);
    strcat(outline, BLANKSTR);
    strcat(outline, entryptr->filename);
    switch(entryptr->dowhat) {
    case O_RDONLY:
      strcat(outline, " w ");
      break;
    case O_WRONLY:
      strcat(outline, " r ");
      break;
    case O_RDWR:
      strcat(outline, " b ");
      break;
    }
    
    sprintf(uidstr, "%u", (unsigned int) entryptr->userid);
    strcat(outline, uidstr);

    argctr = 0;

    while (entryptr->command[argctr] != NULL) {
      strcat(outline, BLANKSTR);
      strcpy(scratch, entryptr->command[argctr]);
      while ((cptr = strchr(scratch, BLANK)) != NULL) *cptr ^= HIBITMASK;
      strcat(outline, scratch);
      argctr++;
    }
    
    fputs(outline, restartfile);
    fprintf(restartfile, "\n");
    entryptr = entryptr->next;
  }
  fclose(restartfile);
  return;
}


#endif  /* RESTARTFILENAME */



/*
 * Return a usage message. Pass the string used to invoke the daemon.
 */
  void
usage(char *exename)
{
  fprintf(stderr, "\nrunpiped %s.\n", VERSIONSTR);
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "    %s [-d ]\n\n", exename);
}


/*
 * Get a pointer to an entry which we are going to delete from the
 * tables, and construct a fake control message to newentry() telling
 * it to drop that entry. Send responses to /dev/null, since there is
 * no client program listening to this. This function is called only
 * by the daemon itself, not in response to a client request.
 */
  void
dropentry(struct pipetable *tableptr)
{
  char message[MAXMESSLEN + 2];
  int nullfd;	  /* We'll send responses to /dev/null */

  strcpy(message, "/ ");
  strcat(message, tableptr->filename);
  strcat(message, " d 0 /");    /* We now delete that entry */
  nullfd = open(NULLDEV, O_WRONLY);
  newentry(message, nullfd);
  close(nullfd);
}


/*
 * All clean up done here. Right now that means delete the socket, and
 * possibly the pipes via the dropentry() call.
 */
  void
cleanup(void)
{
  while (tableroot != NULL) dropentry(tableroot);
  if (unlinkme != NULL) {
    unlink(unlinkme);
#ifdef NOSOCKETPERMS
    *strrchr(unlinkme, '/') = 0;
    rmdir(unlinkme);
#endif
  }
  exit(0);
}


/*
 * Send response in *msg to the client program by writing to the
 * file descriptor 'fildes'.
 */
  void
respond(int fildes, char *msg)
{
  int msglen, wrote;

  if (fildes == -1) return;

/*
 * This looks weird. The socket to the client is non-blocking, and
 * sometimes one end or the other forgets that there is more data to
 * send. This technique keeps writing until the whole message gets through.
 */
  msglen = strlen(msg);
  wrote = 0;
  do {
    wrote += write(fildes, msg, msglen - wrote);
  } while (wrote < msglen);
}


/*
 * Concatenate a string *src to the end of *dest, but don't let *dest
 * get longer than maxlen. Returns 0 if the entire *src is successfully
 * appended to *dest.
 */
  int
strmaxcat(char *dest, const char *src, size_t maxlen)
{
  int totallen;

  totallen = strlen(dest) + strlen(src);
  if (totallen > maxlen - 1) {
    strncat(dest, src, maxlen - strlen(dest) - 1);
    return 1;  /* I'm full */
  }
  strcat(dest, src);
  return 0;
}


/*
 * Parse a message which has come from a client. The message is passed
 * in *message, and the responses will be sent to the socket at the
 * file descriptor 'sockfd'.
 */
  int
newentry(char *message, int sockfd)
{
  char *dir;	  /* Will point to working directory for program to run */
  char *pip;	  /* Points to the name of the pipe to watch */
  char *action;	  /* Points to the action indicator. read, write, read/write, delete, list */
  char *uid;      /* Points to the ASCII representation of the client user-id. */
  char *prog;	  /* Points to the command+arguments to run on the pipe */
  char *cptr, *cptr2;
  char passedpath[MAXPATHLEN+1];    /* The path name received from the client */
  char truepath[MAXPATHLEN+1];	    /* The resolved path		      */
  struct pipetable *curpos, *lastpos;
  uid_t msguid;	      /* Numeric representation of the client user-id */
  int i, j;
  int numargs;	      /* Number of blank-separated entries in command+arguments */
  size_t addchrs;
  static int numofpipes = 0;   /* Number of pipes being watched */


/*
 * The incoming string should have no non-printable characters
 */
  for (cptr = message; *cptr; cptr++)
    if (!isprint(*cptr)) {
      respond(sockfd, "Error parsing command string from client.\n");
      return NE_BADFMT;
    }

/* Parse out the fields and NULL separate them */
  dir = message;

  if ((pip = strchr(dir, FSEP)) == NULL) {
    respond(sockfd, "Error parsing command string from client.\n");
    return NE_BADFMT;
  } else pip++;
  *(pip - 1) = 0;
  if ((action = strchr(pip, FSEP)) == NULL) {
    respond(sockfd, "Error parsing command string from client.\n");
    return NE_BADFMT;
  } else action++;
  *(action - 1) = 0;
  if ((uid = strchr(action, FSEP)) == NULL) {
    respond(sockfd, "Error parsing command string from client.\n");
    return NE_BADFMT;
  } else uid++;
  *(uid - 1) = 0;
  if ((prog = strchr(uid, FSEP)) == NULL) {
    respond(sockfd, "Error parsing command string from client.\n");
    return NE_BADFMT;
  } else prog++;
  *(prog - 1) = 0;

/*
 * None of those strings should be zero length.
 */
  if (!strlen(dir) || !strlen(uid) || !strlen(pip) || !strlen(action) || !strlen(prog)) {
    respond(sockfd, "Error parsing command string from client.\n");
    return NE_BADFMT;
  }

  msguid = (uid_t) strtoul(uid, NULL, 10);

#ifdef USERMODE
  if (msguid && msguid != getuid()) {
    respond(sockfd, "Failed. You are not who you are supposed to be!\n");
    return NE_BADFMT;
  }
#endif

/*
 * List the table of entries being handled. Root can view all, anybody
 * else can view only his own
 */
  if (*action == LISTTAB) {
    for (curpos = tableroot; curpos != NULL; curpos = curpos->next)
      if (!msguid || msguid == curpos->userid) {
	strcpy(response, curpos->filename);
	switch(curpos->dowhat) {
	case O_RDONLY:
	  strcat(response, " w ");
	  break;
	case O_WRONLY:
	  strcat(response, " r ");
	  break;
	case O_RDWR:
	  strcat(response, " b ");
	}
	for (i = 0; (curpos->command)[i] != NULL; i++) {
	  if (strmaxcat(response, (curpos->command)[i], RESPONSELEN - 1)) break;
	  if (strmaxcat(response, " ", RESPONSELEN - 1)) break;
	}
	strcat(response, "\n");
	respond(sockfd, response);
      }
    respond(sockfd, "\n");	    /* Send a blank line to show that we're done */
    return NE_SUCCESS;
  }

/* Get the full path name of the pipe we're watching */
  if (strlen(pip) > MAXPATHLEN) {
    respond(sockfd, "Pipe path name too long\n");
    return NE_BADFMT;
  }

  strcpy(passedpath, pip);
  passedpath[MAXPATHLEN] = 0;
  if (realpath(passedpath, truepath) == NULL || strlen(truepath) > MAXPATHLEN) {
    sprintf(response, "Error parsing pipe file name at: %s\n", truepath);
    respond(sockfd, response);
    return NE_BADFMT;
  }

  if (*action == DELETE) {  /* Kill a field, don't create a new one */
    lastpos = curpos = tableroot;
    while (curpos != NULL) {
      if (!strcmp(curpos->filename, truepath)) {

/*
 * If a nonprivileged user tries to delete somebody else's entry, don't even
 * confirm to him that there is such an entry.
 */
	if (!msguid || (msguid == curpos->userid)) {
	  if (curpos->childpid > 0) kill(curpos->childpid, SIGKILL);  /* Kill any running process on the pipe */
	  if (curpos == tableroot) {
	    tableroot = curpos->next;
	  } else {
	    lastpos->next = curpos->next;
	  }
	  for (i = 0; (curpos->command)[i] != NULL; i++)
	    free((curpos->command)[i]);
	  free(curpos->command);
	  free(curpos);
#ifndef PARANOID
#ifdef KILLPIPES
 	  unlink(truepath);		/* Delete the named pipe */
#endif
#endif
	  numofpipes--;
	  sprintf(response, "%s deleted from runpipe list.\n", curpos->filename);
	  respond(sockfd, response);
	  return NE_SUCCESS;
	}
      }
      lastpos = curpos;
      curpos = curpos->next;
    }
    sprintf(response, "Failed to find pipe: %s\n", truepath);
    respond(sockfd, response);
    return NE_NOTFOUND;
  }

/*
 * OK, we're trying to add a new pipe to the list
 */
  if (numofpipes == MAXPIPES) {
    sprintf(response, "Error: maximum number of pipes, %d, reached.\n", MAXPIPES);
    respond(sockfd, response);
    return NE_BADMEM;
  }

  if (!ismypipe(truepath, msguid)) {
    sprintf(response, "No such pipe belonging to you: %s\n", truepath);
    respond(sockfd, response);
    return NE_NOTFOUND;
  }

  lastpos = tableroot;
  for (curpos = tableroot; curpos != NULL; lastpos = curpos, curpos = curpos->next)
    if (!strcmp(truepath, curpos->filename)) {
      sprintf(response, "Failed. Another process is already attached to pipe %s\n", truepath);
      respond(sockfd, response);
      return NE_NOTFOUND;
    }

  if (lastpos == NULL) {   /* No entries in the table */
    tableroot = (struct pipetable *) malloc(sizeof(struct pipetable));
    if (tableroot == NULL) {
      respond(sockfd, "Failed to allocate memory for new entry in table\n");
      return NE_BADMEM;
    }
    curpos = tableroot;
    curpos->next = NULL;
  } else {
    if ((curpos = (struct pipetable *) malloc(sizeof(struct pipetable))) == NULL) {
      respond(sockfd, "Failed to allocate memory for new entry in table\n");
      return NE_BADMEM;
    }
    curpos->next = lastpos->next;
    lastpos->next = curpos;
  }

  /* Now build new entry in curpos */

  strcpy(curpos->directory, dir);
  strcpy(curpos->filename, truepath);
  curpos->userid = msguid;
  addchrs = strlen(prog) + 1;

/* Count command line arguments */
  for (numargs = 1, cptr = prog; *cptr; cptr++)
    if (*cptr == BLANK) numargs++;

  curpos->command = (char **) malloc((numargs + 1) * sizeof(char *));
  if (curpos->command == NULL) {
    if (lastpos == NULL) {
      tableroot = NULL;
      free(curpos);
    } else {
      lastpos->next = curpos->next;
      free(curpos);
    }
    respond(sockfd, "Failed to allocate memory for new entry in table\n");
    return NE_BADMEM;
  }


  i = strlen(prog);
  prog[i] = BLANK ; prog[i+1] = 0;	/* A kludge. It works. */

  for (cptr = prog, i = 0; i < numargs; i++) {

/* Find the length of the strings, malloc the space, copy the string, and
    switch the hibit blanks back to blanks				    */

    addchrs = (size_t) (strchr(cptr, BLANK) - cptr);


    curpos->command[i] = (char *) malloc((addchrs + 1) * sizeof(char));
    if (curpos->command[i] == NULL) {
      for (j = i-1; j >= 0; j--) free(curpos->command[j]);
      free(curpos->command);
      if (lastpos == NULL) {
	tableroot = NULL;
	free(curpos);
      } else {
	lastpos->next = curpos->next;
	free(curpos);
      }
      respond(sockfd, "Failed to allocate memory for new entry in table\n");
      return NE_BADMEM;
    }
    strncpy(curpos->command[i], cptr, addchrs);
    curpos->command[i][addchrs] = 0;
    cptr += addchrs + 1;

/* Now clear quoted blanks in the command line argument */
    while ((cptr2 = strchr(curpos->command[i], BLANK ^ HIBITMASK)) != NULL)
      *cptr2 ^= HIBITMASK;

  }
  curpos->command[numargs] = NULL;            /* Terminate list of arguments for execvp() */

  switch (*action) {
  case READ:
    curpos->dowhat = O_WRONLY;
    break;
  case WRITE:
    curpos->dowhat = O_RDONLY;
    break;
  case READWRITE:
    curpos->dowhat = O_RDWR;
    break;
  }
  curpos->fildes = -1;
  curpos->childpid = 0;
  curpos->lastt = time(NULL);
  numofpipes++;
  sprintf(response, "%s added to runpipe list.\n", curpos->filename);
  respond(sockfd, response);
  return NE_SUCCESS;
}


/*
 * Mask out four common signals, sending them to the signal handler at
 * cleanup().
 */
  void
masksigs(void)
{
  struct sigaction trappings;

  trappings.sa_handler = (void *) cleanup;
  trappings.sa_mask = 0;
  trappings.sa_flags = SA_RESTART;
  trappings.sa_restorer = NULL;

  sigaction(SIGHUP, &trappings, NULL);
  sigaction(SIGINT, &trappings, NULL);
  sigaction(SIGQUIT, &trappings, NULL);
  sigaction(SIGTERM, &trappings, NULL);
}



/*
 * Act on a pipe read/write action. The 'fd' is the descriptor which was
 * opened by the external process, and tableptr points to the structure
 * holding information about that pipe.
 */
  void
action(int fd, struct pipetable *tableptr)
{
  int nullfd;   /* Those file numbers which aren't sent to the pipe, send to /dev/null */

  switch (tableptr->childpid = fork()) {
  case -1:
#ifdef SYSLOG
    syslog(LOG_ERR, "Failed to fork. %m");
#endif
    return;
  case 0:
    setegid(getgid());
#ifdef SYSTEM
#ifdef NOROOTPIPES
    if (!tableptr->userid) {
      setuid(NOBODY);
    } else {
      setuid(tableptr->userid);
    }
#else
    setuid(tableptr->userid);
#endif
#endif

    nullfd = open(NULLDEV, O_RDWR);

    dup2(nullfd, 2);		      /* Send stderr to /dev/null */
    switch(tableptr->dowhat) {
    case O_RDWR:  dup2(fd, 0);	      /* Send stdout to pipe */
		  dup2(fd, 1);	      /* Send stdin to pipe */
  		  break;
    case O_RDONLY:  dup2(fd, 0);      /* Send stdout to pipe */
		    dup2(nullfd, 1);  /* Send stdin to /dev/null */
		    break;
    case O_WRONLY:  dup2(nullfd, 0);  /* Send stdout to /dev/null */
		    dup2(fd, 1);      /* Send stdin to pipe */
		    break;
    }

    close(fd);			      /* We're done with this, close this copy of the file descriptor */

    if (chdir(tableptr->directory)) {
#ifdef SYSLOG
      syslog(LOG_ERR, "Unable to chdir - %%m.\n");
#endif
      exit(1);
    }
    tableptr->lastt = time(NULL);
    execvp(tableptr->command[0], tableptr->command);
#ifdef SYSLOG
    syslog(LOG_ERR, "Bad call to execvp - %%m.\n");
#endif
    exit(0);
    break;
  default:
    close(fd);
    return;
  }
}


/*
 * Check to see if the socket already exists, and if it does, make sure
 * that the daemon isn't already listening on the pipe. If there's a
 * daemon listening on the socket, abort.
 */
  void
lookforsock(char *socketpath, struct sockaddr_un *binding)
{
  int msgsock, btot, nbytes;
  char message[MAXMESSLEN + 2];

  if (ismysocket(socketpath, getuid())) {
    if ((msgsock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
      fprintf(stderr, "Error: failed to create a socket file descriptor.\n");
      exit(BADPERM);
    }

    if (connect(msgsock, (struct sockaddr *) binding, strlen(socketpath) + 2) < 0) {
      if (unlink(socketpath)) {
	fprintf(stderr, "Problem trying to unlink stale sockets, if any.\n");
	exit(BADPERM);
      }
      return;
    }

/* Check to see if the daemon's already running, by sending it a message */

    fcntl(msgsock, F_SETFL, O_NONBLOCK);
    if (write(msgsock, PING, strlen(PING)) != strlen(PING)) {
      fprintf(stderr, "Error: problem writing to the socket.\n");
      exit(BADUSAGE);
    }

/* Read until we run out of bytes in the socket. */
    btot = 0;
    do {
      nbytes = read(msgsock, message + btot, MAXMESSLEN - btot);
      if (nbytes == -1) break;
      btot += nbytes;
      message[btot] = 0;
    } while (btot < strlen(PONG) && nbytes > 0);

    if (!strcmp(message, PONG)) {
      fprintf(stderr, "Error: there's already a daemon listening on the socket.\n");
      exit(BADUSAGE);
    }

    if (unlink(socketpath)) {
      fprintf(stderr, "Problem trying to unlink stale sockets, if any.\n");
      exit(BADPERM);
    }
  }
}


/*
 * Attempt to make an "illegal" connection onto the socket. If it works,
 * report the error and die. This function will have no body if NOSOCKETPERMS
 * is defined. We don't need to check in that case, because we're using
 * another scheme to protect the socket: hiding it in a subdirectory with
 * restricted permissions.
 */
  void
checkbadperms(struct sockaddr_un *binding)
{
#ifndef NOSOCKETPERMS
  int pid, msgsock, status;

  if ((msgsock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
    fprintf(stderr, "Failed to create a socket file descript: %s\n", strerror(errno));
    exit(BADPERM);
  }

  switch(pid = fork()) {
  case -1:  fprintf(stderr, "Cannot fork: %s\n", strerror(errno));
	    exit(ABORT);
  case 0:
#ifdef SYSTEM
	    if (setuid(NOBODY)) {
	      fprintf(stderr, "Cannot change uid: %s\n", strerror(errno));
	      exit(BADPERM);
	    }
#endif
	    if (connect(msgsock, (struct sockaddr *) binding, strlen(binding->sun_path) + 2) < 0) {
	      exit(NOCONNECT);
	    } else {
	      close(msgsock);
	      exit(CONNECT);
	    }
  default:  wait(&status);
	    if (WIFEXITED(&status) && WEXITSTATUS(&status) == NOCONNECT) return;
	    fprintf(stderr, "OS is insecure. Recompile with \"Nosocketperms\" set.\n");
	    unlink(binding->sun_path);
	    exit(ABORT);
  }
#endif
}


/*
 * Build up a list of pipes which might be open for reading, and get them
 * ready for select()
 */
  int
setupselect(void)
{
  struct pipetable *tptr;
  int foundone;

  foundone = 0;
  for (tptr = tableroot; tptr != NULL; tptr = tptr->next)
    if (tptr->dowhat == O_RDONLY || tptr->dowhat == O_RDWR) {
      if (tptr->fildes >= 0) close(tptr->fildes);
      tptr->fildes = open(tptr->filename, tptr->dowhat | O_NONBLOCK);
      if (tptr->fildes < 0) dropentry(tptr);
      foundone = 1;
    }
  return foundone;
}


/*
 * The toplevel function.
 */
int main(int argc, char **argv)
{
  struct pipetable *tableptr;
  int mysocket;			  /* The file descriptor of the daemon's socket */
  int msgsock;			  /* The copy of the descriptor returned by accept() */
  int inlen, fd, pid, maxrfd, needselect;
  int nbytes, debug;
  char socketpath[UNIX_PATH_MAX+1];
  char message[MAXMESSLEN + 2];
  struct sockaddr_un bindit, inaddr;
  int goback, numread;
  fd_set readfds, writefds, exceptfds;
  struct timeval timeout;


#ifdef PARANOID
  if (setuid(NOBODY)) {
    fprintf(stderr, "Error. Unable to become user %d\n", NOBODY);
    return BADPERM;
  }
#endif

#ifdef SYSTEM
  if (getuid()) {
    fprintf(stderr, "Error. System mode daemon must be run as root.\n");
    return BADUSAGE;
  }
#endif

  debug = needselect = 0;

/* Reject a bad invocation */

  if ((argc != 1 && argc != 2) || ((debug = (argc == 2)) && strcmp(argv[1], DEBUGSWTCH))) {
    usage(argv[0]);
    exit(BADUSAGE);
  }

#ifdef NOSOCKETPERMS
#ifdef USERMODE
  umask(0077);
#else
  umask(0007);
#endif
#endif

  gensockpath(socketpath, DAEMON);

  strcpy(bindit.sun_path, socketpath);
  bindit.sun_family = AF_UNIX;

  lookforsock(socketpath, &bindit);

  if ((mysocket = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
    fprintf(stderr, "Error creating socket: %s\n", strerror(errno));
    exit(BADPERM);
  }

  masksigs();

#ifndef NOSOCKETPERMS
/* We will check to see if protection flags on the socket inode have any force */

#ifndef SYSTEM
  umask(0777);	      /* Nobody except root is allowed to write to the socket */
#else
  umask(0077);	      /* Only owner may write to the socket */
#endif
#endif

/* Bind to the socket */
  if (bind(mysocket, (struct sockaddr *) &bindit,
	   strlen(bindit.sun_path) + 1 + sizeof(bindit.sun_family)) < 0) {
    fprintf(stderr, "Error binding to socket %s: %s\n", bindit.sun_path, strerror(errno));
    exit(BADPERM);
  }

  checkbadperms(&bindit);

#ifndef NOSOCKETPERMS
/* Change the permissions to sane values. */
#ifdef USERMODE
  chmod(socketpath, 0700);
  umask(0077);
#else
  chmod(socketpath, 0770);
  umask(0007);
#endif
#endif

  if (!debug) {
    daemon(0, 1);
  }

#ifdef SYSLOG
  openlog(LOGNAME, LOG_PID, LOG_DAEMON);
#endif

  unlinkme = bindit.sun_path;

/* Set up the non-blocking socket */
  fcntl(mysocket, F_SETFL, O_NONBLOCK);

  listen(mysocket, 5);
  inlen = sizeof(inaddr);


#ifdef RESTARTFILENAME
  read_restart_file();
#endif  /* RESTARTFILENAME */  

/* Enter the main loop */
  while (1) {

    usleep(LOOPDELAY);			/* Wait twenty milliseconds before every attempt */

    pid = waitpid(-1, NULL, WNOHANG);

    for (tableptr = tableroot; tableptr != NULL; tableptr = tableptr->next) {
      if (pid > 0 && tableptr->childpid == pid) {
	tableptr->childpid = 0;
	tableptr->lastt = time(NULL);
	continue;
      }

/* Try to shut down a runaway process */

/* Not working yet, killing the child doesn't close his file descriptors, it
   seems. As a result, the external reading process keeps the pipe open, and
   the next pass through just restarts the process			      */

#ifdef TIMEOUT
      if (tableptr->childpid && difftime(time(NULL), tableptr->lastt) > TIMEOUT) {
	kill(tableptr->childpid, SIGTERM);
	tableptr->childpid = 0;
	tableptr->lastt = time(NULL);
      }
#endif

    }

    msgsock = accept(mysocket, (struct sockaddr *) &inaddr, &inlen);
    if (msgsock >= 0) {   /* There's an instruction coming in, parse it. */
      nbytes = read(msgsock, message, MAXMESSLEN);
      message[MAXMESSLEN - 1] = 0; message[nbytes] = 0;
      if (nbytes < 0)   /* I guess there was no message after all */
	continue;
      if (!strcmp(message, PING)) {
	respond(msgsock, PONG);
	continue;
      }

/* We can't fork off to parse the message, there could be horrible collisions if
   more than one process attempts to change the table entries			  */

      newentry(message, msgsock);
      close(msgsock);
#ifdef RESTARTFILENAME
      write_restart_file();
#endif  /* RESTARTFILENAME */
      needselect = setupselect();
  }

    goback = 0;


/* OK, check pipes open for writing. That is, somebody tried to read the pipe */

    for (tableptr = tableroot; tableptr != NULL; tableptr = tableptr->next) {
      if (!ismypipe(tableptr->filename, tableptr->userid)) {  /* Somebody's trying to cause trouble! */
	dropentry(tableptr);
	needselect = setupselect();
	goback = 1;
	break;   /* Abort the loop, we can check everything on the next pass */
      }

      if ((tableptr->dowhat == O_WRONLY || tableptr->dowhat == O_RDWR) &&
	  !tableptr->childpid && difftime(time(NULL), tableptr->lastt) > 1 &&
	  (tableptr->fildes = open(tableptr->filename, O_WRONLY | O_NONBLOCK)) >= 0) {
	close(tableptr->fildes);
	if ((fd = open(tableptr->filename, tableptr->dowhat)) < 0) {
#ifdef SYSLOG
	  sprintf(response, "Got confused preparing to select pipe %s: %%m", tableptr->filename);
	  syslog(LOG_WARNING, response);
#endif
	  goback = 1;
	  break;
	}

	action(fd, tableptr);
	break;
      }
    }

    if (goback) continue;

/* Check files open for reading. That is, somebody tried to write to the pipe */

    if (!needselect) continue;

    maxrfd = -1;

    FD_ZERO(&readfds); FD_ZERO(&writefds); FD_ZERO(&exceptfds);

    for (tableptr = tableroot; tableptr != NULL; tableptr = tableptr->next) {
      if (!ismypipe(tableptr->filename, tableptr->userid)) {  /* Somebody's trying to cause trouble! */
	dropentry(tableptr);
	needselect = setupselect();
	goback = 1;
	break;   /* Abort the loop, we can check everything on the next pass */
      }

      if ((tableptr->dowhat == O_RDONLY || tableptr->dowhat == O_RDWR) &&
	  !tableptr->childpid && difftime(time(NULL), tableptr->lastt) > 1 &&
	  (tableptr->fildes >= 0)) {
	FD_SET(tableptr->fildes, &readfds);
	if (tableptr->fildes > maxrfd) maxrfd = tableptr->fildes;
      }
    }

    if (goback) continue;

    timeout.tv_sec = WAITSECS;
    timeout.tv_usec = WAITUSECS;
    numread = select(maxrfd + 1, &readfds, &writefds, &exceptfds, &timeout);
    if (numread <= 0) continue;

    for (tableptr = tableroot; tableptr != NULL; tableptr = tableptr->next) {
      if (!ismypipe(tableptr->filename, tableptr->userid)) {
	dropentry(tableptr);
	needselect = setupselect();
	goback = 1;
	break;
      }

      if ((tableptr->dowhat == O_RDONLY || tableptr->dowhat == O_RDWR) &&
	  !tableptr->childpid && difftime(time(NULL), tableptr->lastt) > 1 &&
	  tableptr->fildes >= 0 && FD_ISSET(tableptr->fildes, &readfds))
	    action(tableptr->fildes, tableptr);
    }
  }
}
