/*  Runpipe v1.0b. Copyright May 3, 1995 by Christopher Neufeld		  */

/*  This is the client to the runpipe daemon. It submits preformatted
    command strings to the daemon. As a result, it is considered a trusted
    source of information to the daemon. In SYSTEM or PARANOID modes it
    must be run setgid in order to assure the daemon that the information
    is coming from a valid source.

    The invocation is:

    runpipe <pipename> <command and arguments>

    The daemon expects to see messages on its socket in exactly the following
    format:

    FIELD1 FIELD2 FIELD3 FIELD4 FIELD5 [ FIELD6 ... ]

    FIELD1 should contain the working directory at invocation of the client.

    FIELD2 should contain the complete path name of the pipe.

    FIELD3 should be one of the set "rwbdl".
      r: reads from the pipe are redirected to the command
      w: writes to the pipe are redirected to the command
      b: reads/writes to the pipe are redirected to the command
      d: the table entry for that pipe is deleted
      l: the daemon's list of pipes being handled is printed out

    FIELD4 is the uid of the person who invoked the client program.

    FIELD5 FIELD6 ... are a blank-separated list containing the command
      and its command line arguments. If there are embedded blanks in
      the command line arguments they should be hi-bit quoted. No other
      characters are special. The daemon invokes this with execvp.

    There should be no terminating \n on the command message.

									  */

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/un.h>

#include "runpipe.h"
#include "config.h"


#define VERSIONSTR "v1.0b"

#define SUCCESS 0


void usage(char *exename);
int sendreq(char *message);
int getresp(int fd, char *buff, int bufflen);
int main(int argc, char **argv);


void usage(char *exename)
{
  fprintf(stderr, "\nrunpipe %s.\n", VERSIONSTR);
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "    %s pipename [ %c | %c | %c | %c ] [ prog args... ]\n",
	  exename, READ, WRITE, READWRITE, DELETE);
  fprintf(stderr, "    %s %c\n\n", exename, LISTTAB);
}

int sendreq(char *message)
{
  struct sockaddr_un thesocket;
  int sockfd, messlen;
  char sockpath[UNIX_PATH_MAX + 1];

  if ((sockfd = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
    fprintf(stderr, "Error: failed to create the socket file descriptor.\n");
    return sockfd;
  }
  thesocket.sun_family = AF_UNIX;
  gensockpath(sockpath, CLIENT);

  if (!ismysocket(sockpath,
#ifdef USERMODE
  getuid()
#else
  0
#endif
  )) {
    fprintf(stderr, "Error: my socket is missing.\n");
    exit(BADUSAGE);
  }
  strcpy(thesocket.sun_path, sockpath);
  if (connect(sockfd, (struct sockaddr *) &thesocket, strlen(sockpath) + 2) < 0) {
    fprintf(stderr, "Error: unable to connect to socket.\n");
    exit(BADUSAGE);
  }

  messlen = strlen(message);
  if (write(sockfd, message, messlen) != messlen) {
    fprintf(stderr, "Error: problem writing to the socket.\n");
    exit(BADUSAGE);
  }
  return sockfd;
}

int getresp(int fd, char *buff, int bufflen)
{
  return read(fd, buff, bufflen);
}

int main(int argc, char **argv)
{
  char message[MAXMESSLEN], *cptr;
  char response[RESPONSELEN];
  char pathname[MAXPATHLEN + 2];
  char dirname[MAXPATHLEN + 1];
  int i, sockfd;
  unsigned int messz;
#ifdef MAKEPIPES
  int myerrno;
#endif

  if (argc < 2 || (argc == 2 && (strlen(argv[1]) != 1 || argv[1][0] !=  LISTTAB))) {
    usage(argv[0]);
    return BADUSAGE;
  }

  if (argc == 2) {	  /* It's a table request */
    sprintf(message, "/ / t %ud /", (unsigned int) getuid());
    sockfd = sendreq(message);
    do {
      getresp(sockfd, response, RESPONSELEN - 1);
      fputs(response, stdout);
    } while (strcmp(response, "\n") && strstr(response, "\n\n") == NULL);
    return SUCCESS;
  }

  if (strlen(argv[1]) > MAXPATHLEN) {
    fprintf(stderr, "Error: pipe name %s is too long.\n", argv[1]);
    return BADUSAGE;
  }

  if (argv[2][1] != 0 ||
      (argv[2][0] != READ && argv[2][0] != WRITE && argv[2][0] != READWRITE &&
       argv[2][0] != DELETE)) {
    usage(argv[0]);
    return BADUSAGE;
  }

  if (getcwd(dirname, MAXPATHLEN) == NULL) {
    fprintf(stderr, "Error: your current working directory is too long!\n");
    return BADUSAGE;
  }

#ifdef MAKEPIPES
  myerrno = 0;
  if (((realpath(argv[1], pathname) == NULL) && ((myerrno = errno), 1)) || strlen(pathname) > MAXPATHLEN) {
    if (myerrno != ENOENT || (mknod(argv[1], 0777 | S_IFIFO, 0))) {
      fprintf(stderr, "Error parsing pipe file name at: %s\n", pathname);
      return BADUSAGE;
    }
  }
#else
  if (realpath(argv[1], pathname) == NULL || strlen(pathname) > MAXPATHLEN) {
    fprintf(stderr, "Error parsing pipe file name at: %s\n", pathname);
    return BADUSAGE;
  }
#endif

  sprintf(message, "%s %s %c %u", dirname, pathname, argv[2][0], (unsigned int) getuid());
  messz = strlen(message);
  for (i = 3; i < argc; i++) {
    while((cptr = strchr(argv[i], BLANK)) != NULL) *cptr ^= HIBITMASK;
    messz += strlen(argv[i]) + 1;
  }
  if (messz > MAXMESSLEN - 2) {
    fprintf(stderr, "Error: command too long to parse\n");
    return BADUSAGE;
  }

  if (argc == 3) strcat(message, " /");

  for (i = 3; i < argc; i++) {
    strcat(message, BLANKSTR);
    strcat(message, argv[i]);
  }
  sockfd = sendreq(message);
  getresp(sockfd, response, RESPONSELEN - 1);
  fputs(response, stdout);
  return SUCCESS;
}
