/*
 * startproc.c  Start process(es) of the named program.
 *
 * Was:         daemon [-l log_file] /full/path/to/program
 * Usage:       startproc [+/-<prio>] [-v] [-l log_file|-q] /full/path/to/program
 *
 * Copyright 1994,95 Werner Fink, 1996-98 S.u.S.E. GmbH Fuerth, Germany.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Author:      Werner Fink <werner@suse.de>
 * 1998/05/06 Werner Fink: change name to startproc
 * 1998/05/06 Werner Fink: rework, added "-f" for pid files
 * 1999/08/05 Werner Fink: added "-t" for time to sleep, reenable "-e"
 */

#include "libinit.h"
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/ioctl.h>
#include <grp.h>

#define NOPROCESS	1
#define NOPIDFILE	2
#define NOPIDREAD	4
#define WRGSYNTAX	8
#define NOLOGFILE	16

#define USAGE		"Usage:\n"\
			"\t%s [+/-<prio>] [-s] [-u uid] [-g gid] [-v] [-l log_file|-q] /full/path/to/program\n" \
			, we_are

static int do_fork(const char *name, char *argv[], const char* log,
		   const int nicelvl, const int env);

static int quiet = 1, supprmsg = 0, sess = 0, seconds = 0;
static struct passwd *user = NULL;
struct group *grp = NULL;

int main(int argc, char **argv)
{
    extern char * we_are;
    int c;
    struct stat st;
    char *fullname = NULL, *basename = NULL, *log_file = NULL, *pid_file = NULL;
    int nicelvl = 0, env = 0;

    we_are = base_name(argv[0]);
    openlog (we_are, LOG_OPTIONS, LOG_DAEMON);

/*
 *  We should stat() fullname, because only the path identifies the executable.
 *  If there is one hardlink we have only to stat() the orignal executable.
 *  If there is more than one hardlink and we have to distinguish the
 *  executables by their swapname.  Note if the cmdline of some executables
 *  will changed by the running process its self the name is not clearly
 *  defined ... see libinit.c for more information.
 */

    if (*argv) {
        char **opt = argv;
	if (*(++opt) && (nicelvl = atoi(*opt))) {
	    if (nicelvl > PRIO_MAX)
        	nicelvl = PRIO_MAX;
	    if (nicelvl < PRIO_MIN)
	        nicelvl = PRIO_MIN;
	    argc--, argv++;
	}
    }

    opterr = 0;
    while ((c = getopt(argc, argv, "+ef:l:hqvsu:g:t:")) != -1) { /* `+' is POSIX correct */
	switch (c) {
	    case 'v':
		quiet = 0;
		break;
	    case 'e':
		env = 1;
		break;
	    case 'f':
		/* Allocate here: address optarg (current *argv) isn't freeable */
		if (optarg && optarg[0] != '-' && !pid_file) {
		    pid_file = strdup(optarg);
		} else {
		    nsyslog(LOG_ERR,"Option -f requires pid file to read pid from\n");
		    exit(WRGSYNTAX);
		}
		break;
	    case 'l':
		if (optarg && optarg[0] != '-' && !log_file) {
		    log_file = optarg;
		} else {
		    nsyslog(LOG_ERR,"Option -l requires log file\n");
		    exit(WRGSYNTAX);
		}
		break;
	    case 'q':
	        supprmsg = 1;
	        break;
	    case 's':
		sess = 1;
		break;
	    case 'u':
		if (optarg && optarg[0] != '/' && optarg[0] != '-') {
		    char *endptr;
		    uid_t uid =  (uid_t)strtol(optarg, &endptr, 10);

		    user = getpwnam(optarg);
		    if (!user && (*endptr == '\0')) user = getpwuid(uid);
		    endpwent();

		    if (!user) {
			nsyslog(LOG_ERR,"No such user or user id: %s\n", optarg);
			exit(WRGSYNTAX);
		    }
		} else {
		    nsyslog(LOG_ERR,"Option -u requires user id or user name\n");
		    exit(WRGSYNTAX);
		}
		break;
	    case 'g':
		if (optarg && optarg[0] != '/' && optarg[0] != '-') {
		    char *endptr;
		    gid_t gid =  (gid_t)strtol(optarg, &endptr, 10);

		    grp = getgrnam(optarg);
		    if (!grp && (*endptr == '\0')) grp = getgrgid(gid);
		    endgrent();

		    if (!grp) {
			nsyslog(LOG_ERR,"No such group or group id: %s\n", optarg);
			exit(WRGSYNTAX);
		    }
		} else {
		    nsyslog(LOG_ERR,"Option -g requires group id or group name\n");
		    exit(WRGSYNTAX);
		}
		break;
	    case 't':
		if (optarg && optarg[0] != '/' && optarg[0] != '-') {
		    char *endptr;
		    seconds = (int)strtol(optarg, &endptr, 10);

		    if (strlen(endptr) == strlen(optarg)) {
			nsyslog(LOG_ERR,"Option -t requires number of seconds\n");
			exit(WRGSYNTAX);
		    }

		    switch (*endptr) {
			case 's':
			    endptr++;
			case 'm':
			    endptr++;
			    seconds *= 60;
			    break;
			case 'h':
			    endptr++;
			    seconds *= 60*60;
			    break;
			default:
			    break;
		    }

		    if (strlen(endptr)) {
			nsyslog(LOG_ERR,"Option -t requires number of seconds\n");
			exit(WRGSYNTAX);
		    }

		} else {
		    nsyslog(LOG_ERR,"Option -t requires number of seconds\n");
		    exit(WRGSYNTAX);
		}
		break;
	    case 'h':
		nsyslog(LOG_ERR, USAGE);
		exit(0);
		break;
	    case '?':
		nsyslog(LOG_ERR, USAGE);
		exit(WRGSYNTAX);
		break;
	    default:
		break;
	}
    }

    argv += optind;
    argc -= optind;

    if (*argv) {
	fullname = *argv;
	basename = base_name(fullname);
    }

    if (!fullname) {
	nsyslog(LOG_ERR, USAGE);
	exit(WRGSYNTAX);
    }

    if (!pid_file) {		/* the default pid file */
	pid_file = (char*) xmalloc(DEFPIDLEN+strlen(basename)+1);
	pid_file = strcat(strcat(strcpy(pid_file,DEFPIDDIR),basename),DEFPIDEXT);
    }

    errno = 0;
    if (stat(pid_file, &st) < 0) {
	if (errno != ENOENT) {
	    nsyslog(LOG_ERR, "Can\'t stat %s: %s\n", pid_file, sys_errlist[errno]);
	    exit(NOPIDREAD);
	}
	/* No pid file means that we have to search in /proc/ */
	free(pid_file);
	pid_file = NULL;
    }

    if (pid_file && !st.st_size) {
	nsyslog(LOG_ERR, "Empty pid file %s for %s\n", pid_file, fullname);
	exit(NOPIDREAD);
    }

    getproc();

    if (pid_file) {		/* The case of having a pid file */
	if (verify_pidfile(pid_file,fullname,(DAEMON|PIDOF)) < 0)
	    exit(NOPIDREAD);
    }

    if (!remember) {		/* No process found with pid file */
        if (pidof(fullname,(DAEMON|PIDOF)) < 0)
	    exit(NOPIDREAD);
    }

    if (remember)
	exit(NOPROCESS);

    (void)do_fork(fullname, argv, log_file, nicelvl, env);

    /* Do we have started it? */

check_again:
    /* Here we have to ignore zombies because a zombie isn't that what
       we want to fire of */
    pidof(fullname,(DAEMON|PIDOF|NZOMBIE));

    if (!remember)
	exit(NOPROCESS);

    if (seconds > 0) {
	seconds--;
        sleep(1);
	goto check_again;
    }

    if (!quiet)
	    printf("%d\n", remember->pid);
    exit(0);

} /* end of main */

/* The core function */
static int do_fork(const char *name, char *argv[], const char* log,
		   const int nicelvl, const int env)
{
    extern char * we_are;
    int tty = 255;
    int devnull, olderr, status, n = 0;
    FILE *tmp = NULL;
    pid_t pid;

    devnull = open("/dev/null",O_RDONLY,0);
    if (log) {
	errno = 0;
	if ((tmp = fopen(log,"a")) == NULL) {
	    nsyslog(LOG_ERR,"(do_fork) cannot open %s: %s\n", log, sys_errlist[errno]);
	    exit(NOLOGFILE);
	}
    }

    fflush(stdout);
    fflush(stderr);		/* flush stdout and especially stderr */
    errno = 0;
    switch ((pid = fork())) {
    case 0:
	if (env)
	    set_newenv(name);
	 else
	    set_environ(name);

	if (nicelvl != 0) {
	    errno = 0;
	    if (setpriority(PRIO_PROCESS, getpid(), nicelvl) < 0) {
		nsyslog(LOG_ERR,"(do_fork[1]) cannot set nicelevel: %s\n",
			sys_errlist[errno]);
		exit(NOPROCESS);
	    }
	}
	if (sess) {
	    errno = 0;
	    if (setsid() < 0) {
		nsyslog(LOG_ERR,"(do_fork[1]) cannot create session: %s\n",
			sys_errlist[errno]);
		exit(NOPROCESS);
	    }
	}
	if (grp) {
	    if (setgid(grp->gr_gid) < 0) {
		nsyslog(LOG_ERR,"(do_fork[1]) cannot set group id %u: %s\n",
			grp->gr_gid, sys_errlist[errno]);
		exit(NOPROCESS);
	    }
	}
	if (user) {
	    gid_t ngid = user->pw_gid;
	    if (grp)
		ngid = grp->gr_gid;
	    else {
		if (setgid(ngid) < 0) {
		    nsyslog(LOG_ERR,"(do_fork[1]) cannot set group id %u: %s\n",
			    (unsigned int)ngid, sys_errlist[errno]);
		    exit(NOPROCESS);
		}
	    }
	    if (!getuid()) {
	        if (initgroups(user->pw_name, ngid) < 0) {
		    nsyslog(LOG_ERR,"(do_fork[1]) cannot set supplemental group ids for user %s: %s\n",
			    user->pw_name, sys_errlist[errno]);
		    exit(NOPROCESS);
		}
	    }
	    if (setuid(user->pw_uid) < 0) {
		nsyslog(LOG_ERR,"(do_fork[1]) cannot set user id %u: %s\n",
			(unsigned int)user->pw_uid, sys_errlist[errno]);
		exit(NOPROCESS);
	    }
	}
	dup2(devnull, fileno(stdin));
	if ((log && tmp) || supprmsg) {
	    while (--tty)
		if (isatty(tty)) close(tty);
	    if (log && tmp) {		/* log file for service messages */
		dup2(fileno(tmp), fileno(stdout));
		dup2(fileno(tmp), fileno(stderr));
		setlinebuf(tmp);
	    } else if (supprmsg) {	/* suppress service messages */
		devnull = open("/dev/null",O_WRONLY,0);
		dup2(devnull, fileno(stdout));
		dup2(devnull, fileno(stderr));
	    }
	}
	fflush(stdout);
	fflush(stderr);		/* flush stdout and especially stderr */
	closelog();
	chdir("/");
	errno = 0;
#if DEBUG
	printf("execve(%s, [", name);
	while (argv && *argv) {
	    printf(" %s", *argv);
	    argv++;
	}
	printf(" ], [");
	while (environ && *environ) {
	    printf(" %s", *environ);
	    environ++;
	}
	printf(" ]);\n");
#else
	execve(name, argv, environ);
#endif
	olderr = errno;
	close(fileno(stdout));
	close(fileno(stderr));
	if (tmp)
	    fclose(tmp);
	devnull = open("/dev/tty",O_WRONLY,0);
	dup2(devnull, fileno(stdout));
	dup2(devnull, fileno(stderr));
	openlog (we_are, LOG_OPTIONS, LOG_DAEMON);
	nsyslog(LOG_ERR,"(do_fork[1]) cannot execute %s: %s\n", name, sys_errlist[olderr]);
	exit(NOPROCESS);
	break;
    case -1:
	if (tmp)
	    fclose(tmp);
	fflush(stdout);
	fflush(stderr);		/* flush stdout and especially stderr */
	nsyslog(LOG_ERR,"(do_fork[2]) cannot execute %s: %s\n", name, sys_errlist[errno]);
	exit(NOPROCESS);
	break;
    default:
	if (tmp)
	    fclose(tmp);
	fflush(stdout);
	fflush(stderr);		/* flush stdout and especially stderr */
retry:
	usleep(60*1000);	/* 60 ms time for the child and its child */
	errno = 0;
	switch (waitpid(pid, &status, WNOHANG)) {
	case -1:		/* WNOHANG and hopefully no child but daemon */
	    if (WIFEXITED(status) || WIFSIGNALED(status))
		return status;
	case 0:			/* WNOHANG and no status available */
	    if (errno && errno != ECHILD) {
		nsyslog(LOG_ERR,"(do_fork[3]) cannot execute %s: %s\n", name, sys_errlist[errno]);
		exit (NOPROCESS);
	    }
	    break;
	default:
	    if (++n < 10)
		goto retry;
	    break;
	}
	break;
    }
    return 0;
}
