/*
 * Routines for daemon, killproc, killall5, pidof, and runlevel.
 *
 * Version:	1.8 05-Aug-1999 Fink
 *
 * Copyright 1994,95 Werner Fink, 1996-99 S.u.S.E. GmbH Fuerth, Germany.
 *
 * Some parts of this software are copied out of killall5.c of the
 * sysvinit suite 2.57b, Copyright 1991-1995 Miquel van Smoorenburg.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Author:	Werner Fink <werner@suse.de>, 1994-1999
 *
 * 1998/09/29 Werner Fink: Add kernel thread handling.
 * 1999/02/24 Werner Fink: Advance script search
 * 1999/08/05 Werner Fink: Handle ignoring zombies
 */

#include "libinit.h"  /* Now get the inlined functions */
#ifndef  INITDIR
# define INITDIR	"/sbin/init.d"
#endif

char     * newenvp[MAXENV];
unsigned   newenvc = 0;
PROC     * remember = (PROC*)0;

char * we_are;
unsigned short stopped = 0;
pid_t p_pid, p_sid, p_ppid;

char *sys_signame [NSIG+1];
char *sys_sigalias[NSIG+1];

void init_signames(void)
{
    int n;

    for (n = 0; n < NSIG+1; n++) {
	sys_signame [n] = (char *)0;
	sys_sigalias[n] = (char *)0;
    }

    sys_signame [0] = "EXIT";
    sys_sigalias[0] = "EXIT";

/*     Signal        Value  Action     Comment */
#ifdef SIGHUP    /*    1       A       Hangup detected on controlling terminal or */
                                    /* death of controlling process */
    if (!sys_signame[SIGHUP])
	sys_signame [SIGHUP]    = "HUP";
    else
	sys_sigalias[SIGHUP]    = "HUP";
#endif
#ifdef SIGINT    /*    2       A       Interrupt from keyboard */

    if (!sys_signame[SIGINT])
	sys_signame [SIGINT]    = "INT";
    else
	sys_sigalias[SIGINT]    = "INT";
#endif
#ifdef SIGQUIT   /*    3       A       Quit from keyboard */

    if (!sys_signame[SIGQUIT])
	sys_signame [SIGQUIT]   = "QUIT";
    else
	sys_sigalias[SIGQUIT]   = "QUIT";
#endif
#ifdef SIGILL    /*    4       A       Illegal Instruction */

    if (!sys_signame[SIGILL])
	sys_signame [SIGILL]    = "ILL";
    else
	sys_sigalias[SIGILL]    = "ILL";
#endif
#ifdef SIGABRT   /*    6       C       Abort signal from abort(3) */

    if (!sys_signame[SIGABRT])
	sys_signame [SIGABRT]   = "ABRT";
    else
	sys_sigalias[SIGABRT]   = "ABRT";
#endif
#ifdef SIGFPE    /*    8       C       Floating point exception */
    if (!sys_signame[SIGFPE])
	sys_signame [SIGFPE]    = "FPE";
    else
	sys_sigalias[SIGFPE]    = "FPE";
#endif
#ifdef SIGKILL   /*    9      AEF      Kill signal */

    if (!sys_signame[SIGKILL])
	sys_signame [SIGKILL]   = "KILL";
    else
	sys_sigalias[SIGKILL]   = "KILL";
#endif
#ifdef SIGSEGV   /*    11      C       Invalid memory reference */

    if (!sys_signame[SIGSEGV])
	sys_signame [SIGSEGV]   = "SEGV";
    else
	sys_sigalias[SIGSEGV]   = "SEGV";
#endif
#ifdef SIGPIPE   /*    13      A       Broken pipe: */

    if (!sys_signame[SIGPIPE])
	sys_signame [SIGPIPE]   = "PIPE";/* write to pipe with no readers */
    else
	sys_sigalias[SIGPIPE]   = "PIPE";/* write to pipe with no readers */
#endif
#ifdef SIGALRM   /*    14      A       Timer signal from alarm(1) */

    if (!sys_signame[SIGALRM])
	sys_signame [SIGALRM]   = "ALRM";
    else
	sys_sigalias[SIGALRM]   = "ALRM";
#endif
#ifdef SIGTERM   /*    15      A       Termination signal */

    if (!sys_signame[SIGTERM])
	sys_signame [SIGTERM]   = "TERM";
    else
	sys_sigalias[SIGTERM]   = "TERM";
#endif
#ifdef SIGUSR1   /* 30,10,16   A       User-defined signal 1 */

    if (!sys_signame[SIGUSR1])
	sys_signame [SIGUSR1]   = "USR1";
    else
	sys_sigalias[SIGUSR1]   = "USR1";
#endif
#ifdef SIGUSR2   /* 31,12,17   A       User-defined signal 2 */

    if (!sys_signame[SIGUSR2])
	sys_signame [SIGUSR2]   = "USR2";
    else
	sys_sigalias[SIGUSR2]   = "USR2";
#endif
#ifdef SIGCHLD   /* 20,17,18   B       Child stopped or terminated */

    if (!sys_signame[SIGCHLD])
	sys_signame [SIGCHLD]   = "CHLD";
    else
	sys_sigalias[SIGCHLD]   = "CHLD";
#endif
#ifdef SIGCONT   /* 19,18,25           Continue if stopped */

    if (!sys_signame[SIGCONT])
	sys_signame [SIGCONT]   = "CONT";
    else
	sys_sigalias[SIGCONT]   = "CONT";
#endif
#ifdef SIGSTOP   /* 17,19,23  DEF      Stop process */

    if (!sys_signame[SIGSTOP])
	sys_signame [SIGSTOP]   = "STOP";
    else
	sys_sigalias[SIGSTOP]   = "STOP";
#endif
#ifdef SIGTSTP   /* 18,20,24   D       Stop typed at tty */

    if (!sys_signame[SIGTSTP])
	sys_signame [SIGTSTP]   = "TSTP";
    else
	sys_sigalias[SIGTSTP]   = "TSTP";
#endif
#ifdef SIGTTIN   /* 21,21,26   D       tty input for background process */

    if (!sys_signame[SIGTTIN])
	sys_signame [SIGTTIN]   = "TTIN";
    else
	sys_sigalias[SIGTTIN]   = "TTIN";
#endif
#ifdef SIGTTOU   /* 22,22,27   D       tty output for background process */

    if (!sys_signame[SIGTTOU])
	sys_signame [SIGTTOU]   = "TTOU";
    else
	sys_sigalias[SIGTTOU]   = "TTOU";
#endif
#ifdef SIGTRAP   /*    5       CG      Trace/breakpoint trap */

    if (!sys_signame[SIGTRAP])
	sys_signame [SIGTRAP]   = "TRAP";
    else
	sys_sigalias[SIGTRAP]   = "TRAP";
#endif
#ifdef SIGIOT    /*    6       CG      IOT trap. A synonym for SIGABRT */

    if (!sys_signame[SIGIOT])
	sys_signame [SIGIOT]    = "IOT";
    else
	sys_sigalias[SIGIOT]    = "IOT";
#endif
#ifdef SIGEMT    /*  7,-,7     G */

    if (!sys_signame[SIGEMT])
	sys_signame [SIGEMT]    = "EMT";
    else
	sys_sigalias[SIGEMT]    = "EMT";
#endif
#ifdef SIGBUS    /* 10,7,10    AG      Bus error */

    if (!sys_signame[SIGBUS])
	sys_signame [SIGBUS]    = "BUS";
    else
	sys_sigalias[SIGBUS]    = "BUS";
#endif
#ifdef SIGSYS    /* 12,-,12    G       Bad argument to routine (SVID) */

    if (!sys_signame[SIGSYS])
	sys_signame [SIGSYS]    = "SYS";
    else
	sys_sigalias[SIGSYS]    = "SYS";
#endif
#ifdef SIGSTKFLT /*  -,16,-    AG      Stack fault on coprocessor */

    if (!sys_signame[SIGSTKFLT])
	sys_signame [SIGSTKFLT] = "STKFLT";
    else
	sys_sigalias[SIGSTKFLT] = "STKFLT";
#endif
#ifdef SIGURG    /* 16,23,21   BG      Urgent condition on socket (4.2 BSD) */

    if (!sys_signame[SIGURG])
	sys_signame [SIGURG]    = "URG";
    else
	sys_sigalias[SIGURG]    = "URG";
#endif
#ifdef SIGIO     /* 23,29,22   AG      I/O now possible (4.2 BSD) */

    if (!sys_signame[SIGIO])
	sys_signame [SIGIO]     = "IO";
    else
	sys_sigalias[SIGIO]     = "IO";
#endif
#ifdef SIGPOLL   /*            AG      A synonym for SIGIO (System V) */

    if (!sys_signame[SIGPOLL])
	sys_signame [SIGPOLL]   = "POLL";
    else
	sys_sigalias[SIGPOLL]   = "POLL";
#endif
#ifdef SIGCLD    /*  -,-,18    G       A synonym for SIGCHLD */

    if (!sys_signame[SIGCLD])
	sys_signame [SIGCLD]    = "CLD";
    else
	sys_sigalias[SIGCLD]    = "CLD";
#endif
#ifdef SIGXCPU   /* 24,24,30   AG      CPU time limit exceeded (4.2 BSD) */

    if (!sys_signame[SIGXCPU])
	sys_signame [SIGXCPU]   = "XCPU";
    else
	sys_sigalias[SIGXCPU]   = "XCPU";
#endif
#ifdef SIGXFSZ   /* 25,25,31   AG      File size limit exceeded (4.2 BSD) */

    if (!sys_signame[SIGXFSZ])
	sys_signame [SIGXFSZ]   = "XFSZ";
    else
	sys_sigalias[SIGXFSZ]   = "XFSZ";
#endif
#ifdef SIGVTALRM /* 26,26,28   AG      Virtual alarm clock (4.2 BSD) */

    if (!sys_signame[SIGVTALRM])
	sys_signame [SIGVTALRM] = "VTALRM";
    else
	sys_sigalias[SIGVTALRM] = "VTALRM";
#endif
#ifdef SIGPROF   /* 27,27,29   AG      Profile alarm clock */

    if (!sys_signame[SIGPROF])
	sys_signame [SIGPROF]   = "PROF";
    else
	sys_sigalias[SIGPROF]   = "PROF";
#endif
#ifdef SIGPWR    /* 29,30,19   AG      Power failure (System V) */

    if (!sys_signame[SIGPWR])
	sys_signame [SIGPWR]    = "PWR";
    else
	sys_sigalias[SIGPWR]    = "PWR";
#endif
#ifdef SIGINFO   /* 29,-,-     G       A synonym for SIGPWR */

    if (!sys_signame[SIGINFO])
	sys_signame [SIGINFO]   = "INFO";
    else
	sys_sigalias[SIGINFO]   = "INFO";
#endif
#ifdef SIGLOST   /*  -,-,-     AG      File lock lost */

    if (!sys_signame[SIGLOST])
	sys_signame [SIGLOST]   = "LOST";
    else
	sys_sigalias[SIGLOST]   = "LOST";
#endif
#ifdef SIGWINCH  /* 28,28,20   BG      Window resize signal (4.3 BSD, Sun) */

    if (!sys_signame[SIGWINCH])
	sys_signame [SIGWINCH]  = "WINCH";
    else
	sys_sigalias[SIGWINCH]  = "WINCH";
#endif
#ifdef SIGUNUSED /*  -,31,-    AG      Unused signal */

    if (!sys_signame[SIGUNUSED])
	sys_signame [SIGUNUSED] = "UNUSED";
    else
	sys_sigalias[SIGUNUSED] = "UNUSED";
#endif
}

/* write to syslog file if not open terminal */
void nsyslog(int pri, char *fmt, ...)
{
    extern void vsyslog (/* int, __const char *, __gnuc_va_list */);
    va_list  args;

    va_start(args, fmt);

    /* file descriptor of stderr is 2 in most cases */
    if (ttyname(fileno(stderr)) == NULL)
	vsyslog(pri, fmt, args);
    else {
	fprintf(stderr,"%s: ",we_are);
	vfprintf(stderr, fmt, args);
    }

    va_end(args);
}

/* See if the proc filesystem is there. Mount if needed. */
void getproc()
{
    struct stat st;
    pid_t pid, wst;

    /* Stat /proc/version to see if /proc is mounted. */
    if (stat("/proc/version", &st) < 0) {

	/* It's not there, so mount it. */
	errno = 0;
	if ((pid = fork()) < 0) {
	    nsyslog(LOG_ERR,"cannot fork: %s\n", sys_errlist[errno]);
	    exit(1);
	}
	errno = 0;
	if (pid == 0) {
	    /* Try a few mount binaries. */
	    execl("/sbin/mount", "mount", "-t", "proc", "none", "/proc", NULL);
	    execl("/etc/mount", "mount", "-t", "proc", "none", "/proc", NULL);
	    execl("/bin/mount", "mount", "-t", "proc", "none", "/proc", NULL);

	    /* Okay, I give up. */
	    nsyslog(LOG_ERR,"cannot execute mount: %s\n", sys_errlist[errno]);
	    exit(1);
	}
	/* Wait for child. */
	while(wait(&wst) != pid)
            __asm__ __volatile__("": : :"memory");
	if (WEXITSTATUS(wst) != 0)
	    nsyslog(LOG_ERR,"mount returned not-zero exit status\n");
    }

    /* See if mount succeeded. */
    errno = 0;
    if (stat("/proc/version", &st) < 0) {
	nsyslog(LOG_ERR,"/proc not mounted, failed to mount: %s\n", sys_errlist[errno]);
	exit(1);
    }
}

/* check a given command line for a full path script name */
static char * checkscripts(const char* entry, const size_t len)
{
    char *scrpt = (char *)entry;
    size_t cnt = len;

    if (!len || *scrpt == '/')
	/* do not check empty entries or full paths */
	goto out;
	
    do {
	/* After the first zero we have the first argument
	 * which may be the name of a so what ever script.
	 */
	scrpt = (char *)memchr(scrpt, 0, cnt);
	if (!scrpt || (cnt = len - (++scrpt - entry)) <= 0)
	    goto out;
	if (*scrpt == '/')
	    return scrpt;
    } while (scrpt && cnt > 0);

out:
    return NULL;
}

/* secure read on EINTR */
static ssize_t xread(int fd, void *buf, size_t count)
{
    register ssize_t bytes;
    register int olderr = errno;

    while (1) {
	errno = 0;
	bytes = read(fd, buf, count);
	if (bytes < 0 && errno == EINTR)
	    continue;
	if (bytes < 0)
	    break;
	errno = olderr;
	return bytes;
    }

    nsyslog(LOG_ERR,"xread error: %s\n", sys_errlist[errno]);
    errno = olderr;
    return bytes;
}

/* Remember all pids not being the caller and its parent */
int allpids (void)
{
    DIR *dir;
    struct dirent *d;
    int fp;
    unsigned num = 0;
    pid_t pid, sid;
    char tmp[CMDLLEN];
    PROC *p, *n;

    p_pid  = getpid();
    p_ppid = getppid();

    errno = 0;
    if ((dir = opendir("/proc")) == (DIR *)0) {
	nsyslog(LOG_ERR,"cannot opendir(/proc): %s\n", sys_errlist[errno]);
	return -1;
    }

    n = remember;
    for (p = remember; n; p = n) {
	n = p->next;
	free(p);
    }
    remember = (PROC*)0;

    /* Real System5 killall (also known as killall5) need only this one,
     * this case is not used by killproc, daemon/startproc, pidof/pidofproc */
    while((d = readdir(dir)) != (struct dirent *)0) {
	if ((pid = (pid_t)atoi(d->d_name)) == 0)
	    continue;

	if ((fp = open(stat_entry(tmp, d->d_name),O_RDONLY,0)) != -1) {
	    char entry[MAXNAMLEN];
	    bzero(entry,MAXNAMLEN);
	    xread(fp,entry,MAXNAMLEN);
	    close(fp);
	    /* see `man 5 proc' */
	    if (sscanf(entry,"%*d %*s %*c %*d %*d %d",&sid) != 1) {
		sid = 0;
		nsyslog(LOG_ERR,"%s can\'t read sid for process %d!\n",
		we_are,pid);
            }
	    if (pid == p_pid) p_sid = sid;
	    do_list(pid,sid);
	    num++;
	}

    }
    closedir(dir);
    return num;
}

/* Search proc table with a gotten full path of a running programm
   and remember it */
int pidof ( const char * fullname, const unsigned short flags )
{
    DIR *dir;
    struct dirent *d;
    struct stat full_st, pid_st;
    int fp;
    unsigned num = 0;
    pid_t pid;
    char *swapname = NULL, tmp[CMDLLEN];
    PROC *p, *n;

    p_pid  = getpid();
    p_ppid = getppid();

    errno = 0;
    if ((dir = opendir("/proc")) == (DIR *)0) {
	nsyslog(LOG_ERR,"cannot opendir(/proc): %s\n", sys_errlist[errno]);
	return -1;
    }

    if (!fullname) {
	nsyslog(LOG_ERR,"program or process name required\n");
	return -1;
    }

    n = remember;
    for (p = remember; n; p = n) {
	n = p->next;
	free(p);
    }
    remember = (PROC*)0;

    /* killproc, daemon/startproc, pidof/pidofproc: stat fullname if a
     * real program is handled, skip this if we handle a kernel thread */

    if (!(flags & KTHREAD)) {
	errno = 0;
	if (stat(fullname, &full_st) < 0) {
	    /* stat() follows soft links -> security */
	    nsyslog(LOG_ERR,"cannot stat %s: %s\n", fullname, sys_errlist[errno]);
	    return -1;
	}
    }

    swapname = swap_name((flags & KTHREAD) ? fullname : base_name(fullname));

    /* killproc, daemon/startproc, pidof/pidofproc */
    while((d = readdir(dir)) != (struct dirent *)0) {

	/* Only directories with pid as names */
	if ((pid = (pid_t)atoi(d->d_name)) == 0)
	    continue;

	/* killproc and daemon/startproc should use the full path.
	 * Kernels 2.1 and above do not lost this link even if
	 * the program is swapped out :-) */
	if (stat(exe_entry(tmp, d->d_name), &pid_st) == 0) {
	    /* Kernel threads do not have an readable exe entry
	     * because there is no symbolic link to follow ...
	     * this is not a Kernel thread */
	    if (flags & KTHREAD)
		continue;
	    if (    pid_st.st_dev == full_st.st_dev
		 && pid_st.st_ino == full_st.st_ino ) {
		if (pid_st.st_nlink > 1)
			continue;	/* We don't know the correct name */
		do_list(pid,0);
		num++;
		continue;
	    }
	    /* Device/Inode not identical, do *not* skip it
	     * to catch some running scripts */
	}

	if (!(flags & KTHREAD) &&	/* Not useful for kernel threads */
	    (fp = open(cmd_entry(tmp, d->d_name),O_RDONLY,0)) != -1) {
	    char entry[MAXNAMLEN], *scrpt = NULL;
	    ssize_t len;

	    bzero(entry,MAXNAMLEN);
	    len = xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    if (pid != p_pid)
		scrpt = checkscripts(entry, len);

	    /* Don't blame our boot scripts having the same name */
	    if ((flags & (KILL|DAEMON)) &&
		 scrpt && (strncmp(scrpt, INITDIR, (sizeof(INITDIR) - 1)) == 0))
		continue;

	    if (	  strcmp(entry,fullname) == 0 ||
		(scrpt && strcmp(scrpt,fullname) == 0)  ) {
		do_list(pid,0);
		num++;
		continue;
	    }
	}

	/*
	 * killproc and daemon/startproc should not stat the calling
	 * process or script e.g. /sbin/init.d/xdm ... because the
	 * name found in stat is identical with the real exectuable
	 * (e.g. xdm) and we do not want kill our parent process or script.
	 */
	if ((pid == p_pid || pid == p_ppid || pid == 1) &&
	    (flags & (KILL|DAEMON)))
	    continue;

	/*
	 * High risk ... the name in stat isn't exact enough to
	 * identify a swapped out process, because only the name
	 * without its path is stated which may not be identical
	 * with the executable its self.
         */
	if ((fp = open(stat_entry(tmp, d->d_name),O_RDONLY,0)) != -1) {
	    char entry[MAXNAMLEN], *comm, *state;
	    bzero(entry,MAXNAMLEN);
	    xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    comm  = index(entry,  ' ');
	    state = index(++comm, ' ');
	    *state++ = '\0';

	    if ( (flags & NZOMBIE) && state[0] == 'Z' )
		/* This is a zombie, ignore it */
		continue;

	    if ( strcmp(comm, swapname) == 0 ) {
		do_list(pid,0);
		num++;
		continue;
	    }
	}
    }

    closedir(dir);
    free(swapname);
    return num;
}

/* Open, read, and verify pid file, if pid found remember it */
int verify_pidfile ( const char * pid_file, const char * fullname,
		     const unsigned short flags )
{
    int fp, cnt;
    pid_t pid;
    char buf[CMDLLEN], *bufp, tmp[CMDLLEN];
    struct stat pid_st, full_st;
    PROC *p, *n;
    
    n = remember;
    for (p = remember; n; p = n) {
	n = p->next;
	free(p);
    }
    remember = (PROC*)0;

    errno = 0;
    if ((fp = open (pid_file, O_RDONLY,0)) < 0 ) {
	nsyslog(LOG_ERR, "Can\'t open pid file %s: %s\n",
		pid_file, sys_errlist[errno]);
	return -1;
    }

    errno = 0;
    if ((cnt = xread (fp, buf, CMDLLEN)) < 0) {
	nsyslog(LOG_ERR, "Can\'t read pid file %s: %s\n",
		pid_file, sys_errlist[errno]);
	return -1;
    }
    close(fp);
    buf[CMDLLEN] = '\0';

    bufp = buf;
    while (--cnt && isspace(*bufp)) bufp++;
    strcpy(buf, bufp);

    if ((bufp = strpbrk(buf, "\r\n\f\t\v \0")))
	*bufp = '\0';

    errno = 0;
    if ((pid = (pid_t)atoi(buf)) <= 0) {
	if (errno)
	    nsyslog(LOG_ERR, "Can\'t handle pid file %s with pid %s: %s\n",
		    pid_file, buf, sys_errlist[errno]);
	if (!pid)
	    nsyslog(LOG_ERR, "Can\'t handle pid file %s with pid `%s\'\n",
		    pid_file, buf);
	return -1;
    }

    if (pid == getpid())
	return 0;		/* Don't kill myself */

    if (!fullname) {
	nsyslog(LOG_ERR,"program or process name required\n");
	return -1;
    }

    if (!(flags & KTHREAD)) {
	errno = 0;
	if (stat(fullname, &full_st) < 0) {
	    /* stat() follows soft links -> security */
	    nsyslog(LOG_ERR,"cannot stat %s: %s\n", fullname, sys_errlist[errno]);
	    return -1;
	}
    }

    /* killproc and daemon/startproc should use the full path */
    errno = 0;
    if (stat(exe_entry(tmp, buf), &pid_st) < 0) {
	if (errno != ENOENT) {
	    nsyslog(LOG_ERR, "Can\'t read %s: %s\n", tmp, sys_errlist[errno]);
	    return -1;
	}
    } else {
	/* Kernel threads do not have an readable exe entry
	 * because there is no symbolic link to follow ...
	 * this is not a Kernel thread */
	if (flags & KTHREAD) {
	    nsyslog(LOG_ERR, "The pid %d is not a kernel thread\n", (int)pid);
	    return -1;
	}
	if (    pid_st.st_dev   == full_st.st_dev
	     && pid_st.st_ino   == full_st.st_ino
	     && pid_st.st_nlink == 1 ) {
	    do_list(pid,0);
	    return 0;		/* Done */
	}
	/* Device/Inode not identical, do *not* skip it
	 * to catch some running scripts */
    }

    if (!(flags & KTHREAD) &&		/* Not useful for kernel threads */
	(fp = open(cmd_entry(tmp, buf),O_RDONLY,0)) != -1) {
	char entry[MAXNAMLEN], *scrpt = NULL;
	ssize_t len;

	bzero(entry,MAXNAMLEN);
	len = xread(fp,entry,MAXNAMLEN);
	close(fp);

	scrpt = checkscripts(entry, len);

	if (          strcmp(entry,fullname) == 0 &&
	    (scrpt && strcmp(scrpt,fullname) == 0)  ) {
	    do_list(pid,0);
	    return 0;		/* Done */
	}
    }

    if ((fp = open(stat_entry(tmp, buf),O_RDONLY,0)) != -1) {
	char entry[MAXNAMLEN], *swapname, *comm, *state;
	bzero(entry,MAXNAMLEN);
	xread(fp,entry,MAXNAMLEN);
	close(fp);
	swapname = swap_name((flags & KTHREAD) ? fullname : base_name(fullname));

	comm  = index(entry,  ' ');
	state = index(++comm, ' ');
	*state++ = '\0';

	if ( (flags & NZOMBIE) && state[0] == 'Z' ) {
	    /* This is a zombie, ignore it */
	    free(swapname);
	    return 0;
	}

	if ( strcmp(comm, swapname) == 0 ) {
	    do_list(pid,0);
	    free(swapname);
	    return 0;		/* Done */
	}
	free(swapname);
    }

    return 0;			/* Nothing found */
}

/* Check remembered pids, every pid will be verified */
int check_pids ( const char * fullname, const unsigned short flags )
{
    int fp, c;
    char *swapname = NULL, buf[10], tmp[CMDLLEN];
    struct stat pid_st, full_st;
    PROC *p, *n, *l;

    if (!fullname) {
	nsyslog(LOG_ERR,"program or process name required\n");
	return -1;
    }

    if (!(flags & KTHREAD)) {
	errno = 0;
	if (stat(fullname, &full_st) < 0) {
	    /* stat() follows soft links -> security */
	    nsyslog(LOG_ERR,"cannot stat %s: %s\n", fullname, sys_errlist[errno]);
	    return -1;
	}
    }

    swapname = swap_name((flags & KTHREAD) ? fullname : base_name(fullname));

    n = remember;
    l = (PROC*)0;
    for(p = remember; n; p = n) {
	l = p->prev;
	n = p->next;

	errno = 0;
	if ((c = snprintf(buf, 9, "%d", p->pid)) > 0)
	    buf[c] = '\0';
	else {
	    nsyslog(LOG_ERR,"error in snprintf: %s\n", sys_errlist[errno]);
	    free(swapname);
	    return -1;
	}

	/* killproc and daemon/startproc should use the full path */
	errno = 0;
	if (stat(exe_entry(tmp, buf), &pid_st) == 0) {
	    /* Kernel threads do not have an readable exe entry
	     * because there is no symbolic link to follow ...
	     * this is not a Kernel thread */
	    if (flags & KTHREAD) {
		nsyslog(LOG_ERR, "The pid %d is not a kernel thread\n", (int)(p->pid));
		free(swapname);
		return -1;
	    }
	    if (   pid_st.st_dev   == full_st.st_dev
		   && pid_st.st_ino   == full_st.st_ino
		   && pid_st.st_nlink == 1 ) {
		continue;	/* Found */
	    }
	    /* Device/Inode not identical, do *not* skip it
	     * to catch some running scripts */
	}

	if (!(flags & KTHREAD) &&		/* Not useful for kernel threads */
	    (fp = open(cmd_entry(tmp, buf),O_RDONLY,0)) != -1) {
	    char entry[MAXNAMLEN], *scrpt;
	    ssize_t len;

	    bzero(entry,MAXNAMLEN);
	    len = xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    scrpt = checkscripts(entry, len);

	    if (	  strcmp(entry,fullname) == 0 &&
	        (scrpt && strcmp(scrpt,fullname) == 0)  ) {
		continue;	/* Found */
	    }
	}

	if ((fp = open(stat_entry(tmp, buf),O_RDONLY,0)) != -1) {
	    char entry[MAXNAMLEN], *comm, *state;
	    bzero(entry,MAXNAMLEN);
	    xread(fp,entry,MAXNAMLEN);
	    close(fp);

	    comm  = index(entry,  ' ');
	    state = index(++comm, ' ');
	    *state++ = '\0';

	    if ( ((flags & NZOMBIE) ? state[0] != 'Z' : 1) && strcmp(comm, swapname) == 0 ) {
		continue;	/* Found */
	    }
	}

	/* Remove this entry in remember */
	if (p == remember) {
	    if (n) n->prev = (PROC*)0;
	    remember = n;
	    free(p);
	} else if (l) {
	    if (n) n->prev = l;
	    l->next = n;
	    free(p);
	} else {
	    nsyslog(LOG_ERR, "error in linked list handling\n");
	    free(swapname);
	    return -1;
	}

    }

    free(swapname);
    return 0;			/* Nothing found */
}

void check_su()
{
#if DEBUG
    printf("Real user ID: %d\n",(int)getuid());
#else
    if ( 0 != (int)getuid() ) {
	nsyslog(LOG_ERR,"%s: Only root sould run this program\n",we_are);
	exit(1);
    }
#endif
    return;
}

/* Used in libinit.h in the inlined function set_newenv */
void addnewenv ( const char * name, const char * entry )
{
    unsigned i, len = strlen(name);
    char *cp = (char*)xmalloc(len+2+strlen(entry));
    extern unsigned newenvc;

    bzero(cp,sizeof(cp));
    (void)strcat(strcat(strcpy(cp,name),"="),entry);

    for (i=0; i < newenvc; i++)
	if (strncmp (cp, newenvp[i], len) == 0 &&
	    (newenvp[i][len] == '=' || newenvp[i][len] == '\0'))
		break;

    if (i == MAXENV) {
	puts ("Environment overflow");
	return;
    }

    if (i == newenvc) {
	newenvp[newenvc++] = cp;
	newenvp[newenvc] = (char*)0;
    } else {
	free (newenvp[i]);
	newenvp[i] = cp;
    }
}

/* Used in libinit.h in the inlined functions set_newenv set_environ */
char ** runlevel(const char *file)
{
    struct utmp *ut = (struct utmp*)0;
    char **buf = (char**)xmalloc(2 * sizeof(char*));

    if (file)
	utmpname(file);
    else
	utmpname(UTMP_FILE);

    setutent();
    while ( (ut = getutent()) != (struct utmp*)0 )
	if ( ut->ut_type == RUN_LVL )
	    break;
    endutent();

    buf[0] = (char*)xmalloc(sizeof(char[2]));
    buf[1] = (char*)xmalloc(sizeof(char[2]));

    buf[0][1] = buf[1][1] = '\0';
#if DEBUG
    buf[0][0] = '0';
    buf[1][0] = '3';
#else
    if (ut && ut->ut_pid) {
	buf[0][0] = ut->ut_pid / 256;
	buf[1][0] = ut->ut_pid % 256;
    } else {
	buf[0][0] = 'N';
	buf[1][0] = '?';
    }
#endif

    return buf;
}

/* Used in killproc.c to translate signal names into signal numbers */
int signame_to_signum (const char *sig)
{
    int n;

    init_signames();
    if (!strncasecmp (sig, "sig", 3))
	sig += 3;
    for (n = 1; n < NSIG+1; n++) {
	if ( sys_signame [n] && !strcasecmp (sys_signame [n], sig) )
	    return n;
	if ( sys_sigalias[n] && !strcasecmp (sys_sigalias[n], sig) )
	    return n;
    }

    return -1;
}

/* libinit.c ends here */
