/*
 * killproc.c   Kill all running processes of a named program.
 *
 * Usage:       killproc [-v] [-t<sec>] [-g|-G] [-SIG] /full/path/to/program
 *
 * Copyright 1994,95 Werner Fink, 1996-98 S.u.S.E. GmbH Fuerth, Germany.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Author:      Werner Fink <werner@suse.de>
 *
 * 1998/05/06 Florian La Roche: added "-g" option to kill process groups
 * 1998/05/06 Werner Fink: rework, added "-f" for pid files
 * 1998/15/09 Werner Fink: exit status for killing not running processes is 0
 * 1998/29/09 Werner Fink: Add kernel thread handling.
 */

#include "libinit.h"

#define NOSUCCESS	1
#define NOPIDFILE	2
#define NOPIDREAD	4
#define WRGSYNTAX	8

#define DEFSIG		"TERM"
#define OTHERSIG	"HUP"

#define USAGE		"Usage:\n"\
			"\t%s [-v] [-t<sec>] [-g|-G] [-SIG] /full/path/to/program\n" \
		 	"\t%s -l\n", we_are, we_are

static int do_kill(const char *name, const pid_t proc, const int sig,
		   const int group_leader, const int process_group);

static int quiet = 1, num;

int main(int argc, char **argv)
{
    extern char * we_are;
    int c, snum;
    struct stat st;
    volatile PROC * list;
    char *fullname = NULL, *basename = NULL, *pid_file = NULL;
    char *posixa, *posixb;	/* Don't fool me with posix correct */
    int process_group = 0, group_leader = 0, wait = 5, force_pidfile = 0;
    unsigned short flags = (KILL|PIDOF);

    we_are = base_name(argv[0]);
    openlog (we_are, LOG_OPTIONS, LOG_DAEMON);

    /* If we are not called as killproc use HUP (e.g. for sigproc) */
    if (strcmp(we_are,"killproc") != 0)
	snum = signame_to_signum(OTHERSIG);
    else
	snum = signame_to_signum(DEFSIG);

    /*
     *  We should stat() fullname, because only the path identifies the executable.
     *  If there is one hardlink we have only to stat() the orignal executable.
     *  If there is more than one hardlink and we have to distinguish the
     *  executables by their swapname.  Note if the cmdline of some executables
     *  will changed by the running process its self the name is not clearly
     *  defined ... see libinit.c for more information.
     */

    c = argc;
    while (--c) {
	if (*(argv[c]) == '-') {
	    char *sig = argv[c];
	    int tmp, len = strlen(sig);
	    sig++;
	    if ( (tmp = atoi(sig)) > 0 && tmp < NSIG ) {
		bzero(sig, len);
		*sig = 'q';		/* set dummy option -q */
		snum = tmp;
		break;
	    } else if ( (tmp = signame_to_signum(sig)) > 0 ) {
		bzero(sig, len);
		*sig = 'q';		/* set dummy option -q */
		snum = tmp;
		break;
	    }
	}
    }

    posixa = getenv("_POSIX_OPTION_ORDER"); unsetenv("_POSIX_OPTION_ORDER");
    posixb = getenv("POSIXLY_CORRECT");     unsetenv("POSIXLY_CORRECT");
    opterr = 0;
    while ((c = getopt(argc, argv, "f:gGnhlvqt:")) != -1) {
	switch (c) {
	    case 't':
		wait = atoi(optarg);
		if (wait < 1) {
		    nsyslog(LOG_ERR, USAGE);
		    exit(WRGSYNTAX);
		}
		break;
	    case 'q':
		/* A signal which have handled or the old but unused -q option */
		break;
	    case 'v':
		quiet = 0;
		break;
	    case 'g':
		if (process_group) {
		    nsyslog(LOG_ERR, USAGE);
		    exit(WRGSYNTAX);
		}
		group_leader++;
		break;
	    case 'G':
		if (group_leader) {
		    nsyslog(LOG_ERR, USAGE);
		    exit(WRGSYNTAX);
		}
		process_group++;
		break;
	    case 'n':
		flags |= KTHREAD;
		break;
	    case 'f':
		/* Allocate here: address optarg (current *argv) isn't freeable */
		if (optarg && !pid_file) {
		    pid_file = strdup(optarg);
		    force_pidfile++;
		} else {
		    nsyslog(LOG_ERR,"Option -f requires pid file to read pid from\n");
		    exit(WRGSYNTAX);
		}
		break;
	    case '?':
		nsyslog(LOG_ERR, USAGE);
		exit(WRGSYNTAX);
		break;
	    case 'h':
		nsyslog(LOG_ERR, USAGE);
		exit(0);
		break;
	    case 'l':
		list_signames();
		exit(0);
		break;
	    default:
		break;
	}
    }
    if (posixa) setenv("_POSIX_OPTION_ORDER", posixa, 0);
    if (posixb) setenv("POSIXLY_CORRECT",     posixb, 0);

    argv += optind;
    argc -= optind;

    if (*argv) {
	fullname = *argv;
	basename = base_name(fullname);
    }

    if (!fullname) {
	nsyslog(LOG_ERR, USAGE);
	exit(WRGSYNTAX);
    }

    if (!pid_file) {            /* the default pid file */
	pid_file = (char*) xmalloc(DEFPIDLEN+strlen(basename)+1);
	pid_file = strcat(strcat(strcpy(pid_file,DEFPIDDIR),basename),DEFPIDEXT);
    }

    errno = 0;
    if (stat(pid_file, &st) < 0) {
	if (errno != ENOENT) {
	    nsyslog(LOG_ERR, "Can\'t stat %s: %s\n",
		    pid_file, sys_errlist[errno]);
	    exit(NOPIDREAD);
	}
	/* No pid file means that we have to search in /proc/
	 * but for killing a job only if pid file isn't forced
	 * ... this differs in comparision to startproc! */
	if (force_pidfile) {
	    nsyslog(LOG_ERR, "No pid file %s for %s\n", pid_file, fullname);
	    if (snum == SIGTERM || snum == SIGKILL) /* already terminated */
		exit(0);
	    exit(NOPIDFILE);
	}
	free(pid_file);
	pid_file = NULL;
    }

    if (pid_file && !st.st_size) {
	nsyslog(LOG_ERR, "Empty pid file %s for %s\n", pid_file, fullname);
	exit(NOPIDREAD);
    }

    getproc();

    if (pid_file) {		/* The case of having a pid file */
	if (verify_pidfile(pid_file,fullname,flags) < 0)
	    exit(NOPIDREAD);
    } else {			/* No pid file found or given */
	if (pidof(fullname,flags) < 0)
	    exit(NOPIDREAD);
    }

    if (!remember) {
	if (snum == SIGTERM || snum == SIGKILL)     /* already terminated */
	    exit(0);
	exit(NOSUCCESS);
    }

    num = 0;
    for(list = remember; list; list = list->next)
	do_kill(basename, list->pid, snum, group_leader, process_group);

    if (snum == SIGTERM || snum == SIGKILL) {
	int halfsec = 2*wait;
	/*
	 * Does anybody have a better idea ... something with sigaction()/signal()
	 * and alarm(). Who wakes us if the terminated process is finished?
	 * The process(es) is/are not a child of us.
	 */
	usleep(60*1000);	/* 60 ms time for the process and its childs */
again:
	if (check_pids(fullname,flags) < 0)
	    exit(NOPIDREAD);

	if (!remember)		/* success */
	    goto success;

	fflush(stdout);
	fflush(stderr);
	if (halfsec-- > 0) {	/* wait */
	    usleep(5*100*1000);
	    goto again;
	}

	if (snum == SIGKILL)
	    goto badterm;

	if (check_pids(fullname,flags) < 0)
	    exit(NOPIDREAD);

	for(list = remember; list; list = list->next)
	    do_kill(basename, list->pid, SIGKILL, group_leader, process_group);

	/* Do we have killed them? */

	usleep(60*1000);	/* 60 ms time for the process and its childs */
	if (check_pids(fullname,flags) < 0)
	    exit(NOPIDREAD);

	if (remember)
	    goto badterm;

success:
	if (num) putchar('\n');
	errno = 0;
	if (pid_file && (unlink(pid_file) < 0)) {
	    if (errno != ENOENT)
		nsyslog(LOG_ERR, "Can\'t remove %s: %s\n", pid_file, sys_errlist[errno]);
	}
	exit(0);

badterm:
	if (num) putchar('\n');
	exit(NOSUCCESS);
    }

    if (num)
	putchar('\n');
    exit(0);

} /* end of main */

/* The core function */
static int do_kill(const char *name, const pid_t proc, const int sig,
		   const int group_leader, const int process_group)
{
    pid_t target = proc;
    int stop = (sig == SIGTERM || sig == SIGKILL);

    errno = 0;
    if (group_leader) {
	if ((target = -getpgid(proc)) >= 0) {
	    if (errno != ESRCH)
		nsyslog(LOG_ERR, "Can\'t signal %s to process with pid %d: %s\n",
			sys_signame[sig], (int)proc, sys_errlist[errno]);
	    exit(NOSUCCESS);
	}
    } else if (process_group)
	target = -proc;
#if DEBUG
    printf("kill(%d,%d)\n",(int)target, sig);
#else

    if (stop) kill(target, SIGSTOP);
    errno = 0;
    if (kill(target, sig) < 0) {
	if (errno != ESRCH) {
	    nsyslog(LOG_ERR, "Can\'t signal %s to process with pid %d: %s\n",
		    sys_signame[sig], (int)proc,  sys_errlist[errno]);
	    exit(NOSUCCESS);
	}
    }
    if (stop) kill(target, SIGCONT);

    if (!quiet) {
	if (num++) putchar(' ');
	printf("SIG%s %s(%d)",sys_signame[sig],name,(int)proc);
    }
#endif
    return 0;
}
