/*
 * checkproc.c  Checks process(es) of the named program.
 *
 * Usage:       checkproc [-v] [-k] [-f pid_file] /full/path/to/program
 *
 * Copyright 1994,95 Werner Fink, 1996-98 S.u.S.E. GmbH Fuerth, Germany.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Author:      Werner Fink <werner@suse.de>
 * 1998/05/06 Werner Fink: rework, added "-f" for pid files
 */

#include "libinit.h"

#define NOPROCESS	1
#define NOPIDFILE	2
#define NOPIDREAD	4
#define WRGSYNTAX	8

#define USAGE		"Usage:\n"\
			"\t%s [-v] [-k] [-f pid_file] /full/path/to/program\n" \
			, we_are

int main(int argc, char **argv)
{
    extern char * we_are;
    int c, num;
    struct stat st;
    volatile PROC * list;
    char *fullname = NULL, * basename = NULL, *pid_file = NULL;
    char *posixa, *posixb;	/* Don't fool me with posix correct */
    int quiet = 1, force_pidfile = 0;
    unsigned short flags = (DAEMON|PIDOF|NZOMBIE);

    we_are = base_name(argv[0]);
    openlog (we_are, LOG_OPTIONS, LOG_DAEMON);

/*
 *  We should stat() fullname, because only the path identifies the executable.
 *  If there is one hardlink we have only to stat() the orignal executable.
 *  If there is more than one hardlink and we have to distinguish the
 *  executables by their swapname.  Note if the cmdline of some executables
 *  will changed by the running process its self the name is not clearly
 *  defined ... see libinit.c for more information.
 */

    posixa = getenv("_POSIX_OPTION_ORDER"); unsetenv("_POSIX_OPTION_ORDER");
    posixb = getenv("POSIXLY_CORRECT");     unsetenv("POSIXLY_CORRECT");
    opterr = 0;
    while ((c = getopt(argc, argv, "kf:nhqvz")) != -1) {
	switch (c) {
	    case 'k':
		flags &= ~DAEMON;
		flags |= KILL;
		break;
	    case 'q':
		break;
	    case 'v':
		quiet = 0;
		break;
	    case 'n':
		flags |= KTHREAD;
		break;
	    case 'z':
		flags &= ~NZOMBIE;
		break;
	    case 'f':
		/* Allocate here: address optarg (current *argv) isn't freeable */
		if (optarg && !pid_file) {
		    pid_file = strdup(optarg);
		    force_pidfile++;
		} else {
		    nsyslog(LOG_ERR,"Option -f requires pid file to read pid from\n");
		    exit(WRGSYNTAX);
		}
		break;
	    case 'h':
		nsyslog(LOG_ERR, USAGE);
		exit(0);
		break;
	    case '?':
		nsyslog(LOG_ERR, USAGE);
		exit(WRGSYNTAX);
		break;
	    default:
		break;
	}
    }
    if (posixa) setenv("_POSIX_OPTION_ORDER", posixa, 0);
    if (posixb) setenv("POSIXLY_CORRECT",     posixb, 0);

    argv += optind;
    argc -= optind;

    if (*argv) {
	fullname = *argv;
	basename = base_name(fullname);
    }

    if (!fullname) {
	nsyslog(LOG_ERR, USAGE);
	exit(WRGSYNTAX);
    }

    if (!pid_file) {		/* the default pid file */
	pid_file = (char*) xmalloc(DEFPIDLEN+strlen(basename)+1);
	pid_file = strcat(strcat(strcpy(pid_file,DEFPIDDIR),basename),DEFPIDEXT);
    }

    errno = 0;
    if (stat(pid_file, &st) < 0) {
	if (errno != ENOENT) {
	    nsyslog(LOG_ERR, "Can\'t stat %s: %s\n",
		    pid_file, sys_errlist[errno]);
	    exit(NOPIDREAD);
	}
	/* No pid file means that we have to search in /proc/
	 * if KILL in flags is set this works like startproc ...
	 * if not set this works like killproc */
        if (force_pidfile) {
	    nsyslog(LOG_ERR, "No pid file %s for %s\n", pid_file, fullname);
	    if (flags & KILL)
		exit(NOPIDFILE);
	}
	free(pid_file);
	pid_file = NULL;
    }

    if (pid_file && !st.st_size) {
	nsyslog(LOG_ERR, "Empty pid file %s for %s\n", pid_file, fullname);
	exit(NOPIDREAD);
    }

    getproc();

    if (flags & DAEMON) {	/* Do the verification just like startproc */
	if (pid_file) {		  /* The case of having a pid file */
	    if (verify_pidfile(pid_file,fullname,flags) < 0)
		exit(NOPIDREAD);
	}
	if (!remember) {	  /* No process found with pid file */
	    if (pidof(fullname,flags) < 0)
		exit(NOPIDREAD);
	}
    } else {		/* Now we're act like killproc: restrictive */
	if (pid_file) {		  /* The case of having a pid file */
	    if (verify_pidfile(pid_file,fullname,flags) < 0)
		exit(NOPIDREAD);
	} else {		  /* No pid file found or given */
	    if (pidof(fullname,flags) < 0)
		exit(NOPIDREAD);
	}
    }

    num = 0;	/* If quiet we could test 'remember' and exit appropiate */
    for(list = remember; list; list = list->next) {
	if (!quiet) {
	    if (num) putchar(' ');
	    printf("%d", (int)list->pid);
	}
	num++;
    }

    if (!num)
	exit(NOPROCESS);
    if (!quiet)
	putchar('\n');

    closelog();
    exit(0);
}
