#include <fcntl.h>
#include <unistd.h>

#include <string>
#include <iostream>
#include <iomanip>

#include "init_ipc/sysvq.hh"
#include "init_ipc/User.hh"

#include "error.hh"

#include "config.h"

const char* argv0 = PACKAGE;
using namespace std;

class CommandState:public init_ipc::User
{
  std::string my_action;
  bool my_is_open;
  bool my_has_errors;
  
public:
  CommandState():my_is_open(false),my_has_errors(false)
  {
  }
  void prepare()
  {
    if(!my_is_open){
      if(!open()){
	error(_("unable to communicate with init"));
	exit(1);
      }
      my_is_open = true;
    }
  }
  bool has_errors()
  {
    return my_has_errors;
  }
  
  void service(const std::string&name)
  {
    prepare();
    
    if(my_action=="start"||my_action=="need"){
      if(!start_service(name))
	my_has_errors = true;
      return;
    }
    if(my_action=="stop"||my_action=="r"){
      if(!stop_service(name))
	my_has_errors = true;
      return;
    }
    if(my_action=="forget"){
      if(!forget_service(name))
	my_has_errors = true;
      return;
    }
    if(my_action=="dont-spawn"){
      if(!forget_respawn(name))
	my_has_errors = true;
      return;
    }
    
    error(_("don't know what to do with ") << name);
    return;
  }
  
  void set_action(const std::string&action)
  {
    my_action = action;
  }
}
;

struct Argument
{
  char const* name;
  char const* explanation;
  struct value
  {
    enum type
    {
      service,
      none
    };
  }
  ;
  
  value::type type;
  void (*handler)(const char*name,const char**value,CommandState&state);

  bool is_valid()const
  {
    return handler;
  }
  
  bool try_match(char const*n,const char**arg,CommandState&state)const
  {
    if(name)
      {
	if(strcmp(n,name))
	  return false;
	arg++;
      }
    if((!arg[1]) && (type!=value::none))
      return false;
    handler(n,arg,state);
    return true;
  }
  
  char const*type_to_string()const
  {
    const char* table[]=
    {
      "",
      _("name of a service"),
    };
    return (table[type]);
  }
}
;

namespace clo
{
  static void silent(const char*,const char**,CommandState&)
  {
    int fd = open("/dev/null",O_WRONLY);
    if(fd<0)
      {
	if(!fork())abort(); /* report this error condition in a roundabout way */
	
	close(STDERR_FILENO);
      }
    dup2(fd,STDERR_FILENO);
    close(fd);
  }
  
  static void help(const char*,const char**,CommandState&);
  
    
  static void version(const char*,const char**,CommandState&)
  {
    cout << PACKAGE << " " << VERSION << endl;
  }

  static void list(const char*,const char**,CommandState&state)
  {
    char*list;
    state.prepare();
    if(!(list=state.list_services())){
      error(_("unable to list services"));
      return;
    }
    cout << list << endl;
    delete list;
  }

  static void change_action(const char*name,const char**,CommandState&state)
  {
    const char*i=name;
    while(*i && *i=='-')i++;
    
    state.set_action(i);
  }

  static void do_action(const char*,const char**value,CommandState&state)
  {
    state.service(*value);
  }
  static void respawn(const char*,const char**value,CommandState&state)
  {
    state.prepare();
    bool ok = state.respawn(*value,value+1);
    state.close();
    
    exit( ok  ? EXIT_SUCCESS : EXIT_FAILURE);
  }
  static void fork(const char*,const char**value,CommandState&state)
  {
    switch(::fork()){
    case -1: error(_("unable to fork"));
      state.close();
      exit(10);
    case 0:
      if(setsid()==-1){
	error(_("unable to setsid"));
	state.close();
	exit(10);
      }
      break;
    default:
      exit(EXIT_SUCCESS);
    }
  }
}

static const Argument arguments[]=
{
  {"--help",_("print command line help"),Argument::value::none,clo::help
  },
  {"--version",_("print version info"),Argument::value::none,clo::version
  },
  {"--stop",_("stop following services"),Argument::value::none,clo::change_action
  }
  ,
  {"--start",_("start following services"),Argument::value::none,clo::change_action
  },
  {"--forget",_("forget following services"),Argument::value::none,clo::change_action
  },
  {"--list",_("list services"),Argument::value::none,clo::list
  },
  {"--display-services",_("list services (simpleinit compatibility)"),
   Argument::value::none,clo::list
  },
  {"--need",_("start following services (simpleinit compatibility)"),Argument::value::none,clo::change_action
  },
  {"-r",_("stop following services  (simpleinit compatibility)"),Argument::value::none,clo::change_action
  },
  {"--respawn",_("[id] [program] [argument]* -- tell init to keep the following program running"),Argument::value::none,clo::respawn
  },
  {"--dont-spawn",_("tell init not to keep the following utmp id running"),Argument::value::none,clo::change_action
  },
  {"--not-crucial",_("return before service is started (do not depend on service)"),Argument::value::none,clo::fork
  },
  {"--silent",_("don't whine about problems"),Argument::value::none,clo::silent
  },
  {0,_("default: start service specified"),Argument::value::none,clo::do_action
  },
  {
    0,0,Argument::value::none,0
  }
}
;
static void print_help(ostream& stream)
{
  
  stream << _("Command line help") << endl
	 << endl;
  for(Argument const*i=arguments;i->is_valid();i++)
    stream << setw(12) << (i->name ? i->name : "") << setw(16) << "  " << _(i->explanation) << endl;
}

namespace clo
{
  static void help(const char*,const char**,CommandState&)
  {
    print_help(cout);
  }
}


int main(int argc,char*argv[])
{
  argv0 = argv[0];
  CommandState state;
  char*after_last_slash=argv[0];
  for(char*i=argv[0];*i;i++)
    if(*i=='/')after_last_slash=i+1;
    
  state.set_action(after_last_slash);
  
  srand (time (0));
  
  if(argc<2){
    error(_("specify at least one service to start on the command line (try --help)"));
    return 1;
  }
  
  for(char**i=argv+1;*i;i++)
    {
      for(Argument const*a=arguments;a->is_valid();a++)
	if(a->try_match(i[0],(const char**)i,state))
	  goto have_found;

    havent_found:
      error(_("bad command line argument: ") << i[0]);
      print_help(cerr);
      return 3;
    have_found:
      continue;
    }
  
  return state.has_errors();
}
    
