#define _GNU_SOURCE

#include <unistd.h>
#include <signal.h>
#include <sys/resource.h>
#include <sys/reboot.h>
#include <errno.h>

#include "bootstrap.hh"
#include "pathnames.hh"
#include "init_ipc/Init.hh"
#include "ServiceNotification.hh"
#include "Inittab.hh"
#include "ProcessTab.hh"
#include "SuloginProcess.hh"
#include "Service.hh"
#include "PermitSignals.hh"
#include "simpleio.hh"

sig_atomic_t global_stop_spawning = 0;
const time_t respawn_period = 5*60; /*seconds */
static init_ipc::Init writer;

struct sigspec
{
  int sig;
  sighandler_t handler;
}
;


const int masked_signals[]=
  {
    SIGCHLD,
    SIGTERM,    
    SIGINT,
    SIGQUIT,
    SIGUSR1,
    SIGUSR2,
    SIGALRM,
    SIGTSTP,
    NSIG
    }
;
static void sigint(int),sigquit(int),sigusr1(int),sigusr2(int),sigterm(int);

static const sigspec shutdown_sigs[] =
  {
    { SIGINT, sigint 
    }
    ,
    { SIGQUIT, sigquit 
    }
    ,
    { SIGUSR1, sigusr1 
    }
    ,
    { SIGUSR2, sigusr2 
    }
    ,
    { SIGTERM, sigterm 
    },
    { 0,0 
    }
  };


static void re_exec(const char*arg=0)
{
  execl(_PATH_INIT,_PATH_INIT,arg,0);
}

static void sulogin()
{
  const char*bin = global_inittab.get_sulogin().c_str();
  say(_("emergency! Launching \"%s\""),bin);

  SuloginProcess*login;

  pid_t child = fork();
  char *const argv[]={(char*)bin,0};
  switch(child){
  case -1: say(_("can't fork to launch program, execing in place"));
  case 0: do_exec(bin,argv);
  default:
    login = new SuloginProcess(bin,child);
    login->wait();
    if(!login->is_successful()){
      say(_("can't start emergency shell"));
      pause();
      abort();
    }
    else
      say(_("\"%s\" terminated successfully, continuing boot"),bin);
  }
  delete login;
  return;
}

static void clear_shutdown_signals()
{
  struct sigaction sa;
  sa.sa_flags = 0;
  sigemptyset(&sa.sa_mask);
  sa.sa_handler = SIG_IGN;
  
  for(const sigspec*i=shutdown_sigs; i->handler; i++){
    sigaction(i->sig,&sa,0);
  }
}

static void handle_shutdown()
{
  clear_shutdown_signals();

  global_stop_spawning = 2;//won't be turned off by SIGTSTP
  alarm(0);
  if(!writer.close())
    say(_("trouble closing SysV queue"));

  /* This should kill all services */
  global_processtab.stop_all_services();
}

static void do_reboot(char* howto,char*arg2=0)
{
  const char*shutdown = global_inittab.get_shutdown().c_str();
  execl(shutdown,shutdown,howto,arg2,0);
  say(_("unable to execute \"%s\", doing a very soft boot"),shutdown);
  re_exec();
}

static void dump_core(int signum)
{
  const size_t core_size = 100000;
  if(!fork()){
      clear_signals();
      struct rlimit rl;
      if(!getrlimit (RLIMIT_CORE,&rl)){
	if(!rl.rlim_max)
	  rl.rlim_max = core_size;
	rl.rlim_cur = rl.rlim_max;
	if(setrlimit(RLIMIT_CORE,&rl))
	  abort();
      }
      else
	abort();
      
      raise(signum);
      abort();
  }
}

static void sig_fatal_error(int signum)
{
  /* we can't block real error signals */
  static volatile sig_atomic_t handling_fatal_error = 0;
  if(!handling_fatal_error++){
    say(_("received fatal signal \"%s\", restarting"),sys_siglist[signum]);
    dump_core(signum);
    /* *((int*)0)=0xdeadbeef; */
  }

  
  for(;;){
    re_exec("panic");
    if(handling_fatal_error<4){
      say(_("re-exec failed. You are screwed. Pausing."));
      /* *((int*)0)=0xdeadbeef; */
    }
    pause();/*this can be broken by sending -STOP and -CONT */
  }
}

static void sigint(int)
{
  debug((_("sigint")));
  handle_shutdown();
  do_reboot("reboot");
}
static void sigusr1(int)
{
  debug((_("sigusr1")));
  handle_shutdown();
  do_reboot("halt");
}
static void sigusr2(int)
{
  debug((_("sigusr2")));
  handle_shutdown();
  do_reboot("poweroff");
}
static void sigquit(int)
{
  debug((_("sigquit")));
  handle_shutdown();
  do_reboot("exec",_PATH_INIT);
}
static void sigterm(int)
{
  debug((_("sigterm")));
  handle_shutdown();
  sulogin();
  do_reboot("exec",_PATH_INIT);
}
  
static void sigchld(int)
{
  debug((_("sigchld")));
  global_processtab.wait();
}
static void sigtstp(int)
{
  debug((_("sigtstp")));
  global_stop_spawning ^= 1;
  say(global_stop_spawning ? _("spawning disabled") : _("spawning re-enabled"));

  if(!global_stop_spawning)
    global_processtab.respawn();
}

static void sigalrm(int)
{
  debug((_("sigalrm")));
  global_processtab.respawn();
  alarm(respawn_period);
}

static void sighup(int signum)
{
  debug((_("sighup")));
  dump_core(signum);
}


static void init_signals() 
{
  struct sigaction sa;
  sigset_t ss;
  sigfillset(&ss);
  sigprocmask(SIG_BLOCK,&ss,0); /* don't want half initialized handlers going off */
  
  sa.sa_flags = 0;

  /* SIGINT, SIGUSR1, SIGUSR2 and SIGQUIT bring the machine down. If
     they interrupt each other, huge confusion may result. Other
     signals might as well be blocked because they're no use anyway.
  */
  sigemptyset(&sa.sa_mask);
  sigaddset(&sa.sa_mask,SIGCHLD);
  
  for(const sigspec*i=shutdown_sigs; i->handler; i++)
    sigaddset(&sa.sa_mask,i->sig);
  
  for(const sigspec*i=shutdown_sigs; i->handler; i++){
    sa.sa_handler = i->handler;
    sigaction(i->sig,&sa,0);
  }

  sa.sa_flags = SA_NOCLDSTOP;
  sa.sa_handler = sigchld;
  sigaction(SIGCHLD,&sa,0);

  signal(SIGTSTP,sigtstp);
  signal(SIGALRM,sigalrm);
  signal(SIGHUP,sighup);
  
  alarm(respawn_period);
  
  const int error_signals[] =
    {
      SIGFPE,SIGILL,SIGSEGV,SIGBUS,SIGABRT,SIGIOT,SIGSYS,SIGTRAP
    };

  sigfillset(&sa.sa_mask);
  sa.sa_handler = sig_fatal_error;
  for(unsigned i=0;i<(sizeof(error_signals)/sizeof(int));i++)
    sigaction(error_signals[i],&sa,0);

  sigemptyset(&ss);
  for(const int*s=masked_signals;*s!=NSIG;s++)
    sigaddset(&ss,*s);
  
  sigprocmask(SIG_SETMASK,&ss,0);
}



int main(int argc, char **argv)
{
  bool panicked = false;
  
  init_signals();
  init_gettext();
  reboot(RB_DISABLE_CAD);
  
  say(_("initializing"));

  global_inittab.read(_PATH_INITTAB);
  global_inittab.setenv();
  
  for (char**i = argv+1; *i; i++) {
    if (!strcmp(*i, "single")||!strcmp(*i,"emergency")){
      sulogin();
      break;
    }
    if(!strcmp(*i,"panic")){
      panicked = true;
      break;
    }
    
    say(_("unrecognised option \"%s\""),*i);
  }
  
  while(!writer.open()){
    say(_("unable to start IPC service."));
    say(_("is kernel SYSV IPC capable? Do message queues exist? (Try running ipcs -q)"));
    sulogin();
  }

  if(!panicked){ /* if we bombed out last time, then don't try to boot this time as the last
		  session is probably still around */
    say(_("booting"));
    while(!global_inittab.boot()){
      sulogin();
    }
  }

  for(;;)
    writer.handle_connect();

  return 0;
}

