#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <assert.h>
#include <syslog.h>
#include <sched.h>

#include "simpleio.hh"
#include "bootstrap.hh"

#include "pathnames.hh"

#include "config.h"

void clear_signals()
{
  sigset_t sm;
  struct sigaction sa;
  sigfillset (&sm);
  sigprocmask(SIG_BLOCK,&sm,0);
  
  alarm(0);
  
  sa.sa_handler = SIG_DFL;
  sigemptyset(&sa.sa_mask);
  
  for(int sig = 0; sig < NSIG; sig++)
    sigaction(sig,&sa,0);

  sigprocmask(SIG_UNBLOCK,&sm,0);
}

static void ul_to_a(unsigned long ul,char*a)
{
  sprintf(a,"%lx",ul);
}


static void vprintb(int fd, const char*f,va_list args)
{
  char const*p;
  char const*arg;
  
  for(p=f;p&&*p;p++)
    {
      if(*p=='%'){
	p++;
	switch(*p){
	case 's': arg = va_arg (args, char const*);
	  assert(arg);
	  simpleio::write(fd,arg,strlen(arg));
	  break;
	case 'x': 
	  {
	    unsigned long a = va_arg(args,unsigned long);
	    char buffer[sizeof(unsigned long)*2+5];
	    ul_to_a(a,buffer);
	    simpleio::write(fd,buffer,strlen(buffer));
	  }
	  break;
	case '%':
	default:
	  write(fd,p,1);
	  break;
	}
      }
      else
	write(fd,p,1);
    }
}

void fputsb(int fd, const char*s) 
{
  simpleio::write(fd,s,strlen(s));
}

void fprintb(int fd,const char*f,...)
{
  va_list args;
	
  va_start(args,f);
  vprintb(fd,f,args);
  va_end(args);
}

int open_console(int flags)
{
  int fd;

  /* if we're running as a normal user, we can't usefully open
     /dev/console, so try the tty */
  
  if((fd = open(_PATH_TTY, flags))>=0)
    return fd;

  if((fd = open(_PATH_CONSOLE, flags))>=0)
    return fd;

  return 1; /* OK, we've lost. Perhaps stdout is open? We can't do any
	       harm by spewing out invalid syscalls */
}

void panic(const char*f,...) 
{
  int fd;
  va_list args,save;
  va_start(args,f);

#ifdef __va_copy
  __va_copy (save, args);
#else
  save = args;
#endif
  closelog();
  vsyslog(LOG_MAKEPRI(LOG_DAEMON, LOG_ERR),f,save);
  closelog();
  
  fd = open_console(O_WRONLY);

  fputsb(fd,PACKAGE ": PANIC: ");
  vprintb(fd,f,args);
  va_end(args);
  write(fd,"\n",1);

  close(fd);
  abort();
}
      
void say(const char*f,...)
{
  int fd;
  va_list args,save;
  va_start(args,f);

#ifdef __va_copy
  __va_copy (save, args);
#else
  save = args;
#endif
  closelog();
  vsyslog(LOG_MAKEPRI(LOG_DAEMON, LOG_NOTICE),f,save);
  closelog();
  
  fd = open_console(O_WRONLY);

  fputsb(fd,PACKAGE ": ");
  vprintb(fd,f,args);
  va_end(args);
  write(fd,"\n",1);

  close(fd);
}
void*xmalloc(size_t s)
{
  void*m;
  if(!s)return 0;
  m=malloc(s);
  if(!m)say(_("out of memory!"));
  return m;
}
char*xstrdup(const char*s)
{
  char*t;
  if(!s)return 0;
  t=strdup(s);
  if(!t)say(_("out of memory!"));
  return t;
}

void init_terminal()
{
  int fd=open_console(O_RDWR);
  if(fd<0)return;
  dup2(fd,0);//stdin
  dup2(fd,1);//stdout
  dup2(fd,2);//stderr

  for(;fd>2;fd--)
    close(fd);
}

void do_exec(const char*bin,char*const*args)
{
  clear_signals();
  if(setsid()==-1)
      say(_("unable to setsid for \"%s\""),bin);

  init_terminal();
  execvp(bin,args);
  say(_("unable to execvp \"%s\""),bin);
  _exit(1);
}

void yield(int seconds)
{
  time_t start = time(0);
  time_t end = start+seconds;
  
  while(time(0)<end)
    {
      sched_yield();
      sleep(1);
    }
}

void init_gettext()
{
#ifdef ENABLE_NLS
  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);
#endif
}


  
