#ifndef SERVICE_HH
#define SERVICE_HH

#include <string>
#include <list>
#include <set>

#include <sys/wait.h>

#include "bootstrap.hh"
class ServiceNotification;

class Service
{
public:
  enum Status {
    unstarted, starting, finishing, startup_failed, shutdown_failed, successful, finished, destroyed
  };
private:
  Status my_status;
  std::string my_name;
  class ServiceHelper* my_helper;
  typedef std::list<class ServiceNotification*> my_watchers_type;
  typedef std::set<Service*> my_dependants_type;
  my_dependants_type my_dependants;
  typedef my_dependants_type my_dependencies_type;
  my_dependencies_type my_dependencies;
  my_watchers_type my_watchers;

public:
  Service(const std::string&n)
    ;
  void add_dependant(Service*s)
  {
    if(s==this)return;

    if(my_dependants.find(s)==my_dependants.end())
      my_dependants.insert(s);
  }
  void add_dependency(Service*s)
  {
    if(s==this)return;
    
    if(my_dependencies.find(s)==my_dependencies.end())
      my_dependencies.insert(s);
  }
  void del_dependant(Service*s)
  {
    my_dependants_type::iterator i = my_dependants.find(s);
    if(i!=my_dependants.end())
      my_dependants.erase(i);
    else panic(_("no such dependant"));
  }
  void del_dependency(Service*s)
  {
    my_dependencies_type::iterator i = my_dependencies.find(s);
    if(i!=my_dependencies.end())
      my_dependencies.erase(i);
    else panic(_("no such dependency"));
  }
  
  
  const std::string& get_name()const
  {
    return my_name;
  }

  Status get_status()const
  {
    return my_status;
  }
  bool is_successful()const
  {
    return my_status == successful;
  }
  bool is_failed()const
  {
    return my_status == startup_failed || my_status == shutdown_failed;
  }
  bool is_starting()const
  {
    return my_status == starting;
  }
  bool is_stopping()const
  {
    return my_status == finishing || my_status == finished || my_status == destroyed;
  }
  
  bool started()const
  {
    return my_status!=unstarted && my_status!=starting;
  }
  
  typedef std::list<Service*> service_set;

  std::string list_dependants()
  {
    std::string ret;
    for(my_dependants_type::iterator i=my_dependants.begin();i!=my_dependants.end();i++){
      ret+=(*i)->get_name();
      ret+=' ';
    }
    return ret;
  }
  
  /*returns number of dependants stopping */
  unsigned stop_dependants();
  
  void stop(pid_t pid);
  
  void start(pid_t pid)
    ;

  void add_watcher(ServiceNotification*sn)
  {
    do_event(my_watchers.insert(my_watchers.end(),sn));
  }
  
  void helper_finished(Status status)
  {
    my_helper=0;
    new_status(status);
  }
  
  ~Service();

  const char*strstatus()const
  {
    switch(my_status){
    case starting: return _("startup");
    case finishing: return _("shutdown");
    case startup_failed: return _("startup failed");
    case shutdown_failed: return _("shutdown failed");
    case successful: return _("running");
    default: return _("[undefined service status]");
    }
  }

protected:
  
    void new_status(Status status)
      {
	my_status=status;
	for(my_watchers_type::iterator i=my_watchers.begin();i!=my_watchers.end();){
	  my_watchers_type::iterator j = i++;
	  do_event(j);
	}
	if(my_status==finished)
	  delete this;
      }
  void do_event(my_watchers_type::iterator i);
}
;

#endif
