#ifndef RESPAWNER_HH
#define RESPAWNER_HH

#include <string>
#include <vector>

#include <sys/time.h>
#include <utmp.h>
#include <time.h>

#include "RespawnProcess.hh"
#include "ProcessTab.hh"
#include "bootstrap.hh"

extern sig_atomic_t global_stop_spawning;

class Respawner
{
  typedef std::vector<std::string> my_args_type;
  my_args_type my_args;
  std::string my_bin;
  std::string my_id;
  typedef std::list<time_t> my_spawns_type;
  my_spawns_type my_spawns;
  RespawnProcess*my_helper;
  
public:
  Respawner(const std::string&str=""):my_id(str),
				      my_helper(0)
  {
  }
  void set_bin(const std::string&s)
  {
    my_bin=s;
  }
  void add_arg(const std::string&s)
  {
    my_args.push_back(s);
  }
  void clear_name()
  {
    my_bin = "";
    my_args.clear();
  }
  void start()
  {
    debug((_("starting respawn of \"%s\""),my_id.c_str()));
    global_processtab.add_respawner(my_id,this);
    spawn();
  }
  void stop()
  {
    global_processtab.remove_respawner(this);
  }
  
  void spawn()
  {
    if(my_helper)
      return;
    
    if(global_stop_spawning || too_fast(time(0))){
      say(_("respawn of \"%s\" (%s) is paused"),my_id.c_str(),my_bin.c_str());
      return;
    }

    char**args = new char*[my_args.size()+1];
    char**arg = args;
    for(my_args_type::iterator i=my_args.begin();i!=my_args.end();i++)
      *arg++ = (char*)(*i).c_str();
    *arg = 0;
    pid_t p = fork();
    switch(p){
    case -1: say(_("fork failed: could not respawn \"%s\""),my_bin.c_str());
      break;
    case 0:
      do_exec(my_bin.c_str(),args);
    default:
      my_helper = new RespawnProcess(p,this);
      write_utmp_start(p);
      delete args;
    }
  }
  void died(int status)
  {
    my_helper = 0;
    write_utmp_stop(status);
  }
  
  
  ~Respawner()
  {
    debug((_("stopping respawn of \"%s\""),my_id.c_str()));
    free_helper();
  }
  
protected:
  void free_helper()
  {
    if(my_helper){
      delete my_helper;
      my_helper=0;
    }
  }
  
  void write_utmp_start(pid_t pid)
  {
    struct utmp ut;
    init_utmp(ut);
    ut.ut_type = INIT_PROCESS;
    ut.ut_pid = pid;
    write_utmp(ut);
  }
  
  void write_utmp_stop(int status)
  {
    struct utmp ut;
    init_utmp(ut);
    ut.ut_type = DEAD_PROCESS;
    ut.ut_exit.e_exit = WEXITSTATUS(status);
    ut.ut_exit.e_termination = WTERMSIG(status);
    write_utmp(ut);
  }
  
  
  bool too_fast(time_t time)
  {
    const time_t period = 128; //seconds
    const unsigned too_many = 64; //respawns
    unsigned recent = 0;
    for(my_spawns_type::iterator i=my_spawns.begin();i!=my_spawns.end();)
      if((*i)+period > time){
	recent++;
	i++;
      }
      else{
	i=my_spawns.erase(i);
      }
    
    if(recent >= too_many){
      say(_("\"%s\" respawning too fast"),my_bin.c_str());
      return true;
    }
    
    my_spawns.push_back(time);
    return false;
  }

  void write_utmp(struct utmp&ut)
  {
    setutent();
    if(!pututline(&ut))
      say(_("unable to write utmp entry for \"%s\""),my_id.c_str());
    endutent();
  }
  void init_utmp(struct utmp&ut)
  {
    memset(&ut,0,sizeof ut);
    strncpy(ut.ut_id,my_id.c_str(),sizeof ut.ut_id);
    gettimeofday(&ut.ut_tv,0);
  }
}
;


#endif
