#ifndef INITTAB_HH
#define INITTAB_HH

#include <list>
#include <map>
#include <string>

#include <signal.h>
#include <unistd.h>

#include "pathnames.hh"
#include "simpleio.hh"
#include "bootstrap.hh"
#include "ProcessTab.hh"
#include "SyncProcess.hh"
#include "Service.hh"

class Inittab
{
  typedef std::list< std::pair<std::string,std::string> > my_environ_type;
  my_environ_type my_environ;
  typedef std::map<std::string,std::string> my_variables_type;
  my_variables_type my_variables;
  Service*my_booter;
  
public:
  Inittab(const char* filename=0):my_booter(0)
  {
    if(filename)
      read(filename);
  }
  void setenv()const
  {
    for(my_environ_type::const_iterator i=my_environ.begin();i!=my_environ.end();i++)
      ::setenv((*i).first.c_str(),(*i).second.c_str(),1);
  }
  std::string get_sulogin()const
  {
    return get_var("sulogin",_PATH_SULOGIN);
  }
  std::string get_serviceprefix()const
  {
    return get_var("fileprefix","/etc/jinit/");
  }
  std::string get_boot()const
  {
    return get_var("boot","/etc/rc");
  }
  std::string get_shutdown()const
  {
    return get_var("shutdown","/sbin/init-shutdown");
  }
  bool boot()
    /* Originally there was one normal service that was started at
       bootup and so needed no further handling. This approach is more
       compatible to existing schemes however. */
  {
    if(!launch(get_boot(),"")){
      say(_("boot script \"%s\" failed to start"),get_boot().c_str());
      return false;
    }
    return true;
  }
  /* true on success, false failure */
  bool start_service(Service*service)
  {
    pid_t child = do_service(service->get_name(),"start");
    if(child)
      service->start(child);
    return child;
  }
  /* true on success, false failure */
  bool stop_service(Service*s)
  {
    if(!s) return true;
    if(s->is_stopping())
      return true;
    if(!s->is_successful())
      return false;

    unsigned deps;
    const unsigned slack_time = 10;
    unsigned time_left = slack_time;
    unsigned old_deps=0;
    
    for(;;){
      deps = s->stop_dependants();
      if(!deps)
	break;

      {
	PermitSignals ps;
	yield();
      }
      
      if(deps==old_deps){
	if(!--time_left){
	  say(_("timed out waiting for dependants of \"%s\" to stop"),
	      s->get_name().c_str());
	  /* is there a cyclic dependancy? */
	  break;
	}
      } else{
	time_left += slack_time;
	old_deps = deps;
      }
    }
    
    pid_t child = do_service(s->get_name(),"stop");
    if(child)
      s->stop(child);
    return child;
  }
  void read(const char*filename);
  
protected:
  std::string get_var(const std::string&name,const std::string&def = std::string())const
  {
    my_variables_type::const_iterator i = my_variables.find(name);
    if(i==my_variables.end())
      {
	debug((_("the inittab variable \"%s\" was not defined, default \"%s\""),
	       name.c_str(),def.c_str()));
	return def;
      }
    return (*i).second;
  }
  void set_var(const std::string&n,const std::string&v)
  {
    my_variables[n]=v;
  }
  void set_envvar(const std::string&n,const std::string&v)
  {
    my_environ.push_back(std::pair<std::string,std::string>(n,v));
  }

  pid_t do_service(const std::string&service,const char* action)
  {
    std::string bin = get_serviceprefix()+service;
    return launch(bin,action);
  }
  
  pid_t launch(const std::string& bin, const char*arg)
  {
    pid_t child = fork();

    switch(child){
    case -1: say(_("can't fork to launch \"%s\" %s"),bin.c_str(),arg);
      return 0;
    case 0:
      {
	debug((_("launching \"%s\""),bin.c_str()));
	char *const argv[]={(char*)bin.c_str(),(char*)arg,0};
	do_exec(bin.c_str(),argv);
      }
    default:
      return child;
    }
  }
  
}
;

extern Inittab global_inittab;

#endif
    
  
  
