#include <stdio.h>

#include <unistd.h>
#include <sys/mount.h>
#include <sys/reboot.h>
#include <fcntl.h>

#include <list>
#include <string>

#include "bootstrap.hh"
#include "pathnames.hh"
#include "simpleio.hh"
#include "config.h"

const bool testing = false;

/* The reason this code was taken out of init, is that if you upgrade
   your init, then it can't umount the fs it's on, as it has been
   deleted (valid on kernel 2.4). Therefore it execs this program to
   do it */

static void do_reboot(int howto)
{
  reboot(howto);
  while(1)pause();
}

static void killall()
{
  say(_("killing (TERM) all processes"));
  if(!testing && kill(-1,SIGTERM)!=-1){
    say(_("waiting for them to die"));
    yield(5);

    say(_("killing (KILL) all processes dead"));
    if(kill(-1,SIGKILL)!=-1){
      say(_("waiting for them to be cleaned up"));
      yield(5);
    }
  }
}
 
static bool force_umount(const char*mntpnt) {
  if(umount2(mntpnt,MNT_FORCE)==-1){
    return false;
  }
  sync();
  return true;
}

static bool force_mount_readonly(const char*mntpnt)
{
  if(mount(mntpnt,mntpnt,0,MS_MGC_VAL|MS_REMOUNT|MS_RDONLY,0)==-1){
    say(_("error remounting \"%s\" readonly"),mntpnt);
    return false;
  }
  sync();
  return true;
}

static void umount_all_filesystems()
{
  /* don't want to use getmntent as that'll pull in all of stdio
     FIXME: this is no longer valid now
   */
  const char*mtabs[] = 
    {
      "/proc/mounts",
      "/etc/mtab",
      "/etc/fstab",
      0
    }
  ;
  typedef std::list<std::string> mpoints_type;
  mpoints_type mpoints;
  int fd;
  const char**mtab;
  for(mtab=mtabs;*mtab;mtab++){
    fd=open(*mtab,O_RDONLY);
    if(fd!=-1) {
      goto got_mtab;
    }
    say(_("unable to open \"%s\" to list mount points"),*mtab);
  }
  say(_("unable to find a list of filesystems, trying to umount /"));
  if(!force_umount("/"))
    if(!force_mount_readonly("/"))
      say(_("couldn't umount / or remount / readonly"));
  
  return;
 got_mtab:
  say(_("umounting filesystems in \"%s\""),*mtab);

  char*line=0;
  while(simpleio::read_line(fd,line)){
    if(!line || !line[0])break;
    
    char*first_space=strchr(line,' ');
    if(!first_space){
      say(_("unable to interpret mtab line \"%s\" from %s"),line,*mtab);
      break;
    }
    first_space++;
    char*next_space=strchr(first_space,' ');
    if(!next_space){
      say(_("can't find mount point in \"%s\" from %s"),line,*mtab);
      break;
    }
    *next_space=0;
    mpoints.push_back(first_space);
    free(line);line=0;
  }
  if(line)free(line);
  close(fd);

  bool made_progress;

  do{
    made_progress = false;
    
    for(mpoints_type::iterator i=mpoints.begin();i!=mpoints.end();)
      if(force_umount((*i).c_str())){
	i=mpoints.erase(i);
	made_progress = true;
      }
      else i++;
  }while(made_progress);

  bool failing=false;
  
  for(mpoints_type::iterator i=mpoints.begin();i!=mpoints.end();i++){
    say(_("unable to umount \"%s\", trying to remount readonly"),(*i).c_str());
    if(!force_mount_readonly((*i).c_str()))
      failing=true;
  }
  if(failing)
    {
      say(_("some filesystems couldn't be umounted, fix it for next time"));
      sync();
      sleep(10);
    }
}

static void do_shutdown()
{
  killall();
  
  umount_all_filesystems();
  sync(); /* you can never tell whether the hw buffers are flushed */
  yield();
  sync();
  say(_("that's all folks!"));
  sync();
  sleep(2);
  sync();
}

static void try_exec(const char*cmd)
{
    execl(cmd,cmd,0);
    say(_("could not execute \"%s\""),cmd);
}

static void print_usage(const char*pn)
{
  fprintf(stderr,"%s from %s %s\n",pn,PACKAGE,VERSION);
  fputs("This program kills all processes and umounts all filesystems.\n",stderr);
  fputs("The first argument says what should be done afterwards\n",stderr);
  fputs("one of: halt, reboot, powerdown or exec [program-name]\n",stderr);
  fputs("This program should only be called by init.\n",stderr);
}

static int bad_cmdline(const char**argv)
{
  if(getpid()==1){
    say(_("%s called by init with bad command line: \"%s\" \"%s\""),
	argv[0],
	argv[1] ? argv[1] : "",
	argv[1] && argv[2] ? argv[2] : "" );
    do_shutdown();
    say(_("%s was called by init with bad command line!"),argv[0]);
    do_reboot(RB_HALT_SYSTEM);
    return 1;
  }
  else{
    print_usage(argv[0]);
    return 1;
  }
}

int main(int argc, char** argv)
{
  if(argc<2)
    return bad_cmdline((const char**)argv);

  say(_("shuting down, afterwards going to %s %s"),
      argv[1],argv[2] ? argv[2] : "");
  
  do_shutdown();
  
  char*cmd =argv[1];

  if(!strcmp(cmd,"exec")){
    if(!argv[2])
      return bad_cmdline((const char**)argv);
    
    init_terminal();
    try_exec(argv[2]);
    try_exec(_PATH_INIT);
    say(_("could not exec anything!"));
    if(getpid()==1)
      do_reboot(RB_HALT_SYSTEM);
    return 1;
  }

  if(!strcmp(cmd,"halt"))
    do_reboot(RB_HALT_SYSTEM);
  if(!strcmp(cmd,"reboot"))
    do_reboot(RB_AUTOBOOT);
  if(!strcmp(cmd,"poweroff"))
    do_reboot(RB_POWER_OFF);

  return bad_cmdline((const char**)argv);
}

