/*  queue.o								*/
/*  -------								*/
/*  This file contains routines related to the QUEUE information	*/
/*  derived from the dialer daemon.					*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<time.h>
#include	<syslog.h>
#include	<errno.h>

#include	<netdb.h>
#include	<netinet/in.h>

#include	"dialmon.h"

#define	ADDRLEN	32			/* Maximum address length	*/
#define	NAMELEN	128			/* Maximum name length		*/

/*  The "porttab" array is a hash table, hashed on the port number,	*/
/*  used to map port number to service name. Note that PORTSIZ _must_	*/
/*  be a power of two.							*/

typedef	struct	_pmap
{	struct	_pmap	*next	  ;
	int	port		  ;
	char	service[2]	  ;
}	PORTMAP	;
#define	PORTSIZ	128
static	PORTMAP	*porttab[PORTSIZ] ;

/*  The "addrmap" list contains IP address to name mappings, and is	*/
/*  used to provide names rather than IP addresses for the packet queue	*/
/*  display. This should probably be a hash table as well as an LRU	*/
/*  list.								*/

typedef	struct	_nmap
{	struct	_nmap	*next	  ;
	char	addr	[ADDRLEN] ;
	char	name	[NAMELEN] ;
}	ADDRMAP	;
static	ADDRMAP	*addrmap	  ;
static	int	addrcnt		  ;
#define	ACSIZE	128

/*  The "pktq" array contains the current set of QUEUE entries from	*/
/*  the dialer daemon. "npktq" holds the number of valid entries.	*/

typedef	struct	_pktq
{	char	proto[12]	  ;	/* Protocol			*/
	char	saddr[20]	  ;	/* Source address		*/
	char	sport[ 8]	  ;	/* Source port			*/
	char	daddr[20]	  ;	/* Destination address		*/
	char	dport[ 8]	  ;	/* Destination port		*/
	char	ttl  [12]	  ;	/* Time-to-live			*/
}	PKTQ	;
static	int	npktq		  ;
static	PKTQ	pktq [MAXPKTQ]    ;
static	PKTQ   *pktqp[MAXPKTQ]	  ;

/*  Reverse name lookup is handled by a separate process so that we do	*/
/*  not block. IP address requests are written to the process via one	*/
/*  pipe, and replies come back via another. To ensure that writing to	*/
/*  a pipe never blocks, we limit the number of requests (I assume that	*/
/*  a pipe buffers at least 4K?)					*/

#define	MAXNSLQ	16			/* Maximum outstanding requests	*/

static	int	pptonsl[2]	;	/* Parent to name lookup ...	*/
static	int	pnsltop[2]	;	/* ... and back			*/
static	pid_t	nslpid		;	/* Name lookup process ID	*/
static	int	innslq		;	/* Requests outstanding		*/

static	int	inqueue	;	/* Between QUEUE and END QUEUE		*/
static	int	queueat	;	/* Next cycle to send queue information	*/
#define	MAXQMSG	80		/* Maximum length of processed message	*/
#define	QINTV	5		/* Queue transmission interval		*/

/*  token	: Get next token from text				*/
/*  text	: char *	: Text buffer or NULL to continue last	*/
/*  delim	: char *	: Delimiter string			*/
/*  (returns)	: char *	: Next token or NULL if none		*/

static	char	*token
	(	char	*text,
		char	*delim
	)
{
	char	*p = strtok (text, delim) ;
	if (p != NULL) while (isspace (*p)) p += 1 ;
	return	p  ;
}

/*  mapport	: Map port to service					*/
/*  port	: char *	: Port number as text string		*/
/*  (returns)	: char *	: Pointer at service name		*/

static	char	*mapport
	(	char	*port
	)
{
	int	pn	= atoi (port) ;
	PORTMAP	*s 	;

	/* Scan the hash table for the specified port. If this is	*/
	/* found then we can return a pointer at the service entry in	*/
	/* the table.							*/
	for (s = porttab[pn & (PORTSIZ-1)] ; s != NULL ; s = s->next)
		if (s->port == pn)
			return	s->service ;

	/* If the port is not found then simply return the port number	*/
	/* as passed to us.						*/
	return	port ;
}

/*  mapaddr	: Map address to name					*/
/*  addr	: char *	: IP address				*/
/*  (returns)	: char *	: Name					*/

static	char	*mapaddr
	(	char	*addr
	)
{
	char	abuf[ADDRLEN]	;
	ADDRMAP	*ap, **lvap	;

	/* Scan the existing list of known names. If the address is	*/
	/* found there then return the associated name, and move the	*/
	/* entry to the front of the queue.				*/
	for (lvap = &addrmap ; (*lvap) != NULL ; lvap = &(*lvap)->next)
		if (strcmp (addr, (*lvap)->addr) == 0)
		{	ap	 = *lvap    ;
			*lvap	 = ap->next ;
			ap->next = addrmap  ;
			addrmap	 = ap	    ;
			return	 ap->name   ;
		}

	/* If the lookup queue is not full then build a new entry and	*/
	/* push it onto the queue, with the name initially set to the	*/
	/* address. The address is passed to the lookup process.	*/
	if (innslq < MAXNSLQ)
	{
		ap	 = (ADDRMAP *)malloc(sizeof(ADDRMAP)) ;
		ap->next = addrmap ;
		addrmap	 = ap	   ;
		addrcnt	+= 1	   ;

		copystr	(ap->addr,   addr, ADDRLEN) ;
		copystr	(ap->name,   addr, NAMELEN) ;

		copystr	(abuf,       addr, ADDRLEN) ;
		write	(pptonsl[1], abuf, ADDRLEN) ;
		innslq	+= 1 ;

		/* See if the queue has got too big in which case drop	*/
		/* off the oldest entry.				*/
		if (addrcnt > ACSIZE)
		{
			for (ap = addrmap ; ap->next != NULL ; ap = ap->next) ;
			free	(ap->next) ;
			ap->next  = NULL   ;
			addrcnt  -= 1	   ;
		}
	}

	return	addr	;
}

/*  procqmsg	: Process a QUEUE-type message				*/
/*  msg		: char *	: QUEUE message				*/
/*  (returns)	: void		:					*/

static	void	procqmsg
	(	char	*msg
	)
{
	int	slot	;
	PKTQ	*pqp	= NULL ;

	char	*proto	;
	char	*saddr	;
	char	*sport	;
	char	*daddr	;
	char	*dport	;
	char	*ttl	;

	if ((proto = token (msg,  " \t\n")) == NULL) return ;
	if ((saddr = token (NULL, "/"    )) == NULL) return ;
	if ((sport = token (NULL, " \t\n")) == NULL) return ;
	if ((daddr = token (NULL, "/"    )) == NULL) return ;
	if ((dport = token (NULL, " \t\n")) == NULL) return ;
	if ((ttl   = token (NULL, " \t\n")) == NULL) return ;

	/* If queue packing is enabled then collapse all entries to the	*/
	/* same destination (address:port). Note that Group packing	*/
	/* merges all non-privileged ports; this is useful with the WWW	*/
	/* as a single page may generate multiple requests to the same	*/
	/* server in quick succession (eg., for embedded images).	*/
	if (packq == Group)
	{
		if (atol(sport) > 1023) sport[0] = 0 ;
		if (atol(dport) > 1023) dport[0] = 0 ;
	}

	if (packq != None)
		for (slot = 0, pqp = NULL ; slot < npktq ; slot += 1)
			if ( (strcmp (pktq[slot].daddr, daddr) == 0) &&
			     (strcmp (pktq[slot].dport, dport) == 0) )
		{	pqp	= &pktq[slot] ;
			break	;
		}

	/* If an extant entry was found then we return, except that the	*/
	/* TTL is updated if it is less than that in the latest diald	*/
	/* message.							*/
	if (pqp != NULL)
	{	if (strcmp (pqp->ttl, ttl) < 0) 
			copystr	(pqp->ttl, ttl, sizeof(pqp->ttl)) ;
		return ;
	}

	/* Impose a limit on the number of entries. We ought to check	*/
	/* in case we have a larger TTL than the smallest in the list,	*/
	/* but since we pack the queue by default, the MAXPKTQ value	*/
	/* should be enough.						*/
	if (npktq >= MAXPKTQ)
		return ;

	/* Copy the information from the fragmented diald message into	*/
	/* the next free entry.						*/
	pqp	= &pktq[npktq] ;
	npktq  += 1 ;

	copystr	(pqp->proto, proto, sizeof(pqp->proto)) ;
	copystr	(pqp->saddr, saddr, sizeof(pqp->saddr)) ;
	copystr	(pqp->sport, sport, sizeof(pqp->sport)) ;
	copystr	(pqp->daddr, daddr, sizeof(pqp->daddr)) ;
	copystr	(pqp->dport, dport, sizeof(pqp->dport)) ;
	copystr	(pqp->ttl,   ttl,   sizeof(pqp->ttl  )) ;

	/* We also try to map the source and destination addresses to	*/
	/* names, even though we ignore the result. This will hopefully	*/
	/* get the mappings into the local cache before we send the	*/
	/* queue entries to the clients.				*/
	mapaddr	(pqp->saddr) ;
	mapaddr	(pqp->daddr) ;
}

/*  sortfn	: Queue ordering function				*/
/*  p1		: PKTQ **	:					*/
/*  p2		: PKTQ **	:					*/
/*  (returns)	: ini		: Ordering				*/

static	int	sortfn
	(	const	void	*p1,
		const	void	*p2
	)
{
	return	strcmp	((*(PKTQ **)p2)->ttl, (*(PKTQ **)p1)->ttl) ;
}

/*  queue_rtn	: Handler for QUEUE messages				*/
/*  msg		: char *	: Message in question			*/
/*  (returns)	: void *	: Handler for next message		*/

global	void	*queue_rtn
	(	char	*msg
	)
{
	int	idx	;

	/* The 'inqueue' variable brackets the QUEUE ... END QUEUE	*/
	/* messages from diald. Each time we enter such a set, we	*/
	/* decrement 'queueat' and reset our copy of the queue when it	*/
	/* falls to zero.						*/
	if (!inqueue)
	{	inqueue	= 1 ;
		if ((queueat -= 1) <= 0) npktq	= 0 ;
	}

	/* On receipt of END QUEUE clear the 'inqueue' flag. Then, if	*/
	/* 'queueat' is zero, send the queue to the clients and reset	*/
	/* 'queueat'. The effect is that our queue is sent every QINTV	*/
	/* cycles of QUEUE ... END QUEUE.				*/
	if (strcmp (msg, "END QUEUE") == 0)
	{
		inqueue	= 0 ;

		if (queueat <= 0)
		{
			/* The queue is sorted. Since each item is	*/
			/* quite large, initialise a list of pointers	*/
			/* first.					*/
			for (idx = 0 ; idx < npktq ; idx += 1)
				pktqp[idx] = &pktq[idx] ;
			qsort	  (pktqp, npktq, sizeof(PKTQ *), sortfn) ;
			sendqueue ()    ;
			queueat = QINTV ;
		}

		return	dont_care ;
	}

	/* If 'queueat' is zero then process the queue message, if not	*/
	/* then ignore it. Hence, with the above code, we only process	*/
	/* every QINTV'th block of QUEUE ... END QUEUE messages.	*/
	if (queueat == 0) procqmsg (msg) ;
	return	queue_rtn ;
}

/*  sendqueue	: Send current queue information to all clients		*/
/*  (returns)	: void		:					*/

global	void	sendqueue ()
{
	PKTQ	*pqp		;
	int	idx		;
	char	pqmsg[512]	;

	/* The queue is send if either we are outside of QUEUE ... END	*/
	/* QUEUE, or 'queueat' is greater than zero; this condition is	*/
	/* true of our queue is not being updated at the moment.	*/
	if (!inqueue || (queueat > 0))
	{
		forward	(NULL, "QUEUE", "START", F_QUEUE) ;

		for (idx = 0 ; idx < npktq ; idx += 1)
		{
			pqp	= pktqp[idx] ;

			sprintf	(pqmsg, "%s|%s/%s|%s/%s|%s",
				        pqp->proto,
				        mapaddr (pqp->saddr),
				        mapport (pqp->sport),
				        mapaddr (pqp->daddr),
				        mapport (pqp->dport),
		      		        pqp->ttl ) ;
			forward (NULL, "QUEUE", pqmsg, F_QUEUE) ;
		}

		forward	(NULL, "QUEUE", "END", F_QUEUE) ;
	}
}

/*  loadserv	: Load /etc/service information				*/
/*  (returns)	: void		:					*/

global	void	loadserv ()
{
	struct	servent	    *service	;

	setservent (1) ;

	while ((service = getservent ()) != NULL)
	{
		PORTMAP	*s = (PORTMAP *)malloc(sizeof(PORTMAP)+strlen(service->s_name)) ;
		int	h  = htons(service->s_port) & (PORTSIZ-1) ;
		s->next	   = porttab[h] ;
		porttab[h] = s		;
		s->port	   = ntohs(service->s_port)   ;
		strcpy	(s->service, service->s_name) ;
	}

	endservent ()  ;
}

/*  nslookup	: Name lookup routine					*/
/*  (returns)	: void		:					*/

static	void	nslook ()
{
	char	abuf[ADDRLEN	    ]	;
	char	rbuf[ADDRLEN+NAMELEN]	;
	int	a1, a2, a3, a4		;
	long	addr			;
	struct	hostent *host		;

	/* Loop reading addresses from the parent. If and when this	*/
	/* fails, exit the process.					*/
	while (read (pptonsl[0], abuf, ADDRLEN) == ADDRLEN)
	{
		/* Convert the textual name to a net address and look	*/
		/* it up.						*/
		sscanf	(abuf, "%d.%d.%d.%d", &a1, &a2, &a3, &a4) ;
		addr	= htonl((a1<<24)|(a2<<16)|(a3<<8)|a4) ;
		host	= gethostbyaddr ((char *)&addr, sizeof(addr), AF_INET) ;

		/* Write back the result, comprising the address and	*/
		/* either then name, or an empty string if it was not	*/
		/* found.						*/
		copystr	(&rbuf[0], abuf, ADDRLEN) ; 
		if (host == NULL)
			copystr	(&rbuf[ADDRLEN], "",	       NAMELEN) ;
		else	copystr	(&rbuf[ADDRLEN], host->h_name, NAMELEN) ;

		if (write (pnsltop[1], rbuf, ADDRLEN+NAMELEN) != ADDRLEN+NAMELEN)
		{	logout	(LOG_ERR, "nsl->parent write: %s", STRERR) ;
			break ;
		}
	}

	exit	(0) ;
}

/*  startnsl	: Start name lookup process				*/
/*  argc	: int		: Parent argument count			*/
/*  argv	: char **	: Parent argument vector		*/
/*  (returns)	: void		:					*/

global	void	startnsl
	(	int	argc,
		char	**argv
	)
{
	int	idx	  ;

	pipe	(pptonsl) ;	/* Parent to lookup pipe ...		*/
	pipe	(pnsltop) ;	/* ... and lookup to parent pipe	*/

	switch (nslpid = fork ())
	{
		case -1 :
			return	;

		case  0 :
			/* Close all descriptors except stdin/out/err	*/
			/* and the pipes; the syslog connection is	*/
			/* closed and reopened around this.		*/
			closelog () ;
			for (idx = 3 ; idx < 64 ; idx += 1)
				if ((idx != pptonsl[0]) && (idx != pnsltop[1]))
					close	(idx) ;
			openlog	 ("dialnsl", LOG_PID|LOG_NDELAY, LOG_LOCAL2) ;

			/* Zap the argument vector so that the forked	*/
			/* process gets a suggestive apparent command	*/
			/* line.					*/
			memset	 (argv[0],
				  0,
				  &argv[argc-1][strlen(argv[argc-1])] - &argv[0][0]) ;
			strcpy	 (argv[0], "dialnsl") ;

			nslook	()  ;
			exit	(0) ;

		default	:
			break	;
	}

	close	  (pptonsl[0]) ;
	close	  (pnsltop[1]) ;
	fdmaskset (pnsltop[0], 0) ;
}

/*  checknsl	: See if there is any name information			*/
/*  in		: fd_set *	: Input file descriptor mask		*/
/*  (returns)	: void		:					*/

global	void	checknsl
	(	fd_set	*in
	)
{
	char	rbuf[ADDRLEN+NAMELEN]	;
	ADDRMAP	*ap			;

	if (FD_ISSET(pnsltop[0], in))
	{
		/* If there appears to be input from the lookup process	*/
		/* then read the next result; in the event of an error	*/
		/* just close the connection.				*/
		if (read (pnsltop[0], rbuf, ADDRLEN+NAMELEN) != ADDRLEN+NAMELEN)
		{	logout	  (LOG_ERR, "nsl->parent read: %s", STRERR) ;
			close	  (pptonsl[1]) ;
			close     (pnsltop[0]) ;
			fdmaskclr (pnsltop[0]) ;
			return ;
		}

		/* Scan the current address map. Provided that the	*/
		/* entry is still there (which it really should be) and	*/
		/* that the name was found, the entry can be updated.	*/
		for (ap = addrmap ; ap != NULL ; ap = ap->next)
			if (strcmp (ap->addr, &rbuf[0]) == 0)
				break	;

		if ((ap != NULL) && (rbuf[ADDRLEN] != 0))
			copystr	(ap->name, &rbuf[ADDRLEN], NAMELEN) ;

		innslq	-= 1 ;
	}
}

/*  clrqueue	: Clear the packet queue				*/
/*  (returns)	: void		:					*/

global	void	clrqueue ()
{
	npktq	= 0 ;
}

/*  qtidy	: Tidy up queue information				*/
/*  (returns)	: void		:					*/

global	void	qtidy ()
{
	npktq	= 0 ;		/* No entries in packet queue		*/
	inqueue	= 0 ;		/* Outside QUEUE ... END QUEUE		*/
	queueat	= 0 ;		/* Will force send on first call	*/
}
