/*  logging.c								*/
/*  ---------								*/
/*  This file contains routines related to logging connection		*/
/*  inforamation, and passing it on to clients.				*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<syslog.h>
#include	<time.h>

#include	"dialmon.h"

#define		SECSPDAY	(24*3600)

/*  copylog	: Copy a log file					*/
/*  srce	: char *	: Source file name			*/
/*  dest	: char *	: Destination file name			*/
/*  skip	: int		: Number of leading lines to skip	*/
/*  after	: long		: Earliest time for output		*/
/*  append	: char *	: Possible line to append		*/
/*  lvnl	: int *		: For line count			*/
/*  (returns)	: int		: Non-zero on success			*/

static	int	copylog
	(	char	*srce,
		char	*dest,
		int	skip,
		long	after,
		char	*append,
		int	*lvnl
	)
{
	FILE	*fdin	;
	FILE	*fdout	;
	int	nlines	= 0   ;
	char	buff	[128] ;

	/* If the source file does not appear to exist then this is	*/
	/* probably the first time we have run, so use /dev/null	*/
	/* instead. This makes the routine do everything correctly, if	*/
	/* rather inefficiently for this case.				*/
	if (access (srce, F_OK) != 0)
		srce	= "/dev/null" ;

	if ((fdin  = fopen (srce, "r")) == NULL)
	{	syslog	(LOG_ERR, "Cannot open %s: %m", srce) ;
		return	0 ;
	}

	if ((fdout = fopen (dest, "w")) == NULL)
	{	syslog	(LOG_ERR, "Cannot open %s: %m", dest) ;
		fclose	(fdin) ;
		return	0 ;
	}

	/* Loop copying lines, except that lines which are both within	*/
	/* the range to be skipped and are before the cutoff time will	*/
	/* be dropped. We also note the time of the first output line.	*/
	while (fgets (buff, sizeof(buff), fdin) != NULL)
	{
		if ((skip > 0) && (atol (buff) <= after))
			continue ;

		fputs	(buff, fdout) ;
		nlines	+= 1 ;
		skip	-= 1 ;
	}

	if (append != NULL) fputs (append, fdout) ;

	fclose	(fdin ) ;
	fclose	(fdout) ;
	*lvnl	= nlines;
	return	1	;
}

/*  addtolog	: Add a connection entry to the log file		*/
/*  stamp	: long		: Entry time stamp			*/
/*  duration	: long		: Connection duration in seconds	*/
/*  conf	: char *	: Configuration				*/
/*  tx		: int		: Total bytes transmitted		*/
/*  rx		: int		: Total bytes received			*/
/*  (returns)	: void		:					*/

void	addtolog
	(	long	stamp,
		long	duration,
		char	*conf,
		int	tx,
		int	rx
	)
{
	char	logf[256]	;
	char	temp[256]	;
	char	line[128]	;
	int	nlines		;
	int	skip		;
	long	after		;

	sprintf	(logf, "%s",     LOGFILE) ;
	sprintf	(temp, "%s.tmp", LOGFILE) ;

	/* Make a copy of the existing log file. In the process we	*/
	/* count the number of lines in the file.			*/
	if (!copylog (logf, temp, 0, 0, NULL, &nlines))
		return	;

	/* If a maximum line count is specified and the log file will	*/
	/* exceed this then arrange to skip the approprate number of	*/
	/* lines at the beginning of the file.				*/
	if ((definfo.logsize > 0) && (nlines + 1 > definfo.logsize))
		skip	= nlines + 1  - definfo.logsize ;
	else	skip	= 0 ;

	/* If an earliest log time is specified then calculate the time	*/
	/* before which log entries should be discarded.		*/
	if (definfo.logtime > 0)
		after	= time (NULL) - definfo.logtime * SECSPDAY ;
	else	after	= 0 ;

	/* Generate the line to be appended to the log file and then	*/
	/* copy back the log file, chopping the front and appending the	*/
	/* new entry. Note that a null configuration is converted to	*/
	/* "Unknown" so that clients can simply use strtok to split up	*/
	/* lines.							*/
	sprintf	(line, "%ld|%ld|%s|%d|%d\n",
			stamp,
			duration,
			((conf == NULL) || (conf[0] == 0)) ? "Unknown" : conf,
			tx, rx) ;
	copylog	(temp, logf, skip, after, line, &nlines) ;
	unlink	(temp) ;
}

/*  sendlog	: Send log information to specified client connection	*/
/*  fd		: int		: Connection to client			*/
/*  (returns)	: void		:					*/

void	sendlog
	(	int	fd
	)
{
	FILE	*logfd	  ;
	char	buff[128] ;

	if ((logfd  = fopen (LOGFILE, "r")) == NULL)
	{	syslog	(LOG_ERR, "Cannot open %s: %m", LOGFILE) ;
		return	;
	}

	write	(fd, "LOG START\n", 10) ;
	while (fgets (buff, sizeof(buff), logfd))
	{	write	(fd, "LOG ", 4) ;
		write	(fd, buff, strlen(buff)) ;
	}
	write	(fd, "LOG END\n",    8) ; 
	fclose	(logfd) ;
}
