/*  DialMon :	A diald monitor daemon for Linux			*/
/*		Mike Richardson (mike@quaking.demon.co.uk) 1997		*/
/*									*/
/*  History :	Apr-97	First version, including Win31/Win95 client.	*/
/*		Apr-97	Better information to clients.			*/
/*		May-97	Improved configuration, plus tcl/tk client.	*/
/*		Feb-98  Jem: Forward LOAD messages to clients		*/
/*		Mar-98	Shadow passwords (Matt Kressel matty@inch.com)	*/
/*		Mar-98	DialD dead/alive and restart			*/
/*		Aug-98	Relesed 0.4					*/
/*		Oct-98	Packet queue extensions and multiple files	*/
/*		Dec-98	Connection log					*/
/*		Feb-99	PAM Authentication <slyglif@magerealm.com>	*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<strings.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<fcntl.h>
#include	<sys/types.h>
#include	<sys/time.h>
#include	<sys/stat.h>
#include	<sys/wait.h>
#include	<syslog.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<netdb.h>
#include	<signal.h>
#include	<pwd.h>
#include	<errno.h>

#include	"dialmon.h"

global	char	laststate[256]	    ;	/* Last state from diald	*/
global	char	lastmsg  [256]	    ;	/* Last message from diald	*/
global	char	lastload [256]	    ;	/* Last load from diald		*/
global	char	lastiface[256]	    ;	/* Last interface message	*/
static	int	tx_total	    ;	/* Total bytes transmitted in K	*/
static	int	rx_total	    ;	/* Total bytes received in K	*/
static	int	tx_last		    ;	/* Bytes transmitted last second*/
static	int	rx_last		    ;	/* Bytes received last second	*/
static	int	contime		    ;	/* Time of last CONNECT status	*/
static	int	state_up	    ;	/* 1 if link (state) is "up"	*/
global	int	bwidth	 = DEFBW    ;	/* Link bandwidth bytes/sec	*/
global	int	extcon	 	    ;	/* External connection present	*/

global	int	forcecnt = 0	    ;	/* Force forever count		*/
static	int	debug	 = 0	    ;	/* Debug (foreground) mode	*/
global	char	*dmcfile = DMCFILE  ;	/* Configuration file name	*/
global	char	*dmufile = DMUFILE  ;	/* User file name		*/

global	DMode	dmode	 = Normal   ;	/* Initially normal operation	*/
global	DMode	amode	 = Normal   ;	/* Initially normal operation	*/
global	AuthTyp	authtyp	 = SAuto    ;	/* Automatic shadow detection	*/
global	Packing	packq	 = Simple   ;	/* Simple queue packing enabled	*/

static	int	connfd	 	    ;	/* Listening socket		*/
static	fd_set	currcon	 	    ;	/* Current connections		*/
static	fd_set	currcl	 	    ;	/* Current client connections	*/

static	char	if_iface [32]	    ;	/* Interface name		*/
static	char	if_local [24]	    ;	/* Local IP address		*/
static	char	if_remote[24]	    ;	/* Remote IP address		*/


/*  errexit	: Error logging						*/
/*  fmt		: char *	: Message format			*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: void		: Never returns				*/
	
global	void	errexit
	(	char	*fmt,
		...
	)
{
	va_list	  aptr 		;
	char	  buff[256]	;

	va_start (aptr, fmt) 	;
	vsprintf (buff, fmt, aptr)   ;

	if (debug)
		fprintf	(stderr, "dialmon: %s\n", buff) ;
	else	syslog	(LOG_ERR, buff) ; 

	unlink	 (DMPFILE) ;
	exit	 (1) ;
}

/*  logout	: Information logging					*/
/*  what	: int		: Type argument for syslog		*/
/*  fmt		: char *	: Message format			*/
/*  ...		: ...		: Arguments				*/
/*  (returns)	: void		: Never returns				*/
	
global	void	logout
	(	int	what,
		char	*fmt,
		...
	)
{
	va_list	  aptr 		;
	char	  buff[256]	;

	va_start (aptr, fmt) 	;
	vsprintf (buff, fmt, aptr)   ;

	if (debug)
		fprintf	(stderr, "dialmon: %s\n", buff) ;
	else	syslog	(what, "%s", buff) ; 

}

/*  copystr	: Like strncpy but ensures termination			*/
/*  dest	: char *	: Destination				*/
/*  srce	: char *	: Source				*/
/*  dlen	: int		: Destination length			*/
/*  (returns)	: void		:					*/

global	void	copystr
	(	char	*dest,
		char	*srce,
		int	dlen
	)
{
	strncpy	(dest, srce, dlen) ;
	dest[dlen-1] = 0	   ;
}

/*  sigexit	: Exit signal handler routine				*/
/*  sig		: int		: Signal caught				*/
/*  (returns)	: void		: Never returns				*/

static	void	sigexit
	(	int	sig
	)
{
	killall () ;
	errexit	("exiting for signal") ;
}


/*  tidyup	: Tidy up diald connection stuff			*/
/*  (returns)	: void		:					*/

global	void	tidyup ()
{
	qtidy	() ;
	dtidy	() ;
	ctidy	() ;
	dmode	= Normal ;
}

/*  forward	: Forward a message to the connected clients		*/
/*  buf		: char *	: Buffer for formatted message		*/
/*  msgtype	: char *	: Message type				*/
/*  msgparm	: char *	: Message parameters			*/
/*  flags	: int		: Request flags				*/
/*  (returns)	: void		:					*/

void	forward
	(	char	*buf,
		char	*msgtype,
		char	*msgparm,
		int	flags
	)
{
	int	s		;
	int	l		;
	char	_buf	[256]	;

	if (buf == NULL) buf = &_buf[0] ;

	strcpy	(buf, msgtype) ;
	if (msgparm != NULL)
	{	strcat	(buf, " "    ) ;
		strcat	(buf, msgparm) ;
	}
	strcat	(buf, "\n")    ;
	l = strlen (buf)       ;

	/* The information is sent to each connected client that has	*/
	/* requested the particular type of information.		*/
	for (s = 0 ; s < MAXFD ; s += 1)
		if (FD_ISSET(s, &currcl) && clientreq (s, flags))
			write (s, buf, l) ;
}

/*  sendmode	: Send current operation mode				*/
/*  (returns)	: void		:					*/

global	void	sendmode ()
{
	/* Should send mode based on diald "status" but this does not	*/
	/* seem to work at present, so use our own idea ...		*/
	switch (dmode /*amode*/)
	{	case Normal : forward (NULL, "MODE", "NORMAL", 0) ; break ;
		case Force  : forward (NULL, "MODE", "FORCE",  0) ; break ;
		case Block  : forward (NULL, "MODE", "BLOCK",  0) ; break ;
		default	    : forward (NULL, "MODE", "ERROR",  0) ; break ;
	}
}

/*  logconn	: Generate connection log entry				*/
/*  (returns)	: void		:					*/

global	void	logconn ()
{
	if (contime != 0)
	{	addtolog  (contime, time (NULL) - contime,
				    ddconf == NULL ? "" : ddconf,
				    tx_total,
				    rx_total) ;
		contime	= 0 ;
	}
}

/*  state_rtn	: Handler for STATE message				*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*state_rtn
	(	char	*msg
	)
{
	/* Clear xmit/recv counts and the packet queue if the link goes	*/
	/* up or down.							*/
	if	(strcmp (msg,   "UP") == 0)
	{	tx_total = rx_total = 0 ;
		state_up = 1 ;
		sendqueue () ;
	}
	else if (strcmp (msg, "DOWN") == 0)
	{	tx_total = rx_total = 0 ;
		state_up = 0 ;
		clrqueue  () ;
		sendqueue () ;
	}
	else
	{
		if	(strcmp (msg, "CONNECT") == 0)
			contime	= time (NULL) ;
		else if	(strcmp (msg,   "CLOSE") == 0)
			logconn	() ;
	
		state_up = 0 ; 
	}

	forward (laststate, "STATE", msg, 0)  ;
	return	dont_care ;
}

/*  iface3_rtn	: Handler 3 for INTERFACE message			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*iface3_rtn
	(	char	*msg
	)
{
	char	text[256] ;
	copystr	(if_remote, msg, sizeof(if_remote)) ;
	sprintf	(text, "%s %s %s", if_iface, if_local, if_remote) ;
	forward	(lastiface, "INTERFACE", text, 0)   ;
	return	dont_care ;
}

/*  iface2_rtn	: Handler 2 for INTERFACE message			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*iface2_rtn
	(	char	*msg
	)
{
	copystr	(if_local, msg, sizeof(if_local)) ;
	return	iface3_rtn ;
}

/*  iface1_rtn	: Handler 1 for INTERFACE message			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*iface1_rtn
	(	char	*msg
	)
{
	copystr	(if_iface, msg, sizeof(if_iface)) ;
	return	iface2_rtn ;
}

/*  message_rtn	: Handler for MESSAGE message				*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*message_rtn
	(	char	*msg
	)
{
	forward	(lastmsg, "MESSAGE", msg, 0) ;
	return	dont_care ;
}

/*  load2_rtn	: Handler for second LOAD message			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*load2_rtn
	(	char	*msg
	)
{
	char	parms[64] ;

	/* The second parameter is the reception count. The transmit	*/
	/* count will have been set by the "load1_rtn" function.	*/
	rx_last    = atoi (msg)		;
	rx_total  += rx_last		;

	/* Send LOAD message to all clients if the link is up ...	*/
	if (state_up)
	{	sprintf	(parms, "%d %d %d %d",
			 tx_last, rx_last, tx_total, rx_total) ;

		forward	(lastload, "LOAD", parms, F_LOAD);
	}

	return	dont_care ;
}

/*  load1_rtn	: Handler for first LOAD message			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*load1_rtn
	(	char	*msg
	)
{
	/* The transmit count is the first parameter, The reception	*/
	/* count will follow in the second message, which will be	*/
	/* handled by "load2_rtn".					*/
	tx_last	    = atoi (msg)	;
	tx_total   += tx_last		;

	return	   load2_rtn		;
}

/*  status9_rtn	: Handle the STATUS message line 9			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status9_rtn
	(	char	*msg
	)
{
	/* Connection timeout ...					*/
	return	dont_care ;
}

/*  status8_rtn	: Handle the STATUS message line 8			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status8_rtn
	(	char	*msg
	)
{
	/* Forcing timeout ...						*/
	return	status9_rtn ;
}

/*  status7_rtn	: Handle the STATUS message line 7			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status7_rtn
	(	char	*msg
	)
{
	/* Impluse timeout ...						*/
	return	status8_rtn ;
}

/*  status6_rtn	: Handle the STATUS message line 6			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status6_rtn
	(	char	*msg
	)
{
	/* Impluse fuzz ...						*/
	return	status7_rtn ;
}

/*  status5_rtn	: Handle the STATUS message line 5			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status5_rtn
	(	char	*msg
	)
{
	/* Impluse secondary timer ...					*/
	return	status6_rtn ;
}

/*  status4_rtn	: Handle the STATUS message line 4			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status4_rtn
	(	char	*msg
	)
{
	/* Impluse primary timer ...					*/
	return	status5_rtn ;
}

/*  status3_rtn	: Handle the STATUS message line 3			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status3_rtn
	(	char	*msg
	)
{
	/* Impluse mode ...						*/
	return	status4_rtn ;
}

/*  status2_rtn	: Handle the STATUS message line 2			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status2_rtn
	(	char	*msg
	)
{
	DMode	nmode	;

	switch (atoi (msg))
	{	case 0  : nmode = Normal ; break ;
		case 1  : nmode = Force  ; break ;
		case 2  : nmode = Block  ; break ;
		default : break ;
	}

	if (amode != nmode)
	{	amode	= nmode	;
		sendmode ()	;
	}

	return	 status3_rtn ;
}

/*  status1_rtn	: Handle the STATUS message line 1			*/
/*  msg		: char *	: Message				*/
/*  (returns)	: void *	: Next state function			*/

static	void	*status1_rtn
	(	char	*msg
	)
{
	/* This is diald's idea of whether the link should be up or	*/
	/* down .... start a scan through the following lines ...	*/
	return	status2_rtn ;
}

/*  dont_care	: Main diald message handler				*/
/*  msg		: char *	: Diald message				*/
/*  (returns)	: void *	: Next state function			*/

global	void	*dont_care
	(	char	*msg
	)
{
	/* Ignore any diald messages we don't care about. Prepare to	*/
	/* handle any forthcoming messages we do care about.		*/
	if	(strcmp (msg, "STATE"    ) == 0) return state_rtn   ;
	else if (strcmp (msg, "MESSAGE"  ) == 0) return message_rtn ;
	else if (strcmp (msg, "LOAD"     ) == 0) return load1_rtn   ;
	else if (strcmp (msg, "QUEUE"    ) == 0) return queue_rtn   ;
	else if (strcmp (msg, "INTERFACE") == 0) return iface1_rtn  ;
	else if (strcmp (msg, "STATUS"	 ) == 0) return status1_rtn ;

	return	dont_care ;
}

/*  sigpipe	: SIGPIPE handler					*/
/*  sig		: int		: Signal				*/
/*  (returns)	: void		:					*/

static	void	sigpipe
	(	int	sig
	)
{
	signal	(SIGPIPE, sigpipe) ;
}

/*  fdmaskset	: Add descriptor to select mask				*/
/*  fd		: int		: Descriptor				*/
/*  client	: int		: Non-zero for client descriptors	*/
/*  (returns)	: void		:					*/

global	void	fdmaskset
	(	int	fd,
		int	client
	)
{
	FD_SET (fd, &currcon) ;
	if (client) FD_SET (fd, &currcl) ;
}

/*  fdmaskclr	: CLear descriptor from select mask			*/
/*  fd		: int		: Descriptor				*/
/*  (returns)	: void		:					*/

global	void	fdmaskclr
	(	int	fd
	)
{
	FD_CLR (fd, &currcon) ;
	FD_CLR (fd, &currcl ) ;
}


/*  dialmon	: DialMon main routine					*/
/*  (returns)	: void		: Never returns				*/

static	void	dialmon ()
{
	struct	sockaddr_in connaddr	;
	struct	servent	    *service	;
	int		    s_port	;

	int		    on		;
	FILE		    *fd		;

	openlog	("dialmon", LOG_PID|LOG_NDELAY, LOG_LOCAL2) ;

	/* First set up a socket on which to listen. By default we use	*/
	/* the port "dialmon"; the socket is a stream socket in the	*/
	/* internet domain, and accepts calls from anywhere. Make it	*/
	/* reusable so we can bounce up and down.			*/
	if (isdigit (definfo.port[0]))
		s_port = ntohs(atol(definfo.port)) ;
	else	if ((service = getservbyname (definfo.port, "tcp")) == 0)
			errexit	("can't find service entry for `%s'", "dialmon") ;
		else	s_port = service->s_port ;

	if ((connfd = socket (AF_INET, SOCK_STREAM, 0)) < 0)
		errexit	("can't create internet domain socket: %s", STRERR)  ;

	on = 1 ;
	if (setsockopt (connfd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) < 0)
		errexit	("cannot set SO_REUSEADDR: %s", STRERR)  ;

	memset	(&connaddr, 0, sizeof(connaddr));
	connaddr.sin_family	 = AF_INET	;
	connaddr.sin_port	 = s_port	;
	connaddr.sin_addr.s_addr = INADDR_ANY	;

	if (bind (connfd, (struct sockaddr *)&connaddr, sizeof(connaddr)) < 0)
		errexit	("can't bind socket to %s: %s", "localhost", STRERR)  ;

	if (listen (connfd, SOMAXCONN) < 0)
		errexit	("cannot listen: %s", STRERR)  ;

	/* If no initial dialer daemon configuration is specified then	*/
	/* we reply on it already running, so initialise communications	*/
	/* with the dialer daemon, and set the current dialer daemon	*/
	/* status.							*/
	if (ddconf == NULL)
		forward (NULL, ddinit () ? DDALIVE : DDDEAD, NULL, 0) ;
	else	forward (NULL, DDDEAD, NULL, 0) ;

	/* Record our process ID in the pid file. However, don't bother	*/
	/* too much whether it works or not.				*/
	if ((fd = fopen (DMPFILE, "w")) != NULL)
	{	fprintf	(fd, "%d\n", getpid()) ;
		fclose	(fd) ;
	}
	else	logout	(LOG_INFO, "unable to create run file") ;

	signal	(SIGINT,  sigexit) ;
	signal	(SIGTERM, sigexit) ;

	/* Set a handler for SIGPIPE; this is needed in case diald	*/
	/* crashes while we are writing to the control fifo, and such	*/
	/* cases, and a SIGCHLD in case we run the dialer daemon.	*/
	signal	(SIGPIPE, sigpipe) ;
	signal	(SIGCHLD, sigchld) ;

	FD_SET	(connfd, &currcon) ;

	for (;;)
	{
		int	nfds		;
		fd_set	in		= currcon;
		struct	timeval	tv	;

		/* See if we need to reboot the dialer daemon if it has	*/
		/* gone down.						*/
		autoboot () ;

		tv.tv_sec  = 2		;
		tv.tv_usec = 0	 	;

		/* Wait for something to happen. The only "error" case	*/
		/* that we ignore is if the call to select should be	*/
		/* interrupted.						*/
		if ((nfds = select (MAXFD, &in, NULL, NULL, &tv)) < 0)
		{	if (errno == EINTR) continue ;
			errexit	("error in select: %s", STRERR)  ;
		}

		/* First check for is a new connecton. Accept any such	*/
		/* connection; there is nothing further that we		*/
		/* actually need to do at this stage.			*/
		if (FD_ISSET (connfd, &in))
			if (newconnection (connfd))
				continue ;

		/* Secondly check for completed name lookups ...	*/
		checknsl (&in) ;

		/* Next check for events related to (a) communications	*/
		/* with the dialer daemon and (b) other status stuff.	*/
		ddcomms	 (&in, forcecnt) ;

		/* Now check for events on the client connections. We	*/
		/* check for broken connections, and a limited range of	*/
		/* commands.						*/
		clientcomms (&in) ;
	}
}

/*  help	: Display help						*/
/*  (returns)	: void		:					*/

static	void	help ()
{
	fprintf	(stdout, "DialD 0.7 monitor daemon - Mike Richardson 1999\n") ;
	fprintf	(stdout, USAGE) ;
	fprintf	(stdout, "  -D          run in foreground for debugging\n"  ) ;
	fprintf	(stdout, "  -b          link bandwidth in bits/sec\n"	    ) ;
	fprintf	(stdout, "  -f          don't pack queue entries\n"	    ) ;
	fprintf	(stdout, "  -i          allow link to be idled\n"	    ) ;
	fprintf	(stdout, "  -p          pppd interface to reset\n"	    ) ;
	fprintf	(stdout, "  -r          dialer daemon configuration\n"	    ) ;
	fprintf	(stdout, "  -P          force normal password checking\n"   ) ;
	fprintf (stdout, "  -S          force shadow password checking\n"   ) ;
	fprintf	(stdout, "  -A          force PAM authentication\n"	    ) ;
	fprintf	(stdout, "  -v,-V       display version\n"		    ) ;
	fprintf	(stdout, "  control     specify dialmon control file root\n") ;
}

/*  nextarg	: Check for next argument				*/
/*  lvargc	: int *		: Argument count pointer		*/
/*  lvargv	: char ***	: Argument vector pointer		*/
/*  lvopts	: char **	: Options pointer			*/
/*  (returns)	: void		:					*/

static	void	nextarg
	(	int	*lvargc,
		char	***lvargv,
		char	**lvopts
	)
{
	/* I really should learn how to use the getopt routines, but	*/
	/* for the moment ...						*/
	if ((*lvopts)[0] == 0)
	{
		if ((*lvargc) <= 0)
		{	fprintf	(stderr, USAGE) ;
			exit	(1) ;
		}

		*lvopts  = **lvargv ;
		*lvargv += 1 ;
		*lvargc -= 1 ;
	}
}

/*  main	: Standard entry routine				*/
/*  argc	: int		: Argument count			*/
/*  argv	: char **	: Argument vector			*/
/*  (returns)	: int		: Exit code				*/

int	main
	(	int	argc,
		char	**argv
	)
{
	int	_argc	= argc	;
	char	**_argv	= argv	;

	int	pid	;
	DIALCNF	*dialcnf;

	argc	-= 1	;
	argv	+= 1	;

	while ((argc > 0) && (argv[0][0] == '-'))
	{
		char	*opts	= *argv	;

		/* Check for the fairly-standard --help option and the	*/
		/* possibly-standard --version option.			*/
		if (strcmp (opts, "--help")    == 0)
		{	help	() ;
			return	0  ;
		}
		if (strcmp (opts, "--version") == 0)
		{	fprintf	(stdout, VERSION) ;
			return	0 ;
		}

		argv	+= 1	;
		argc	-= 1	;
		opts	+= 1	;

		while (*opts) switch (*opts++)
		{
			case 'D' :
				/* 'D' specifies debug mode, in which	*/
				/* case we will run in the foreground.	*/
				debug	= 1 ;
				break	;

			case 'b' :
				/* Set the link bandwidth used for the	*/
				/* transfer bars. This is converted to	*/
				/* bytes/sec assuming 10 bits/byte.	*/
				nextarg	(&argc, &argv, &opts) ;
				bwidth	= strtol (opts, &opts, 10) / 10 ;
				break	;

			case 'i' :
				/* Allows the connection to be idled	*/
				/* provided that the -r option is set.	*/
				ddidle	= 1 ;
				break	;

			case 'r' :
				/* This option selects automatic	*/
				/* restart of the dialer daemon if it	*/
				/* dies.				*/
				nextarg	(&argc, &argv, &opts) ;
				ddconf	= strdup (opts)	;
				opts	= ""	;
				break	;

			case 'p' :
				/* Where automatic restart is enabled.	*/
				/* this option specifies the PPP daemon	*/
				/* to be zapped.			*/
				nextarg	(&argc, &argv, &opts) ;
				pppd	= opts	;
				opts	= ""	;
				break	;

			case 'f' :
				/* This option causes full (ie., not	*/
				/* packed) queue output ....		*/
				packq	= None	;
				break	;

			case 'F' :
				/* ... and this collapses all non-	*/
				/* privileged ports.			*/
				packq	= Group	;
				break	;

			case 'P' :
				/* Force password checking with the	*/
				/* normal (/etc/passwd) stuff.		*/
				authtyp	= SPasswd ;
				break	;

			case 'S' :
				/* Force password checking with the	*/
				/* normal (/etc/passwd) stuff.		*/
				authtyp	= SShadow ;
				break	;

			case 'A' :
				/* For password checking using PAM	*/
				/* authentication.			*/
				authtyp	= SPam	  ;
				break	;

			case 'v' :
			case 'V' :
				/* Print version and exit ...		*/
				fprintf	(stdout, VERSION) ;
				return	0 ;

			default	 :
				fprintf	(stderr, USAGE) ;
				exit	(1) ;
		}
	}

	if (ddconf == NULL) ddidle = 0 ;

	/* There can be one non-option argument, which specifies the	*/
	/* name of the dialmon control file.				*/
	switch (argc)
	{
		case 0 :
			break	 ;

		case 1 :
			dmcfile = malloc  (strlen(argv[0]) + 8) ;
			dmufile = malloc  (strlen(argv[0]) + 8) ;
			sprintf	 (dmcfile, "%s.conf",  argv[0]) ;
			sprintf	 (dmufile, "%s.users", argv[0]) ;
			break	;

		default	:
			fprintf	(stderr, USAGE) ;
			exit	(1) ;
	}

	/* If automatic password detection was specified then see if	*/
	/* there is an /etc/shadow file ....				*/
	if (authtyp == SAuto)
	{
		char	*t ;

		if	(access ("/etc/pam.d/dialmon", F_OK) == 0)
		{	authtyp	= SPam	  ;
			t	= "PAM"	  ;
		}
		else if (access ("/etc/shadow",        F_OK) == 0)
		{	authtyp	= SShadow ;
			t	= "shadow password" ;
		}
		else
		{	authtyp = SPasswd ;
			t	= "normal password" ;
		}

		syslog	(LOG_ERR|LOG_PID, "dialmon: using %s authentication", t) ;
	}

	definfo.perm  = 0	  ;
	definfo.force = 120	  ;
	strcpy	(definfo.port, DMPORT ) ;
	strcpy	(definfo.fifo, DDCFIFO) ;

	/* Read the dialmon control file, if it exists. This will	*/
	/* default anything that has now been specified.		*/
	if (!scanfor (dmcfile, "host", &definfo, MHOST))
		errexit	("Dialmon configuration error: %s", dmcfile) ;

	if (definfo.logsize < 16) definfo.logsize = 16 ;
	if (definfo.logtime <  1) definfo.logtime =  1 ;

	/* If a dialer daemon boot configuration was specified then	*/
	/* check that it exists.					*/
	if (ddconf != NULL)
		if ((dialcnf = findconf (ddconf)) == NULL)
			if (!ddidle || (strcmp (ddconf, IDLE) != 0))
				errexit	("No diald configuration: %s", ddconf) ;

	loadserv () ;
	startnsl (_argc, _argv) ;

	/* If debugging then don't fork ... this gives me a chance to	*/
	/* debug it ...							*/
	if (debug)
	{	dialmon	() ;
		return	0  ;
	}

	/* Fork and run dialmon in the child process in order to	*/
	/* background it ...						*/
	if ((pid = fork ()) < 0)
		errexit	("dialmon: unable to fork: %s", STRERR) ;

	if (pid == 0) dialmon () ;
	return 0 ;
}
