/*  ddctrl.c								*/
/*  --------								*/
/*  This file contains routines related to controlling the diald (and	*/
/*  possibly pppd) processes.						*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<sys/types.h>
#include	<sys/wait.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	<signal.h>
#include	<errno.h>
#include	<syslog.h>
#include	<time.h>

#include	"dialmon.h"

static	pid_t	ddpid		;	/* Dialer daemon PID		*/
static	int	ddboot		;	/* Dialer daemon booting flag	*/
global	char	*pppd		;	/* Restart PPP device		*/
global	char	*ddconf		;	/* Dialer demon restart config	*/
global	int	ddidle		;	/* Allow diald idling		*/

/*  report	: Report restart result					*/
/*  fd		: int		: Originating connection or none	*/
/*  text	: char *	: Report text				*/
/*  (returns)	: void		:					*/

static	void	report
	(	int	fd,
		char	*text
	)
{
	if (fd >= 0)
		write	(fd, text, strlen (text)) ;
	else	logout	(LOG_ERR, "%s", text)	  ;
}

/*  ddwaitdn	: Wait for diald to go down					*/
/*  delay	: int		: Delay in seconds				*/
/*  (returns)	: void		:						*/

static	void	ddwaitdn
	(	int	delay
	)
{
	for ( ; (delay > 0) && (ddpid != 0) ; delay -= 1)
		sleep	(1) ;
}

/*  ddwaitup	: Wait for diald to come up					*/
/*  delay	: int		: Delay in seconds				*/
/*  (returns)	: void		:						*/

static	void	ddwaitup
	(	int	delay
	)
{
	for ( ; (delay > 0) && (ddpid == 0) ; delay -= 1)
		sleep	(1) ;
}

/*  killdd	: Kill dialer daemon					*/
/*  fd		: int		: Connection to caller			*/
/*  (returns)	: int		: Non-zero on success			*/

static	int	killdd
	(	int	fd
	)
{
	/* First we attempt to be nice about it, sending a terminate	*/
	/* signal to diald, and waiting a few seconds			*/
	if ((kill (-ddpid, SIGTERM) != 0) && (errno != ESRCH))
	{	report	(fd, DDKFAIL) ;
		return	0 ;
	}

	forward	(NULL, DDGOING, NULL, 0) ;
	ddwaitdn(5)	  ;

	/* If diald still seems to be there then we become nastier,	*/
	/* sending it a kill signal ....				*/
	if (ddpid != 0)
	{	if ((kill (-ddpid, SIGKILL) != 0) && (errno != ESRCH))
		{	report	(fd, DDKFAIL) ;
			return	0 ;
		}
		ddwaitdn(2) ;
	}

	/* By this time is really should be dead and buried but check	*/
	/* anyway ...							*/
	if (ddpid != 0)
	{	report	(fd, DDKFAIL) ;
		return	0 ;
	}

	return	1 ;
}

/*  killppp	: Kill possible PPP daemon				*/
/*  fd		: int		: Connection to caller			*/
/*  (returns)	: int		: Non-zero on success			*/

static	int	killppp
	(	int	fd
	)
{
	char	buff	[128] ;
	pid_t	pppid	;
	FILE	*pppfd	;

	sprintf	(buff, PPPFILE, pppd) ;

	if ((pppfd = fopen (buff, "r")) == NULL)
		return	1 ;

	buff[0] = 0 ;
	fgets	(buff, sizeof(buff), pppfd) ;
	fclose	(pppfd) ;

	if ((pppid = (pid_t)atoi(buff)) <= 0)
		return	1 ;

	/* First we attempt to be nice about it, sending a terminate	*/
	/* signal to pppd, and waiting a few seconds			*/
	if ((kill (pppid, SIGTERM) != 0) && (errno != ESRCH))
	{	report	(fd, PPKFAIL) ;
		return	0 ;
	}

	sleep	(3) ;

	/* Now be nasty about it ...					*/
	if ((kill (pppid, SIGKILL) != 0) && (errno != ESRCH))
	{	report	(fd, PPKFAIL) ;
		return	0 ;
	}

	sleep	(2) ;
	return	1 ;
}

/*  parse	: Simple command line parser				*/
/*  line	: char *	: Command line				*/
/*  (returns)	: char **	: Parsed command			*/

static	char	**parse
	(	char	*line
	)
{
	static	char	cmdbuf	[256]	;
	static	char	*argv	[ 64]	;

	char	*cp	= cmdbuf	;
	int	argc	= 0		;

	strcpy	(cmdbuf, line)		;

	while (*cp != 0)
	{
		if (isspace (*cp))
		{	while (isspace (*cp)) cp += 1 ;
			continue ;
		}

		argv[argc] = cp ;
		argc += 1	;

		while ((*cp != 0) && !isspace (*cp)) cp += 1 ;
		if (*cp != 0) *cp++ = 0 ;
	}

	argv[argc] = NULL ;
	return	argv	  ;
}

/*  restart	: Restart dialer daemon					*/
/*  fd		: int		: Stream originating command		*/
/*  conf	: char *	: Diald configuration to use		*/
/*  (returns)	: int		: Non-zero on diald running		*/

int	restart
	(	int	fd,
		char	*conf
	)
{
	pid_t	pid	;
	int	fid	;
	char	**argv	;
	DIALCNF	*dialcnf;

	if ((conf == NULL) || (conf[0] == 0))
		conf	= ddconf ;

	/* Scan for the selected configuration. We should never get	*/
	/* here unless the name is valid, except maybe for users	*/
	/* telnet'ing in.						*/
	if ((dialcnf = findconf (conf)) == NULL)
		if (!ddidle || (strcmp (conf, IDLE) != 0))
		{	report	(fd, DDCONFL) ;
			return	0 ;
		}

	/* If we believe there to be an extant dialer daemon then we	*/
	/* try to kill it, plus possibly an instance of the PPP daemon.	*/
	if (ddpid != 0   )
	{	logconn () ;
		if (!killdd ( fd)) return 0 ;
	}
	if (pppd  != NULL)
		if (!killppp (fd)) return 0 ;

	tidyup	 ()   ;
	sendmode (-1) ;

	/* Send some common control stuff that applies whatever		*/
	/* configuration we are switching to.				*/
	forward  (lastiface, "INTERFACE", "none none none", 0     ) ;
	forward	 (lastload,  "LOAD",      "0 0 0 0"	  , F_LOAD) ;
	qtidy	 () ;
	sendqueue() ;

	/* Check for the special case that we have idled the dialer	*/
	/* daemon. In this case, we need to send enough control stuff	*/
	/* to clear things up at the client display end.		*/
	if (strcmp (conf, IDLE) == 0)
	{
		if ((ddconf != NULL) && (ddconf != conf)) free (ddconf) ;
		ddconf	= NULL    ;

		forward  (NULL,      DDIDLE,      NULL,     	 0) ;
		forward	 (laststate, "STATE",     "IDLE",   	 0) ;
		forward	 (lastmsg,   "MESSAGE",   "DialD idled", 0) ;

		addtolog (time (NULL), -1, IDLE, 0, 0) ;
		return	 0  ;
	}

	forward	(laststate, "STATE", "DOWN", 0) ;
 
	argv	= parse (dialcnf->cmd) ;

	switch (pid = fork ())
	{
		case -1 :
			/* Error, unable to fork. Report the error to	*/
			/* the caller or to the error log if none.	*/
			report	(fd, DDSFAIL) ;
			return	0 ;

		case  0 :
			/* Close all streams and then open the standard	*/
			/* streams to /dev/null. This is needed since,	*/
			/* when not in daemon mode, diald will log	*/
			/* to stderr ...				*/
			for (fid = 0 ; fid < MAXFD ; fid += 1) close (fid) ;
			fd = open ("/dev/null", O_RDWR) ;
			if (fd != 0) dup2 (fd, 0) ;
			if (fd != 1) dup2 (fd, 1) ;
			if (fd != 2) dup2 (fd, 2) ;

			/* Make this a new process group, so that the	*/
			/* parent can kill everyting with a single kill	*/
			setpgid (0, 0) ;
			execv	(argv[0], argv) ;
			exit	(1) ;

		default	:
			break	;
	}

	/* Note the PID for the dialer daemon and then sleep for a few	*/
	/* seconds to give it chance to start up. If it fails then the	*/
	/* PID should have been reset by the time we send the status.	*/
	/* Set the 'ddboot' flag which is used to prevent restart-	*/
	/* fail-restart loops.						*/
	ddboot	= 1	  ;
	ddpid	= pid	  ;
	ddwaitup (5)	  ;

	if (ddpid == 0)
	{	forward (NULL, DDBOOTF, NULL, 0) ;
		return	0 ;
	}

	/* We seem to need to sleep a bit after it comes up to ensure	*/
	/* that we will be able to open communication with it.		*/
	forward (NULL, DDRESTART, NULL, 0) ;
	sleep	(2)	     ;
	forward	(NULL, DDALIVE,   NULL, 0) ;
	ddboot	= 0	     ;

	/* Note the new configuration. We copy the string as it may	*/
	/* come from a user command.					*/
	if ((ddconf != NULL) && (ddconf != conf)) free (ddconf) ;
	ddconf	 = strdup (conf) ;

	addtolog (time (NULL), -1, ddconf, 0, 0) ;
	return	 1 ;
}

/*  autoboot	: Handle auto-reboot of the dialer daemon		*/
/*  (returns)	: void		:					*/

global	void	autoboot ()
{
	/* If a dialer daemon restart configuration is specified, the	*/
	/* dialer daemon is not running, and we are not in the process	*/
	/* of restarting it, then do so.				*/
	if ((ddconf != NULL) && (ddpid == 0) && !ddboot)
	{	if (restart (-1, ddconf))
			forward (NULL, ddinit () ? DDALIVE : DDDEAD, NULL, 0) ;
		sendconf (-1) ;
	}
}

/*  dtidy	: Clean up dialer daemon stuff				*/
/*  (returns)	: void		:					*/

global	void	dtidy ()
{
	ddpid	= 0 ;
}

/*  killall	: Kill everything off					*/
/*  (returns)	: void		:					*/

global	void	killall ()
{
	if (ddpid != 0)
	{	kill	(-ddpid, SIGTERM) ;
		sleep	(2) ;
		kill	(-ddpid, SIGKILL) ;
		ddpid	= 0 ;
	}
}

/*  sigchld	: Child signal handler					*/
/*  sig		: int		: Signal number				*/
/*  (returns)	: void		:					*/

global	void	sigchld
	(	int	sig
	)
{
	int	status	;
	pid_t	pid	;

	signal	(SIGCHLD, sigchld) ;
	while ((pid = waitpid (-1, &status, WNOHANG)) > 0)
		if (pid == ddpid) ddpid = 0 ;
}
