/*  ddcomms.c								*/
/*  ---------								*/
/*  This file contains routines related to communication with the	*/
/*  dialer daemon.							*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<time.h>
#include	<fcntl.h>
#include	<syslog.h>
#include	<errno.h>

#include	"dialmon.h"

static	time_t	forcetil ;		/* Force-until time		*/
static	int	ctrlfd	 = -1	;	/* diald control fifo		*/
static	int	infofd	 = -1	;	/* diald information fifo	*/
static	int	statfd	 = -1	;	/* dialmon status fifo		*/
global	DIALINF	definfo		;

static	char	infobuf  [256]	;	/* diald information buffer	*/
static	int	infolen		;	/* Amount of data in buffer	*/
static	char	statbuf  [256]	;	/* dialmon status buffer	*/
static	int	statlen		;	/* Amount of data in buffer	*/

static	void	*(*state_func) (char *) = dont_care ;

/*  ddinit	: Initialise connection to dialer daemon		*/
/*  (returns)	: void		:					*/

global	int	ddinit ()
{
	char	path[128]	;
	char	buff[256]	;
	int	tries		;
	int	mask		= umask (0) ;

	/* NOTE:							*/
	/* See notes below for significance of maxtry ...		*/
	int	maxtry		= ddconf == NULL ? 5 : 30 ;

	/* We now create a named pipe from which we will receive diald	*/
	/* status information, and open it read-write. We don't ever	*/
	/* read from is, but read-write prevents hanging. Neat!		*/
	umask	(0002) ;
	sprintf	(path, "%s/diald.info", PIPEDIR) ;
	unlink	(path) ;
	if (mknod (path, S_IFIFO|S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH, 0) < 0)
		errexit	("mknod %s failed: %s", path, STRERR)  ;

	if ((infofd = open (path, O_RDWR)) < 0)
		errexit	("unable to open pipe %s: %s", path, STRERR)  ;

	/* Open a connection to diald via its diald.ctl pipe, and write	*/
	/* a monitor command to it, requesting that data be sent to the	*/
	/* pipe we just created.					*/

	/* NOTE:							*/
	/* There is a possibility of a race condition here; this may	*/
	/* occur if diald is started from dialmon, and we get here	*/
	/* before diald creates its pipe. So, try a few times with a	*/
	/* short pause before each. Thanks to Matija Nalis for this	*/
	/* bit; I set maxtries depending on whether we are starting	*/
	/* diald, or it is already running.				*/
	for (tries = 0 ; tries < maxtry ; tries += 1)	
	{	if ((ctrlfd = open (definfo.fifo, O_WRONLY|O_NDELAY)) >= 0)
			break	;
		sleep	(1)	;
	}

	if (ctrlfd < 0)
	{	close	(infofd) ;
		infofd	= -1	 ;
		return	0 ;
	}

	/* NOTE:							*/
	/* Matija noted another possible problem here; if you bang out	*/
	/* a command to diald too soon after starting it, the command	*/
	/* may get ignored. So, a further short delay.			*/
	sleep	(3) ;
	sprintf	(buff, "monitor %s/diald.info\n", PIPEDIR) ; 
	if (write (ctrlfd, buff, strlen(buff)) != strlen(buff))
	{	close	(ctrlfd) ;
		close	(infofd) ;
		ctrlfd	= -1	 ;
		infofd	= -1	 ;
		return	0	 ;
	}

	/* Also create a named pipe which is used to signal status	*/
	/* inforamtion to dialmon by the helper programs such as	*/
	/* loginstub and gettystub. Again, read-write prevents hanging.	*/
	umask	(0003) ;
	sprintf	(path, "%s/dialmon.stat", PIPEDIR) ;
	unlink	(path) ;
	if (mknod (path, S_IFIFO|S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH, 0) < 0)
		errexit	("mknod %s failed: %s", path, STRERR)  ;

	if ((statfd = open (path, O_RDWR)) < 0)
		errexit	("unable to open pipe %s: %s", path, STRERR)  ;

	/* Add the pipe descriptors to the global select mask so that	*/
	/* we receive data from it.					*/
	fdmaskset (infofd, 0) ;
	fdmaskset (statfd, 0) ;
	umask	  (mask) ;

	return	1 ;
}

/*  dialdinfo	: Handle incoming information from diald		*/
/*  (returns)	: void		:					*/

static	void	dialdinfo ()
{
	int	len	;
	char	*lp	;

	if (infolen >= sizeof(infobuf)) infolen = 0 ;

	if ((len = read (infofd, &infobuf[infolen], sizeof(infobuf) - infolen - 1)) <= 0)
	{	tidyup	() ;
		logout	(LOG_ERR, "error reading information: %s", STRERR) ;
		return	;
	}

	infolen	+= len ;

	/* Process the new data so long as we have at least one		*/
	/* complete line, ie., there is a newline character in the	*/
	/* buffer.							*/
	while ((lp = (char *)memchr (infobuf, '\n', infolen)) != NULL)
	{
		*lp++ = 0 ;

		/* Process the line using the state machine (code by	*/
		/* jem - much neater and more extensible).		*/
		state_func = (*state_func) (infobuf) ;

		memmove	(infobuf, lp, infolen - (lp - infobuf)) ;
		infolen	-= lp - infobuf ;
	}
}

/*  dialmstat	: Handle incoming status information for dialmon		*/
/*  (returns)	: void		:					*/

static	void	dialmstat ()
{
	int	len	;
	char	*lp	;

	if (statlen >= sizeof(statbuf)) statlen = 0 ;

	if ((len = read (statfd, &statbuf[statlen], sizeof(statbuf) - statlen - 1)) <= 0)
	{	tidyup	() ;
		logout	(LOG_ERR, "error reading status data: %s", STRERR) ;
		return	;
	}

	statlen	+= len ;

	/* Process the new data so long as we have at least one		*/
	/* complete line, ie., there is a newline character in the	*/
	/* buffer.							*/
	while ((lp = (char *)memchr (statbuf, '\n', statlen)) != NULL)
	{
		*lp++ = 0 ;

		switch (statbuf[0])
		{
			case 'E' :
				/* This is a message indicating the	*/
				/* presence or otherwise of an		*/
				/* external (presumably incoming	*/
				/* connection).				*/
				switch (statbuf[1])
				{	case '+' : extcon = 1 ; break ;
					case '-' : extcon = 0 ; break ;
					default	 :		break ;
				}
logout (LOG_ERR, "extern: %s %d\n", statbuf, extcon) ;
				forward (NULL, extcon ? EXTERNUP : EXTERNDOWN, NULL, 0) ;
				break	;

			case 'M' :
				/* Broadcast a message to all clients;	*/
				/* the message follows on the line.	*/
				forward (lastmsg, "MESSAGE", &statbuf[1], 0) ;
				break	;

			default	 :
				/* Ignore anything else ...		*/
				break	;
		}
 
		memmove	(statbuf, lp, statlen - (lp - statbuf)) ;
		statlen	-= lp - statbuf ;
	}
}

/*  dialdup	: Send diald up request					*/
/*  (returns)	: void		:					*/

global	void	dialdup ()
{
	write	(ctrlfd, "force\n",    6) ;
	forcetil = time (NULL) + definfo.force ;
}

/*  dialddn	: Send diald down request				*/
/*  (returns)	: void		:					*/

global	void	dialddn ()
{
	write	(ctrlfd, "down\n",     5) ;
}

/*  dialdforce	: Send diald force request				*/
/*  (returns)	: void		:					*/

global	void	dialdforce ()
{
	write	(ctrlfd, "force\n",    6) ;
}

/*  dialdblock	: Send diald block request				*/
/*  (returns)	: void		:					*/

global	void	dialdblock ()
{
	write	(ctrlfd, "block\n",    6) ;
}

/*  dialdunfrc	: Send diald un-force request				*/
/*  (returns)	: void		:					*/

global	void	dialdunfrc ()
{
	write	(ctrlfd, "unforce\n",  8) ;
}

/*  dialdunblk	: Send diald un-block request				*/
/*  (returns)	: void		:					*/

global	void	dialdunblk ()
{
	write	(ctrlfd, "unblock\n",  8) ;
}

/*  ctidy	: Tidy up communications				*/
/*  (returns)	: void		:					*/

global	void	ctidy ()
{
	state_func	= dont_care ;
	infolen		= 0 ;

	if (infofd >= 0)
	{	close     (infofd) ;
		fdmaskclr (infofd) ;
		infofd = -1 ;
	}
	if (statfd >= 0)
	{	close     (statfd) ;
		fdmaskclr (statfd) ;
		statfd = -1 ;
	}
	if (ctrlfd >= 0)
	{	close     (ctrlfd) ;
		ctrlfd = -1 ;
	}
}

/*  ddcomms	: Handle dialer daemon communications			*/
/*  in		: fd_set *	: Input descriptor mask			*/
/*  forcecnt	: int		: Current clients-forcing count		*/
/*  (returns)	: void		:					*/

global	void	ddcomms
	(	fd_set	*in,
		int	forcecnt
	)
{
	/* First see if it is time to un-force. The criteria for this	*/
	/* are:								*/
	/* (a)	We have a control connection				*/
	/* (b)	The force-until time has been reached			*/
	/* (c)	We are in normal mode					*/
	if ( (ctrlfd   >= 0) &&
	     (forcetil  > 0) && (forcetil <= time(NULL)) && (dmode == Normal) )
	{
		if (write (ctrlfd, "unforce\n", 8) != 8)
		{	close	(ctrlfd) ;
			ctrlfd	= -1	 ;
		}
		forcetil = -1 ;
	}

	/* Secondly, see if there is any data arriving on the dialer	*/
	/* daemon information pipe. 					*/
	if ((infofd >= 0) && FD_ISSET (infofd, in)) dialdinfo () ;

	/* ... and thirdly, look for any other incoming status		*/
	/* infomation.							*/
	if ((statfd >= 0) && FD_ISSET (statfd, in)) dialmstat () ;
}

