/*  client.c								*/
/*  --------								*/
/*  This file contains routines related to communication with the	*/
/*  dialmon clients.							*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#include	<sys/types.h>
#include	<sys/socket.h>
#include	<netinet/in.h>
#include	<netdb.h>
#include	<syslog.h>
#include	<pwd.h>
#include	<errno.h>
#include	<time.h>
#include	<shadow.h>

#include	"dialmon.h"

static	Peer	*peerlst [MAXFD];	/* Peer names and buffering	*/

extern	char		*crypt	 (const char *, const char *) ;
extern	struct	passwd	*checkpw (struct passwd *, char *, int, char *[]) ;

/*  sendperm	: Send permissions					*/
/*  fd		: int		: File descriptor			*/
/*  perm	: int		: Permissions mask			*/
/*  (returns)	: void		:					*/

static	void	sendperm
	(	int	fd,
		int	perm
	)
{
	if (perm & A_DOWN)
		write (fd, "ALLOW_DOWN YES\n", 15) ;
	else	write (fd, "ALLOW_DOWN NO\n",  14) ;
	if (perm & A_UP  )
		write (fd, "ALLOW_UP YES\n",   13) ;
	else	write (fd, "ALLOW_UP NO\n",    12) ;
	if (perm & A_SET )
		write (fd, "ALLOW_SET YES\n",  14) ;
	else	write (fd, "ALLOW_SET NO\n",   13) ;
	if (perm & A_CTRL)
		write (fd, "ALLOW_CTRL YES\n", 15) ;
	else	write (fd, "ALLOW_CTRL NO\n",  14) ; 
}

/*  login	: Log in user on connection				*/
/*  fd		: int		: Connection to peer			*/
/*  user	: char *	: User name and password		*/
/*  (returns)	: void		:					*/

static	void	login
	(	int	fd,
		char	*user
	)
{
	char	*pwd	;
	char	*lp	;
	DIALINF	dialinf	;
	
	struct	passwd	*p_pwd	;
	struct	spwd	*s_pwd	;
	char		*e_pwd	;

	if ((pwd = strchr (user, ':')) == NULL)
	{	logout	(LOG_ERR, "Bad user format: %s", user) ;
		write	(fd, BADUSER, sizeof(BADUSER) - 1) ;
		return	;
	}

	*pwd++	= 0	;
	if ((lp = strchr (pwd, '\n')) != NULL) *lp = 0 ;
	if ((lp = strchr (pwd, '\r')) != NULL) *lp = 0 ;

	dialinf.perm	= 0 ;
	strcpy	(dialinf.pwd, user) ;

	if ( !scanfor (dmufile, user,      &dialinf, MUSER) &&
	     !scanfor (dmufile, "default", &dialinf, MUSER) )
	{
		logout	(LOG_ERR, "No user access: %s", user) ;
		write	(fd, BADPWD, sizeof(BADPWD) - 1) ;
		return	;
	}

	/* First check on the user. Whether we are using normal, shadow	*/
	/* or PAM authentication, the user name must exist in the	*/
	/* password file.						*/
	if ((p_pwd = getpwnam (dialinf.pwd)) == NULL)
	{	logout	(LOG_ERR, "Unknown user: %s", user) ;
		write	(fd, BADPWD, sizeof(BADPWD) - 1) ;
		return	;
	}
	e_pwd	= p_pwd->pw_passwd ;

	switch (authtyp)
	{
		case SShadow	:
			/* If using shadow passwords then look up the	*/
			/* shadow entry. This should not fail but who	*/
			/* knows ...					*/
			if ((s_pwd = getspnam (dialinf.pwd)) == NULL)
			{	logout	(LOG_ERR, "Missing shadow: %s", user) ;
				write	(fd, BADPWD, sizeof(BADPWD) - 1) ;
				return	;
			}
			e_pwd	= s_pwd->sp_pwdp   ;

			/* Drop through to check the actual password.	*/

		case SPasswd	:
			/* Normal or shadow password. In either case we	*/
			/* check the encrypted password against that	*/
			/* proferred.					*/
			if (strcmp (crypt (pwd, e_pwd), e_pwd) != 0)
			{
				logout	(LOG_ERR, "Bad password: %s", user) ;
				write	(fd, BADPWD, sizeof(BADPWD) - 1) ;
				return	;
			}

			break	;

		case SPam	:
			/* PAM authentication ....			*/
			if (checkpw (p_pwd, pwd, 0, NULL) == NULL)
			{
				logout	(LOG_ERR, "PAM auth failed: %s", user) ;
				write	(fd, BADPWD, sizeof(BADPWD) - 1) ;
				return	;
			}

			break	;

		default	:
			/* This is actually a dialmon error, since we	*/
			/* should always have a valid authtype.		*/
			logout	(LOG_ERR, "Bad password setting") ;
			write	(fd, PWDERR, sizeof(PWDERR) - 1)  ;
			return	;
	}

	/* All OK. Note the user name and the permissions, and send	*/
	/* the latter to the client.					*/
	peerlst[fd]->user    = strdup (user) ;
	peerlst[fd]->perm    = dialinf.perm  ;
	peerlst[fd]->version = 0	     ;
	peerlst[fd]->subver  = 0	     ;
	sendperm (fd, dialinf.perm)	     ;
}

/*  options	: Turn on or off an option				*/
/*  cmd		: char *	: Command				*/
/*  lvfl	: int *		: Flags					*/
/*  (returns)	: void		:					*/

static	void	options
	(	char	*cmd,
		int	*lvfl
	)
{
	int	add	= -1 ;
	int	flag	;

	/* The first character indicates how to change the flags. The	*/
	/* options are:							*/
	/*	+	Add flags					*/
	/*	-	Remove flags					*/
	/*	=	Set specified flags (clear and add)		*/
	switch (*cmd++)
	{
		case '+' :
			add	= 1 ;
			break	;

		case '-' :
			add	= 0 ;
			break	;

		case '=' :
			add	= 1 ;
			*lvfl	= 0 ;

		default	 :
			return	;
	}

	/* If the flags are still set to F_ALL then clear them back to	*/
	/* zero. Doing things this provides backwards compatability	*/
	/* with onler dialm clients.					*/
	if (*lvfl == F_ALL) *lvfl = 0 ;

	/* Run through processing each specified flag. Any that are	*/
	/* unrecognised are silently ignored.				*/
	while (*cmd != 0)
	{
		switch (*cmd++)
		{
			case 'Q' :
				flag	= F_QUEUE ;
				break	;

			default	:
				continue ;
		}

		if (add)
			*lvfl	|=  flag ;
		else	*lvfl	&= ~flag ;
	}
}

static	void	verflags
	(	Peer	*peer
	)
{
	static	int	vmap[] =
	{	0,	7,	F_LOAD,
		-1
	}	;

	int	*vp	;

	for (vp = &vmap[0] ; vp[0] >= 0 ; vp += 3)
		if ((peer->version == vp[0]) && (peer->subver == vp[1]))
		{	peer->flags = vp[2] ;
			return	;
		}

	peer->flags	= F_ALL ;
}

/*  command	: Handle command from client				*/
/*  fd		: int		: Client stream				*/
/*  cmd		: char *	: Command string			*/
/*  (returns)	: void		:					*/

global	void	command
	(	int	fd,
		char	*cmd
	)
{
	Peer	*peer	= peerlst[fd] ;
	char	*cr	;

	if ((cr = strchr (cmd, '\r')) != NULL) *cr = 0 ;

	switch (cmd[0])
	{
		case 'U' :
			/* This is a request to bring the link up. See	*/
			/* if the client is allowed.			*/
			if ((peer->perm & A_UP  ) == 0)
			{	write	(fd, NOUP,   sizeof(NOUP  ) - 1) ;
				break	;
			}

			if (dmode == Normal) dialdup () ;
			break	;

		case 'D' :
			/* Similarly to take the link down ...		*/
			if ((peer->perm & A_DOWN) == 0)
			{	write	(fd, NODOWN, sizeof(NODOWN) - 1) ;
				break	;
			}

			if (dmode == Normal) dialddn () ;
			break	;

		case 'F' :
			/* This is a request to force the link up until	*/
			/* further notice.				*/
			if ((peer->perm & A_SET ) == 0)
			{	write	(fd, NOSET,  sizeof(NOSET ) - 1) ;
				break	;
			}

			if (dmode == Block) dialdunblk () ;
			dmode	= Force ;
			dialdforce ()   ;
			sendmode   ()   ;
			break	;

		case 'B' :
			/* This is a request to block the link up until	*/
			/* further notice.				*/
			if ((peer->perm & A_SET ) == 0)
			{	write	(fd, NOSET,  sizeof(NOSET ) - 1) ;
				break	;
			}

			if (dmode == Force) dialdunfrc () ;
			dmode	= Block ;
			dialdblock ()   ;
			sendmode   ()   ;
			break	;

		case 'N' :
			/* This request simply returns the mode to	*/
			/* normal if it was not already.		*/
			if ((peer->perm & A_SET ) == 0)
			{	write	(fd, NOSET,  sizeof(NOSET ) - 1) ;
				break	;
			}

			switch (dmode)
			{	case Force : dialdunfrc () ; break ;
				case Block : dialdunblk () ; break ;
				default	   :		     break ;
			}
			dmode	= Normal ;
			sendmode   ()    ;
			break	;

		case 'L' :
			/* Login user ...				*/
			login	(fd, &cmd[1]) ;
			break	;

		case 'R' :
			/* Restart the dialer daemon			*/
			if ((peer->perm & A_CTRL) == 0)
			{	write	(fd, NOCTRL, sizeof(NOCTRL) - 1) ;
				break	;
			}

			if (restart (fd, &cmd[1]))
				forward (NULL, ddinit () ? DDALIVE : DDDEAD, NULL, 0) ;
			sendconf (-1) ;
			break	;
 
		case 'Q' :
			/* This is just a way of closing a client	*/
			/* connection, useful when testing from a	*/
			/* telnet client.				*/
			close	  (fd) ;
			fdmaskclr (fd) ;
			break	;

		case 'V' :
			/* This supplies version inforamtion in case we	*/
			/* get into the game of handling multiple	*/
			/* client versions.				*/
			sscanf	(&cmd[1], "%d.%d", &peer->version, &peer->subver) ;
			verflags(peer) ;
			break	;

		case 'I' :
			/* Request to send connection logging		*/
			/* inforamation for display by the client.	*/
			sendlog	(fd) ;
			break	;

		case 'O' :
			/* This is a request to turn on or off options	*/
			/* such as queue transmission.			*/
			options	(&cmd[1], &peer->flags) ;
			break	;

		default	 :
			/* Silently ignore anything else ...		*/
			break	;
	}

}

/*  newconnection: Handle new connection				*/
/*  connfd	 : int		: Listening socket			*/
/*  (returns)	 : int		: Non-zero on EINTR			*/

global	int	newconnection
	(	int	connfd
	)
{
	struct	sockaddr_in peer	;
	struct	hostent	    *peerent	;

	char	host[128] ;
	int	s	  ;
	int	len	  ;
	DIALINF	dialinf	  ;

	memset	(&peer, 0, sizeof(peer));
	peer.sin_family = AF_INET	;

	len = sizeof(peer) ;
	if ((s = accept (connfd, (struct sockaddr *)&peer, &len)) < 0)
	{	if (errno == EINTR) return 1 ;
		errexit	("error in accept: %s", STRERR)  ;
	}

	len = sizeof(peer) ;
	if (getpeername (s, (struct sockaddr *)&peer, &len) >= 0)
	{
		if ((peerent = gethostbyaddr ((char *)&peer.sin_addr.s_addr,
					      sizeof(peer.sin_addr.s_addr),
					      AF_INET)) != NULL)
			strcpy	(host, peerent->h_name) ;
		else	sprintf	(host, "%d.%d.%d.%d",
				 (int)((ntohl(peer.sin_addr.s_addr) >> 24) & 0xff),
				 (int)((ntohl(peer.sin_addr.s_addr) >> 16) & 0xff),
				 (int)((ntohl(peer.sin_addr.s_addr) >>  8) & 0xff),
				 (int)((ntohl(peer.sin_addr.s_addr) >>  0) & 0xff)) ;

		logout	(LOG_INFO, "connection from %s", host) ;
	}
	else
	{	logout	(LOG_INFO, "connection from unknown peer") ;
		strcpy	(host, "<unknown host>") ;
	}

	/* Scan the configuration file for this host. If it is not	*/
	/* found then use the default permissions; also, initialise	*/
	/* the flags to send all information.				*/
	dialinf.perm  = 0     ;
	if (!scanfor (dmcfile, host, &dialinf, MCLNT))
		dialinf.perm = definfo.perm ;


	{	Peer	*p = (Peer *)calloc(1, sizeof(Peer)) ;
		p->host    = strdup (host) ;
		p->user	   = NULL	   ;
		p->perm	   = dialinf.perm  ;
		p->flags   = F_ALL	   ;
		p->force   = 0		   ;
		p->blen    = 0		   ;
		peerlst[s] = p		   ;
	}

	fdmaskset  (s, 1) ;

	sendperm   (s, peerlst[s]->perm) ;
	write	   (s, laststate, strlen(laststate)) ;
	write	   (s, lastmsg,   strlen(lastmsg  )) ;
	write	   (s, lastload,  strlen(lastload )) ;
	write	   (s, lastiface, strlen(lastiface)) ;

	sendconf   (s) ;
	sendmode   ( ) ;

	if (extcon)
		write	(s, EXTERNUP,   strlen(EXTERNUP  )) ;
	else	write	(s, EXTERNDOWN, strlen(EXTERNDOWN)) ;
	write	   (s, "\n", 1) ;

	return	0 ;
}

/*  clientdata	: Handle data from a client connection			*/
/*  s		: int		: Client connection file descriptor	*/
/*  (returns)	: void		:					*/

static	void	clientdata
	(	int	s
	)
{
	int	len	;
	char	*lp	;
	Peer	*peer	= peerlst[s] ;

	/* Paranoia check for end of buffer ...				*/
	if (peer->blen >= sizeof(peer->buff)) peer->blen = 0 ;

	/* Read up to the amount of free space in the peer buffer. If	*/
	/* this fails or hits an end-of-file then close the connection	*/
	/* and release allocated space.					*/
	if ((len = read (s, &peer->buff[peer->blen], sizeof(peer->buff) - peer->blen)) <= 0)
	{
		close	  (s) ;
		fdmaskclr (s) ;
		free	  (peer->host ) ;
		free	  (peer->user ) ;
		free	  (peer	      ) ;
		peerlst[s] = NULL ;
		return	;
	}

	peer->blen += len ;

	/* Loop processing all complete lines in the buffer. Most	*/
	/* likely there will be one (and exactly one) but who knows	*/
	/* what some future client might do.				*/
	while ((lp = (char *)memchr (peer->buff, '\n', peer->blen)) != NULL)
	{
		*lp++ = 0 ;

		command	(s, peer->buff) ;

		memmove	(peer->buff, lp, peer->blen - (lp - peer->buff)) ;
		peer->blen -= lp - peer->buff ;

	}
}

/*  clientcomms	: Handle client communications				*/
/*  in		: fd_set *	: Input file descriptor mask		*/
/*  (returns)	: void		:					*/

global	void	clientcomms
	(	fd_set	*in
	)
{
	int	s	;

	for (s = 0 ; s < MAXFD ; s += 1)
		if ((peerlst[s] != NULL) && FD_ISSET(s, in))
			clientdata (s) ;
}

/*  clientreq	: See if client has requested particular information	*/
/*  s		: int		: Connection descriptor			*/
/*  flags	: int		: Information type flags		*/
/*  (returns)	: int		: Non-zero if requested			*/

global	int	clientreq
	(	int	s,
		int	flags
	)
{
	if (flags == 0)
		return	1 ;
	else	return	(peerlst[s] != NULL) && ((peerlst[s]->flags & flags) != 0) ;
}
