/*
 * Program:	Pluggable Authentication Modules login services
 *
 * Author:	Michael K. Johnson
 *		Red Hat Software
 *		Internet: johnsonm@redhat.com
 *
 * Date:	17 June 1996
 * Last Edited:	18 June 1996
 *
 * Copyright 1996 Red Hat Software
 * Copyright 1995 by the University of Washington
 *
 *  Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both the
 * above copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the University of Washington not be
 * used in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  This software is made available
 * "as is", and
 * THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
 * NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <security/pam_appl.h>

/* Static variables used to communicate between the conversation function
 * and the server_login function
 */
static char *PAM_username;
static char *PAM_password;
static int PAM_error = 0;

/* PAM conversation function
 * Here we assume (for now, at least) that echo on means login name, and
 * echo off means password.
 */
static int PAM_conv (int num_msg,
                     const struct pam_message **msg,
		     struct pam_response **resp,
		     void *appdata_ptr) {
  int replies = 0;
  struct pam_response *reply = NULL;

  #define COPY_STRING(s) (s) ? strdup(s) : NULL

  reply = malloc(sizeof(struct pam_response) * num_msg);
  if (!reply) return PAM_CONV_ERR;

  for (replies = 0; replies < num_msg; replies++) {
    switch (msg[replies]->msg_style) {
      case PAM_PROMPT_ECHO_ON:
        reply[replies].resp_retcode = PAM_SUCCESS;
	reply[replies].resp = COPY_STRING(PAM_username);
          /* PAM frees resp */
        break;
      case PAM_PROMPT_ECHO_OFF:
        reply[replies].resp_retcode = PAM_SUCCESS;
	reply[replies].resp = COPY_STRING(PAM_password);
          /* PAM frees resp */
        break;
      case PAM_TEXT_INFO:
        /* fall through */
      case PAM_ERROR_MSG:
        /* ignore it, but pam still wants a NULL response... */
        reply[replies].resp_retcode = PAM_SUCCESS;
	reply[replies].resp = NULL;
        break;
      default:
        /* Must be an error of some sort... */
        free (reply);
        PAM_error = 1;
        return PAM_CONV_ERR;
    }
  }
  *resp = reply;
  return PAM_SUCCESS;
}
static struct pam_conv PAM_conversation = {
    &PAM_conv,
    NULL
};

/* Server log in
 * Accepts: user name string
 *	    password string
 * Returns: T if password validated, NULL otherwise
 */
struct passwd *checkpw (struct passwd *pw,char *pass,int argc,char *argv[])
{
  pam_handle_t *pamh;
  int pam_error;

  /* PAM only handles authentication, not user information. */
  if (!(pw && pw->pw_uid && pw->pw_name && strlen(pw->pw_name)))
      return NULL;

  /* validate password */

  /* Now use PAM to do authentication.  For now, we won't worry about
   * session logging, only authentication.  Bail out if there are any
   * errors.  Since this is a limited protocol, and an even more limited
   * function within a server speaking this protocol, we can't be as
   * verbose as would otherwise make sense.  It would be nice if we
   * could return a string for the server to pass the the client and/or
   * log, but that doesn't exist right now.  If it is ever added, PAM
   * could make good use of it.
   * Query: should we be using PAM_SILENT to shut PAM up?
   */
  #define PAM_BAIL if (PAM_error || (pam_error != PAM_SUCCESS)) { \
     pam_end(pamh, 0); return NULL; \
   }
  PAM_password = pass;
  PAM_username = pw->pw_name;
  pam_error = pam_start("dialmon", pw->pw_name, &PAM_conversation, &pamh);
  PAM_BAIL;
  pam_error = pam_authenticate(pamh, 0);
  PAM_BAIL;
  pam_error = pam_acct_mgmt(pamh, 0);
  PAM_BAIL;
  pam_error = pam_setcred(pamh, PAM_ESTABLISH_CRED);
  PAM_BAIL;
  pam_end(pamh, PAM_SUCCESS);
  /* If this point is reached, the user has been authenticated. */
  return pw;
}

