/*
 ipc.c

 DFC -- Dedicated File Cache -- IPC wrappers

 Copyright (c) 2000 Edward V. POPKOV
 Distributed under GPL, see COPYING for details

 "Edward V. POPKOV" <evpopkov@carry.netnet.lv>

 $Id: ipc.c,v 0.5 2000/09/22 14:47:43 root Exp root $

 $Log: ipc.c,v $
 Revision 0.5  2000/09/22 14:47:43  root
 Fixed permissions on pidfile

 Revision 0.4  2000/09/21 21:56:07  root
 Fixed `after file' bug

 Revision 0.3  2000/04/19 20:49:26  root
 Optimized code, fixed includes/defines

 Revision 0.2  2000/04/18 01:08:17  root
 Bugfix (buffer overrun) in dfcd

 Revision 0.1  2000/04/07 15:00:00  root
 Signal handling workaround, bugfixes, added sample conf, backup on upgrade

 Revision 0.0  2000/04/06 13:45:28  root
 Initial check-in

*/

#ifndef _IPC_
#define _IPC_

#include <dfc.h>

#define DFC_IPC_SHM_CREATE IPC_CREAT | IPC_EXCL | S_IRUSR | S_IWUSR
#define DFC_IPC_SHM_SETOWN S_IRUSR | S_IRGRP

int dfc_ipc_shm_destroy( int * )
	__attribute__(( __regparm__( 1 )));

int dfc_ipc_ftok( key_t * key, char * path )
{
	int r = FALSE;

DFC_PARAM_CHECK( key != NULL && (* key) == DFC_ERROR && len_c(path) > DFC_EMPTY)

	r = ((* key) = ftok( path, DFC_FTOK_SYM )) != DFC_ERROR;

DFC_R_PERROR( r )
DFC_PARAM_ERROR

	return r;

}/* dfc_ipc_ftok */

int dfc_ipc_shm_create( int * mem, char * * addr, key_t key, int size )
{
	int r = FALSE;

DFC_PARAM_CHECK( mem != NULL && (* mem) == DFC_ERROR &&
			addr != NULL && (* addr) == NULL &&
			key != DFC_ERROR && size > DFC_EMPTY )

	if (((* mem) = shmget(key, size, DFC_IPC_SHM_CREATE)) != DFC_ERROR ) {
		r = ((* addr) = shmat((* mem), NULL, DFC_EMPTY )) != NULL

#ifdef DFC_SHM_LOCK

		&& shmctl((* mem), SHM_LOCK, NULL ) != DFC_ERROR

#endif /*DFC_SHM_LOCK*/

		;
	}

DFC_R_PERROR( r )
DFC_PARAM_ERROR

	return r;

}/* dfc_ipc_shm_create */

int dfc_ipc_shm_setown( const int mem, const uid_t uid, const gid_t gid )
{
	int r = FALSE;

DFC_PARAM_CHECK( mem != DFC_ERROR )

	struct shmid_ds ds;

	if ( shmctl( mem, IPC_STAT, &ds ) != DFC_ERROR ) {
		ds.shm_perm.uid  = uid;
		ds.shm_perm.gid  = gid;
		ds.shm_perm.mode = DFC_IPC_SHM_SETOWN;
		r = shmctl( mem, IPC_SET, &ds ) != DFC_ERROR;
	}

DFC_R_PERROR( r )
DFC_PARAM_ERROR

	return r;

}/* dfc_ipc_shm_setown */

int dfc_ipc_shm_destroy( int * mem )
{
	int r = FALSE;

DFC_PARAM_CHECK( mem != NULL && (* mem) != DFC_ERROR )

	if (

#ifdef DFC_SHM_LOCK

		shmctl((* mem), SHM_UNLOCK, NULL ) != DFC_ERROR &&

#endif /*DFC_SHM_LOCK*/

		shmctl((* mem), IPC_RMID, NULL ) != DFC_ERROR ) {
			(* mem) = DFC_ERROR;
			r       = TRUE;
	}

DFC_R_PERROR( r )
DFC_PARAM_ERROR

	return r;

}/* dfc_ipc_shm_destroy */

int dfc_ipc_shm_attach( char * * addr, const key_t key, const int size )
{
	int r = FALSE;

DFC_PARAM_CHECK( addr != NULL && (* addr) == NULL && key != DFC_ERROR &&
			size > DFC_EMPTY )

	int m = DFC_ERROR;

	if (( m = shmget( key, size, DFC_IPC_SHM_SETOWN )) != DFC_ERROR ) {
		(* addr) = shmat( m, NULL, SHM_RDONLY );
		r = (* addr) != NULL;
	}

DFC_R_PERROR( r )
DFC_PARAM_ERROR

	return r;

}/* dfc_ipc_shm_attach */

int dfc_ipc_shm_detach( char * * addr )
{
	int r = FALSE;

DFC_PARAM_CHECK( addr != NULL && (* addr) != NULL )

	r = shmdt((* addr)) != DFC_ERROR;
	if ( r )
		(* addr) = NULL;
	else
		dfc_perror;

DFC_PARAM_ERROR

	return r;

}/* dfc_ipc_shm_detach */

#endif /*_IPC_*/

/* End of ipc.c */
