/*
 dfcd.c

 DFC -- Dedicated File Cache -- Main

 Copyright (c) 2000 Edward V. POPKOV
 Distributed under GPL, see COPYING for details

 "Edward V. POPKOV" <evpopkov@carry.netnet.lv>

 $Id: dfcd.c,v 0.5 2000/09/22 14:47:43 root Exp root $

 $Log: dfcd.c,v $
 Revision 0.5  2000/09/22 14:47:43  root
 Fixed permissions on pidfile

 Revision 0.4  2000/09/21 21:56:07  root
 Fixed `after file' bug

 Revision 0.3  2000/04/19 20:49:26  root
 Optimized code, fixed includes/defines

 Revision 0.2  2000/04/18 01:08:17  root
 Bugfix (buffer overrun) in dfcd

 Revision 0.1  2000/04/07 15:00:00  root
 Signal handling workaround, bugfixes, added sample conf, backup on upgrade

 Revision 0.0  2000/04/06 13:45:28  root
 Initial check-in

*/

#ifndef _DFCD_
#define _DFCD_

#include <dfc.h>

int    pidfile = FALSE;
int    shm_ary[ DFC_SHM_COUNT ];

void dfcd_errexit( const char * )
	__attribute__(( __regparm__( 1 ), __noreturn__ ));

void dfcd_errexit( const char * errmsg )
{
	dfc_err( errmsg );
	exit( EXIT_FAILURE );
}/* dfcd_errexit */

void dfcd_exit()
{
	int sp = DFC_BASE;

	while ( sp < DFC_SHM_COUNT && shm_ary[ sp ] != DFC_ERROR ) {
		dfc_ipc_shm_destroy( shm_ary + sp );
		sp++;
	}
		
	if ( pidfile ) dfc_dmn_cleanup( DFC_PID_FNAME );

	return;
}/* dfcd_exit */

int dfcd_to_shmem( int * mem, char * fname, const fstat_t fstat )
{
	int r = FALSE;

DFC_PARAM_CHECK( mem != NULL && (* mem) == DFC_ERROR &&
			len_c( fname ) > DFC_EMPTY )

	key_t   ky = DFC_ERROR;
	char  * bf = NULL;
	off_t   ft = fstat.fs_off + sizeof( ft );
	int     fd = DFC_ERROR;

	if ( dfc_ipc_ftok( &ky, fname )) {
		if ( dfc_ipc_shm_create( mem, &bf, ky, ft )) {
			if ( dfc_file_open( &fd, fname, O_RDONLY )) {
				memcpy((void *) bf, (void *) &ft, sizeof(ft));
				r = dfc_file_get( fd, ( bf + sizeof( ft )),
					fstat.fs_off );
				dfc_file_close( &fd );
				if ( ! r ) {
					dfc_ipc_shm_destroy( mem );
				} else {
					r = dfc_ipc_shm_setown((* mem),
						fstat.fs_uid,fstat.fs_gid ) &&
						dfc_ipc_shm_detach( &bf );
				}
			}
		}
	}

DFC_R_PERROR( r )
DFC_PARAM_ERROR

	return r;

}/* dfcd_to_shmem */

int dfcd_shm_fill( int * shma, const char * cname )
{
	int r = FALSE;

DFC_PARAM_CHECK( shma != NULL && len_c( cname ) > DFC_EMPTY )

	int       fd = DFC_ERROR;
	int       sp = DFC_BASE;
	char    * dn = NULL;
	DIR     * dr = NULL;
	char    * fn = NULL;
	fstat_t   fs;
	int       mi = DFC_ERROR;

	if ( dfc_file_open( &fd, cname, O_RDONLY )) {
		while ( dfc_cnf_entry( fd, &dn )) {
			if ( dfc_dir_open( &dr, dn )) {
				while ( dfc_dir_entry( dr, &fn, dn )) {

					fs.fs_off = DFC_ERROR;
					fs.fs_uid = ~ ( DFC_ROOTID );
					fs.fs_gid = ~ ( DFC_ROOTID );

					if ( ! dfc_file_isfile( fn, &fs )) {
						dfc_mem_free((void * *) &fn );
						continue;
					}

					if ((r = dfcd_to_shmem(&mi, fn, fs))) {
						shm_ary[ sp++ ] = mi;
						mi = DFC_ERROR;
					}

					dfc_mem_free((void *) &fn );

					if ( ! r || sp == DFC_SHM_COUNT )
						break;
				}

				r = dfc_dir_close( &dr ) && r;
				if ( ! r || sp == DFC_SHM_COUNT )
					break;
			}
		}
		dfc_file_close( &fd );
	}

DFC_R_PERROR( r )
DFC_PARAM_ERROR

	return r;

}/* dfcd_shm_fill */

int main()
{
	if ( atexit( dfcd_exit ) == DFC_ERROR )
		dfcd_errexit( "Unable to register at-exit routine" );

	{
		register int shmp = DFC_BASE;

		do {
			shm_ary[ shmp++ ] = DFC_ERROR;
		}
		while ( shmp < DFC_SHM_COUNT );
	}

	if ( ! dfc_dmn_bkground())
		dfcd_errexit( "Unable to fork into background" );

	if ( ! ( pidfile = dfc_dmn_writepid( DFC_PID_FNAME )))
		dfcd_errexit( "Unable to write pidfile" );

	if ( ! dfcd_shm_fill( shm_ary, DFC_CNF_FNAME ))
		dfcd_errexit( "Unable to create shmem areas" );

	dfc_dmn_loop = TRUE;
	dfc_dmn_conf = FALSE;

	if ( dfc_dmn_sigact( SIGTERM, TRUE, dfc_dmn_sigterm ) &&
		dfc_dmn_sigact( SIGHUP, FALSE, dfc_dmn_sighup )) {
		do {
			while ( dfc_dmn_loop )
				pause();

			if ( dfc_dmn_conf ) {

				register int sp = DFC_BASE;

				while ( sp < DFC_SHM_COUNT &&
						shm_ary[ sp ] != DFC_ERROR ) {
					dfc_ipc_shm_destroy( shm_ary + sp );
					sp++;
				}
				dfc_dmn_loop = dfcd_shm_fill( shm_ary,
								DFC_CNF_FNAME );
			}
		}
		while ( dfc_dmn_conf );
	}

	return EXIT_SUCCESS;

}/* main */

#endif /*_DFCD_*/

/* End of dfcd.c */
