/* do: "gcc -O2 -Wall -o update bdflush.c" to make the executable */

/*
 * v1.0 - released to Eric Youngdale 17 Jan 1994
 * v1.1 - added exit() statements after two loops, DEBUG, argv0 diags
 * v1.2 - added getopt and a buch of flags to set params
 * v1.3 - Call setsid after each fork, print version number.
 * v1.4 - Minor bugfix - sync_delay and flush_delay were not being used.
 * v1.5 - only sync & flush if euid != 0, added better help
 * v1.6 - added support for notebooks (experimental)
 *      -  2.0+ kernels no longer need calling bdflush( 0, 0 ) 
 *           (enable by defining symbol 'NEWKERNEL')
 *                                 Pavel Machek <pavel@elf.mj.gts.cz>
 */

/* Define symbol NEWKERNEL for 2.0+ kernels    */
#undef NEWKERNEL

/* Select this to set where spindown binaries go    */
#define RCPATH "/etc/rc/"

/* Spindown:
    This feature will be probably used primarily on notebooks, but I
    started developing it on normal desktop. Now I have notebook, too.

    How it works: 
    This thing computes time from last read from disk. If times gets
    higher than sleep_delay, RCPATH/rc.spindown is invoked, and
    background updating is stopped.

    After RCPATH/rc.spindown terminates, first read from disk will 
    cause this to think that disk is back up, invoke RCPATH/rc.spinup 
    and start writing changed buffers to disk regurally.

    Comments / thanx / bug reports / modifications to
                                   Pavel Machek <pavel@elf.mj.gts.cz>
 */

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <limits.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <linux/unistd.h>
#include <sys/wait.h>
#include <syslog.h>
#include <time.h>

#define environ __environ
/*
 * bdupdate: an attempt to unify the update and bdflush programs; this
 * should be a drop-in replacement for "update", but calls bdflush (if
 * it works) every 5 seconds, too.
 *
 * If invoked with an argument, it prints stats as-per bdflush.
 *
 * - alec <Alec.Muffett@UK.Sun.COM>
 */

_syscall2(int, bdflush, int, func, int, data);

static char * version_string = "1.6.2 "
#ifdef NEWKERNEL
                               "for kernels 2.0+"
#endif
                               ;

time_t time_spin_down;
int sleep_delay = 0;
int sleep_countdown, last_reads = 0, sleep_now = 0;
char is_spinned_down = 0;

char *bdparam[] =
{
    "Max fraction of LRU list to examine for dirty blocks",
    "Max number of dirty blocks to write each time bdflush activated",
    "Num of clean buffers to be loaded onto free list by refill_freelist",
    "Dirty block threshold for activating bdflush in refill_freelist",
    "Percentage of cache to scan for free clusters",
    "Time for data buffers to age before flushing",
    "Time for non-data (dir, bitmap, etc) buffers to age before flushing",
    "Time buffer cache load average constant",
    "LAV ratio (used to determine threshold for buffer fratricide).",
    (char *) 0
};

void
inststr(char *dst[], int argc, char *src)
{
    if (strlen(src) <= strlen(dst[0]))
    {
        char *ptr;

        for (ptr = dst[0]; *ptr; *(ptr++) = '\0');

        strcpy(dst[0], src);
    } else
    {
        /* stolen from the source to perl 4.036 (assigning to $0) */
        char *ptr, *ptr2;
        int count;
        ptr = dst[0] + strlen(dst[0]);
        for (count = 1; count < argc; count++) {
            if (dst[count] == ptr + 1)
                ptr += strlen(++ptr);
        }
        if (environ[0] == ptr + 1) {
            for (count = 0; environ[count]; count++)
                if (environ[count] == ptr + 1)
                    ptr += strlen(++ptr);
        }
        count = 0;
        for (ptr2 = dst[0]; ptr2 <= ptr; ptr2++) {
            *ptr2 = '\0';
            count++;
        }
        strncpy(dst[0], src, count);
    }
}

void
usage(argv)
     char ** argv;
{
    fprintf(stderr,"usage: %s [-012345678] [-d] [-s sync-delay] [-f flush-delay] [-vnw]\n",argv[0]);
    fprintf(stderr,"  -0  Max fraction of LRU list to examine for dirty blocks \n");
    fprintf(stderr,"  -1  Max number of dirty blocks to write each time bdflush activated \n");
    fprintf(stderr,"  -2  Num of clean buffers to be loaded onto free list by refill_freelist \n");
    fprintf(stderr,"  -3  Dirty block threshold for activating bdflush in refill_freelist \n");
    fprintf(stderr,"  -4  Percentage of cache to scan for free clusters \n");
    fprintf(stderr,"  -5  Time for data buffers to age before flushing \n");
    fprintf(stderr,"  -6  Time for non-data (dir, bitmap, etc) buffers to age before flushing \n");
    fprintf(stderr,"  -7  Time buffer cache load average constant \n");
    fprintf(stderr,"  -8  LAV ratio (used to determine threshold for buffer fratricide) \n");
    fprintf(stderr,"  -d  Display kernel parameters\n");
    fprintf(stderr,"  -f  Call flush this often (in seconds)\n");
    fprintf(stderr,"  -h  Give this help\n");
    fprintf(stderr,"  -s  If acting as update then call sync this often (in seconds)\n");
    fprintf(stderr,"  -v  Display version information\n");
    fprintf(stderr,"  -n  Sleep after this time in seconds (read docs!), enables spindown\n");
    exit(1);
}

int
num_reads(void)
{
   FILE *stat;
   int succ = 0;
   char s[160]; 
   int num1 = 0, num2 = 0, num3 = 0, num4 = 0;
   stat = fopen( "/proc/stat", "r" );
   /* old version didn't work on SMP systems!!! */
   while (!succ && !feof(stat)) {
      if (fscanf( stat, "disk_rio %d %d %d %d\n", &num1, &num2, &num3, &num4 )==4)
    succ = 1;
      else
    fgets( s, 160, stat );
   }
   if (!succ)
     syslog(LOG_ERR, "error reading /proc/stat" );
   fclose( stat );
   return num1+num2+num3+num4;
}

void
spin_it_down(int wait_pid_num)
{
   is_spinned_down = 1;

   if (wait_pid_num > 0) {
      /* wait for child to spin down disks */
      int ret;
      ret = waitpid(wait_pid_num, NULL, 0);
      if (ret == -1)
    syslog(LOG_ERR, "waitpid: %m");
      else
    if (ret != wait_pid_num)
      syslog(LOG_ERR, "waitpid returns unexpected value %i", ret);
   }
   last_reads = num_reads(); /* save last_reads */
}

void
spin_down( void )
{
   int pid;

   sleep_now = 0;

   if ((pid = fork()) == -1)
       syslog(LOG_ERR, "spindown can't fork: %m");
   else
       if (!pid) {
       execl( RCPATH "rc.spindown", RCPATH "rc.spindown", NULL );
       syslog(LOG_ERR, "spindown can't exec: %m");
       exit(1);
       } else {
       spin_it_down(pid);
       }
}

/*
   If we receive signal 1, we'll make disk spin down...
 */

void
hup_handler( int i )
{
sleep_now = 1;
signal(SIGHUP, &hup_handler);
}

void
spin_up( void )
{
    int pid;
    syslog(LOG_NOTICE, "spinup (after %imin of spindown)", (time(NULL)-time_spin_down)/60);
    if ((pid = fork()) == -1)
        syslog(LOG_ERR, "spinup can't fork: %m");
    else
        if (!pid) {
        execl( RCPATH "rc.spinup", RCPATH "/rc.spinup", NULL );
        syslog(LOG_ERR, "spinup can't exec: %m");
        exit(1);
    }
    is_spinned_down = 0;
}


int
main(int argc, char *argv[])
{
    int sync_delay = 30, flush_delay = 5;
    int i;
    int go = 0;
    int ch;

    /* shame on whomever wrote getopt - global variables */
    extern char * optarg;
    extern int optopt;

    /* Start by parseing command line options */
    while((ch = getopt(argc, argv, "vdhs:f:1:2:3:4:5:6:7:8:0:n:")) != EOF)
        switch(ch) {
    case 'h':
        usage(argv);
        break;
    case 'v':
    case 'd':
            /* print out the current parameters */
        printf("bdflush version %s\n", version_string);
        go = 1; 
        if( ch=='v' ) break;
        for (i = 0; bdparam[i]; i++)
            {
                int j;
                int data;
                int bdflag;
            
                bdflag = 2 + (i << 1);
            
                j = bdflush(bdflag, (int) &data);
            
                printf("%d: %5d %s\n", i, data, bdparam[i]);
            
                if (j)
                {
                    fprintf(stderr, "bdupdate:(%d) bdflush(%08x,&d)=%d %d\n",
                            i, bdflag, j, errno);
                    break;
                }
            }
            break;
    case 's':
        sync_delay = atoi(optarg);
            break;
    case 'f':
            flush_delay = atoi(optarg);
        break;
    case '1': case '2': case '3': case '4':
    case '5': case '6': case '7': case '8':
    case '0':

               if ((i = bdflush(1+((ch - '0' + 1) <<1 ), atoi(optarg))))
            {
                fprintf(stderr, "bdupdate: bdflush(%d,%d)=%d %d\n",1+((ch - '0') <<1 ), atoi(optarg), i, errno);
            exit(1);
            }

        break;
    case 'n':
            sleep_delay = atoi(optarg);
            if (!sleep_delay)
            {
                fprintf(stderr, "Wrong argument to -n\n" );
                exit( 1 );
            }
            break;
    case ':':
            fprintf(stderr, "Option -%c requires an argument!\n",
                    optopt);
        usage(argv);
            break;
        default:
            usage(argv);
            break;
        }

    if(go) exit(0);

    if(geteuid() != 0)    /* These 7 lines were repeated in bdflush-1.5 release. Why?!    */
    {
    bdflush(1, 0);
    sync();
    fprintf(stderr, "%s: not run as superuser, bdflush() and sync() called.\n",argv[0]);
    exit(0);
    }

    if (sleep_delay)
    {
        sleep_delay /= flush_delay;
        if (!sleep_delay) sleep_delay = 1;
        sleep_countdown = sleep_delay;
    }
#ifndef DEBUG
    for (i = 0; i < OPEN_MAX; i++)
    {
        close(i);
    }
#endif

    chdir("/");

    /* I didn't like idea of SIGTERM being ignored    */
    signal(SIGINT, SIG_IGN);
    signal(SIGCLD, SIG_IGN);
    signal(SIGHUP, &hup_handler);
#ifndef NEWKERNEL
    if (!fork())                /* fork off a daemon */
    {
        inststr(argv, argc, "bdflush (daemon)");
        setsid();  /* Release terminal */

          /* This call seems to be ignored in 2.0+ kernels    */
        i = bdflush(0, 0); /* should never return */
#ifdef DEBUG                    /* this should not happen */
        if (i)  
        {
            fprintf(stderr, "bdupdate: bdflush(0,0)=%d %d\n", i, errno);
        }
        fprintf( stderr, "bdflush(0,0) returned\n" );
#endif
        exit(1);
    }
#endif

    if (!fork())                /* fork off a daemon */
    {
        int has_bdflush = 1;

        inststr(argv, argc, "update (bdflush)");
        setsid();  /* Release terminal */

        openlog("update", LOG_CONS | LOG_PID, LOG_DAEMON );

        for (;;)                /* flush expired buffers */
        {
            if (has_bdflush)
            {
                sleep(flush_delay);
                if (sleep_delay)
                {
                    i = num_reads();
                    if (i != last_reads)
                    {
                        last_reads = i;
                        sleep_countdown = sleep_delay;
                        if (is_spinned_down) {
                            inststr(argv, argc, "update (bdflush)");
                            spin_up();
                        }
                    }
                    sleep(1);
                    if ((!is_spinned_down) && ((sleep_countdown && (!(--sleep_countdown))) || sleep_now)) {
                        inststr(argv, argc, "update (spindown)" );
			if (sleep_now)
			    syslog(LOG_NOTICE, "spindown (on explicit request)" );
			else
			    syslog(LOG_NOTICE, "spindown (after %imin idle)", sleep_delay*flush_delay/60);
                        spin_down();
			time_spin_down = time(NULL);
                    }
                }
                if ((!sleep_delay) || (!is_spinned_down))
                    if ((i = bdflush(1, 0)))
                    {
                        has_bdflush = 0;    /* oopsie! */

                        inststr(argv, argc, "update (sync)");
                        syslog( LOG_ERR, "Bdflush(1,0) returned %d/%d, falling back to sync.", i, errno );
                    }
            } else
                {
                    sleep(sync_delay);

                    sync();
                }
        }

        exit(1);
    }
    /* parent exits */
    return (0);
}
