/*
 * privdb.h  -  Header file for netboot library database interface
 *
 * Copyright (C) 2003-2007 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: privdb.h,v 1.13 2007/01/06 18:31:38 gkminix Exp $
 */

#ifndef PRIVDB_H
#define PRIVDB_H



/*
 *********************************************************************************
 *
 *		Database section name definitions
 *
 *********************************************************************************
 */


/*
 * Characters used in searching for a section name
 */
#define DBMATCH_SINGLE	'?'		/* Matches one single character */
#define DBMATCH_MULTI	'*'		/* Matches multiple characters */




/*
 *********************************************************************************
 *
 *		Database record and item definitions
 *
 *********************************************************************************
 */


/*
 * Database record item types
 */
enum itemtype {
	item_none,			/* empty item */
	item_string,			/* string value */
	item_integer,			/* integer number value */
	item_boolean,			/* boolean value */
};



/*
 * Definition of one database record item
 */
struct dbitem {
	char             *name;		/* name of item */
	enum itemtype     type;		/* type of item */
	union {
	   char    *s;			/* string value */
	   long     i;			/* integer value */
	   int      b;			/* boolean value */
	}                 val;
	struct dbitem    *next;		/* next item record */
};




/*
 *********************************************************************************
 *
 *		Database configuration definitions
 *
 *********************************************************************************
 */


/*
 * Open mode for database
 */
enum dbmode {
	dbmode_none,			/* no mode specified */
	dbmode_ro,			/* read-only */
	dbmode_rw			/* read-write */
};



/*
 * Types of database interfaces
 */
enum dbtype {
	dbtype_none,			/* no database type specified */
	dbtype_text,			/* text database */
	dbtype_bdb,			/* Berkeley db */
	dbtype_sql			/* SQL database */
};



/*
 * Configuration information for a text database
 */
struct dbtext_config {
	char            *fname;		/* file name */
	char            *lockfile;	/* name of lock file */
};



/*
 * Configuration information for Berkeley DB
 */
struct bdb_config {
	char            *fname;		/* file name */
};



/*
 * Configuration information for ODBC SQL database
 */
struct sqldb_config {
	char             *dsn;		/* data source name */
	char             *username;	/* user name */
	char             *password;	/* password */
	char             *systable;	/* name of system table */
	char             *paramtable;	/* name of parameter table */
	int               logintimeout;	/* login timeout */
	int               conntimeout;	/* connection timeout */
};




/*
 * Record to store database configuration information
 */
struct dbconfig {
	char            *name;		/* name of database configuration */
	enum dbtype      type;		/* type of database */
	enum dbmode      mode;		/* opening mode */
	union {
	  struct dbtext_config text;
	  struct bdb_config    bdb;
	  struct sqldb_config  sql;
	} config;
	struct dbconfig *next;
};




/*
 *********************************************************************************
 *
 *		Database handle definitions
 *
 *********************************************************************************
 */


/*
 * Maximum number of databases enclosed in one handle
 */
#define DBHANDLE_MAX	8



/*
 * Database flags
 */
#define DBFLAGS_GLOBAL	0x00FF		/* global database flags */
#define DBFLAGS_ADD	0x0001		/* database to add records to */
#define DBFLAGS_OPT	0x0002		/* database is optional */
#define DBFLAGS_RO	0x0004		/* open database readonly */

#define DBFLAGS_PRIVATE	0xFF00		/* private database flags */
#define DBFLAGS_FORCERO	0x0100		/* force database readonly */



/*
 * Action codes for cursor bookmark routine
 */
#define BOOKMARK_SET		0	/* set new mark */
#define BOOKMARK_JUMP		1	/* jump to mark */
#define BOOKMARK_RELEASE	2	/* release old mark */



/*
 * Routine to terminate processing of a database handle in case of error
 */
typedef void (*dbterm) __P((DBHDL handle));



/*
 * Database ID type
 */
#define DBID		voidstar
#define DBID_NULL	NULL



/*
 * Structure holding database handling routine pointers
 */
struct dbprocs {
	DBID           (*openptr)    __P((struct dbconfig *conf, int flags));
	int            (*findptr)    __P((DBID dbid, const char *recname,
								int first));
	int            (*nextptr)    __P((DBID dbid));
	int            (*prevptr)    __P((DBID dbid));
	int            (*firstptr)   __P((DBID dbid));
	int            (*lastptr)    __P((DBID dbid));
	char          *(*curposptr)  __P((DBID dbid));
	int            (*markptr)    __P((DBID dbid, int action));
	void           (*delptr)     __P((DBID dbid));
	struct dbitem *(*readptr)    __P((DBID dbid, char **recname));
	void           (*writeptr)   __P((DBID dbid, struct dbitem *items));
	void           (*addptr)     __P((DBID dbid, const char *recname,
							struct dbitem *items));
	void           (*releaseptr) __P((DBID dbid));
	void           (*closeptr)   __P((DBID dbid));
	char          *(*errmsgptr)  __P((DBID dbid, const char *recname,
						const struct dbitem *item));
};


/*
 * Structure holding the definitions for one database handle
 */
struct dbhandle {
	int              dbnum;			/* number of databases */
	DBID             dbids[DBHANDLE_MAX];	/* database id numbers */
	int              flags[DBHANDLE_MAX];	/* database flags */
	int              dbread;		/* current read database */
	int              dbmark;		/* marked database */
	dberr            errhandler;		/* error handler */
	dbterm           termhandler;		/* termination handler */
	char            *lastfind;		/* last found record name */
	struct dbconfig *configs[DBHANDLE_MAX];	/* database configurations */
	struct dbprocs  *procs[DBHANDLE_MAX];	/* procedure pointers */
	struct dbhandle *next;
};




/*
 *********************************************************************************
 *
 *		Database routines which are internal to the netboot library
 *
 *********************************************************************************
 */


/*
 * Section parser error routine
 */
typedef void (*secterr) __P((const char *msg, const char *recname,
						const struct dbitem *item));



/*
 * Routines for internal use by the netboot library
 */
extern int nblib_parse_sect __P((const char *recname,
					struct dbitem *itemlist,
					struct sectdef *sects,
					secterr errhandler));

extern void nblib_db_error __P((const char *msg, dberrclass class));

extern int nblib_db_match __P((const char *pattern, const char *string));


#endif

