/*
 * md5.c  -  MD5 checksum routines
 *
 * Copyright (C) 2004-2007 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: md5.c,v 1.6 2007/01/06 18:31:38 gkminix Exp $
 *
 *********************************************************************************
 *
 * This code implements the MD5 message-digest algorithm.
 * The algorithm is due to Ron Rivest.	This code was
 * written by Colin Plumb in 1993, no copyright is claimed.
 * This code is in the public domain; do with it what you wish.
 *
 * Equivalent code is available from RSA Data Security, Inc.
 * This code has been tested against that, and is equivalent,
 * except that you don't need to include two pages of legalese
 * with every copy.
 *
 *********************************************************************************
 */

#include <common.h>
#include <nblib.h>



/*
 * Structure holding an MD5 checksum context
 */
struct md5context {
	__u32 buf[4];
	__u32 bits[2];
	__u8  in[64];
};

static struct md5context *ctx = NULL;



/*
 * Forward function declaration
 */
static void md5transform __P((__u32 *buf, __u32 *in));



/*
 * Function to reverse byte order on big-endian machines
 * Note: this code is harmless on little-endian machines.
 */
#ifndef WORDS_BIGENDIAN
# define bytereverse(buf, len)
#else
static inline void bytereverse __F((buf, len), __u8 *buf AND size_t len)
{
  __u32 t;

  do {
	t = (__u32)(((__u16)(buf[3] << 8 | buf[2])) << 16 |
			((__u16)(buf[1] << 8 | buf[0])));
	*((__u32 *)buf) = t;
	buf += sizeof(__u32);
    } while (--len);
}
#endif



/*
 * Start MD5 accumulation. Set bit count to 0 and buffer to mysterious
 * initialization constants.
 */
void md5init __F_NOARGS
{
  /* Allocate new MD5 context structure */
  if (ctx != NULL)
	free(ctx);
  ctx = (struct md5context *)nbmalloc(sizeof(struct md5context));

  /* Initialize buffer constants */
  ctx->buf[0] = 0x67452301;
  ctx->buf[1] = 0xefcdab89;
  ctx->buf[2] = 0x98badcfe;
  ctx->buf[3] = 0x10325476;

  /* Reset bit count */
  ctx->bits[0] = 0;
  ctx->bits[1] = 0;
}



/*
 * Update context to reflect the concatenation of another buffer full
 * of bytes.
 */
void md5update __F((buf, len), __u8 *buf AND size_t len)
{
  size_t l;
  __u32 t;

  /* MD5 context has to be initialized */
  assert(ctx != NULL);

  /* Check for reasonable length */
  assert(len <= (0xffffffff >> 3));
  if (len == 0)
	return;

  /* Update bitcount */
  t = ctx->bits[0];
  ctx->bits[0] += (__u32)len << 3;
  ctx->bits[0] &= 0xffffffff;
  if (ctx->bits[0] < t)
	ctx->bits[1]++;
  ctx->bits[1] += ((__u32)len >> 29) & 0x00000007;
  l = (size_t)((t >> 3) % sizeof(ctx->in));

  /* Handle any leading odd-sized chunks */
  if (l > 0) {
	__u8 *p = &(ctx->in[l]);

	l = sizeof(ctx->in) - l;
	if (len < l) {
	    memcpy(p, buf, len);
	    return;
	}
	memcpy(p, buf, l);
	bytereverse(ctx->in, (sizeof(ctx->in) / sizeof(__u32)));
	md5transform(ctx->buf, (__u32 *)(ctx->in));
	buf += l;
	len -= l;
  }

  /* Process data in chunks of one input buffer */
  while (len >= sizeof(ctx->in)) {
	memcpy(ctx->in, buf, sizeof(ctx->in));
	bytereverse(ctx->in, (sizeof(ctx->in) / sizeof(__u32)));
	md5transform(ctx->buf, (__u32 *)(ctx->in));
	buf += sizeof(ctx->in);
	len -= sizeof(ctx->in);
  }

  /* Handle any remaining bytes of data. */
  memcpy(ctx->in, buf, len);
}



/*
 * Final wrapup - pad to 64-byte boundary with the bit pattern 
 * 1 0* (64-bit count of bits processed, MSB-first)
 */
void md5final __F((digest), __u8 *digest)
{
  size_t count;
  __u8 *p;

  /* MD5 context has to be initialized */
  assert(ctx != NULL);

  /* Compute number of bytes mod 64 */
  count = (size_t)((ctx->bits[0] >> 3) % sizeof(ctx->in));

  /* Set the first char of padding to 0x80 */
  p = &(ctx->in[count]);
  *p++ = 0x80;

  /* Bytes of padding needed to make 64 bytes */
  count = sizeof(ctx->in) - 1 - count;

  /* Pad out to 56 mod 64 */
  if (count < 8) {
	/* Two lots of padding: Pad the first block to 64 bytes */
	memset(p, 0, count);
	bytereverse(ctx->in, (sizeof(ctx->in) / sizeof(__u32)));
	md5transform(ctx->buf, (__u32 *)(ctx->in));
	/* Now fill the next block with 56 bytes */
	memset(ctx->in, 0, (sizeof(ctx->in) - 8));
  } else {
	/* Pad block to 56 bytes */
	memset(p, 0, count - 8);
  }
  bytereverse(ctx->in, ((sizeof(ctx->in) - 8) / sizeof(__u32)));

  /* Append length in bits and transform */
  p = &(ctx->in[sizeof(ctx->in) - 8]);
  ((__u32 *)p)[0] = ctx->bits[0];
  ((__u32 *)p)[1] = ctx->bits[1];
  md5transform(ctx->buf, (__u32 *)(ctx->in));
  bytereverse((__u8 *)(ctx->buf), (sizeof(ctx->buf) / sizeof(__u32)));
  memcpy(digest, ctx->buf, sizeof(ctx->buf));

  /* Free MD5 context structure */
  free(ctx);
  ctx = NULL;
}



/*
 * The four core functions
 */
#define F1(x, y, z)	(z ^ (x & (y ^ z)))
#define F2(x, y, z)	F1(z, x, y)
#define F3(x, y, z)	(x ^ y ^ z)
#define F4(x, y, z)	(y ^ (x | ~z))



/*
 * This is the central step in the MD5 algorithm
 */
#define MD5STEP(f, w, x, y, z, data, s) \
	( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x )



/*
 * The core of the MD5 algorithm, this alters an existing MD5 hash to
 * reflect the addition of 16 longwords of new data.  MD5Update blocks
 * the data and converts bytes into longwords for this routine.
 */
static void md5transform __F((buf, in), __u32 *buf AND __u32 *in)
{
  register __u32 a, b, c, d;

  a = buf[0];
  b = buf[1];
  c = buf[2];
  d = buf[3];

  MD5STEP(F1, a, b, c, d, in[0] + 0xd76aa478, 7);
  MD5STEP(F1, d, a, b, c, in[1] + 0xe8c7b756, 12);
  MD5STEP(F1, c, d, a, b, in[2] + 0x242070db, 17);
  MD5STEP(F1, b, c, d, a, in[3] + 0xc1bdceee, 22);
  MD5STEP(F1, a, b, c, d, in[4] + 0xf57c0faf, 7);
  MD5STEP(F1, d, a, b, c, in[5] + 0x4787c62a, 12);
  MD5STEP(F1, c, d, a, b, in[6] + 0xa8304613, 17);
  MD5STEP(F1, b, c, d, a, in[7] + 0xfd469501, 22);
  MD5STEP(F1, a, b, c, d, in[8] + 0x698098d8, 7);
  MD5STEP(F1, d, a, b, c, in[9] + 0x8b44f7af, 12);
  MD5STEP(F1, c, d, a, b, in[10] + 0xffff5bb1, 17);
  MD5STEP(F1, b, c, d, a, in[11] + 0x895cd7be, 22);
  MD5STEP(F1, a, b, c, d, in[12] + 0x6b901122, 7);
  MD5STEP(F1, d, a, b, c, in[13] + 0xfd987193, 12);
  MD5STEP(F1, c, d, a, b, in[14] + 0xa679438e, 17);
  MD5STEP(F1, b, c, d, a, in[15] + 0x49b40821, 22);

  MD5STEP(F2, a, b, c, d, in[1] + 0xf61e2562, 5);
  MD5STEP(F2, d, a, b, c, in[6] + 0xc040b340, 9);
  MD5STEP(F2, c, d, a, b, in[11] + 0x265e5a51, 14);
  MD5STEP(F2, b, c, d, a, in[0] + 0xe9b6c7aa, 20);
  MD5STEP(F2, a, b, c, d, in[5] + 0xd62f105d, 5);
  MD5STEP(F2, d, a, b, c, in[10] + 0x02441453, 9);
  MD5STEP(F2, c, d, a, b, in[15] + 0xd8a1e681, 14);
  MD5STEP(F2, b, c, d, a, in[4] + 0xe7d3fbc8, 20);
  MD5STEP(F2, a, b, c, d, in[9] + 0x21e1cde6, 5);
  MD5STEP(F2, d, a, b, c, in[14] + 0xc33707d6, 9);
  MD5STEP(F2, c, d, a, b, in[3] + 0xf4d50d87, 14);
  MD5STEP(F2, b, c, d, a, in[8] + 0x455a14ed, 20);
  MD5STEP(F2, a, b, c, d, in[13] + 0xa9e3e905, 5);
  MD5STEP(F2, d, a, b, c, in[2] + 0xfcefa3f8, 9);
  MD5STEP(F2, c, d, a, b, in[7] + 0x676f02d9, 14);
  MD5STEP(F2, b, c, d, a, in[12] + 0x8d2a4c8a, 20);

  MD5STEP(F3, a, b, c, d, in[5] + 0xfffa3942, 4);
  MD5STEP(F3, d, a, b, c, in[8] + 0x8771f681, 11);
  MD5STEP(F3, c, d, a, b, in[11] + 0x6d9d6122, 16);
  MD5STEP(F3, b, c, d, a, in[14] + 0xfde5380c, 23);
  MD5STEP(F3, a, b, c, d, in[1] + 0xa4beea44, 4);
  MD5STEP(F3, d, a, b, c, in[4] + 0x4bdecfa9, 11);
  MD5STEP(F3, c, d, a, b, in[7] + 0xf6bb4b60, 16);
  MD5STEP(F3, b, c, d, a, in[10] + 0xbebfbc70, 23);
  MD5STEP(F3, a, b, c, d, in[13] + 0x289b7ec6, 4);
  MD5STEP(F3, d, a, b, c, in[0] + 0xeaa127fa, 11);
  MD5STEP(F3, c, d, a, b, in[3] + 0xd4ef3085, 16);
  MD5STEP(F3, b, c, d, a, in[6] + 0x04881d05, 23);
  MD5STEP(F3, a, b, c, d, in[9] + 0xd9d4d039, 4);
  MD5STEP(F3, d, a, b, c, in[12] + 0xe6db99e5, 11);
  MD5STEP(F3, c, d, a, b, in[15] + 0x1fa27cf8, 16);
  MD5STEP(F3, b, c, d, a, in[2] + 0xc4ac5665, 23);

  MD5STEP(F4, a, b, c, d, in[0] + 0xf4292244, 6);
  MD5STEP(F4, d, a, b, c, in[7] + 0x432aff97, 10);
  MD5STEP(F4, c, d, a, b, in[14] + 0xab9423a7, 15);
  MD5STEP(F4, b, c, d, a, in[5] + 0xfc93a039, 21);
  MD5STEP(F4, a, b, c, d, in[12] + 0x655b59c3, 6);
  MD5STEP(F4, d, a, b, c, in[3] + 0x8f0ccc92, 10);
  MD5STEP(F4, c, d, a, b, in[10] + 0xffeff47d, 15);
  MD5STEP(F4, b, c, d, a, in[1] + 0x85845dd1, 21);
  MD5STEP(F4, a, b, c, d, in[8] + 0x6fa87e4f, 6);
  MD5STEP(F4, d, a, b, c, in[15] + 0xfe2ce6e0, 10);
  MD5STEP(F4, c, d, a, b, in[6] + 0xa3014314, 15);
  MD5STEP(F4, b, c, d, a, in[13] + 0x4e0811a1, 21);
  MD5STEP(F4, a, b, c, d, in[4] + 0xf7537e82, 6);
  MD5STEP(F4, d, a, b, c, in[11] + 0xbd3af235, 10);
  MD5STEP(F4, c, d, a, b, in[2] + 0x2ad7d2bb, 15);
  MD5STEP(F4, b, c, d, a, in[9] + 0xeb86d391, 21);

  buf[0] += a;
  buf[1] += b;
  buf[2] += c;
  buf[3] += d;
}

