/*
 * misc.c  -  Miscellaneous routines
 *
 * Copyright (C) 2002-2007 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: misc.c,v 1.9 2007/01/06 18:31:22 gkminix Exp $
 */

#define NEED_BINARY 1
#define NEED_DIR 1
#define NEED_TIME 1
#include <common.h>
#include <nblib.h>
#include "mknbi.h"
#include "dir.h"

#ifndef MKNBI_H_DOS
#error Included wrong header file
#endif



/*
 * Compare two host character strings
 */
static int cmpstr __F((s1, s2, n), const hchar_t *s1 AND const hchar_t *s2 AND size_t n)
{
  while (n > 0) {
	if (*s1++ != *s2++)
		return(FALSE);
	n--;
  }
  return(TRUE);
}



/*
 * Convert a string name into an array of host characters. Returns TRUE
 * if the string name fits into the buffer array.
 */
int cvtstr __F((hname, str, n),
				hchar_t *hname AND
				const char *str AND
				int n)
{
  const char *cp = str;
  int i = 0;
  hchar_t c;

  if (cp != NULL) {
	while (*cp && i < n) {
		if ((c = charcollect(*cp)) != 0)
			hname[i++] = c;
		cp++;
	}
  }
  while (i < n)
	hname[i++] = ' ';
  return(cp == NULL || !*cp);
}



/*
 * Compare a name with host characters against a string
 */
int cmpname __F((fsp, name, ext),
				struct fname_struct *namep AND
				const char *name AND
				const char *ext)
{
  hchar_t buf[8];

  /* Check file name */
  if (!cvtstr(buf, name, 8))
	return(FALSE);
  if (!cmpstr(namep->name, buf, 8))
	return(FALSE);

  /* Check file extension */
  if (!cvtstr(buf, ext, 3))
	return(FALSE);
  return(cmpstr(namep->ext, buf, 3));
}



/*
 * Find a file in the directory tree, or just in the current
 * directory.
 */
struct file_struct *findfile __F((dsp, recursive, name, ext),
				struct dir_struct *dsp AND
				int recursive AND
				const hchar_t *name AND
				const hchar_t *ext)
{
  struct file_struct *fsp = dsp->files;
  struct dir_struct *subdir;

  /* Search for file in current directory */
  if (!recursive) {
	while (fsp != NULL) {
		if (cmpstr(name, fsp->name.name, 8) &&
		    cmpstr(ext, fsp->name.ext, 3))
			return(fsp);
		fsp = fsp->next;
	}
	return(NULL);
  }

  /* Scan all directory subtrees */
  fsp = findfile(dsp, FALSE, name, ext);
  if (fsp == NULL) {
	subdir = dsp->subdirs;
	while (subdir != NULL) {
		fsp = findfile(subdir, TRUE, name, ext);
		if (fsp != NULL)
			break;
		subdir = subdir->next;
	}
  }
  return(fsp);
}



/*
 * Find a file according to it's name
 */
struct file_struct *findfname __F((dsp, recursive, fname),
				struct dir_struct *dsp AND
				int recursive AND
				const char *fname AND
				const char *fext)
{
  struct file_struct *ret = NULL;
  hchar_t name[8];
  hchar_t ext[3];

  if (cvtstr(name, fname, 8) && cvtstr(ext, fext, 3))
	ret = findfile(dsp, recursive, name, ext);
  return(ret);
}



/*
 * Get file modification time and convert it into MS-DOS format
 */
void gettime __F((mtime, date, time),
				time_t mtime AND
				unsigned int *date AND
				unsigned int *time)
{
  struct tm *tmp;

  tmp = localtime(&mtime);
  *time = tmp->tm_hour * 2048 + tmp->tm_min * 32 + tmp->tm_sec / 2;
  *date = (tmp->tm_year - 80) * 512 + (tmp->tm_mon + 1) * 32 + tmp->tm_mday;
}

