/*
 * boot.h  -  Definition of DOS boot sector
 *
 * Copyright (C) 2003-2007 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: boot.h,v 1.7 2007/01/06 18:31:22 gkminix Exp $
 *
 */

#ifndef HEADERS_BOOT_H
#define HEADERS_BOOT_H


/*
 * Check the compile system and turn structure packing on
 */
#include <compiler.h>
#include <pack.h>



/*
 * Layout of a DOS boot sector. The boot sector of mknbi-dos uses a private
 * boot sector signature to determine if the boot sector implements local
 * extensions to the BPB structures. mknbi-dos will then change the signature
 * back to the official value.
 */
#define BOOT_OFFSET	0x7C00		/* offset for boot block in segment 0 */
#define BOOT_MAGIC_OFS	0x01EE		/* offset to magic 7.1 compat. value */
#define BOOT_SIG_OFS	0x01FE		/* offset of boot signature */

#define BOOT_MAGIC	0x017F		/* magic word for 7.1 compatibility */
#define BOOT_SIG_PRIV	0x50A0		/* private boot signature */
#define BOOT_SIG	0xAA55		/* boot signature */

#define OEMNAME_LEN	8		/* length of OEM name */
#define VOLNAME_LEN	11		/* length of volume name */
#define FSNAME_LEN	8		/* length of file system name */

#define BOOT_FAT12_MAX	0x0fef		/* max clust # for 12 bit FAT */
#define BOOT_FAT16_MAX	0xffef		/* max clust # for 16 bit FAT */
#define BOOT_FAT12_NAME	"FAT12   "	/* FAT id for 12 bit FATs */
#define BOOT_FAT16_NAME	"FAT16   "	/* FAT id for 16 bit FATs */
#define BOOT_VOL_NAME	"NO NAME    "	/* default volume name */
#define BOOT_OEM_NAME	"MSDOS5.0"	/* default OEM name */


#ifndef _ASSEMBLY_

/* Official boot record */
struct boot_record {
  __u8          jmp_opcode[3];		/* jump opcode */
  __u8          oem_name[OEMNAME_LEN];	/* OEM name */
  __u16         bytes_per_sect;		/* bytes per sector */
  __u8          sect_per_cluster;	/* sectors per cluster */
  __u16         reserved_sect;		/* reserved sectors */
  __u8          fat_num;		/* number of FATs */
  __u16         dir_num;		/* dir entries */
  __u16         sect_num;		/* number of sectors */
  __u8          media_id;		/* media ID */
  __u16         sect_per_fat;		/* sectors per FAT */
  __u16         sect_per_track;		/* sectors per track */
  __u16         head_num;		/* number of heads */
  struct i_long hidden_num;		/* hidden sectors */
  struct i_long sect_num_32;		/* sects if part > 32MB */
  __u8          boot_id;		/* id of boot disk */
  __u16         dev_attr;		/* device attribute */
  struct i_long vol_num;		/* volume number */
  __u8          vol_name[VOLNAME_LEN];	/* volume name */
  __u8          fat_name[FSNAME_LEN];	/* FAT name */
} PACKED;


/* Private extensions to boot record */
struct priv_boot {
  struct boot_record boot_rec;		/* official boot record */
  __u16              io_sys_sects;	/* no of sectors in IO.SYS */
  __u16              io_sys_ofs;	/* load offset for IO.SYS */
} PACKED;

#else

			.struct	0
bootrec_jmp_opcode:	.struct	.+3		/* jump opcode */
bootrec_oem_name:	.struct	.+OEMNAME_LEN	/* OEM name */
bootrec_bpb:					/* BPB start in boot sector */
bootrec_bytes_per_sect:	.struct	.+2		/* bytes per sector */
bootrec_sect_per_clust:	.struct	.+1		/* sectors per cluster */
bootrec_reserved_sect:	.struct	.+2		/* reserved sectors */
bootrec_fat_num:	.struct	.+1		/* number of FATs */
bootrec_dir_num:	.struct	.+2		/* dir entries */
bootrec_sect_num:	.struct	.+2		/* number of sectors */
bootrec_media_id:	.struct	.+1		/* media ID */
bootrec_sect_per_fat:	.struct	.+2		/* sectors per FAT */
bootrec_sect_per_track:	.struct	.+2		/* sectors per track */
bootrec_head_num:	.struct	.+2		/* number of heads */
bootrec_hidden_num:	.struct	.+4		/* hidden sectors */
bootrec_sect_num_32:	.struct	.+4		/* sects if part > 32MB */
bootrec_boot_id:	.struct	.+1		/* id of boot disk */
bootrec_dev_attr:	.struct	.+2		/* device attribute */
bootrec_vol_num:	.struct	.+4		/* volume number */
bootrec_vol_name:	.struct	.+VOLNAME_LEN	/* volume name */
bootrec_fat_name:	.struct	.+FSNAME_LEN	/* FAT name */
bootrec_io_sys_sects:	.struct	.+2		/* no of sectors in IO.SYS */
bootrec_io_sys_ofs:	.struct	.+2		/* load offset for IO.SYS */

#endif /* _ASSEMBLY_ */



/*
 * Definitions for disk layout values. The standard values for a boot sector
 * are those for a 1.44 MB floppy disk.
 */
#define DISKID_FLOPPY	0x00		/* BIOS ID of first floppy disk drive */
#define DISKID_HD	0x80		/* BIOS ID of first hard disk drive */

#define MEDIAID_FLOPPY	0xF0		/* Media ID of a floppy disk */
#define MEDIAID_HD	0xF8		/* Media ID of a hard disk */

#define BOOT_SPC	1		/* sectors per cluster */
#define BOOT_SPFAT	9		/* sectors per FAT */
#define BOOT_SPTRK	18		/* sectors per track */
#define BOOT_FATNUM	2		/* number of FATs */
#define BOOT_HEADS	2		/* number of heads */
#define BOOT_DIRNUM	224		/* number of root dir entries */
#define BOOT_SECNUM	2880		/* total number of sectors */
#define BOOT_FORMAT	MEDIAID_FLOPPY	/* format ID for floppy disk */
#define BOOT_ID		DISKID_FLOPPY	/* BIOS id of boot floppy disk */



/*
 * Restore all definitions for data structure packing
 */
#include <unpack.h>

#endif

