/*
 * makerom.h  -  Main include file for makerom program
 *
 * Copyright (C) 1997-2007 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: makerom.h,v 1.19 2007/02/01 12:09:21 gkminix Exp $
 */

#ifndef MAKEROM_H
#define MAKEROM_H


/*
 **************************************************************************
 *
 *			General definitions
 *
 **************************************************************************
 */

/* Section name for systems database */
#define DBSECTNAME	"makerom"



/* Size of I/O buffers */
#define BLKSIZE		512



/* Length of PnP device identifier string */
#define PNPIDLEN	7



/*
 * Network driver types. These numbers have to correspond with the names
 * in the dtypes array.
 */
#define DRVTYPE_NONE	0
#define DRVTYPE_PD	1
#define DRVTYPE_NDIS	2
#define DRVTYPE_UNDI	3

#define DRVTYPE_MIN	1		/* lowest usable driver type */
#define DRVTYPE_MAX	3		/* highest usable driver type */



/*
 * Bus types. These numbers correspond with those defined in the PXE
 * specification plus 1 (to care for BUSTYPE_NONE). They also have to
 * correspond with the names in the btypes array.
 */
#define BUSTYPE_NONE	0
#define BUSTYPE_ISA	1
#define BUSTYPE_EISA	2
#define BUSTYPE_MCA	3
#define BUSTYPE_PCI	4

#define BUSTYPE_MIN	1		/* lowest usable bus type */
#define BUSTYPE_MAX	4		/* highest usable bus type */




/*
 **************************************************************************
 *
 *	Definitions used to describe a bootrom specification
 *
 **************************************************************************
 */
#define MAXPROGS	8		/* maximum number of driver programs */
#define MAXLOADERS	3		/* maximum number of loader programs */

#define OUT_NONE	0		/* no output file type set */
#define OUT_BINARY	1		/* output file as binary */
#define OUT_IHEX	2		/* output file as Intel hex */
#define OUT_MHEX	3		/* output file as Motorola hex */
#define OUT_THEX	4		/* output file as Tektronix hex */
#define OUT_BIOS	5		/* output file as BIOS include */
#define OUT_FLASH	6		/* output file as flash loader */
					/* OUT_FLASH has to come last! */
#define OUT_DEF		OUT_BINARY	/* default output file type */
#define OUT_MIN		OUT_BINARY	/* lowest usable output file type */
#define OUT_MAX		OUT_FLASH	/* highest usable output file type */


/* Structure holding all definitions for one loader */
struct loaderdef {
	char *name;			/* name of loader binary file */
	char *outname;			/* name of output file */
	int   outsize;			/* size of output file in kB */
	int   outtype;			/* type of output file (see above) */
	int   useint18;			/* nonzero if we should use int 18h */
	int   nobbs;			/* nonzero if to NOT use BBS */
	int   bootask;			/* nonzero if to ask for net boot */
	int   asktime;			/* timeout for asking user */
	int   devnum;			/* device number */
};


/* Structure holding definitions for loadable DOS programs */
struct progdef {
	int   prognum;			/* number of DOS programs */
	char *prognames[MAXPROGS];	/* DOS program file names */
	char *progargs[MAXPROGS];	/* DOS program arguments */
	long  minsizes[MAXPROGS];	/* minimum execution sizes */
	long  maxsizes[MAXPROGS];	/* maximum execution sizes */
};


/* Structure holding all definitions for packet driver loader */
struct pktdrvdef {
	int             drvindex;	/* index of packet driver */
	struct progdef  progs;		/* loadable DOS programs */
};


/* Structure holding all definitions for NDIS driver loader */
struct ndisdef {
	char           *protocolini;	/* protocol.ini file contents */
	int             drvindex;	/* index of packet driver */
	struct progdef  progs;		/* loadable DOS programs */
};


/* Structure holding all definitions for UNDI driver loader */
struct undidef {
	char *name;			/* name of UNDI driver file */
};


/* Structure holding all definitions for the network driver */
struct netdrvdef {
	char *name;			/* name of network driver file */
	int   drivertype;		/* type of network driver */
	int   driverflags;		/* network driver flags */
	union {
		struct pktdrvdef pd;	/* definitions for packet driver */
		struct ndisdef   ndis;	/* definitions for NDIS driver */
		struct undidef   undi;	/* definitions for UNDI driver */
	} driverdefs;
};


/* Structure holding all information for one bootrom specification */
struct bootdef {
	char            *name;			/* name of bootrom spec */
	char            *kernelname;		/* name of kernel file */
	char		*pnp_devid;		/* PnP device ID string */
	int              pci_vendid;		/* PCI vendor ID */
	int              pci_devid;		/* PCI device ID */
	int              bus_type;		/* bus type of network card */
	int              kernel_flags;		/* kernel flags */
	int		 canflash;		/* true if flash available */
	int              loadernum;		/* number of loader specs */
	struct loaderdef loaders[MAXLOADERS];	/* loader specifications */
	struct netdrvdef netdrv;		/* network driver specs */
};




/*
 **************************************************************************
 *
 *			External variables
 *
 **************************************************************************
 */
extern unsigned long serno;		/* bootrom serial number */




/*
 **************************************************************************
 *
 *			External routines
 *
 **************************************************************************
 */
extern struct bootdef *getdb __P((const char *name));
extern struct bootdef *getuser __P((void));
extern void putdb __P((const struct bootdef *bp));
extern void dopasses __P((const struct bootdef *bp));
extern void makehex __P((char *outname, int tempfile, int outtype));
extern void makeflash __P((char *outname, int tempfile, unsigned long ldsize));
extern int checkflash __P((const struct bootdef *bp));
extern char *instdrv __P((char *drvname, int drvtype));
extern unsigned long freeze __P((int infile, int outfile));
extern unsigned long donetdrv __P((const struct netdrvdef *np, 
					unsigned long *parareq,
					unsigned long *paratext,
					int outfile));


#endif

