/*
 * doconfig.c  -  Read configuration file for makerom
 *
 * Copyright (C) 1998-2007 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: doconfig.c,v 1.18 2007/02/01 12:09:21 gkminix Exp $
 */

#include <common.h>
#include <nblib.h>
#include "makerom.h"
#include "doconfig.h"



/*
 * Default path and file name definitions
 */
#define DRIVERFILE	"//netboot.drivers"
#define MD5SUMFILE	"//netdrvr/MD5SUMS"
#define BINDIR		"//binaries"
#define NETDRVDIR	"//netdrvr"
#define UTILSDIR	"//misc://utils"

#define INSTPROG	"instdrv"
#define KERNELFILE	"kernel.bin"
#define ROMFILE		"rom.bin"
#define FLOPPYFILE	"floppy.bin"



/*
 * Global variables
 */
struct confstruct config;		/* configuration information */



/*
 * Local variables
 */
static char *tmpdrvdbname = NULL;	/* temporary drivers file name */
static char *tmpmd5fname = NULL;	/* temporary MD5 sums file name */



/*
 * Definition of makerom:general section in configuration file
 */
static struct paramdef general_params[] = {
  { "driverdb",		par_string,	NULL,	{&tmpdrvdbname}},
  { "md5sumfile",	par_string,	NULL,	{&tmpmd5fname}},
  { "bindir",		par_string,	NULL,	{&config.bindir}},
  { "netdrvdir",	par_string,	NULL,	{&config.netdrvdir}},
  { "utilsdir",		par_string,	NULL,	{&config.utilsdir}},
  { "helpfile",		par_file,	NULL,	{&config.helpfile}},
  { "instprog",		par_file,	NULL,	{&config.instprog}},
  { "kernel",		par_file,	NULL,	{&config.files[FILE_KERNEL]}},
  { "rom",		par_file,	NULL,	{&config.files[FILE_ROM]}},
  { "floppy",		par_file,	NULL,	{&config.files[FILE_FLOPPY]}},
  { NULL,		par_null,	NULL,	{NULL}}
};

static struct sectdef confsects[] = {
  { "makerom:general",	general_params,	NULL,	NULL },
  { NULL,		NULL,		NULL,	NULL }
};



/*
 * Default values for configuration structure
 */
static const struct {
	char **cnfptr;
	char  *defstr;
} filedefaults[] = {
	{ &config.bindir,				BINDIR },
	{ &config.netdrvdir,				NETDRVDIR },
	{ &config.utilsdir,				UTILSDIR },
	{ &config.instprog,				INSTPROG },
	{ &config.files[FILE_KERNEL],			KERNELFILE },
	{ &config.files[FILE_ROM],			ROMFILE },
	{ &config.files[FILE_FLOPPY],			FLOPPYFILE },
	{ NULL,						NULL }
};



/*
 * Parse options and read configuration file
 */
void doconfig __F((argc, argv, opts),
				int argc AND
				char **argv AND
				struct cmdopt *opts)
{
  int i;

  /* Clearout configuration structure */
  memzero(&config, sizeof(config));

  /* Copy default strings into structure */
  for (i = 0; filedefaults[i].defstr != NULL; i++)
	copystr(filedefaults[i].cnfptr, filedefaults[i].defstr);

  /* Parse options and read configuration file */
  nbsetup(argc, argv, opts, confsects);

  /* Normalize main directory names */
  setpath(&config.bindir, nbhomedir);
  setpath(&config.netdrvdir, nbhomedir);
  setpath(&config.utilsdir, nbhomedir);

  /* Normalize name of help file */
  checkaccess(&config.helpfile, nbdatadir, ACCESS_FILE_READ);

  /* Normalize binary file names */
  for (i = 0; i < FILE_NUMBER; i++)
	checkaccess(&config.files[i], config.bindir, ACCESS_FILE_READ);

  /* Let the command line file name override the one from the config file */
  if (tmpmd5fname != NULL) {
	if (config.md5fname == NULL)
		config.md5fname = tmpmd5fname;
	else
		free(tmpmd5fname);
	tmpmd5fname = NULL;
  } else if (config.md5fname == NULL)
	copystr(&config.md5fname, MD5SUMFILE);

  /* Let the command line file name override the one from the config file */
  if (tmpdrvdbname != NULL) {
	if (config.drvdbname == NULL)
		config.drvdbname = tmpdrvdbname;
	else
		free(tmpdrvdbname);
	tmpdrvdbname = NULL;
  } else if (config.drvdbname == NULL)
	copystr(&config.drvdbname, DRIVERFILE);
}

