/*
 * bootrom.h  -  Memory layout definitions for the bootrom
 *
 * Copyright (C) 1995-2007 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: bootrom.h,v 1.12 2007/02/01 12:09:21 gkminix Exp $
 */

#ifndef SYSTEM_BOOTROM_H
#define SYSTEM_BOOTROM_H


/*
 **************************************************************************
 *
 * The memory layout of the bootrom looks like this:
 *
 *      +------------------+                     (00000:0)
 *      | Int & BIOS data  |    2kB
 *      +------------------+            LOWMEM   (01000:0)
 *      | OS loading area  |  variable
 *      +------------------+
 *      | Bootrom code     |  variable
 *      +------------------+            MEMEND   (0A000:0)
 *      | Video memory     |
 *      | Compressed ROM   |  384kB
 *      | BIOS             |
 *      +------------------+            EXTMEM   (10000:0)
 *      | OS loading area  |
 *      +------------------+            physical end of memory
 *
 *
 * During initialization and decompression of the bootrom code (either
 * out of the real rom or a floppy image), the lower OS loading area
 * looks like this:
 *
 *      +------------------+            DECOMPMEM (02000:0)
 *      | Decompressed ROM |  256kB
 *      +------------------+            FLOPMEM   (05000:0)
 *      | Floppy image     |  128kB
 *      +------------------+            FLOPEND   (07000:0)
 *
 * Note that FLOPEND should not exceed 07000:0, as the decompression code
 * puts itself right behind that address, and it requires 64kB of free
 * memory.
 */
#define LOWMEM		0x007C0	/* beginning of OS loading area	*/
#define DOSEXTMEM	0x01000	/* beginning of DOS extended area	*/
#define DECOMPMEM	0x02000	/* beginning of decompressed image	*/
#define DECOMPEND	0x05000	/* end of decompressed image area	*/
#define FLOPMEM		0x05000	/* beginning of floppy loading area	*/
#define FLOPEND		0x07000	/* end of floppy loading area		*/
#define OSENDMEM	0x08000	/* end of guaranteed OS loading area	*/
#define MEMEND		0x0A000	/* end of ram on typical PC system	*/
#define ROMSIZE		0x08000	/* max. size of rom area in bytes	*/
#define EXTMEM		0x10000	/* beginning of extended memory area	*/




/*
 **************************************************************************
 *
 * The ROM startup code, the kernel image and the network driver startup
 * code all need some values inserted at their beginning by the binary
 * patch program. Define the offsets to those values from the beginning
 * of each file.
 *
 * Note that these values have to correspond to the actual offsets
 * in the corresponding bootrom assembler files!
 *
 * The version number values in each file will be checked by the makerom
 * program. They should be at different offsets, so that an error will be
 * generated when an invalid file is fed into makerom.
 */
#define DATAOFS		0x00008	/* offset to special data	*/

/* Offsets into the ROM startup code */
#define ROMLENOFS	0x00002		/* offset to rom length byte	*/
#define ROMMAJOROFS	(DATAOFS +  0)	/* offset to major version num	*/
#define ROMMINOROFS	(DATAOFS +  1)	/* offset to minor version num	*/
#define ROMSEROFS	(DATAOFS +  2)	/* offset to rom serial num	*/
#define ROMVECTOFS	(DATAOFS +  6)	/* offset to bootrom vector	*/
#define ROMCOMPSIZEOFS	(DATAOFS +  8)	/* offset to compressed size	*/
#define ROMTOTSIZEOFS	(DATAOFS + 12)	/* offset to total size in para	*/
#define ROMPCIPTROFS	(DATAOFS + 16)	/* offset to ptr to PCI struct	*/
#define ROMPNPPTROFS	(DATAOFS + 18)	/* offset to ptr to PnP header	*/
#define ROMIDOFS	(DATAOFS + 20)	/* offset to ID string		*/
#define ROMBOOTFLGOFS	(DATAOFS + 28)	/* offset to boot flags		*/

/* Offsets into the kernel code */
#define KRNAPIENTRY	0x0000		/* offset to kernel API entry	*/
#define KRNLOADER	0x0003		/* offset to kernel loader code	*/
#define KRNTEXTOFS	(DATAOFS +  0)	/* offset to kernel text size	*/
#define KRNDATAOFS	(DATAOFS +  2)	/* offset to kernel data size	*/
#define KRNROMIDOFS	(DATAOFS +  4)	/* offset to kernel ROM ID ofs	*/
#define KRNSIZEOFS	(DATAOFS +  6)	/* offset to total kernel size	*/
#define KRNMAJOROFS	(DATAOFS +  8)	/* offset to major version num	*/
#define KRNMINOROFS	(DATAOFS +  9)	/* offset to minor version num	*/
#define KRNFLAGOFS	(DATAOFS + 10)	/* offset to kernel flags	*/

/* Offsets into the network driver code */
#define NETLOADER	0x0003		/* offset to driver loader code	*/
#define NETTEXTOFS	(DATAOFS +  0)	/* offset to driver text size	*/
#define NETDATAOFS	(DATAOFS +  2)	/* offset to driver data size	*/
#define NETROMIDOFS	(DATAOFS +  4)	/* offset to driver ROM ID ofs	*/
#define NETSIZEOFS	(DATAOFS +  6)	/* offset to total driver size	*/
#define NETMAJOROFS	(DATAOFS +  8)	/* offset to major version num	*/
#define NETMINOROFS	(DATAOFS +  9)	/* offset to minor version num	*/
#define NETBUSTYPEOFS	(DATAOFS + 10)	/* offset to bus type		*/
#define NETDEVIDOFS	(DATAOFS + 11)	/* offset to device ID		*/
#define NETFLAGOFS	(DATAOFS + 15)	/* offset to driver flags	*/

/* Offsets into the flash loader image */
#define FLOADSIZEOFS	0x0002		/* offset to size of flash load	*/




/*
 **************************************************************************
 *
 * ID string at the beginning of a netboot bootrom. This string has
 * to be 7 bytes long.
 */
#define NBROMID		"netboot"
#define NBROMIDLEN	7



/*
 **************************************************************************
 *
 * Bootrom loader flags:
 */
#define ROMFLG_ASKNET	0x0001		/* ask for network boot		*/
#define ROMFLG_ASKTIME	0x001E		/* timeout for keystroke	*/
#define ROMFLG_FLOADER	0x0020		/* flash loader installed	*/
#define ROMFLG_NOBBS	0x0040		/* dont support BBS		*/
#define ROMFLG_DEVNUM	0x0380		/* device number		*/

#define ROMFLG_SHIFT_ASKTIME	0	/* ASKTIME shift bits		*/
#define ROMFLG_SHIFT_DEVNUM	7	/* DEVNUM shift bits		*/



/*
 **************************************************************************
 *
 * Bootrom kernel flags:
 */
#define KRNFLG_NOUID	0x0001		/* avoid sending a client UID	*/
#define KRNFLG_OPTS	0x0002		/* send DHCP options request	*/
#define KRNFLG_NOBPEXT	0x0004		/* avoid to load BOOTP ext file	*/



/*
 **************************************************************************
 *
 * Network driver flags. These flags are driver dependant.
 */
#define DRVFLG_NORESET	0x0001		/* do not use hard reset	*/
#define DRVFLG_NOMODE0	0x0002		/* do not use receive mode 0	*/
#define DRVFLG_IRQPCI	0x0004		/* override IRQ with PCI value	*/


#endif

