#include <arpa/inet.h>
#include <ctype.h>
#include <netinet/in.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <termios.h>
#include <unistd.h>
#include "os.h"
#include "cmdline.h"
#include "modem.h"
#include "output.h"
#include "status.h"
#include "varlist.h"

#ifdef __linux__
#include <linux/vt.h>
#include <sys/ioctl.h>
#endif

int modem_signals;

extern char initstring[256];
extern int device_delay, timeout, online, remote, node;
extern varlist list;
extern vmodem *modem;

extern void timed_out();

void vmodem::sig(int s)
{
  struct termios state;
  
  tcgetattr(fd, &state);
  if(s)
    state.c_lflag |= ISIG;
  else
    state.c_lflag &=~ISIG;
 
  tcsetattr(fd, TCIOFLUSH, &state);
}

void vmodem::clean()
{
  struct termios state;
  
  tcgetattr(fd, &state);
  state.c_iflag &= ~(INLCR);
  state.c_oflag &= ~(ONLCR | ONLRET);

  // #%@$ !!!
  tcsetattr(fd, TCSAFLUSH, &state);
}

int vmodem::poll(int d)
{
  fd_set rdfd;
  struct timeval tv;
  
  FD_ZERO(&rdfd);
  FD_SET(fd, &rdfd);
  tv.tv_sec = 0;
  tv.tv_usec = d << 10;
  return select(fd + 1, &rdfd, NULL, NULL, &tv);
}

void vmodem::flush()
{
  fd_set rdfd;
  struct timeval tv;
  char c;
  int ret;
  
  for(;;) {
    FD_ZERO(&rdfd);
    FD_SET(fd, &rdfd);
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    ret = select(fd + 1, &rdfd, NULL, NULL, &tv);
    if(ret == 0)
      break; 
      
    if(ret == -1) // EINTR
      continue; 
  
    read(fd, &c, 1);
    dprintf(">flush: %c (%d)\n", c, c);
  }
}

int vmodem::read_char()
{
  fd_set rdfd;
  struct timeval tv;
  unsigned char c;
  int ret, timeouts = 0;
   
  for(;;) {
    FD_ZERO(&rdfd);
    FD_SET(fd, &rdfd);

    tv.tv_sec  = 1;
    tv.tv_usec = 0;
    ret = select(fd + 1, &rdfd, NULL, NULL, &tv);
      
    if(ret) {
      if(ret == -1)
	if(errno == EINTR)
	  continue;
        else
	  fatal_error("select returned error");
       
      ret = read(fd, &c, 1);
      if(ret <= 0)
        if(online)
	  raise(SIGHUP);
        else 
	  init_error("read error");

      return c;
    }

    if(remote || modem -> type != DEV_LOCAL) 
      timeouts++;
     
    if(timeouts == timeout)
      if(!online)
        return TIMEOUT;
      else
        timed_out();
  }
}

vmodem::~vmodem()
{
}

void vmodem::raw(char c)
{
  write(fd, &c, 1);
}

int vmodem::raw(char *s)
{
  int l = strlen(s);
  write(fd, s, l);
  return l;
}

int vmodem::wait4call()
{
  list.add_sys("baudrate", 0);
  return 0;
}

void vmodem::reset()
{
}

localmodem::localmodem(char *d)
{
  type = DEV_LOCAL;  
  device = (char *)malloc(32);
  strncpy(device, d, 31);
     
#if defined(__linux__) && defined(VT_OPENQRY) 
  int vt_num;   
  if(!strcmp(d, "auto")) {
    if(exists("firstvc"))
      sprintf(device, "/dev/tty%d", number("firstvc") + node);
    else { 
      if(ioctl(0, VT_OPENQRY, &vt_num))
        fatal_error("VT_OPENQRY failed");   
      sprintf(device, "/dev/tty%d", vt_num);
    }
    dprintf(">localmodem: using device %s\n", device);
    fp = fopen(device, "r+");
  }
  else
    fp = fopen(device, "r+");
#else
  fp = fopen(device, "r+");
#endif  
   
  if(!fp) {
    perror("fopen");
    fatal_error("could not open device \"%s\"", device);
  }
   
  fd = fileno(fp);
  setvbuf(fp, NULL, _IONBF, 0);
  initialized = 0;
}

int localmodem::init()
{ 
  if(!initialized) {  
    if(tcgetattr(fd, &state) || tcgetattr(fd, &oldstate))
      return 1;
  }
   
  /* off */
  state.c_iflag &= ~(PARMRK | INPCK | INLCR | IGNCR | ICRNL | IUCLC | IXON
		     | IXOFF | IXANY | IMAXBEL);
  state.c_oflag &= ~(OLCUC | OCRNL | ONOCR);
  state.c_cflag &= ~(CSTOPB | PARENB | PARODD | CLOCAL);
  state.c_lflag &= ~(ISIG | ICANON | ECHO | PENDIN);
		     
  /* on */
  state.c_cflag |= (CRTSCTS | CS8 | CREAD);
  state.c_iflag |= (IGNPAR | IGNBRK | INLCR);
  state.c_oflag |= (HUPCL | ONLCR | ONLRET);

//  if(modem_signals)
//    state.c_lflag |= ISIG;
   
  state.c_cc[VMIN] = 0;
  state.c_cc[VTIME] = 0;
   
  if(tcsetattr(fd, TCSAFLUSH, &state))
    return 1;

  initialized = 1;
   
  return 0;
}

void localmodem::restore()
{
  tcsetattr(fd, TCSAFLUSH, &oldstate);  
}

localmodem::~localmodem()
{
  free(device);
  dprintf(">resetting terminal state\n");
  restore();
  dprintf(">closing terminal\n");
  fclose(fp);
}

realmodem::realmodem(char *device)
{
  type = DEV_MODEM;
  initialized = 0;
  realmodem::device = device;
   
  dprintf(">opening modem: %s\n", device);
   
  fp = fopen(device, "r+");
  
  if(!fp) {
    perror("fopen");
    init_error("error opening modem");
  }
  
  fd = fileno(fp);

  dprintf(">modem fd = %d\n>disabling buffer\n", fd);
  setvbuf(fp, NULL, _IONBF, 0);
}

/* R I G H T ! 
 * We're going to initialize the modem, and we ain't taking no prisoners!
 */
int realmodem::init()
{
  if(!initialized) {
    if(tcgetattr(fd, &state) || tcgetattr(fd, &oldstate))
      return 1;
  }
   
  /* off */
  state.c_iflag &= ~(PARMRK | INPCK | IGNCR | ICRNL | IUCLC | IXON
		     | IXOFF | IXANY | IMAXBEL);
  state.c_oflag &= ~(OLCUC | OCRNL | ONOCR);
  state.c_cflag &= ~(CSTOPB | PARENB | PARODD | CLOCAL);
  state.c_lflag &= ~(ISIG | ICANON | ECHO | PENDIN);
		     
  /* on */
  state.c_cflag |= (CRTSCTS | CS8 | CREAD);
  state.c_iflag |= (IGNPAR | IGNBRK | INLCR);
  state.c_oflag |= (HUPCL | ONLCR | ONLRET);
  state.c_cc[VMIN] = 0;
  state.c_cc[VTIME] = 0;
  
  cfsetospeed(&state, list["lockedspeed"] -> get_i());
   
  if(tcsetattr(fd, TCSAFLUSH, &state))
    return 1;

  initialized = 1; 
  
  return 0;
}

void realmodem::reset()
{
  command(initstring);
 
  drain();
   
  if(!strstr(buf, "OK")) {
    fprintf(stderr, "modem response: %s\n", buf);  
    init_error("modem reset error");
  }
}

void realmodem::restore()
{
  cfsetospeed(&oldstate, B0);
  tcsetattr(fd, TCSAFLUSH, &oldstate);  
}

void realmodem::drain()
{
  int c, t, d;

  timeout = 60;
   
  c = read_char();
  if(c == TIMEOUT) {
    sprintf(buf, "(timeout)");
    return;
  }
   
  buf[0] = c;

  timeout = 1;
   
  for(t = 1;t < 200;t++) {
    c = read_char();
    if(c == TIMEOUT) 
      break;
    buf[t] = c;
  }

  buf[t] = 0;
   
  dprintf(">modem: ");
  for(d = 0;d < t;d++)
    if(iscntrl(buf[d]))
      dprintf("[%d]", buf[d]);
    else
      dprintf("%c", buf[d]);
  dprintf("\n");
}

void realmodem::command(char *cmd)
{
  int c;

  timeout = 5;
  dprintf(">modem command: %s ", cmd);
   
  while(*cmd) {
    raw(*cmd);
    dprintf("%c", *cmd);
    usleep(device_delay * 1000);
    c = read_char();
   
    if(c == TIMEOUT)
      init_error("modem command timeout");
    
    if(c != *cmd) {
      dprintf("*error*\n");
      drain();
      init_error("modem command error");
    }
     
    cmd++;
  }
   
  dprintf("\n");
   
  raw(13);
  read_char();
}

realmodem::~realmodem()
{
  dprintf(">resetting modem state\n");
  restore();
  dprintf(">closing modem\n");
  fclose(fp);
  usleep(1000 * 1000);
}

int realmodem::connect()
{
  char *speed, *caller, speed_buf[40], caller_buf[40], proto_buf[40], *proto;
  int t;
    
  drain();
  
  extern void connect_getty(char *);
  connect_getty(buf);

                     //1234567
  speed = strstr(buf, "CONNECT");
  if(!speed)
    return 1;
    
  speed += 8; //space
  if(isdigit(*speed)) {
    for(t = 0;t < 39 && isdigit(speed[t]);t++)
      speed_buf[t] = speed[t];
    speed_buf[t] = 0;
    list.add_sys("baudrate", atoi(speed_buf));
    proto = speed + t;    
    if(*proto == '/') {
      proto++;
      for(t = 0;t < 39 && proto[t] != ' ' && proto[t] != '\r' 
                && proto[t] != '\n' && proto[t] != 0;t++) 
        proto_buf[t] = proto[t];
      proto_buf[t] = 0;
      list.add_sys("mprotocol", proto_buf);
    }
  }
  else
    list.add_sys("baudrate", 300);
    
                      //123456789ABCDEF
  caller = strstr(buf, "CALLER NUMBER: ");
  if(caller) {
    caller += 15;
    for(t = 0;t < 39 && caller[t] != '\r';t++)
      caller_buf[t] = caller[t];
    caller_buf[t] = 0; 
    list.add_sys("caller", caller_buf);
  }

  return 0;
}

int realmodem::wait4call()
{
  int c;

  timeout = 60;
   
  while((c = read_char()) == TIMEOUT);
     
  drain();
  if(!strstr(buf, "ING"))
    return 1;

  log("incoming call");
  
  command("ATA");

  if(connect())
    return 1;

  log("connect ok, callerid=%s, speed=%d, protocol=%s", string("caller"), 
      string("baudrate"), string("mprotocol"));

  return 0;
}
