/* viewfile.h - header file for file viewer
 *
 *   do_read_file          - entry point for file viewer
 *   read_file             - read a file
 *
 *   rf_read_line          - read single line from file
 *   rf_lineout            - output single line from file to screen
 *   rf_get_current_line   - find current line
 *   rf_get_line_positions - read file line offsets
 *   rf_redraw_screen      - redraw screen
 *   rf_footer             - draw bottom line
 *   rf_wipe_footer        - clear bottom line
 *   rf_goto_top           - go to top of file and redraw
 *   rf_goto_end           - go to end of file and redraw
 *   rf_scroll_back        - scroll back "n" lines
 *   rf_scroll_forward     - scroll forward "n" lines
 *   rf_percent            - return position as a percentage
 *   rf_skipedit_back      - skip to previous edit
 *   rf_skipedit_forward   - skip to next edit
 *
 *   rf_edit_lock          - lock file
 *   rf_edit_unlock        - unlock file
 *
 *   rf_view_index         - [I] - view index of file
 *   rf_goto               - [G] - go to line / percentage
 *   rf_search             - [/] - find string
 *   rf_search_backwards   - [\] - find string
 *   rf_edit_file          - [.] - edit file
 *   rf_add_to_file        - [A], [E] - add to file
 *   rf_entry_copy         - [K] - copy from file
 *   rf_mail_file          - [M] - mail file
 *   rf_view_help          - [?] - view help
 *   rf_delete_entry       - [.][D] - delete entry
 *
 *   mview_edit_file       - run an editor on a file
 *
 * $Id: viewfile.h,v 1.1.1.1 1999/12/02 20:03:07 ivarch Exp $
 */

#ifndef _MT_VIEWFILE_H
#define _MT_VIEWFILE_H

#ifndef _MT_MENU_H
#include "menu.h"
#endif	/* _MT_MENU_H */

#ifndef _MT_LDB_H
#include "ldb.h"
#endif	/* _MT_LDB_H */

#define MFILE_LINE_NORMAL	0	/* normal file line */
#define MFILE_LINE_DATESTAMP	1	/* datestamp line */
#define MFILE_LINE_FROM		2	/* from line */
#define MFILE_LINE_TO		3	/* to line */
#define MFILE_LINE_SUBJECT	4	/* subject line */
#define MFILE_LINE_BLANK	5	/* blank or whitespace-only line */

struct Line_Array;		/* element of array of positions of lines */
struct RF_Data;			/* data necessary for reading a file */

typedef struct RF_Data * rf_data_t;


struct Line_Array {		/* element of array of positions of lines */
  long offset;				/* offset from start of file */
  time_t datestamp;			/* if non-zero, value of datestamp */
  unsigned long len;			/* line length */
  unsigned short type;			/* type of line */
};


struct RF_Data {		/* data necessary for reading a file */
  int fd;				/* file descriptor */
  long num_lines;			/* number of lines in file */
  time_t last_modified;			/* time file was last modified */
  time_t last_scanned;			/* time line list was last made */
  char linebuf [1024];			/* line buffer */
  char tmp [1024];			/* workspace */
  char search_buf [1024];		/* buffer for search string */
  struct Line_Array * line_pos;		/* array of line positions */
  struct LDB * ldb_entry;		/* pointer to LDB entry of file */
  char * title;				/* file title */
  char * commentfile;			/* commentfile to add to */
  int done;				/* 1 if finished reading */
  int aborted;				/* 1 if reading aborted */
  unsigned long flags;			/* file flags */
};

extern int rf_redraw;		/* flag, set if need to redraw */

extern char * current_user;	/* current username */

int do_read_file (ldb_t, char *, unsigned long, char *); /* entry point */

int read_file (ldb_t, char *, unsigned long, char *);	/* read a file */

void rf_read_line (rf_data_t, long);		/* read a line from the file */
void rf_lineout (rf_data_t, long);		/* output a line */
int  rf_get_current_line (rf_data_t);		/* find current line */
void rf_get_line_positions (rf_data_t);		/* store line positions */
void rf_redraw_screen (rf_data_t);		/* redraw screen */
void rf_footer (rf_data_t);			/* draw bottom line */
void rf_wipe_footer (rf_data_t);		/* undraw bottom line */
void rf_goto_top (rf_data_t);			/* go to top of file, redraw */
void rf_goto_end (rf_data_t);			/* go to bottom and redraw */
void rf_scroll_back (rf_data_t, int);		/* scroll back n lines */
void rf_scroll_forward (rf_data_t, int);	/* scroll forward n lines */
int  rf_percent (rf_data_t);			/* position as a percentage */
void rf_skipedit_back (rf_data_t);		/* skip to previous edit */
void rf_skipedit_forward (rf_data_t);		/* skip to next edit */

int  rf_edit_lock (char *);			/* lock file */
void rf_edit_unlock (void);			/* unlock file */

void rf_view_index (rf_data_t);			/* [I] view index of file */
void rf_goto (rf_data_t);			/* [G] go to a line / %age */
void rf_search (rf_data_t);			/* [/] find a string */
void rf_search_backwards (rf_data_t);		/* [\] find a string */
void rf_edit_file (rf_data_t);			/* [.] edit file */
void rf_add_to_file (rf_data_t, int);		/* [A] add to file */
void rf_entry_copy (rf_data_t);			/* [K] copy from file */
void rf_mail_file (rf_data_t);			/* [M] mail file to self */
void rf_view_help (rf_data_t);			/* [?] view help */
void rf_delete_entry (rf_data_t);		/* [.][D] delete entry */

void rf_fudge_linescan (rf_data_t, char *);	/* fudge file line scanning */

int  rf_scan_entries (rf_data_t, long *, char *, char *, int, char *);

int  mview_edit_file (char *, long, int, char *, char *);

#endif	/* _MT_VIEWFILE_H */

/* EOF */
