/* display.c - display functions
 *
 * $Id: display.c,v 1.5 2000/07/31 13:28:19 ivarch Exp $
 */

#include <stdio.h>
#include <string.h>
#include "mstring.h"
#include "terminal.h"
#include "viewfile.h"

extern int rf_viewing_messages;


/* Redraw the screen.
 */
void rf_redraw_screen (rf_data_t data) {
  long i, j;

  t_clear ();						/* clear screen */

  if (data->ldb_entry->current_line > data->num_lines)
    data->ldb_entry->current_line = data->num_lines;

  i = data->ldb_entry->current_line - (t_rows - 1);
  if (i < -2) i = -2;

  for (j = i; j < (i + (t_rows - 1)); j ++) rf_lineout (data, j);

  if (j > data->num_lines) j = data->num_lines;

  data->ldb_entry->current_line = j;

  rf_footer (data);
}


/* Wipe the bottom line menu.
 */
void rf_wipe_footer (rf_data_t data) {
  t_goto (0, t_rows - 1);
  t_clrtoeol ();
  t_goto (0, t_rows - 1);
}


/* Return the current position in the file, as a percentage.
 */
int rf_percent (rf_data_t data) {
  if (data->num_lines <= 0) return (100);
  return ((100 * data->ldb_entry->current_line) / data->num_lines);
}


/* Draw the bottom line menu.
 */
void rf_footer (rf_data_t data) {
  char tmp[128];
  char tmp2[128];
  int percent;

  percent = rf_percent (data);

  strcpy (tmp, "[\035BQ\035b]");
  if (percent == 100) strcat (tmp, "/[\035BRET\035b]");

  strcat (tmp, ":Exit  ");

  if ((rf_viewing_messages) && !(data->flags & MENU_STATUS_BARINFO)) {
    strcat (tmp, "[\035BS\035b]:Send/Reply  ");
  }

  if ((!rf_viewing_messages) && (!(data->flags & MENU_STATUS_ADD))
      && (!(data->flags & MENU_STATUS_EDIT)) && (percent == 100)) {
    strcat (tmp, "[\035BC\035b]:Continue  ");
  }

  if (data->flags & MENU_STATUS_ADD) strcat (tmp, "[\035BA\035b]:Add  ");

  if ((data->flags & MENU_STATUS_EDIT)
      || ((data->flags & MENU_STATUS_ADD)
          && !(data->flags & MENU_STATUS_BARDELETE)))
    strcat (tmp, "[\035B.\035b]:Edit  ");

  if (!(data->flags & MENU_STATUS_BARINFO))
    strcat (tmp, "[\035B?\035b]:Help  ");

  sprintf (tmp2, "(%d%%)", percent);
  strcat (tmp, tmp2);

  if (percent == 100) {
    if ((t_cols - mstrlen (tmp)) >= 8) {
      strcpy (tmp2, tmp);
      strcpy (tmp, "[\035B-\035b]:Up  ");
      strcat (tmp, tmp2);
    }
  } else {
    if ((t_cols - mstrlen (tmp)) >= 14) {
      strcpy (tmp2, tmp);
      strcpy (tmp, "[\035BSPACE\035b]:More  ");
      strcat (tmp, tmp2);
    }
  }

  t_goto ((t_cols - mstrlen (tmp)) / 2, t_rows - 1);
  t_write (0);
  t_write (tmp);
}


/* Output the given line from the current file, or a header if the line < 0.
 * Note that to display the file, 0 <= line < rf_num_lines.
 */
void rf_lineout (rf_data_t data, long line) {
  int a, b;

  if ((line >= data->num_lines) || (line < -2)) {
    t_lineout (0, 0);		/* line out of range - output blank line */
    return;
  }

  if (line == -2) {		/* top line - file header */

    data->linebuf[0] = 0;

    if (data->title) {
      b = (t_cols - mstrlen (data->title)) / 2;
      while ((b > 0)
             && (strlen (data->linebuf) < (sizeof (data->linebuf) - 16))) {
        strcat (data->linebuf, " ");
        b --;
      }
      strncat (data->linebuf, data->title, sizeof (data->linebuf) - 16);
      if (mstrlen (data->linebuf) >= t_cols) {
        data->linebuf[mstrindex (data->linebuf, t_cols - 1)] = 0;
        strcat (data->linebuf, "\035R>\035r");
      }
      strcat (data->linebuf, "\035a\035CA\035Ca");
    }

  } else if (line == -1) {	/* second line - header underline */

    strcpy (data->linebuf, "\035CB");
    for (b = 0;
         (b < t_cols) && (b < (sizeof (data->linebuf) - 16));
         b ++) data->linebuf[b+3] = '~';
    data->linebuf[b+3] = 0;
    strcat (data->linebuf, "\035CA");

  } else {			/* the rest - part of the file */

    rf_read_line (data, line);

    data->linebuf[sizeof (data->linebuf) - 16] = 0;	/* terminate string */

    a = 0;
    while (data->linebuf[a] != 0) {		/* remove naughty characters */
      if (((data->linebuf[a] < 32) && (data->linebuf[a] >= 0))
          || (data->linebuf[a] == 127)) {
        if (!miscntl(data->linebuf[a]) && (data->linebuf[a]!=8))
          data->linebuf[a] = ' ';
      }
      a ++;
    }

    if (data->line_pos[line].datestamp != 0) {		/* hit a datestamp */
      data->ldb_entry->last_datestamp = data->line_pos[line].datestamp;
      data->ldb_entry->datestamp_line = line;
    }

    if (mstrlen (data->linebuf) >= t_cols) {	/* chop long lines */
      a = mstrindex (data->linebuf, t_cols - 1);
      strcpy (data->linebuf + a, "\035a\035CA\035Ca\035R>\035r");
    }

  }

  t_lineout (data->linebuf, (data->flags & MENU_STATUS_ANONYMOUS) ? 1 : 0);
}

/* EOF */
