// $Id: misc.cc,v 1.5 1998/07/18 19:56:23 jvuokko Exp $
/*****************************************************************************
 * *
 * *      MODULE:     misc.cc
 * *                  -------
 * ***************************************************************************
 * *
 * *
 * *      COPYRIGHT (C) 1997 JUKKA VUOKKO
 * *      ALL RIGHTS RESERVED
 * ***************************************************************************
 * *
 * *      This module contains (independent) miscallenous functions.
 * *      Before use, initialize error handler function to call right
 * *      error handler by calling function
 * *      `set_misc_lib_error_handler( error_function_t *)'
 * *
 *****************************************************************************/

#include <stdio.h>      // sprintf(), rand(), srand(), tmpnam()
#include <time.h>       // time()
#include <stdlib.h>     // EXIT_FAILURE, EXIT_SUCCESS
#include <signal.h>
#include <string.h>     // strerror
#include <errno.h>

#ifdef __WATCOMC__      // Watcom stuff
#   include <io.h>         // access()
#   include <direct.h>
#elif defined (__EMX__)  // EMX-stuff
#   include <unistd.h>   // F_OK
#   include <sys/types.h>
#   include <dirent.h>
#else                    // UNIX
#   include <sys/types.h>
#   include <sys/wait.h>    // wait()
#   include <sys/utsname.h> // uname()
#   include <unistd.h>   // F_OK
#   include <dirent.h>
#   include <fcntl.h>    // open, write, creat, read
#endif

#include "system.h"
#include "misc.hh"

// Required literals:
//   DIR_SEP               -directory separator.
//   DOS_PATH              -if system uses \ in a path separator
//   LITTLE_ENDIAN_SYS     -if systems byte order is little endian (like intel)




//-------------------------------
// for set current diskdrive and copying file
#ifdef OS2
//#   define INCL_DOSERRORS
#   define INCL_ERRORS
#   define INCL_DOSFILEMGR
#   include <os2.h>
#endif

#if defined (WIN32) || defined (DOS)
#   include <dos.h>
#endif

#ifdef __unix__
#   include <unistd.h>  // getcwd, chdir
#endif

static void default_err_handler( const char* );

//****************** GLOBAL VARIABLES *************************************/




static error_function_t *error_handler = default_err_handler;


void
set_misc_lib_error_handler( error_function_t *err_handler )
{
        if ( err_handler != 0 ) {
                error_handler = err_handler;
        }
}


static void
default_err_handler( const char* errmsg )
{
        cerr << "\nERROR (misc library): " << errmsg << endl;
        abort();
}


/***************************************************************************
 * FUNCTION: correct_path
 ***************************************************************************
 *
 * DESCRIPTION: Converts path to unix- or dos-style and appends trailing
 *              '/' or '\' to end of path.
 *
 *              If first character of string is '~', then character will
 *              be replaced with contents of environment variable $HOME.
 *
 ***************************************************************************/
void
correct_path (String &str)
{
        char ch;
        int len;
        expand_path( str );
        len = str.length();
        ch = str.nget (len-1);
        if (ch != '/' && ch != '\\') {
                str += "/";
        }
#ifdef DOS_PATH
        str.replace_ch ('/', '\\');
#endif
}


//*************************************************************************/
// FUNCTION: expand_path
//*************************************************************************/
//
// Expands '~' in start if path if found. Original path will replaced with
// expanded one. Be CAREFULLY!
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : 
//   OUT: 
//
// RETURN: true if tilde-expansion was done, false if not.
//*************************************************************************/
bool
expand_path (String& path)
{
        char *home = NULL;
        int pos;

        if (path.length() == 0 || path[0] != '~') {
                return false;
        }

        pos = path.findch( DIR_SEP );

        if (path.length() >= 2 && pos == 1) {
                home = getenv( "HOME" );
        }

        if (!home) {
                return false;
        }

        String fullpath( home );

        if (pos == path.length()-1) {
                path = fullpath;
                return true;
        }

        if (fullpath.length() == 0 || fullpath.last() != DIR_SEP) {
                fullpath += DIR_SEP;
        }

        fullpath += path.substr( pos + 1 );

        path = fullpath;
        return true;
}

//*************************************************************************/
// FUNCTION: have_full_path
//*************************************************************************/
//
// Checks if given path or filename has full absolute path, i.e. path that
// starts from root of drive, like '/home/foobar/nnn' or 'C:\foo\bar\nnn'
//
// RETURN: true if path is absolute, false if path is relative
//*************************************************************************/
bool
have_full_path( const String& path )
{
        bool rc = false;
        
        String tmp = path;
        if ( tmp[0] == DIR_SEP ) {  // Check if path starts with '/'
                rc = true;
        } 
#ifdef DOS_PATH
        else if ( tmp[1] == ':' ) { // Check dos-style disk drive
                rc = true;
        }
#endif
        return rc;
}


//*************************************************************************/
// FUNCTION: get_int32
//*************************************************************************/
//
// Returns 32-bit unsigned integer value from byte_t array. Order of
// bytes in array must be little-endian.
//
//*************************************************************************/
unsigned int
get_le_int32 (byte_t *s)
{
        unsigned int value;
#ifdef LITTLE_ENDIAN_SYS
        value = *((unsigned int*) s);
#else
        value = (unsigned int) s[0] +
                ( ( (unsigned int) s[1] ) << 8 ) +
                ( ( (unsigned int) s[2] ) << 16 ) +
                ( ( (unsigned int) s[3] ) << 24 );
#endif
        return value;
}

//*************************************************************************/
// FUNCTION: put_le_int32
//*************************************************************************/
//
// This function puts an 32-bit little-endian integer value to given 
// character array. Size of array must be 4 bytes.
//
// RETURN: Pointer to character array
//*************************************************************************/
byte_t*
put_le_int32 (byte_t *s, unsigned int value)
{
#ifdef LITTLE_ENDIAN_SYS
        unsigned int *ss = (unsigned int*) s;
        *ss = value;
#else
        s[0] = (char) (value & 0xff);
        s[1] = (char) ((value >> 8) & 0xff);
        s[2] = (char) ((value >> 16) & 0xff);
        s[3] = (char) ((value >> 24) & 0xff);
#endif
        return s;
}


/***************************************************************************
 * FUNCTION: change_dir
 ***************************************************************************
 *
 * DESCRIPTION: Change to given directory. This understands driveletters.
 *
 *
 * RETURN : result of chdir
 ***************************************************************************/
int
change_dir (const char *dir)
{
        int rc;
        int l = strlen(dir);
        char *d = new char [l + 1];
        strcpy (d,dir);

        // on some platforms, chdir does not work if given directory
        // ends with separator / or \. So cut it out.
        if (d[l-1] == '\\' || d[l-1] == '/') {
                d[l-1] = '\0';
        }
        // for stubid systems, that have drive letter in front of path
#ifdef DOS_PATH
#   ifdef OS2
        ULONG drive;
#   else
        unsigned int drive;
        unsigned int total;
#   endif
        if (strchr (dir, ':') != NULL) {
                //  need to change disk drive
                drive = (unsigned) toupper (dir[0]);
                drive -= 64;
#   ifdef OS2
                DosSetDefaultDisk (drive);
#   else
                 _dos_setdrive (drive, &total);
#   endif

         }
#endif
        rc = chdir (d);
        delete[] d;
        return rc;
}


/***************************************************************************
 * FUNCTION: wc_access
 ***************************************************************************
 *
 * DESCRIPTION: Better access-function to Watcom C/C++.
 *
 *
 * RETURN : result of original access function
 ***************************************************************************/
#ifdef __WATCOMC__
int wc_access (char *path, int mode)
{
        char *ptr;
        char *buf;
        int i;
        buf = new char [strlen (path) + 1];
        strcpy (buf, path);
        ptr = strchr (buf, 0);
        ptr--;
        if (*ptr == '\\') {
                *ptr = 0;
        }
        i = access (buf, mode);
        delete[] buf;
        return i;
}
#endif


//**************************************************************************/
//  
//  FUNCTION: check_matches
// 
// 
//  DESCRIPTION: Checks if given string *a matches with pattern. Understands
//               ?, * and [Xx] wildcards.
// 
// 
//  EXTERNAL REFERENCES
//  IN :  -
//  OUT:  -
// 
//  RETURNS: true if matches.
// 
//**************************************************************************/
bool
check_matches( char *a, char* pattern )
{
        char *ptr = pattern;
        char *aptr = a;

        while (*ptr && *aptr) {
                switch (*ptr) {
                case '?':
                        ++aptr;
                        ++ptr;
                        break;
                case '[':
                        ++ptr;
                        while (*ptr) {
                                if (*ptr == ']') {
                                        return false;
                                }
                                if (*aptr == *ptr) {
                                        break;
                                }
                                ++ptr;
                        }
                        while (*ptr != ']' && *ptr) {
                                ++ptr;
                        }
                        ++ptr;
                        ++aptr;
                        break;
                case '*':
                        ++ptr;
                        if (*ptr != '\0') {
                                while (*aptr && *aptr != *ptr) {
                                        ++aptr;
                                }
                                if (*aptr == '\0') {
                                        return false;
                                }
                                ++aptr;
                                ++ptr;
                        } else {
                                // rest of string matches
                                return true;
                        }
                        break;
                default:
                        if (*ptr != *aptr) {
                                return false;
                        } else {
                                ++aptr;
                                ++ptr;
                        }
                }
                
        }
        if (*aptr != '\0' || (*ptr != '\0' && *ptr != '?' && *ptr != '*' )) {
                return false;
        }
        return true;
}

//*************************************************************************/
// FUNCTION: rm_files
//*************************************************************************/
//
// This function removes files that matches with given wildcard expression.
//
// NOTE: Under Os/2, this function is case sensitive.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : 
//   OUT: 
//
// RETURN: true if success, false if failed.
//*************************************************************************/
bool
rm_files ( String& expr )
{
        if (expr.is_empty() || expr.last() == '\\' || expr.last() == '/') {
                return false;
        }

        String path;
        String file_expr;
        String *str;
        int    i;
        DIR    *dir;
        struct dirent *ent;
        List<String> filelist;
        
        DEBUG("rm_files: Removing: "<<expr.get());
        i = expr.length() - 1;
        while (i && expr[i] != DIR_SEP ) {
                --i;
        }

        if ( i ) {
                path.copy( expr, 0, i );
        } else {
                path = ".";
        }
        file_expr.copy( expr, i + 1);

        // Read contents of the directory to a list
        if ( (dir = opendir( path.get() )) == NULL ) {
                String tmp = "Unable to open directory: " + path;
                error_handler( tmp.get() );
        }
        
        while ( (ent = readdir( dir )) != NULL ) {
                if (ent->d_name[0] != '.') {
                        if ( check_matches( ent->d_name, file_expr.get()) ) {
                                str = new String;
                                str->put( ent->d_name );
                                filelist.add( str );
                        }
                }
        }

        if ( closedir( dir ) ) {
                String tmp = "Unable to close directory : " + path;
                error_handler( tmp.get() );
        }

        path += DIR_SEP;
        if ( filelist.first() == true ) {
                String tmp;
                do {
                        assert( filelist.get() );
                        tmp = path +  *(filelist.get());
                        DEBUG("rm_files: Removing : "<<tmp );
                        remove( tmp.get() );
                } while ( filelist.next() == true );
        }
        filelist.destroy();

        return true;
}

/***************************************************************************
 * FUNCTION: copy_file
 ***************************************************************************
 *
 * DESCRIPTION: copy file to other. If destination file cannot be
 *              created then program is halted.
 *
 * RETURN : false if write failed (disk full).
 ***************************************************************************/
bool
copy_file (char *src, char *dest)
{
#if defined (__unix__)
        int src_handle, dest_handle;
        int bytes;
        bool rc = true;
        char *buf = new char [4096];

        src_handle = open (src, O_RDONLY);
        dest_handle = creat (dest, 0666);
        if (-1 == src_handle || -1 == dest_handle) {
                sprintf (buf, "Cannot copy file %s to %s.",src, dest);
                error_handler( buf );
        }
        DEBUG("Copying : "<<src<<" --> "<<dest);
        while ( (bytes = read (src_handle, buf, 4096)) > 0) {
                if (bytes > write (dest_handle, buf, bytes)) {
                        if ( bytes != -1 ) {
                                errno = ENOSPC;
                        }
                        rc = false;
                        break;
                }
        }
        delete[] buf;
        close( src_handle );
        close( dest_handle );
        return rc;
#elif defined (OS2)
        DEBUG("Copying : "<<src<<" --> "<<dest);
        APIRET rc = DosCopy ( (PSZ) src, (PSZ) dest, DCPY_EXISTING);
        if (rc == ERROR_DISK_FULL) {
                errno = ENOSPC;
                return false;
        } else if (rc) {
                char buf[1024];
                sprintf (buf, "Cannot copy file %s to %s.",src, dest);
                error_handler( buf );
        }
        return true;        
#else
        // not implemented for Win32 and DOS!
        fooooooooooo
#endif
}

/**************************************************************************
 * FUNCTION: get_tmpname
 **************************************************************************
 *
 * This function creates a unique name for temprary file. In Os/2 and DOS
 * environment variable TMP or TEMP is required.
 *
 * If name cannot be created, then error is reported and program is halted.
 *
 **************************************************************************/
char*
get_tmpname()
{
        char *ptr;
#ifdef __unix__
        ptr = tmpnam (NULL);
        if (NULL == ptr) {
                error_handler( "Cannot get unique name for a "\
                              "temporary file." );
        }
        return ptr;
#else
        // And suprise! MS-familian systems need again some extra work...
        
        static char *tmpname = 0;
        String current, tmpdir;
        
        if (tmpname != 0) {
                delete tmpname;
        }

        // get temporary directory
        
        char *tmp = getenv ("TMP");
        if (tmp == NULL) {
                char *tmp = getenv ("TEMP");
                if (tmp == NULL) {
                        error_handler( "Environment variable TMP is not set!");
                }
        }

        // store current directory
        char *cptr = getcwd (NULL, 0);
        if (cptr == NULL) {
                error_handler("Cannot get current directory entry");
        }

        // if getcwd() did not include driveletter, then get it here
        if (cptr[1] != ':') {
                DEBUG("Need to solve drive letter!");
#ifdef OS2
                ULONG drive, map;
                if (DosQueryCurrentDisk (&drive, &map)) {
                        error_handler("Cannot get current drive!");
                }
                drive += 64;
                DEBUG("Drive : "<< (char) drive);
                char letter[2];
                letter[0] = (char) drive;
                letter[1] = 0;
                current = letter;
                current += ":";
                current += cptr;
                DEBUG("Current : "<<current.get());
#else
                // not yet implemented!
                foobar_fooooo;
#endif
        } else {
                current = cptr;
        }
        tmpdir = tmp;

        // change to temporary directory
        if (-1 == change_dir (tmp)) {
                error_handler("Cannot change to temporary directory");
        }

        // try to create unique name for file
        ptr = tmpnam (NULL);
        
        if (NULL == ptr) {
                error_handler("Cannot get unique name for a "\
                              "temporary file." );
        }
        DEBUG("Tmp : "<<tmpdir.get());
        
        // back to current working directory
        DEBUG("Current : "<<current.get());
        if (-1 == change_dir (current.get())) {
                error_handler("Cannot return to current directory");
        }

        // create temporary file name including path
        tmpname = new char [strlen (tmp) + strlen (ptr) + 10];
        correct_path (tmpdir);
        strcpy (tmpname, tmpdir.get());
        strcat (tmpname, ptr);
        
        return tmpname;
#endif
}




//*************************************************************************/
// FUNCTION: _system
//*************************************************************************/
//
// Executes external command using fork() and execvp() in unix systems and
// system() in others.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : Pid
//   OUT: Pid
//
// RETURN: 0 if success. If failed, then errno is set correctly...
//*************************************************************************/
int 
_system( const char* cmd )
{
        assert( cmd );
        String tmp = cmd;
        int i = 0;

#ifdef __unix__        
        // build array from command line
        List<String>* args = tmp.split();
        char **argv = new char*[ args->count_items() + 1 ];
        do {
                argv[i++] = args->get()->c_str();
        } while (args->next());
        argv[i] = 0;
        
#ifdef USE_DEBUG
        i = 0;
        while (argv[i]) {
                DEBUG("ARGV " << i << ": "<<argv[i]);
                ++i;
        }
#endif /* USE_DEBUG */
        DEBUG("Executing: " << cmd );

        pid_t pid;
        static char buf[512];
        struct sigaction act, saved_sigquit, saved_sigint;
        act.sa_handler = SIG_IGN;
        act.sa_flags = 0;
        sigemptyset( &act.sa_mask );

           // ignore signals SIGINT and SIGQUIT
        if ( sigaction( SIGINT, &act, &saved_sigint ) < 0 ||
             sigaction( SIGQUIT, &act, &saved_sigquit )  < 0 ) {
                sprintf( buf, "Cannot set up signal handler: %s",
                         strerror( errno ) );
                error_handler( buf );
                exit( EXIT_FAILURE );
        }

           // block signal SIGCHLD
        sigset_t chldmask, savemask;
        sigemptyset( &chldmask );
        sigaddset( &chldmask, SIGCHLD );
        if ( sigprocmask( SIG_BLOCK, &chldmask, &savemask ) < 0 ) {
                error_handler( "Cannot block SIGCHLD!" );
                exit( EXIT_FAILURE );
        }


        if (0 == (pid = fork()) ) {
                   // restore signal handlers for child process and execute
                   // command.
                sigaction( SIGINT, &saved_sigint, NULL );
                sigaction( SIGQUIT, &saved_sigquit, NULL );
                sigprocmask( SIG_SETMASK, &savemask, NULL );

                i = execvp( argv[0], argv );
                error_handler("Execvp failed!! This should NEVER happen!");
                _exit( EXIT_FAILURE );
        } else if ( pid < 0 ) {
                error_handler( "Cannot create childprocess." );
                exit( EXIT_FAILURE );
        } else {
                int status;
                while ( waitpid( pid, &status, 0 ) < 0 ) {
                        DEBUG( "Error in waitpid" );
                        if ( EINTR == errno ) {
                                continue;
                        }
                        sprintf( buf, "Error during executing command: %s",
                                 strerror( errno ) );
                        error_handler( buf );
                        exit( EXIT_FAILURE );
                }
        }
        
           // restore signals...
        if ( sigaction( SIGINT, &saved_sigint, NULL ) < 0 ||
             sigaction( SIGQUIT, &saved_sigquit, NULL ) < 0 ||
             sigprocmask( SIG_SETMASK, &savemask, NULL ) < 0 ) {
                sprintf( buf, "Cannot restore signal handlers: %s",
                         strerror( errno ) );
                error_handler( buf );
        }

        DEBUG("exec status: " << i);
        delete args;
        delete[] argv;
        if (i > 0 ) {
                i = 0;
        }
#else  /* not __unix__ */
        i = system ( cmd );
#endif /* __unix__ */
        return i;
}


//
// Front end for read function, this handles blocks that are larger
// than SSIZE_MAX.
//
int
blockread( int fd, void *buf, size_t size )
{
           // FIXME: varmista toiminta kunnon testein!
        size_t n = size;
        int cnt = 0;
        char *pbuf = (char*) buf;
        int rc;
        while ( n > SSIZE_MAX ) {
                rc = read( fd, pbuf+cnt, SSIZE_MAX );
                if ( -1 == rc ) {
                        return -1;
                }
                cnt += rc;
                if ( rc < SSIZE_MAX ) {
                        n = 0;
                } else {
                        n -= SSIZE_MAX;
                }
        }
        if ( n > 0 ) {
                rc = read( fd, pbuf+cnt, n );
                return rc == -1 ? -1 : cnt + rc;
        }
        return cnt;
}

//
// Watcom C does not have popen() and pclose(). And it does not have
// pipe() which is needed to implemented popen. So this is very Q&D
// function that works somewhat similar like popen() in most situations.
// But this uses temporary file instead of pipe.
// NOTE: Nested calls are not allowed!!
#ifdef __WATCOMC__

// FIXME: replace these with something that is more re-entrant....
static FILE *pipefp;
static char *pipename;

FILE*
popen( const char* cmd, const char *mode )
{
        assert( !pipefp );
        if ( *mode != 'r' && *mode != 'w' ) {
                return 0;
        }

        char *pipefile = get_tmpname();
        if ( !pipefile ) {
                return 0;
        }

        String command = cmd;
        command += " > ";
        command += pipefile;

        DEBUG( "Popen uses command: " << command );

        _system( command.c_str() );

        FILE *fp = fopen( pipefile, mode );
        pipename = pipefile;
        pipefp = fp;
        return fp;
}

int
pclose( FILE *fp )
{
        assert( fp == pipefp );
        DEBUG( "Deleting pipe file" );

        int rc = fclose( fp );
        pipefp = 0;
        if ( pipename ) {
                DEBUG( "Removing: '" << pipename << "'" );
                remove( pipename );
                DEBUG( "Pipefile removed." );
        }
        pipename = 0;
        DEBUG( "pclose result: " << rc );
        return rc;
}

#endif
        
