// 	$Id: String.hh,v 1.6 1998/05/25 20:06:12 jvuokko Exp $	

/****************************************************************************
 * *
 * *  MODULE : String.hh
 * *
 * *  Copyright (c) 1996-1998 Jukka Vuokko <jvuokko@iki.fi> 
 * *  ALL RIGHTS RESERVED
 * *
 ****************************************************************************
 * *
 * *  Class for handling strings.
 * *
 * *
 * *  Pitisi dokumentoida kaikki toiminnot....
 * *
 * *
 ***************************************************************************/

#ifndef __STRING_CLASS_H__
#define __STRING_CLASS_H__

#include <string.h>
#include <ctype.h>
#include <iostream.h>


#include "my_types.h"
#include "List.hh"

// character sets
#define IBM_PC_CHARSET     0
#define ISO_LATIN_CHARSET  1
#define ASCII_7BIT_CHARSET 2
#define MACINTOSH_CHARSET  3

#define CHARACTER_SETS     4
#define CONVERT_CHARACTERS 11

#define NOTHING -1  // for backwards compatibilty
#define MAX_NUMBER_LEN 10

/**
 * Class for handling string.
 */
class String {
public:
           /**
            * Value npos marks invalid index.
            */
#ifdef NO_STATIC_CONST
        enum { npos = NOTHING };
#else
        static const int npos = NOTHING;
#endif

           /// Type for characer set.
        typedef enum charset {
                   /// IBM-PC character set.
                ibm_pc = 0,
                   /// ISO-8859.1 Character set.
                latin1 = 1,
                   /// 7Bit ascii character set.
                asc7bit   = 2,
                   /// Macintosh character set.
                mac    = 3 

        } charset_t;


           /** @name Constructors */
           //@{

           /**
            * Default constructor. Initialize empty string.
            */
        String ();

           /**
            * Initilize string using given c-style string.
            *
            * @param s Pointer to null terminated character array.
            */
        String (const char *s);

           /**
            * Copy constructor. Initialize string with contents of
            * a given string.
            *
            * @param src Reference to the source string.
            */
        String (String const &src);
           //@}
        
           /**
            * Get pointer to current c-string. 
            *
            * @return Pointer to null terminated c-string.
            */
        char* get () const { return str; }

           /**
            * Get pointer to current c-string. 
            *
            * @return Pointer to null terminated c-string.
            */
        char* c_str() const { return str; }

           /**
            * Allocate memory for given pointer *s and copy contents of 
            * the current string to it.
            *
            * @param s Pointer where string is to be copied.
            * @return Pointer to allocated string.
            */
        char* get (char **s) const;

           /**
            * Allocate character array and copy contents of string to it.
            * 
            * @return Pointer to allocated character array.
            */
        char* new_cstr() const {
                char *s = new char[len+1];
                strcpy( s, str );
                return s;
        }

           /**
            * Copy contents of the current string to the memory block pointed
            * by the pointer s. 
            * NOTE: {\em Memory is not allocated here, it is assumed
            * that s has enough room for whole string and null terminator.}
            *
            * @param s Pointer to memory block (big enough)
            * @return s
            */
        char* get_at (char *s) const { strcpy (s, str); return s; }

           /**
            * Copy first {\tt max } characters from the current string 
            * to the memory block pointed by the pointer s.
            * NOTE: {\em Memory is not allocated here, it is assumed
            * that s has enough room for whole string and null terminator.}
            *
            * @param s Pointer to memory block (big enough)
            * @return Pointer to the memory block (value of the s)
            */
        char* get_at (char *s, int max) const;

           /**
            * Get last character of the string.
            *
            * @return Reference to the last character.
            */
        char& last() { return str[len - 1]; }

           /**
            * Get character from given position.
            *
            * @param pos Index of character in the string.
            * @return Character or 0 if given index is out of range.
            */
        char nget (const int pos) const; 

           /**
            * Get length of the string.
            *
            * @return Number of characters in the string.
            */
        int length () const { return len; }

           /**
            * Get substring that contains first n characters of the 
            * current string.
            *
            * @param n Length of the substring
            * @return Substring
            */
        String substr (const int n) const;

           /** @name Appending */
           //@{
           /**
            * Append given string to the current string.
            *
            * @param src Reference to the string to append.
            * @return Reference to the current string
            */
        const String& append (String const &src);
           /**
            * Append given c-string to the current string.
            *
            * @param s Pointer to the c-string to append.
            * @return Reference to the current string
            */
        const String& append (const char *s);
           /**
            * Append first {\tt maxlen } characters from the c-string
            * to the current string.
            *
            * @param s Pointer to the c-string to append.
            * @param maxlen Maximum number of the appended characters.
            * @return Reference to the current string
            */
        const String& append (const char *s, const int maxlen);
           /**
            * Append given character to the current string.
            *
            * @param ch Character to append.
            * @return Reference to the current string
            */
        const String& append (const char ch);
           /**
            * Append given character to the end of the current string
            * {\tt count} times.
            *
            * @param ch Character to append.
            * @param count Number of copies of the character to append.
            * @return Reference to the current string
            */
        const String& append (const char ch, const int count);
           /**
            * Append given integer to the end of the current string. If given
            * integer has fewer digits than is specified in the parameter 
            * {\tt min\_len}, then leading zeroes are inserted to the
            * number. (e.g. if n is 7 and min\_len is 3, then appended value
            * will be 007).
            * 
            *
            * @param n Integer value to append.
            * @param min_len Minimum count of the digits in the integer.
            * @return Reference to the current string
            */
	const String& append (int n, int min_len = 1);
           //@}

           /** @name Operators */
           //@{
           /// Return character of the given index in the string.
        const char& operator [] (int n) const;
           /// Put given string to the current string.
        const String& operator=(String const &src);
           /// Put given c-string to the current string.
        const String& operator=(const char *s);
           /// Put given integer to the current string.
        const String& operator=(const int n);
           /// Append given string to the current string.
        const String& operator += (String const &src);
           /// Append given c-string to the current string.
        const String& operator += (const char *src);
           /// Append given character to the current string.
        const String& operator += (const char ch);
           /// Append given integer to the current string.
        const String& operator += (const int n);

           /// Concatenate two strings.
        friend String operator + (String const &str1, String const &str2);
           /// Concatenate string and c-string.
        friend String operator + (String const &str, const char *s);
           /// Concatenate c-string and String.
        friend String operator + (const char *s, String const &str);

           /// Equal operator for String and String.
        friend int operator == (String const &str1, String const &str2);
           /// Equal operator c-string and String.
        friend int operator == (const char *s, String const &str);
           /// Equal operator String and c-string.
        friend int operator == (String const &str, const char *s);
           /// NOT operator for String and String.
        friend int operator != (String const &str1, String const &str2);
           /// NOT operator for String and c-string.
        friend int operator != (String const &str, const char *s);
           /// NOT operator for c-string and String.
        friend int operator != (const char *s, String const &str);
           /// Less than operator for Strings
        friend int operator < (String const &str1, String const &str2);
           /// Less than operator for c-string and String 
        friend int operator < (const char *s, String const &str);
           /// Less than operator for String and c-string
        friend int operator < (String const &str, const char *s);
           /// Less than or equal operator.
        friend int operator <= (String const &str1, String const &str2);
           /// Less than or equal operator.
        friend int operator <= (String const &str, const char *s);
           /// Less than or equal operator.
        friend int operator <= (const char *s, String const &str);
           /// Greater than operator.
        friend int operator > (String const &str1, String const &str2);
           /// Greater than operator.
        friend int operator > (const char *s, String const &str);
           /// Greater than operator.
        friend int operator > (String const &str, const char *s);
           /// Greater than or equal operator.
        friend int operator >= (String const &str1, String const &str2);
           /// Greater than or equal operator.
        friend int operator >= (String const &str, const char *s);
           /// Greater than or equal operator.
        friend int operator >= (const char *s, String const &str);

           /**
            * Redirection operator. Redirect contents of string to the
            * ostream-object.
            */
        friend ostream& operator << (ostream &os, String const &str);
        
           //@}
        
           /** @name comparing */
           //@{
           /**
            * Compare current string to the given string.
            *
            * @param src Compared string.
            *
            * @return 0 if strings are equals. <1 if current string is 
            * less than given. >1 if given string is less than current.
            */
        int compare (String const &src) const;        
           /**
            * Compare current string to the given c-string.
            *
            * @param src Compared c-string.
            *
            * @return 0 if strings are equals. <1 if current string is 
            * less than given. >1 if given string is less than current.
            */
        int compare (const char *s) const;
           /**
            * Compare current string to given without case sensitivity.
            *
            * @param src Compared string.
            *
            * @return 0 if strings are equals. <1 if current string is 
            * less than given. >1 if given string is less than current.
            */
        int icompare (String const &src) const;
           /**
            * Compare string to given c-string without case sensitivity.
            *
            * @param src Compared c-string.
            *
            * @return 0 if strings are equals. <1 if current string is 
            * less than given. >1 if given string is less than current.
            */
        int icompare (const char *s) const;

           /**
            * Match given string to the pattern using regexp (or something
            * like it.) {\em This is not yet fully implemented }
            *
            * @param Experession in c-string.
            * @return true if matches, false if not.
            */
        bool match (const char *pattern) const;        
           //@}
        
           /** @name Initializing */
           //@{
           /**
            * Put contents of the given string to the current string.
            *
            * @param src Source string.
            * @return Length of the new string.
            */
        int put (String const &src);
           /**
            * Put contents of the given c-string to the current string.
            *
            * @param s Null terminated c-string.
            * @return Length of the new string.
            */
        int put (const char *s);               
           /**
            * Put given integer to the current string. If number has fewer
            * digits than min\_len is, then leading zeroes are insterted.
            *
            * @param n Integer to put.
            * @param min_len Minumum number of digits.
            * @return Length of the new string.
            */
        int put (int n, int min_len = 1);
           /**
            * Put first n characters from the given character array to the
            * current string.
            * WARNING: {\em If given array is shorter than l, then result
            * is unknown.}
            *
            * @param s Pointer to the character array.
            * @param l Maximum length of the new string.
            * @return Length of the new string.
            */
        int put (const char *s, const int l);
           /**
            * Put first n characters from the given byte array to the
            * current string.
            * WARNING: {\em If given array is shorter than l, then result
            * is unknown.}
            *
            * @param s Pointer to the byte array.
            * @param l Maximum length of the new string.
            * @return Length of the new string.
            */
        int put (const byte *s, const int l);
           //@}

           /** @name Copying */
           //@{
           /**
            * Copy contents of the given string to the current string. 
            * If the start position is not specified (s\_pos), then 
            * whole string is copied. Otherwise
            * copying is started from the position s\_pos.
            *
            * @param src Source string.
            * @param s_pos Start position in the source string.
            *
            * @return Reference to the current String.
            */
        const String& copy (String const &src, const int s_pos = 0);
           /**
            * Copy contents of the given c-string from position s\_pos 
            * to the current string. 
            * If the start position is not specified (s\_pos), then 
            * whole string is copied.
            *
            * @param src Source c-string.
            * @param s_pos Start position in the source string.
            *
            * @return Reference to the current String.
            */
        const String& copy (const char *s, const int s_pos = 0);

           /**
            * Copy max num characters from the given string to the
            * current string. Copying is started from the position s\_pos.
            *
            * @param src Source String.
            * @param s_pos Start position.
            * @param num Number of characters to copy.
            * @return Reference to the current String.
            */
        const String& copy (String const &src, const int s_pos, const int num);
           //@}

        void s_downcase();                // isoiksi tai pieniksi
        void s_upcase();
        String downcase() const;
        String upcase() const;

        String capitalize() const;
        void s_capitalize(); // muuten pieniksi, mutta sanojen ekat
        // kirjaimet isoiksi
        void cut_tail (const char ch=32);// poistaa tekstin lopusta vlilynnit
        void cut_first (const char ch=32); // poistaa alusta vlilynnit
        void ncut_first(const int n);
        void ncut_end (const int n); // poistaa lopusta n merkki.
        bool starts_with( const char *s );
        bool starts_with( const String& s );

           /** @name Finding */
           //@{
           /**
            * Find first occuring substring s. If substring found, then
            * index of it is returned. If not, then String::npos is returned.
            *
            * @param s c-string to find.
            * @return Index of substring or String::npos
            */
        int find (const char *s) const;
           /**
            * Find occuring substring s.  If substring found, then
            * index of it is returned. If not, then String::npos is returned.
            *
            * @param s String to find.
            * @return Index of substring or String::npos
            */
        int find (String const &src) const;

           /**
            * Find substring ignoring case sensitivity. 
            *
            * @param src Substring to find.
            * @return Index of substring or String::npos if not found.
            */ 
        int ifind (String const &src) const;

           /**
            * Find substring ignoring case sensitivity. 
            *
            * @param s Substring to find.
            * @return Index of substring or String::npos if not found.
            */ 
        int ifind (const char *s) const;

           /**
            * Find first occuring character {\tt *s} from current string.
            *
            * @param s Pointer to character to find.
            * @return Index of character or String::npos if not found.
            */
        int findch (const char *s) const;

           /**
            * Find first occuring character {\tt n} from current string.
            *
            * @param n Integer value of the character to find.
            * @return Index of character or String::npos if not found.
            */
        int findch (const int n) const;

           /**
            * Seek for any character that is in the given null terminated
            * character array {\tt list }. Seeks only withing first
            * {\tt max } characters of the current string.
            *
            * @param list Characters to find in array.
            * @param max  Max. position in string to use when seeking.
            * @return true if found, false if not.
            */
        bool find_any_ch (const char *list, const int max ) const; 
           //@}
        
           /**
            * Replace every character {\tt a}  with character <tt> b </tt>.
            *
            * @param a Character to be replaced.
            * @param b Character to replace with.
            * @return true if success, false if not.
            */
        bool replace_ch (const char a, const char b);

           /**
            * Count number of lines in the string. Actually counts number
            * of the line feeds in the string.
            *
            * @return Number of lines in the current string.
            */
        int count_lines () const;

           /**
            * Count length of the longest line within current string.
            * Ofcourse only when string contains more than on lines.
            *
            * @return Lenght of the longest line.
            */
        int max_line_len() const;

           /**
            * Get length of the all lines of the current string to the
            * integer array. The size of array will be same as number of
            * lines is. Memory for array is allocated in this method, but
            * it must free somewhere else.
            * NOTE: {\em index of first line in the array is 0}.
            *
            * @return Integer array that contains length of all lines.
            */
        int* get_all_lens() const;
        
           /**
            * Remove every occuring of the character {\tt ch } from the
            * current string.
            *
            * @param ch Character to remove.
            */
        void strip_ch (const char ch);

           /**
            * Checks whether string is empty or not.
            *
            * @return true if empty.
            */
        bool is_empty () const;

           /**
            * Checks if string contains only numbers.
            *
            * @return true if string contains only numbers.
            */
        bool is_number() const;

           /**
            * Convert character set of the string. The characterset is
            * converted from the {\tt src} to {\tt dest}.
            *
            * @param src  Source character set
            * @param dest Destination character set
            */
        void convert_charset (const int src, const int dest);

           /**
            *
            */
        List<String>* split( char fs = ' ' );

           /**
            *
            */
        void rot13();

           /**
            * Destructor. Frees all resources of the string.
            */
        ~String ();
private:
           /**
            * Pointer to null terminated string
            */
        char *str;
           /**
            * Length of the string 
            */
        int len;
};




#endif



