// 	$Id: String.cc,v 1.3 1998/05/15 21:19:46 jvuokko Exp $	

/****************************************************************************
 * *
 * *  MODULE : String.cc
 * *
 * *  Copyright (c) 1996-1998 Jukka Vuokko <jvuokko@iki.fi>
 * *  ALL RIGHTS RESERVED.
 * *
 ****************************************************************************
 * *
 * * Functions for handling String-object.
 * *
 * *
 * *
 * *
 * *
 ***************************************************************************/

#include "String.hh"


//*************************************************************************/
// FUNCTION: ch_tolower
//*************************************************************************/
//
// tolower, that handles extended pc and latin1 character sets.
//
//
// RETURN: integer value of character
//*************************************************************************/
static int
ch_tolower (const char s)
{
        unsigned char ch;
        int rc = (int) s;
        ch = (unsigned char) s;
        if (isalpha (ch)) {
                return tolower (ch);
        } 
        switch (ch) {
                // First test for IBMPC character set
        case 0x8f: //  A'
                rc = 0x86;
                break;
        case 0x8e: // A"
                rc = 0x84;
                break;
        case 0x99: // O"
                rc = 0x94;
                break;
                // Then test for ISO Latin-1
        case 0xc5: // A'
                rc = 0xe5;
                break;
        case 0xc4: // A"
                rc = 0xe4;
                break;
        case 0xd6: // O"
                rc = 0xf6;
                break;
        }
        return rc;
}


//*************************************************************************/
// FUNCTION: ch_toupper
//*************************************************************************/
//
// This version of toupper() works fine with characters .
//
// RETURN: value of character in upper case.
//*************************************************************************/
static int
ch_toupper (const char s)
{
        unsigned char ch;
        int rc = (int) s;
        ch = (unsigned char) s;
        if (isalpha (ch)) {
                return toupper (ch);
        } 
        switch (ch) {
                // First test for IBMPC character set
        case 0x86: //  A'
                rc = 0x8f;
                break;
        case 0x84: // A"
                rc = 0x8e;
                break;
        case 0x94: // O"
                rc = 0x99;
                break;
                // Then test for ISO Latin-1
        case 0xe5: // A'
                rc = 0xc5;
                break;
        case 0xe4: // A"
                rc = 0xc4;
                break;
        case 0xf6: // O"
                rc = 0xd6;
                break;
        }
        return rc;
}

//*************************************************************************/
// FUNCTION: strcmp_i
//*************************************************************************/
//
// My version of stricmp()-function. This works with string that contains
// characters  and 
//
//
// RETURN: <0 if string *a is less, 0 if strings are equal, >0 if *b are less.
//*************************************************************************/
static int
strcmp_i (const char *a, const char *b)
{
        register int aa,bb;
        aa = ch_tolower (*a);
        bb = ch_tolower (*b);                
        while (aa) {
                if (aa ^ bb)
                        break;
                a++;
                b++;
                aa = ch_tolower (*a);
                bb = ch_tolower (*b);                
        }
        return aa-bb;
}

//*************************************************************************/
// FUNCTION: int2asc
//*************************************************************************/
//
// Converts number <n> which max.lenght is MAX_NUMBER_LEN to c-string, and
// places string in given address <s>.
//
//*************************************************************************/
static void
int2asc (char *s, int n)
{
        char tmp[MAX_NUMBER_LEN+1]={0};
        char *ptmp=tmp;
        char *ps=s;
        char v=0;
        
        // is given number negative ?
        if (n<0) {
                *ps++='-';
                n=-n;
        }

        // kydn luku lpi, jolloin tmp:n sisllksi tulee
        // numeroita vastaavat ascii-arvot, kneisess jrjestyksess 
        while (1) {
                *ptmp++= (char) ((unsigned int) n % 10);
                if ((n= (unsigned int) n / 10) == 0)
                        break;
        }
   
        // kopioidaan tmp:n sislt pinvastaisessa jrjestyksess lopulliseen
        // stringiin, ja listn arvoihin 0-merkin ascii arvon verran 
        while (ptmp!=tmp) {
                v=*(--ptmp);
                *ps++ = (char) (v+'0');
        }
        *ps=0x00; // loppumerkki
}



/***************************************************************************
 * FUNCTION : String::String
 ***************************************************************************
 *
 * description : constructor for class String. Initializes string with
 *               null terminator.
 *
 ***************************************************************************/
String::String ()
{
        str = new char[1];
        *str = '\0';
        len = 0;
}

/***************************************************************************
 * FUNCTION : String::String
 ***************************************************************************
 *
 * description : Constructor. Initializes string using string *s.
 *
 *
 *
 * parameters : *s string to store.
 *
 ***************************************************************************/
String::String (const char *s)
{
        len = strlen (s);
        str = new char [len+1];
        strcpy (str, s);
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: String
// 
//  DESCRIPTION: Constructor. Initializes string using String& src.
// 
//**************************************************************************/
String::String (String const &src)
{
        if (this != &src) {
                len = src.len;
                str = new char [len+1];
                strcpy (str, src.str);
        }
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: get
// 
//  DESCRIPTION: Allocates space for pointer *s, and copy current string
//               to it.
// 
//  RETURNS: Current string in array *s.
//**************************************************************************/
char* String::get (char **s) const
{
        *s = new char [len +1 ];
        strcpy (*s, str);
        return *s;
}

char* String::get_at (char *s, int max) const
{
        strncpy (s, str, max);
        *(s + max) = 0x00;
        return s;
}


const char& String::operator [] (int n) const
{
        static char null = '\0';
        if (n >= len || n < 0) {
                return null;
        }
        return str[n];
}

/***************************************************************************
 * FUNCTION : String::append
 ***************************************************************************
 *
 * description : Appends given string to current string.
 *
 *
 *
 * parameters : String object.
 *
 *
 * return : appended object.
 ***************************************************************************/
const String &String::append (String const &src)
{
        char *tmp = new char [len + 1];
        strcpy (tmp, str);
        if (this == &src) { // self assigment (this replicates contents of str)
                delete[] str;
                str = new char [len + len + 1];
                strcpy (str, tmp);
                strcat (str, tmp);
        } else {
                delete [] str;
                str = new char [len + src.len + 1];
                strcpy (str, tmp);
                strcat (str, src.str);
        }
        delete [] tmp;
        len = strlen (str);
        return *this;
}
/***************************************************************************
 * FUNCTION : String::append
 ***************************************************************************
 *
 * description : Appends given c-string to current String.
 *
 *
 * parameters : Null terminated string.
 *
 * return : Refernce to current String.
 ***************************************************************************/
const String &String::append (const char *s)
{
        char *tmp = new char [len + 1];
        strcpy (tmp, str);

        if (s == str) { // self assigment
                delete [] str;
                str = new char [len + len + 1];
                strcpy (str, tmp);
                strcat (str, tmp);
        } else {
                delete [] str;
                str = new char [len + strlen(s) + 1];
                strcpy (str, tmp);
                strcat (str,s);
        }
        len = strlen (str);
        delete[] tmp;
        return *this;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: append
// 
//  DESCRIPTION: Appends character <ch> to end of string.
// 
// 
//  RETURNS: current string
//**************************************************************************/
const String &String::append (const char ch)
{
        char *tmp = new char [len + 2];
        strcpy (tmp, str);
        delete [] str;
        *(tmp+len++) = ch;
        *(tmp+len) = '\0';
        str = tmp;
        return *this;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: append
// 
//  DESCRIPTION: appends max <maxlen> characters fron string <*s> to end
//               of current string
// 
//  RETURNS: current string
//**************************************************************************/
const String& String::append (const char *s, const int maxlen)
{
        char *tmp = new char [len + 1];
        strcpy (tmp, str);
        if (str == s) { // self assigment
                delete[] str;
                str = new char [len + maxlen + 1];
                strcpy (str, tmp);
                strncpy (str+len, tmp, maxlen);
                str[len + maxlen] = 0x00;
        } else {
                delete [] str;
                str = new char [len + maxlen + 1];
                strcpy (str, tmp);
                strncpy (str+len, s, maxlen);
                str[len+maxlen] = '\0';
        }
        len = strlen (str);
        delete[] tmp;
        return *this;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: append
// 
//  DESCRIPTION: Appends character <ch> to end of string <count> times.
// 
//  RETURNS: current string
//**************************************************************************/
const String& String::append (const char ch, const int count)
{
        int i;
        char *tmp = new char [len + 1];
        strcpy (tmp, str);
        delete [] str;
        str = new char [len + count + 1];
        strcpy (str, tmp);
        for (i=0; i < count; i++) {
                str[len+i] = ch;
        }
        str[len+count] = '\0';
        len = strlen (str);
        delete[] tmp;
        return *this;
}

/***************************************************************************
 * FUNCTION : String::append
 ***************************************************************************
 *
 * description : Appedends number at end of string.
 *
 *
 *
 * parameters : Null terminated string.
 *
 *
 * return : Reference to current String 
 ***************************************************************************/
const String&
String::append (
                int n,      // number to append
                int min_len // minimum amount of digits in number. Default is 1
                )
{
        char *tmp = new char [len + 1];
        char number[MAX_NUMBER_LEN+1];
        int numlen,i;
        if (min_len >= MAX_NUMBER_LEN) {
                min_len = MAX_NUMBER_LEN;
        }
        strcpy (tmp, str);
        delete [] str;
        int2asc (number, n);
        numlen = strlen (number);

        // Insert possible zeros to start of number.
        if (numlen < min_len) {
                memmove (number + (min_len - numlen), number, numlen+1);
                for (i=0; i < (min_len - numlen); i++) {
                        *(number + i) = '0';
                }
        }
        
        str = new char [len + strlen(number) + 1];
        strcpy (str, tmp);
        strcat (str,number);
        len = strlen (str);
        delete[] tmp;
        return *this;
}

/***************************************************************************
 * FUNCTION : String::operator+=
 ***************************************************************************
 *
 * description : Same as String::append
 *
 ***************************************************************************/
const String &String::operator+=(String const &src)
{
        return append (src);
}

/***************************************************************************
 * FUNCTION : String::operator+=
 ***************************************************************************
 *
 * description : Same as String::append
 *
 ***************************************************************************/
const String &String::operator+=(const char *src)
{
        return append (src);
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: operator +=
// 
//  DESCRIPTION: Same as String::Append
// 
//**************************************************************************/
const String &String::operator+=(const char ch)
{
        return append (ch);
}

/***************************************************************************
 * FUNCTION : String::operator+=
 ***************************************************************************
 *
 * description : Same as String::append
 *
 ***************************************************************************/
const String& String::operator+=(const int n)
{
        return append (n);
}

/***************************************************************************
 * FUNCTION : String::compare
 ***************************************************************************
 *
 * description : Compares string &src with current string.
 *
 *
 *
 * parameters : String object.
 *
 *
 * return : 0, if strings are identical, <1 if current string is less than
 *             target string. >1 if target string is bigger than current
 *             string.
 ***************************************************************************/
int String::compare (String const &src) const
{
        return strcmp (str, src.str);
}

/***************************************************************************
 * FUNCTION : String::compare
 ***************************************************************************
 *
 * description : Compares current string with c-string.
 *
 *
 * parameters : Null terminated string
 *
 * return : Negative if s > this, Positive if this > s. 0 if equal
 ***************************************************************************/
int String::compare (const char *s) const
{
        return strcmp (str, s);
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: icompare
// 
//  DESCRIPTION: Compares string <src> with current string without case 
//               sensivity.
// 
// 
//  RETURNS: Negative if s > this, Positive if this > s. 0 if equal
//**************************************************************************/
int String::icompare ( String const &src) const
{
        return strcmp_i (str, src.str);
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: icompare
// 
//  DESCRIPTION: Ignores case and compares current string with string <*s>
// 
// 
//  RETURNS: Negative if s > this, Positive if this > s. 0 if equal
//**************************************************************************/
int String::icompare (const char *s) const
{
        return strcmp_i (str, s);
}

//
// Regexp. Hyvksyy seuraavat:
// ^ rivin alku (jos alussa)
// $ rivin loppu, jos lopussa
// . mik tahansa
// * edellist merkki 0 tai enemmn. Ei ymmrr tallennettuja \1 - \9.
// ? edellist 0 tai korkeintaan 1 kpl.
// + edellist pit olla vhintn 1 kpl.
// {n} tasan n kpl edellist
// {n,} n tai enemmn
// {,m} 0 - m kappaletta
// {n,m} n - m kappaletta
// [xyz] mik tahansa joukosta
// [^xyz] mik tahansa joukkoon kuulumaton
// [0-9]  numero vlilt 0-9
// \ Est seuraavaa toimimasta erikoismerkkin
// | yhdist regexpin toiseen
//

// this is not yet fully implemented :-)
bool String::match (const char *pattern) const
{
        if (*pattern == 0x00) {
                return false;
        }
        if (*pattern == '~') {
                if (ifind (pattern+1) == npos) {
                        return false;
                }
        } else if (find (pattern) == npos) {
                return false;
        }
        return true;
}


/***************************************************************************
 * FUNCTION : String::put
 ***************************************************************************
 *
 * description : Initializes current String with given string.
 *
 * parameters : String object
 *
 * return :  Length of the string. 
 ***************************************************************************/
int String::put (String const &src)
{
        if (this != &src) {
                delete[] str;
                len = src.len;
                str = new char [len + 1];
                strcpy (str, src.str);
        }
        return len;
}

/***************************************************************************
 * FUNCTION : String::put
 ***************************************************************************
 *
 * description : Initializes current String with given c-string.
 *
 * parameters : null terminated string
 *
 * return : Number of placed characters.
 ***************************************************************************/
int String::put (const char *s)
{
        if (str != s) {
                delete[] str;
                len = strlen(s);
                str = new char[len+1];
                strcpy (str, s);
        }
        return len;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: put
// 
//  DESCRIPTION: Replaces current string with given number. Inserts zeroes at
//               start of number, if number is too short.
//
// 
//  RETURNS: Number of placed characters (lenght of string).
//**************************************************************************/
int
String::put (
             int n,        // number to put
             int min_len)  // minimum amount of digits in number. Default is 1
{
        char number[MAX_NUMBER_LEN+1];
        int numlen,i;
        if (min_len >= MAX_NUMBER_LEN) {
                min_len = MAX_NUMBER_LEN;
        }
        delete [] str;
        int2asc (number, n);
        numlen = strlen (number);
        
        // Insert possible zeros to start of number.
        if (numlen < min_len) {
                memmove (number + (min_len - numlen), number, numlen+1);
                for (i=0; i < (min_len - numlen); i++) {
                        *(number + i) = '0';
                }
        }
        
        str = new char [strlen(number) + 1];
        strcpy (str,number);
        len = strlen (str);
        return len;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: put
// 
//  DESCRIPTION: Puts <l> characters from string <*s> to current string.
//
//  NOTE:    If string <*s> is shorter than <l>, then result is unknown.
// 
// 
//  RETURNS: Amount of placed characters
//**************************************************************************/
int String::put (const char *s, const int l)
{
        if (str == s) {
                // self assigment
                str[l] = 0x00;
                len = l;
        } else {
                int i;
                delete[] str;
                str = new char [l+1];
                len = l;
                for (i=0; i < l; i++) {
                        str[i] = s[i];
                }
                str[i] = 0;
        }
        return len;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: put
// 
//  DESCRIPTION: Puts <l> characters from unsigned 
//               char-array <*s>  to current string.
// 
//  RETURNS: Amount of placed characters
//**************************************************************************/
int String::put (const byte *s, const int l)
{
        int i;
        delete[] str;
        str = new char [l+1];
        len = l;
        for (i=0; i < l; i++) {
                str[i] = (char) s[i];
        }
        str[i] = 0;
        return len;
}

/***************************************************************************
 * FUNCTION : String::operator =
 ***************************************************************************
 *
 * description : Initialize current string using given c-string.
 *
 * parameters : null terminated string
 *
 * return : current String object.
 ***************************************************************************/
const String& String::operator=(const char *s)
{
        if (str == s) {
                return *this;
        }
        delete[] str;
        len = strlen(s);
        str = new char [len +1];
        strcpy (str, s);
        return *this;
}

/***************************************************************************
 * FUNCTION : String::operator =
 ***************************************************************************
 *
 * description : Initialize current string with given string
 *
 * parameters : String object
 *
 * return : current String object.
 ***************************************************************************/
const String& String::operator=(String const &src)
{
        if (&src == this) {
                return *this;
        }
        delete[] str;
        str = new char [src.len + 1];
        len = src.len;
        strcpy (str, src.str);
        return *this;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: operator =
// 
//  DESCRIPTION: Initialize current string with given integer.
// 
// 
//  RETURNS: current string
//**************************************************************************/
const String& String::operator=(const int n)
{
        char tmp[MAX_NUMBER_LEN+1];
        delete[] str;
        int2asc (tmp, n);
        str = new char [(len = strlen (tmp)) + 1];
        strcpy (str, tmp);
        return *this;
}

/***************************************************************************
 * FUNCTION : String::copy
 ***************************************************************************
 *
 * description : Copies contents of given string to current. If start
 *               position is not specified, then whole string is copied.
 *               (without s_pos, this is equal to string::put).
 *               If source is smaller than given start position, then
 *               nothing is copied.
 *
 * parameters : String object, position to start copy from.
 *
 *
 * return : current object.
 ***************************************************************************/
const String& String::copy (String const &src, const int s_pos)
{
        if (this == &src) {
                // Copy part of object itself.
                if ( s_pos > len ) {
                        delete[] str;
                        str = new char;
                        *str = 0x00;
                        len = 0;
                } else {
                        strcpy( str, str+s_pos );
                        len = len - s_pos;
                }
                return *this;
        }
        if (s_pos > src.len) {
                // if start position is bigger than source's length, then
                // remove contents of current string.
                delete[] str;
                str = new char;
                *str = 0x00;
                len = 0;
        } else {
                int i = src.len;
                delete[] str;
                len = i - s_pos;
                str = new char [len +1];
                strcpy (str, src.str+s_pos);
        }
        return *this;
}
/***************************************************************************
 * FUNCTION : String::copy
 ***************************************************************************
 *
 * description : Copies contents of given c-string from position <s_pos>
 *               to current string.
 *               
 * parameters : null terminated string, position to start copy from.
 *
 * return : current object.
 ***************************************************************************/
const String &String::copy (const char *s, const int s_pos)
{
        if (s == str) {
                // FIXME: muuta shiftaukseksi 
                return *this;
        }
        int i = strlen (s);
        delete[] str;
        if (s_pos <= i) {
                str = new char [(len = i - s_pos)  +1];
                strcpy (str, s+s_pos);
        } else {
                str = new char;
                *str = 0x00;
                len = 0;
        }
        return *this;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: copy
// 
//  DESCRIPTION: Copies <num> characters from string <src>, starting at
//               position <s_pos>.
// 
//  RETURNS: current string
//**************************************************************************/
const String& String::copy (String const &src, const int s_pos,
                            const int num)
{
        if (this == &src) {
                // FIXME
                return *this; // thnkin joku vippaskonsti!
        }
        if (s_pos > src.len) {
                delete[] str;
                str = new char;
                *str = 0x00;
                len = 0;
                return *this;
        } else {
                int i;
                delete[] str;
                i = src.len;
                len = num;
                str = new char [len + 1];
                strncpy (str, src.str+s_pos, len);
                str[len] = '\0';
		len = strlen (str);
                return *this;
        }
}

/***************************************************************************
 * FUNCTION : String::s_upcase
 ***************************************************************************
 *
 * description : Changes all characters of current string to upper case.
 *
 ***************************************************************************/
void String::s_upcase()
{
         for (char *ptr = str; *ptr != '\0';
             *ptr = char (::ch_toupper (*ptr)), ptr++);
}

/***************************************************************************
 * FUNCTION : String::s_downcase
 ***************************************************************************
 *
 * description : Changes all characters of current string to lower case
 *
 ***************************************************************************/
void String::s_downcase()
{
        for (char *ptr = str; *ptr != '\0';
             *ptr = char (::ch_tolower (*ptr)), ptr++);
}

//**************************************************************************/
// CLASS: String
// MEMBER FUNCTION: upcase
//**************************************************************************/ 
//
// Returns upper case version of current string
// 
//**************************************************************************/
String String::upcase() const
{
        String tmp = *this;
        for (char *ptr = tmp.str; *ptr != '\0';
             *ptr = char (::ch_toupper (*ptr)), ptr++);
        return tmp;
}
//**************************************************************************/
// CLASS: String
// MEMBER FUNCTION: downcase
//**************************************************************************/ 
//
// Returns down case version of current string
// 
//**************************************************************************/
String String::downcase() const
{
        String tmp = *this;
        for (char *ptr = tmp.str; *ptr != '\0';
             *ptr = char (::ch_tolower (*ptr)), ptr++);
        return tmp;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: s_capitalize
// 
//  DESCRIPTION: Sets first letter of each word in current string to
//               upper character.
// 
//**************************************************************************/
void String::s_capitalize()
{
        s_downcase();
        char *ptr = str;
        do {
                // skip over spaces
                while (*ptr && *ptr == ' ') {
                        ptr++;
                }
                if (*ptr) {
                        *ptr = (char) ::ch_toupper (*ptr);
                        ptr++;
                }
                // skip over rest of current word
                while (*ptr && *ptr != ' ') {
                        ptr++;
                }
        } while (*ptr);
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: capitalize
// 
//  DESCRIPTION: Returns String which is copy of current string, with the
//               first letter of each word set to uppercase and others set
//               to downcase.
// 
//**************************************************************************/
String String::capitalize() const
{
        String tmp = downcase();
        char *ptr = tmp.str;
        do {
                // skip over spaces
                while (*ptr && *ptr == ' ') {
                        ptr++;
                }
                if (*ptr) {
                        *ptr = (char) ::ch_toupper (*ptr);
                        ptr++;
                }
                // skip over rest of current word
                while (*ptr && *ptr != ' ') {
                        ptr++;
                }
        } while (*ptr);
        return tmp;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: cut_tail
// 
//  DESCRIPTION: Cuts all characters <ch> from end of string
// 
//**************************************************************************/
void String::cut_tail (const char ch)
{
        char *tmp;
        tmp = str;
        if (len > 0) {
                int i = len-1;
                while (tmp[i] == ch && i) {
                        tmp[i] = 0x00;
                        --i;
                }
                if (i == 0 && tmp[0] == ch) {
                        tmp[0] = 0x00;
                }
                str = new char [i+2];
                strcpy( str, tmp );
                //str[i] = 0x00;
                //while (i) {
                //        --i;
                //        str[i] = tmp[i];
                //}
                delete[] tmp;
                len = strlen (str);
        }
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: cut_first
// 
//  DESCRIPTION: Cuts all characters <ch> from start of string
// 
//**************************************************************************/
void
String::cut_first (const char ch)
{
        char *tmp;
        char *ptr = str;
        while (*ptr == ch && *ptr) {
                ++ptr;
        }
        len = strlen (ptr);
        tmp = new char [len+1];
        strcpy (tmp, ptr);
        delete[] str;
        str = tmp;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: ncut_first
// 
//  DESCRIPTION: Cuts n characters from start of string
// 
//**************************************************************************/
void
String::ncut_first(const int n)
{
    char *tmp;
    char *ptr = str + (n>len?len:n);
    len = strlen(ptr);
    tmp = new char[len+1];
    strcpy(tmp, ptr);
    delete[] str;
    str = tmp;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: ncut_end
// 
//  DESCRIPTION: Cuts <n> characters from end of string
// 
//**************************************************************************/
void
String::ncut_end (const int n)
{
        if (len < n) {
                return;
        }
        str[len-n] = '\0';
        len = strlen (str);
}

/***************************************************************************
 * FUNCTION : String::find
 ***************************************************************************
 *
 * description : Returns index to first occuring substring <s>.
 *
 *
 *
 * parameters : null terminated string.
 *
 *
 * return : Index to founded string. If string not found, returns npos.
 ***************************************************************************/
int String::find (const char *s) const
{
        char *ptr;
        if ((ptr = strstr (str, s)) != NULL) {
                return int (ptr) - int (str);
        } else {
                return npos;
        }
}

/***************************************************************************
 * FUNCTION : String::find
 ***************************************************************************
 *
 * description : Returns index of first occuring substring <src>
 *
 * parameters : String object.
 *
 * return : npos if substring not found.
 ***************************************************************************/
int String::find (String const &src) const
{
        char *ptr;
        if ((ptr = strstr (str, src.str)) != NULL) {
                return int (ptr) - int (str);
        } else {
                return npos;
        }
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: ifind
// 
//  DESCRIPTION: Find without case sensivity.
// 
//  RETURNS: Index of first letter of string if or npos if not found.
//**************************************************************************/
int String::ifind (String const &src) const
{
        String tmp, tmp2;
        tmp = *this;
        tmp2 = src;
        tmp.s_downcase();
        tmp2.s_downcase();
        return (tmp.find (tmp2));
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: ifind
// 
//  Non case-sensitive find.
// 
//  RETURNS: Index of first letter of string if or npos if not found.
//**************************************************************************/
int String::ifind (const char *s) const
{
        String tmp, tmp2;
        tmp = *this;
        tmp2 = s;
        tmp.s_downcase();
        tmp2.s_downcase();
        return (tmp.find (tmp2));
}


/***************************************************************************
 * FUNCTION : String::findch
 ***************************************************************************
 *
 * description : Returns index to given character.
 *
 * parameters : character to find.
 *
 * return : index to character, or npos if not found.
 ***************************************************************************/
int String::findch (const char *s) const
{
        char *ptr;
        ptr = strchr (str, *s);
        if (ptr == NULL) {
                return npos;
        } else {
                return int(ptr) - int (str);
        }
}

/***************************************************************************
 * FUNCTION : String::findch
 ***************************************************************************
 *
 * description : Returns index of first occuring character <n>
 *
 * parameters : integer value of character to find.
 *
 ***************************************************************************/
int String::findch (const int n) const
{
        char *ptr;
        ptr = strchr (str, char (n));
        if (ptr == NULL) {
                return npos;
        } else {
                return int(ptr) - int (str);
        }
}


//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: find_any_ch
// 
//  DESCRIPTION: Seeks for any character in given string <*list> within
//               first <max> characters of current string.
// 
// 
//  RETURNS: true if found
//**************************************************************************/
bool String::find_any_ch (const char *list, const int max) const
{
        int i;
        const char *ptr = list;
        int last = MIN(max, len);
        while (*ptr) {
                for (i=0; i < last; i++) {
                        if (*ptr == str[i]) {
                                return true;
                        }
                }
                ptr++;
        }
        return false;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: replace_ch
// 
//  DESCRIPTION: Replaces every char <a> with character <b>
// 
// 
//  RETURNS: true if one or more character is replaced.
//**************************************************************************/
bool String::replace_ch (const char a, const char b)
{
        char *ptr = str;
        bool rc = false;
        while (*ptr) {
                if (*ptr == a) {
                        *ptr = b;
                        rc = true;
                }
                ptr++;
        }
        return rc;
}

/***************************************************************************
 * FUNCTION : String::is_empty
 ***************************************************************************
 *
 * description : Checks if current string is empty
 *
 *
 * return : true if string is empty, otherwise false.
 ***************************************************************************/
bool String::is_empty () const
{
        if (len < 1) {
                return true;
        } else {
                return false;
        }
}

/***************************************************************************
 * FUNCTION : String:nget
 ***************************************************************************
 *
 * description : Returns character from position pos.
 *
 * parameters : Position
 *
 * return : If position is out of range, returns 0.
 ***************************************************************************/
char
String::nget (const int pos) const
{
        if (pos >= len) {
                return 0;
        } else {
                return str[pos];
        }
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: count_lines
// 
//  DESCRIPTION: Returns number of lines in string (if string contains
//               line-feeds).
//**************************************************************************/
int
String::count_lines() const
{
        int i = 1;
        char *ptr = str;
        while (*ptr != 0x00) {
                if (*ptr == '\n') {
                        i++;
                }
                ptr++;
        }
        return i;
}
//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: max_line_len
// 
//  DESCRIPTION: Returns length of longes line within string (if string
//               contains more than one lines).
// 
//**************************************************************************/
int
String::max_line_len() const
{
        int l = 0;
        int cl = 0;
        char *ptr = str;
        while (*ptr != 0x00) {
                if (*ptr != '\n') {
                        cl++;
                        if (cl >= l) {
                                l = cl;
                        }
                } else {
                        cl = 0;
                }
                ptr++;
        }
        return l;        
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: get_all_lens
// 
//  DESCRIPTION: Returns lenght of every line in current string at
//               array. Space of array is allocated here, but must free
//               somewhere else.
//               NOTE: index of first line in array is 0.
// 
//**************************************************************************/
int*
String::get_all_lens() const
{
        int i = 0;
        int lines = count_lines();
        int *lens = new int [lines];
        char *ptr = str;
        lens[i] = 0;
        while (*ptr != 0x00) {
                if (*ptr != '\n') {
                        lens[i]++;
                } else {
                        i++;
                        lens[i] = 0;
                }
                ptr++;
        }
        return lens;
}


// returns true, if string contains only numbers
bool
String::is_number() const
{
        if (len < 1) {
                return false;
        }
        int i = 0;
        do {
                if (isdigit( str[i] ) == 0) {
                        return false;
                }
                ++i;
        } while (str[i]);
        return true;
}
        

/***************************************************************************
 * FUNCTION : String::~String
 ***************************************************************************
 *
 * description : Destructor
 *
 ***************************************************************************/
String::~String ()
{
        if (str != NULL) {
                delete[] str;
        }
        str = NULL;
        len = npos;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: strip_ch
// 
//  DESCRIPTION: Removes every character <ch> from string
// 
//**************************************************************************/
void
String::strip_ch (const char ch)
{
        char *tptr, *sptr;
        char *tmp = new char [len + 1];
        memset (tmp, 0, len+1);
        tptr = tmp;
        sptr = str;
        while (*sptr) {
                if (*sptr != ch) {
                        *tptr = *sptr;
                        tptr++;
                }
                sptr++;
        }
        len = strlen (tmp);
        delete[] str;
        str = tmp;
}

//**************************************************************************/
//  CLASS: String
//  MEMBER FUNCTION: convert_charset
// 
//  DESCRIPTION: Converts character set of string from set <dest> 
//               to set <src>.
// 
//**************************************************************************/
void
String::convert_charset (const int src, const int dest)
{
        char table [CHARACTER_SETS][CONVERT_CHARACTERS] = {
                // Order of characters in array: 
		// a' a'' o'' A' A'' O'' e' u'' E' U'' sharp-s
                // IBM-PC codepage 437 
                {0x86,0x84,0x94,0x8f,0x8e,0x99,0x82,0x81,0x90,0x9a,0xe1},
                // ISO Latin-1 
                {0xe5,0xe4,0xf6,0xc5,0xc4,0xd6,0xe9,0xfc,0xc9,0xdc,0xdf},
                // ASCII 7-bit
                {0x7d,0x7b,0x7c,0x5d,0x5b,0x5c,0x60,0x7e,0x40,0x5e,0x73},
                // Apple Macintosh
                {0x8c,0x8a,0x9a,0x81,0x80,0x85,0x8e,0x9f,0x83,0x86,0xa7}
        };
        int i;
        for (i=0; i < CONVERT_CHARACTERS; i++) {
                replace_ch (table[src][i], table[dest][i]);
        }
}


//           *********************************************************
//           **                                                     **
//           **  friend functions                                   **
//           **                                                     **
//           *********************************************************



//*************************************************************************/
// FUNCTION: operator +
//*************************************************************************/
//
// Concatenates two String objects to one.
//
//
// RETURN: Concatenated string.
//*************************************************************************/
String operator + (String const &str1, String const &str2)
{
        String tmp = str1;
        tmp += str2;
        return tmp;
}

//*************************************************************************/
// FUNCTION: operator +
//*************************************************************************/
//
// Concatenates string object with null terminated c-string.
//
// RETURN: String
//*************************************************************************/
String operator + (String const &str, const char *s)
{
        String tmp = str;
        tmp += s;
        return tmp;
}

//*************************************************************************/
// FUNCTION: operator +
//*************************************************************************/
//
// Concatenates null terminated c-string with String object
//
//
// RETURN: string object
//*************************************************************************/
String operator + (const char *s, String const &str)
{
        String tmp = s;
        tmp += str;
        return tmp;
}


//*************************************************************************/
// FUNCTION: operator ==
//*************************************************************************/
//
// Compares two String objects.
//
// RETURN: 1 if equals, 0 if not.
//*************************************************************************/
int operator == (String const &str1, String const &str2)
{
        if  (!strcmp (str1.str, str2.str)) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator ==
//*************************************************************************/
//
// Compares null terminated c-string with String object
//
//
// RETURN: 1 if equals, 0 if not.
//*************************************************************************/
int operator == (const char *s, String const &str)
{
        if  (!strcmp (s, str.str)) {
                return 1;
        } 
        return 0;
}


//*************************************************************************/
// FUNCTION: operator ==
//*************************************************************************/
//
// Compares String object with c-string
//
//
// RETURN: 1 if equals, 0 if not.
//*************************************************************************/
int operator == (String const &str, const char *s)
{
        return ( s == str );
}

//*************************************************************************/
// FUNCTION: operator !=
//*************************************************************************/
//
// Compares two String objects.
//
//
// RETURN: 1 if not equals, 0 if equals
//*************************************************************************/
int operator != (String const &str1, String const &str2)
{
        if (strcmp (str1.str, str2.str)) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator !=
//*************************************************************************/
//
// Compares String object with c-string
//
//
// RETURN: 1 if not equals, 0 if equals.
//*************************************************************************/
int operator != (String const &str, const char *s)
{
        if (strcmp (str.str, s)) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator !=
//*************************************************************************/
//
// Compares c-string with String object.
//
//
// RETURN: 1 if not equals, 0 if equals.
//*************************************************************************/
int operator != (const char *s, String const &str)
{
        return ( str != s );
}


//*************************************************************************/
// FUNCTION: operator <
//*************************************************************************/
//
// Checks if first string is less than second.
//
// RETURN: 1 if less, 0 if bigger.
//*************************************************************************/
int operator < (String const &str1, String const &str2)
{
        if  (strcmp (str1.str, str2.str) < 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator <
//*************************************************************************/
//
// Checks if first string object is less than second c-string.
//
// RETURN: 1 if less, 0 if bigger.
//*************************************************************************/
int operator < (String const &str, const char *s)
{
        if  (strcmp (str.str, s) < 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator <
//*************************************************************************/
//
// Checks if string *s < string str
//
// RETURN: 1 if less, 0 if bigger.
//*************************************************************************/
int operator < (const char *s, String const &str)
{
        if (strcmp (s, str.str) < 0) {
                return 1;
        }
        return 0;
}

//*************************************************************************/
// FUNCTION: operator <=
//*************************************************************************/
//
// Checks if first string is equal or less than second.
//
// RETURN: 1 if less or equal, 0 if bigger.
//*************************************************************************/
int operator <= (String const &str1, String const &str2)
{
        if  (strcmp (str1.str, str2.str) <= 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator <=
//*************************************************************************/
//
// Checks if first string object is equal or less than second c-string.
//
// RETURN: 1 if less or equal, 0 if bigger.
//*************************************************************************/
int operator <= (String const &str, const char *s)
{
        if  (strcmp (str.str, s) <= 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator <=
//*************************************************************************/
//
// Checks if string *s <= string str
//
// RETURN: 1 if less or equal, 0 if bigger.
//*************************************************************************/
int operator <= (const char *s, String const &str)
{
        if (strcmp (s, str.str) <= 0) {
                return 1;
        }
        return 0;
}

//*************************************************************************/
// FUNCTION: operator >
//*************************************************************************/
//
// Checks if first string is more than second.
//
// RETURN: 1 if ore, 0 if less.
//*************************************************************************/
int operator > (String const &str1, String const &str2)
{
        if  (strcmp (str1.str, str2.str) > 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator >
//*************************************************************************/
//
// Is String > c-string ?
//
// RETURN: 1 if it is bigger, 0 if not.
//*************************************************************************/
int operator > (String const &str, const char *s)
{
        if  (strcmp (str.str, s) > 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator >
//*************************************************************************/
//
// Is c-string > String ?
//
// RETURN: 1 if bigger, 0 if not.
//*************************************************************************/
int operator > (const char *s, String const &str)
{
        if (strcmp (s, str.str) > 0) {
                return 1;
        }
        return 0;
}


//*************************************************************************/
// FUNCTION: operator >=
//*************************************************************************/
//
// Is String >= String ?
//
// RETURN: 1 if first is bigger or if both are equal, 0 if first is less.
//*************************************************************************/
int operator >= (String const &str1, String const &str2)
{
        if  (strcmp (str1.str, str2.str) >= 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator >=
//*************************************************************************/
//
// Is String >= c-string ?
//
// RETURN: 1 if String is bigger or if both are equal, 0 if String is less.
//*************************************************************************/
int operator >= (String const &str, const char *s)
{
        if  (strcmp (str.str, s) >= 0) {
                return 1;
        } 
        return 0;
}

//*************************************************************************/
// FUNCTION: operator >=
//*************************************************************************/
//
// Is c-string >= String ?
//
// RETURN: 1 if c-string is bigger if both are equals, 0 if c-string is less.
//*************************************************************************/
int operator >= (const char *s, String const &str)
{
        if (strcmp (s, str.str) >= 0) {
                return 1;
        }
        return 0;
}


//*************************************************************************/
// FUNCTION: operator <<
//*************************************************************************/
//
// Sents contents of string to ostream
//
//
// RETURN: reference to ostream-object
//*************************************************************************/
ostream& operator << (ostream &os, String const &str)
{
        if (str.len > 0) {
                os << str.str;
        }
        return os;
}

//**************************************************************************/
// CLASS: String
// MEMBER FUNCTION: substr
//**************************************************************************/ 
//
// Gets substring from current string.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  -
//   OUT:  -
// 
// PARAMETERS: Index of first letter of the substring
//
// RETURN: Substring 
//**************************************************************************/
String String::substr (const int n) const
{
        String tmp;
        if (n < len) {
                tmp = (str + n);               
        }
        return tmp;
}

//**************************************************************************/
// CLASS: String
// MEMBER FUNCTION: split
//**************************************************************************/ 
//
// Splits string to fields using given separator.
//
// NOTE: remember delete returned list when it is no longer needed!
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  -
//   OUT:  -
// 
// PARAMETERS: fs   = field separator, default is ' '.
//
// RETURN: Pointer to list<String> that contains fields.
//**************************************************************************/
List<String>*
String::split( char fs )
{
        DEBUG("Splitting \'"<<str<<"\'");
        DEBUG("Separator: \'"<<fs<<"\'");
        List<String> *fields = new List<String>;

        if (len < 1) {
                String *field = new String;
                fields->add( field );
        } else {
                int i = 0;
                String *field;
                do {
                        field = new String;
                        while ( str[i] && str[i] != fs) {
                                *field += str[i++];
                        }
                        fields->add( field );
                        while ( str[i] && str[i] == fs ) {
                                ++i;
                        }
                } while (str[i]);
                fields->first();
        }
#ifdef USE_DEBUG
        fields->first();
        int ii = 0;
        do {
                DEBUG("Field "<<ii<<" = "<<*(fields->get()));
                ++ii;
        } while( fields->next());
        fields->first();
#endif
        return fields;
}

//**************************************************************************/
// CLASS: String
// MEMBER FUNCTION: rot13
//**************************************************************************/ 
//
// Encodes/Decodes current string using Rot13 method.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  -
//   OUT:  -
// 
// PARAMETERS: -
//
// RETURN: -
//**************************************************************************/
void
String::rot13()
{
        char rot13src[]="abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
        char rot13dst[]="nopqrstuvwxyzabcdefghijklmNOPQRSTUVWXYZABCDEFGHIJKLM";
        if (len < 1 ) {
                return;
        }
        char *ptr = str;
        
        int index;
        while (*ptr) {
                index = 0;
                while (rot13src[index]) {
                        if (rot13src[index] == *ptr ) {
                                *ptr = rot13dst[index];
                                break;
                        }
                        ++index;
                }
                ++ptr;
        }
}

bool
String::starts_with( const char* s )
{
        bool rc = false;
        int slen = strlen( s );
        if ( len >= slen ) {
                if ( !strncmp( str, s, slen ) ) {
                        rc = true;
                }
        }
        return rc;
}

bool
String::starts_with( const String& s )
{
        bool rc = false;
        if ( this == &s ) {
                rc = true;
        } else if ( len >= s.len ){
                if ( !strncmp( str, s.str, s.len ) ) {
                        rc = true;
                }
        }
        return rc;
}

