// 	$Id: terminal_io.hh,v 1.14 1998/04/30 16:30:09 jvuokko Exp $	

/****************************************************************************
 * *
 * *  MODULE : terminal_io.hh
 * *
 * *  Copyright (c) 1997 Jukka Vuokko
 * *  See file COPYING for more information about copyrights.
 * *
 ****************************************************************************
 * *
 * *  Class for handling console-io and terminal output.
 * *
 * *
 * *
 * *
 * *
 ***************************************************************************/


#ifndef __TERMINAL_IO_HH__
#define __TERMINAL_IO_HH__

#include "../utilib/my_types.h"
#include "../utilib/String.hh"
#include "../utilib/List.hh"

#include <iostream.h>
#include <stdio.h>
#include <stdarg.h>
#ifdef __unix__
#   include <termios.h>
#   include <sys/types.h>
#   include <sys/uio.h>
#   include <unistd.h>
//#   include <termcap.h> // Linux tarvitsee tt
//#   include <term.h>    // Ja suurin osa muista unixeista tt...
// Ehk on paras toteuttaa esittelemll tarvittavat funktiot tss:
extern "C" {
         extern int   tgetent( void*, char*);
         extern int   tgetnum( const char*);
         extern int   tgetflag( const char* );
         extern char* tgetstr( const char*, char**);
         extern char* tgoto( const char*, int, int );
         extern int   tputs( const char*, int, int(*)(int) );
};
#else
#   include <conio.h>
#   define DOS_KEYS
#endif

#ifdef __unix__
#   define USE_TERMCAP
//#   define USE_ANSI
#else
//#   define USE_ANSI // uncomment this, if you want use ansi with Os/2
#endif

#if defined( USE_ANSI ) || defined( USE_TERMCAP )
#   define USE_STDOUT
#endif

// real keycodes
#define CODE_CTRL_A 1
#define CODE_CTRL_B 2
#define CODE_CTRL_D 4
#define CODE_CTRL_E 5
#define CODE_CTRL_F 6
#define CODE_CTRL_H 8
#define CODE_CTRL_K 11
#define CODE_CTRL_L 12
#define CODE_CTRL_V 22
#define CODE_BS     CODE_CTRL_H
#ifdef __unix__
#   define LINE_FEED 0x0a
#else
#   define LINE_FEED 0x0d
#endif



// jmr's own keycodes.
#define CODE_UP       0x001000
#define CODE_DOWN     0x002000
#define CODE_RIGHT    0x004000
#define CODE_LEFT     0x008000
#define CODE_HOME     0x010000
#define CODE_END      0x020000
#define CODE_PPAGE    0x040000
#define CODE_NPAGE    0x080000
#define CODE_DEL      0x100000


#define ESCAPE (char) 27

// colors
#ifdef USE_STDOUT
#   define BLACK         0
#   define RED           1
#   define GREEN         2
#   define YELLOW        3
#   define BLUE          4
#   define MAGENTA       5
#   define CYAN          6
#   define WHITE         7
#else                 // PC VGA colors
#  define BLACK          0
#  define BLUE           1
#  define GREEN          2
#  define CYAN           3
#  define RED            4
#  define MAGENTA        5
#  define YELLOW         6
#  define WHITE          7
#endif


// other attributes
#define INVERSE       0x80000
#define UNDERLINE     0x40000
#define BLINK         0x20000
#define BOLD          0x10000


// color bits
#define FCOLOR_BITS 0xF000
#define BCOLOR_BITS 0x0F00

// Convert color value to actual value.
#define FOREGROUND(a) ( ((a) << 12) & FCOLOR_BITS )
#define BACKGROUND(a) ( ((a) << 8 ) & BCOLOR_BITS )

// Get only attribute bits 
#define ATTRIBUTES(a) ( (a) & ~0xff )

// get colors from attribute
#define GET_FCOLOR(a) ( ((a) >> 12 ) & 0x7 )
#define GET_BCOLOR(a) ( ((a) >> 8  ) & 0x7 )

// defaults
#define DEFAULT_FCOLOR 0x8000
#define DEFAULT_BCOLOR 0x0800
#define DEFAULT_COLORS (0x8800)
#define DEFAULT_ATTR DEFAULT_COLORS 



// flag for Terminal_screen::refresh()
#define REDRAW_ALL 1


// flags for window styles
#define WIN_DEFAULT  0x00
#define WIN_TITLEBAR 0x01
#define WIN_BORDERS  0x02
#define WIN_SLINE    0x04

#define NO_PERCENT  -1
#define ALL_PERCENT 200
#define TOP_PERCENT 300
#define BOT_PERCENT 100

#ifdef USE_STDOUT
#   define SET_TERM_ATTR(a) set_term_attr(a)
#else
#   define SET_TERM_ATTR(a) 
#endif

// type for screen attribute. 
typedef unsigned int attr_t;

/*
 * This inline function converts jmr's internal attribute to
 * attribute-character pair that PC understands.
 */
#ifndef USE_STDOUT
inline unsigned short get_pc_attr_cell( attr_t a )
{
        register ushort_t pca = (ushort_t) (a & 0xff);

        if ( a & DEFAULT_FCOLOR ) {
                pca |= (ushort_t) 0x700;
        } else {
                pca |= (ushort_t) ( (a >> 4) & 0x700 );  // foreground bits
        }
        
        if (! (a & DEFAULT_BCOLOR) ) {
                pca |= (ushort_t) ( (a << 4) & 0x7000);  // background bits
        }

        if ( a & INVERSE ) {
                // get foreground bits and switch them to backround
                ushort_t tmp = (ushort_t) ( ((pca << 4) & 0xf000) |
                                            (pca & 0xff)); 
                pca = (ushort_t) ( ((pca >> 4) & 0xf00) | tmp) ;
        }

        if ( a & BOLD ) {
                pca |= (ushort_t) 0x800;
        }
        
        if ( a & BLINK ) {
                pca |= 0x8000;
        }

        return pca;
}
#endif /* USE_STDOUT */


class Terminal_screen;
class Window;

void attr_memset( attr_t *buf, attr_t a, int n );

//**************************************************************************/
//
// CLASS: Winlist
//
// DERIVED FROM:
//
// Class for handling list of windows.
//
//**************************************************************************/
class Winlist {
public:
        Winlist() {}
        ~Winlist() { }
        bool add( Window *win ) { windows.last(); return windows.add( win ); }
        bool del( Window *win );
        bool first() { return windows.first(); }
        bool next()  { return windows.next(); }
        bool last()  { return windows.last(); }
        bool raise( Window *win );
        Window* get() { return windows.get(); }
private:
        List<Window> windows;
};


class Statusline {
public:
        Statusline(int maxlen) : sline(0) { init(maxlen); }
        virtual ~Statusline() { delete[] sline;}
        void init(int maxlen);
        void set_infotext( const char *msg ) { infotext = msg;}
        void set_percent( int pr ) { percent = pr; }
        void set_extra_info( const char *msg) { extra_info =  msg;}
        char* get_sline();
protected:
        int width;
        String extra_info;
        String infotext;
        char *sline;
        int percent;
};


//**************************************************************************/
//
// CLASS: Window
//
// DERIVED FROM:
//
// 
//
//**************************************************************************/
class Window {
public:
        Window( Terminal_screen *handle, int x, int y, int w, int h,
                int style = WIN_DEFAULT, const char* title = 0);
        ~Window() {  delete[] _win; if(_sline) delete _sline; }


        void set_border_colors( int fg, int bg );
        
        void set_sline_text( const char* text );
        void set_sline_percent( int pros );
        void set_sline_info( const char* msg);
                                                        

        void disable()   { is_visible = false; }
        void enable() { is_visible = true; }
        
        void inverse( int n=1 ) { n ? _attr |= INVERSE : _attr &= ~INVERSE; }
        void bold( int n=1 )    { n ? _attr |= BOLD : _attr &= ~BOLD;  }
        void blink( int n=1 )   { n ? _attr |= BLINK : _attr &= ~BLINK; }
        void reset_attr()       { _attr = DEFAULT_ATTR; }
        void textcolor (int color);
        void textcolor (int fg, int bg);
        void back_color (int color);
        
        int print( const char *fmt, ... );
        int vprint( const char *fmt, va_list ap );
        bool addstr ( const String& str ) { return addbytes( str.c_str(),
                                                             str.length() ); }
        bool addstr ( const char* str ) { return addbytes( str,
                                                           strlen( str ) ); }
        bool addch ( char ch, int count = 1 );
        bool addbytes ( const char*, int );

        //void refresh( bool cleared = false);
        void update();
        
        void clear();
        //void erase();
        void clr_line();
        void clr_eol();
        void bottom() { _x = _xmin; _y = _ymax; }
        void scroll_down();
        bool down( int n=1);
        bool up( int n=1);
        void goto_x (const int x) { _x = x < _xmax ? x : _xmax; }
        char* getline(char *buffer, int max, int minlen = 1);
        void forward (const int n=1);
        void back (const int n=1);
        bool gotoxy(int x, int y );
        int get_x() { return _x - _xmin; }
        int get_y() { return _y - _ymin; }
        int get_xmax() { return _xmax - _xmin; }
        int get_ymax() { return _ymax - _ymin; }
private:
        void draw_borders();
        attr_t *_win;
        attr_t _attr;
        attr_t _border_attr;
        int _width;
        int _height;
        int _xbeg;
        int _ybeg;
        int _xmax;
        int _ymax;
        int _xmin;
        int _ymin;
        int _x;
        int _y;
        int _size;
        int _cwidth;
        int _style;
        bool is_visible;
        bool _border_changed;

        String _title;
        Statusline *_sline;
        Terminal_screen * _mainwin;

};

//**************************************************************************/
//
// CLASS: Terminal_screen
//
// DERIVED FROM:
//
// 
//
//**************************************************************************/
class Terminal_screen {
public:
        Terminal_screen();
        void set_rawmode ();
        void set_origmode();
        void force_colormode() {color_term = true;} // pakottaa ansi-tilaan
        bool is_color_term() const { return color_term; }
        int get_ch();
        int get_xch() { return xkey; } // palauttaa ESC:i seuranneen
        void refresh( int flag = 0);
        void update( attr_t *win, int x, int y, int w,
                     int h, int xpos, int ypos);
        void enable() { screen_is_active = true; stdscr->enable(); }
        void disable() { screen_is_active = false; stdscr->disable(); }
#ifdef USE_STDOUT
        void set_term_attr( int attr );
#endif
        ~Terminal_screen();

        Window *newwin( int x, int y, int w, int h,
                        int style = WIN_DEFAULT, const char *title = 0);
        bool endwin( Window *win) { return winlist.del( win ); }

        Window *get_screen() { return stdscr; }
        void inverse( int n=1) { stdscr->inverse( n ); }
        void bold( int n=1) { stdscr->bold( n ); }
        void blink( int n=1 ) { stdscr->blink( n ); }
        void textcolor (int color) { stdscr->textcolor( color ); }
        void textcolor (int fg, int bg) { stdscr->textcolor( fg, bg ); }
        void back_color (int color) { stdscr->back_color( color ); }
        void reset_attr () { stdscr->reset_attr(); }
        
        void clear() { stdscr->clear(); cleared = true; }
        void erase() { stdscr->clear(); }
        bool addstr ( const String& str ) { return stdscr->addstr( str ); }
        bool addstr ( const char* str ) { return stdscr->addstr( str ); }
        bool addch ( char ch, int c=1 ) { return stdscr->addch( ch, c ); }
        int print (const char *fmt, ...);
        void empty_line() { stdscr->clr_line(); }
        void clr_eol() { stdscr->clr_eol(); }
        int get_xmax() { return _xmax; }
        int get_ymax() { return _ymax; }
        int get_x() { return stdscr->get_x(); }
        int get_y() { return stdscr->get_y(); }
        void bottom() { stdscr->bottom(); }
        void down (int n = 1) { stdscr->down(n); }
        void up (int n = 1) { stdscr->up(n); }
        void goto_x (const int x) { stdscr->goto_x(x); }
        char* getline(char *buf, int max, int minlen = 1) { return stdscr->getline( buf, max, minlen); }
        void gotoxy (int x, int y) { stdscr->gotoxy(x,y); }
        void forward (const int n=1) { stdscr->forward(n); }
        void back (const int n=1) { stdscr->back(n); }

private:
        void set_cur_pos(int, int );
        void clear_screen();
        void drawline( attr_t*, int );
        void drawscreen();
        void get_screen_size();
        void get_scr_size();
        int raw_readch();
        int raw_getch();
#ifdef __unix__
        void get_term_data();
#endif

        
#ifdef __unix__
        struct termios t_orig;
        struct termios t_new;
#endif
        char *term_ce; // clear to end of line
        char *term_cl; // clear screen
        char *term_cm; // move cursor to x, y
        char *term_mb; // blink on
        char *term_md; // bold mode on
        char *term_me; // normal mode
        char *term_mr; // reverse mode
        char *term_us; // underline mode
        
        bool rawmode;
        bool color_term;
        bool cleared;
        bool screen_is_active;
        int _ymax;
        int _xmax;
        int _height;
        int _width;
        int _xpos;
        int _ypos;
        int xkey;
        Window *stdscr;
        attr_t *currscr;
        attr_t *lastscr;

        Winlist winlist;
};


inline void
Window::set_sline_text(const char* text )
{
        if(_sline) {
                _sline->set_infotext(text);
                _border_changed = true;
        }
}

inline void
Window::set_sline_percent( int pros )
{
        if(_sline) {
                _sline->set_percent(pros);
                _border_changed = true;
        }
}

inline void
Window::set_sline_info( const char *msg )
{
        if(_sline) {
                _sline->set_extra_info( msg );
                _border_changed = true;
        }
}



inline void
Window::update()
{
        if (_border_changed == true) {
                draw_borders();
        }
        if (is_visible == true ) {
                _mainwin->update( _win, _xbeg, _ybeg, _width, _height, _x, _y);
        }
}

#endif
