// 	$Id: terminal_io.cc,v 1.18 1998/05/25 20:06:07 jvuokko Exp $	

/****************************************************************************
 * *
 * *  MODULE : terminal_io.cc
 * *
 * *  Copyright (c) 1997 Jukka Vuokko
 * *  See file COPYING for more information about copyrights.
 * *
 ****************************************************************************
 * *
 * *  Functions for read input from console and for setting attributes
 * *  of terminal.
 * *
 * *
 * *
 * *
 ***************************************************************************/
#include <stdlib.h> // getenv(), free()
#include <stdio.h> // perror()
#include <signal.h>
#include <stdarg.h> 
#ifndef __unix__
#   include <io.h>    // read(), write()
#endif

#include "constants.h"
#include "jmr.hh"
#ifdef OS2
#   define INCL_VIO
#   include <os2.h> // for handling screen
#endif

#if defined (__WATCOMC__) && defined (DOS)
#   include <graph.h> // for getting size of screen
#endif

#include "terminal_io.hh"

// no debugging stuff are currently needed from this module
#undef DEBUG
#define DEBUG(ex)

//*************************************************************************/
// FUNCTION: attr_memset
//*************************************************************************/
//
// Fills given buffer (size n) with a given attribute value.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : 
//   OUT: 
//
// RETURN: -
//*************************************************************************/
void
attr_memset( attr_t *buf, attr_t a, int n )
{
        register int i = n;
        while (i) {
                buf[--i] = a;
        }
}

//*************************************************************************/
// FUNCTION: newcstr
//*************************************************************************/
//
// This function dublicates given c-string.
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : 
//   OUT: 
//
// RETURN: Pointer to new copy of a string
//*************************************************************************/
static char*
newcstr( const char* str )
{
        char *newstr = new char[strlen (str) +1];
        strcpy( newstr, str );
        return newstr;
}



//*************************************************************************/
// FUNCTION: get_termstr
//*************************************************************************/
//
// This function gets termcap-string for a given termcap-item and
// cuts padding information off.
// For example, if termcap string is "\e[%i%d;%dH$<5>", returned string
// are "\e[%i%d;%dH"
//
// EXTERNAL VARIABLE REFERENCES:
//   IN : 
//   OUT: 
//
// RETURN: 
//*************************************************************************/
#ifdef USE_TERMCAP
static char*
get_termstr( const char* tc )
{
        static char buffer[BUFSIZ];
        static char _tc[10];
        char *str;
        char *p = buffer;
        strcpy( _tc, tc );
        p = tgetstr( _tc, &p );
        if (!p) {
                str = new char;
                *str = 0;
                DEBUG("No termcap capability for : " << tc );
                return str;
        }

        DEBUG("Full termstr = " << p );

        // skip over possible pad info. It should not be needed with
        // todays terminals...
        while ((*p == '.') || ((*p >= '0') && (*p <= '9'))) {
                ++p;
        }
        
        if (*p == '*') ++p;

        // remove terminal padding ( $<...> ) tputs does not handle them.
        char *ptr = p;
        while (*ptr) {
                if ((*ptr++ == '$') && (*ptr == '<')) {
                        char *tmp;
                        tmp = ptr - 1;
                        while (*ptr && (*ptr != '>')) {
                                ++ptr;
                        }
                        if (!*ptr) {
                                break;
                        }
                        ++ptr;
                        strcpy( tmp, ptr );
                }
        }

        DEBUG("Returning termcap str: " << p );
        str = newcstr( p );
        return str;
}
#endif /* USE_TERMCAP */


//**************************************************************************/
// CLASS: Winlist
// MEMBER FUNCTION: del
//**************************************************************************/ 
//
// This function removes given window from list of windows. Note that data
// of the window is also removed.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: false if failed.
//**************************************************************************/
bool
Winlist::del( Window *win )
{
        DEBUG("Deleting window");
        if ( windows.first() == false ) {
                return false;
        }
        Window *tmp;
        do {
                tmp = windows.get();
                if ( tmp == win ) {
                        windows.remove();
                        DEBUG("Window deleted!");
                        return true;
                }
        } while (windows.next() == true );
        DEBUG("Deleting failed!");
        return false;
}

//**************************************************************************/
// CLASS: Winlist
// MEMBER FUNCTION: raise
//**************************************************************************/ 
//
// Brings given window to top of the screen.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: false if failed.
//**************************************************************************/
bool
Winlist::raise( Window *win )
{
        if ( windows.first() == false) {
                return false;
        }
        Window *tmp;
        do {
                tmp = windows.get();
                if (tmp == win) {
                        windows.leavenode();
                        windows.last();
                        windows.add( tmp );
                        return true;
                }
        } while ( windows.next() == true );
        return false;
}





//**************************************************************************/
//
// CLASS: Terminal_screen
// MEMBER FUNCTION: Terminal_screen
//
// description: Constructor. Initializes terminal for use.
//
//
//**************************************************************************/
Terminal_screen::Terminal_screen()
{
        stdscr = 0;
#ifdef __unix__
        rawmode = false;

        get_term_data();

        int istat, ostat;
        istat = tcgetattr(0, &t_orig);
        if (istat != 0) {
                perror ("tcgetattr");
                exit(EXIT_FAILURE);
        }
        rawmode = true;

        // set console to rawmode
        t_new = t_orig;
        t_new.c_lflag &= ~ICANON;
        t_new.c_lflag &= ~ECHO;
        //        t_new.c_lflag &= ~ISIG;
        t_new.c_cc[VMIN] = 1;
        t_new.c_cc[VTIME] = 0;

        ostat = tcsetattr(0, TCSANOW, &t_new);
        // return here if something goes wrong with rawmode
        if (ostat != 0) {
                perror ("tcsetattr");
                exit(EXIT_FAILURE);
        }



#else
        color_term = true;
        rawmode = true;
        term_ce = newcstr ("\033[K" );
        term_cl = newcstr ("\033[2J" );
        term_cm = newcstr ("\033[%d;%df");
        term_mb = newcstr ("\033[5m");
        term_md = newcstr ("\033[1m");
        term_me = newcstr ("\033[0m");
        term_mr = newcstr ("\033[7m");
        term_us = newcstr ("\033[4m");
#endif /* __unix__ */
        
        get_screen_size();
        
#ifdef __NOCOLORS__
        color_term = false;
#endif
        
        cleared = true;
        screen_is_active = false;
        stdscr = new Window( this, 0, 0, _width, _height);
        currscr = new attr_t[ _width * _height ];
        lastscr = new attr_t[ _width * _height ];

        attr_memset( currscr, DEFAULT_ATTR | ' ', _width * _height );
        attr_memset( lastscr, DEFAULT_ATTR | ' ', _width * _height );
        _xpos = _ypos = 0;
}

//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: get_screen_size
// 
//  DESCRIPTION: Reads size of terminal screen.
// 
// 
//**************************************************************************/
void
Terminal_screen::get_screen_size()
{
#ifdef __unix__
        char co[] = "co";
        char li[] = "li";
        _xmax = tgetnum (co) -1;
        _ymax = tgetnum (li) -1;
#elif defined (OS2)
        VIOMODEINFO vioinfo;
        vioinfo.cb = sizeof (vioinfo);
        VioGetMode ((PVIOMODEINFO) &vioinfo, (HVIO) 0);
        _xmax = vioinfo.col-1;
        _ymax = vioinfo.row-1;
#elif defined (__GO32__)
        struct text_info ti;
        gettextinfo(&ti);
        _ymax=ti.screenheight;
        _xmax=ti.screenwidth;
#elif defined (__WATCOMC__) && defined (DOS)
        struct videoconfig vc;
        _getvideoconfig (&vc);
        _xmax = vc.numtextcols -1;
        _ymax = vc.numtextrows -1;
#else
        _xmax = 79;
        _ymax = 23;
#endif
#if defined( USE_ANSI ) && !defined( __unix__)
        --_ymax; // in DOS & Os/2 bottom line is not useful!
#endif
        _height = _ymax+1;
        _width  = _xmax+1;
}
//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: get_scr_size
// 
// 
//  DESCRIPTION: Reads size of screen using VT-100 secuences.
//               NOTE: At unix, console must be in raw mode.
// 
//  NOTE: This method uses external function handle_error, which displays
//        error message and terminates program.
//
//        It can be replaced with cerr and exit(EXIT_FAILURE).
//**************************************************************************/
void
Terminal_screen::get_scr_size()
{
        char response[80];
        char x[5];
        char y[5];
        char *ptr;
        int i;
        //gotoxy(200,200);
        cout << ESCAPE << "[200;200f";
        if (rawmode == false) {
                handle_error ("Keyboard must be in raw mode when reading cursor position.",HALT_FL);
        }
        
        cout <<"\033[6n"<<flush;
        i = 0;
#ifdef __unix__
        char c;
        do {
                read (0, &c, 1);
                response[i] = c;
                i++;
                if (c == 'R') {
                        break;
                }
        } while (1);
        response[i] = '\0';
#else
        cin >> response;
#endif /* __unix__ */
        // parse output string
        if (response[0] == 27 && response[1] == '[') {
                ptr = response+2;
                i = 0;
                while (isdigit (*ptr) && *ptr && i < 4) {
                        y[i] = *ptr;
                        i++;
                        ptr++;
                }
                y[i] = '\0';
                ptr++;
                i=0;
                while (isdigit (*ptr) && *ptr && i < 4) {
                        x[i] = *ptr;
                        i++;
                        ptr++;
                }
                x[i] = '\0';
                _xmax = atoi (x) -1;
                _ymax = atoi (y) -1;
        }
}

//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: get_term_data
// 
//  DESCRIPTION: Gets necessary data from terminal, and checks if terminal
//               have capability to show colors.
// 
//**************************************************************************/
#ifdef __unix__
void
Terminal_screen::get_term_data()
{
        static char _terminal[2048];
        static char buffer[1024];
        int colors;
        char termname[40], *p;
        char *ptr = buffer;
        if ((p = (char *) getenv ("TERM")) == NULL) {
                cerr << "TERM variable must be set to use screen capabilities";
                exit (EXIT_FAILURE);
        }
        if (strcpy (termname, p) == NULL) {
                cerr <<"Can't get TERM variable\n";
                exit(EXIT_FAILURE);
        }
        if (tgetent (_terminal, termname) != 1) {
                cerr <<"Can't get entry for TERM\n";
                exit(EXIT_FAILURE);
        }

        // get control sequences
#ifdef USE_TERMCAP
        term_cl = get_termstr( "cl" );
        if (!*term_cl) {
                cerr << "Terminal must have clear screen capability!";
                exit( EXIT_FAILURE );
        }

        term_cm = get_termstr( "cm" );
        if ( !*term_cm ) {
                cerr << "Terminal must have cursor movement capability!";
                exit( EXIT_FAILURE );
        }

        term_ce = get_termstr( "ce" );
        if ( !*term_ce ) {
                cerr << "Terminal must have clear to end of line capability!";
                exit( EXIT_FAILURE );
        }
        term_mb = get_termstr( "mb" );
        term_md = get_termstr( "md");
        term_me = get_termstr( "me");
        term_mr = get_termstr( "mr");
        term_us = get_termstr( "us" );
#else
        term_ce = newcstr ("\033[K" );
        term_cl = newcstr ("\033[2J" );
        term_cm = newcstr ("\033[%d;%df");
        term_mb = newcstr ("\033[5m");
        term_md = newcstr ("\033[1m");
        term_me = newcstr ("\033[0m");
        term_mr = newcstr ("\033[7m");
        term_us = newcstr ("\033[4m");
#endif
        // check if terminal accept colors
        color_term = false;
  again:
        DEBUG( "Checking for color capability" );
        char Co[] = "Co";
        char tc[] = "tc";
        if ((colors = tgetnum (Co)) >= 1) {
                DEBUG("COLORS!");
                color_term = true;
        } else {
                p = tgetstr (tc, &ptr);
                if (p != NULL) {
                        DEBUG( "New entry: " << _terminal );
                        if (tgetent (_terminal, p) != 1) {
                                cerr << "\nCannot get terminal!";
                                exit(EXIT_FAILURE);
                        } else {
                                DEBUG( "Checking it..." );
                                goto again;
                        }
                }
        }
}
#endif /* __unix__ */

//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: ~Terminal_screen
// 
// 
//  DESCRIPTION: Destructor. Returns console to normal mode.
// 
// 
//**************************************************************************/
Terminal_screen::~Terminal_screen()
{
        DEBUG("Destructor of class Terminal_screen");
        if ( stdscr ) {
                refresh();
                set_cur_pos( 0, _ymax );
                printf( term_ce );
        }

#ifdef __unix__
        if (rawmode == true)
                tcsetattr(0, TCSANOW, &t_orig);
#endif
        DEBUG("Deleting terminal strings");
        delete term_ce ;
        delete term_cl ;
        delete term_cm ;
        delete term_mb ;
        delete term_md ;
        delete term_me ;
        delete term_mr ;
        delete term_us ;
        DEBUG("Deleting stdscr");
        delete stdscr;
        DEBUG("Deleting currscr");
        delete[] currscr;
        DEBUG("Deleting lastscr");
        delete[] lastscr;
        DEBUG("Screen destroyed!");
}

//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: set_rawmode
// 
// 
//  DESCRIPTION: Sets console to raw mode
// 
// 
//**************************************************************************/
void
Terminal_screen::set_rawmode()
{
#ifdef __unix__
        tcsetattr(0, TCSANOW, &t_new);
#endif
        rawmode = true;
}
//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: set_origmode
// 
// 
//  DESCRIPTION: Sets console to original mode
// 
// 
//**************************************************************************/
void
Terminal_screen::set_origmode()
{
#ifdef __unix__
        tcsetattr (0, TCSANOW, &t_orig);
#endif
        rawmode = false;
}


//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: get_ch
// 
// 
//  DESCRIPTION: Reads keyboard.
// 
// 
//  RETURNS: Pressed key.
// 
//**************************************************************************/
int
Terminal_screen::get_ch()
{
        int c;
        refresh();
loop:
        c = raw_getch();
        // convert ctrl-v to page up.
        if (c == CODE_CTRL_V) {
                c = CODE_PPAGE;
        }
        // This makes BS key work.
        if (c == 127) {
                c = CODE_BS;
        }

        if (c == CODE_CTRL_L ) {
                refresh( REDRAW_ALL );
                goto loop;
        }
        return c;
}

//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: raw_readch
// 
// 
//  DESCRIPTION: Reads character fron console
// 
// 
//  RETURNS: integer value of character.
// 
//**************************************************************************/
int
Terminal_screen::raw_readch()
{
#ifdef __unix__
        
       unsigned char c;
       read(0, &c, 1);
       return (int) c;
       
#elif defined (DOS_KEYS)
       int c;
#   if defined (__EMX__)
        c = _read_kbd (0,1,1);
#   else
        c = ::getch();
#   endif
        
        if (c == 0 
#   ifdef __WATCOMC__
            || c == 224
#   endif /* __WATCOMC__ */
            ) {
#   ifdef __EMX__
                c = _read_kbd( 0, 1, 1 );
#   else
                c = ::getch();
#   endif /* __EMX__ */
                switch (c) {
                case 80:
                        c = CODE_DOWN;
                        break;
                case 72:
                        c = CODE_UP;
                        break;
                case 75:
                        c = CODE_LEFT;
                        break;
                case 77:
                        c = CODE_RIGHT;
                        break;
                case 71:
                        c = CODE_HOME;
                        break;
                case 79:
                        c = CODE_END;
                        break;
                case 73:
                        c = CODE_PPAGE;
                        break;
                case 81:
                        c = CODE_NPAGE;
                        break;
                }
        }
        return c;
#endif /* __unix__ */      
}

//**************************************************************************/
//  CLASS: Terminal_screen
//  MEMBER FUNCTION: raw_getch
// 
// 
//  DESCRIPTION: Getch that handles ESC-sequences
// 
//  RETURNS: Pressed key.
// 
//**************************************************************************/
int
Terminal_screen::raw_getch()
{
	int ch;
	int keyvalue;
        xkey = 0;
        keyvalue =  raw_readch ();
        if (keyvalue == 27) {
                ch = raw_readch();
                if (ch == '[') {
                        ch = raw_readch();
                }
                switch (ch) {
                case 'A':
                case 'i':
                        keyvalue = CODE_UP;
                        break;
                case 'B':
                        keyvalue = CODE_DOWN;
                        break;
                case 'D':
                        keyvalue = CODE_LEFT;
                        break;
                case 'C':
                        keyvalue = CODE_RIGHT;
                        break;
                case 'I':   // ansi  PgUp 
                case 'V':   // at386 PgUp
                        keyvalue = CODE_PPAGE;
                        break;
                case 'G':   // ansi  PgDn
                case 'U':   // at386 PgDn
                case 'v':   // emacs PgDn
                        keyvalue = CODE_NPAGE;
                        break;
                case 'H':   // at386 Home
                case '<':   // emacs Home
                        keyvalue = CODE_HOME;
                        break;
                case 'F':   // ansi  End
                case 'Y':   // at386 End
                case '>':   // emacs End
                        keyvalue = CODE_END;
                        break;
                case '5':   // vt200 PgUp
                        ch = raw_readch ();	
                        keyvalue = CODE_PPAGE;
                        break;
                case '6':   // vt200 PgDn
                        ch = raw_readch ();	
                        keyvalue = CODE_NPAGE;
                        break;
                case '1':   // vt200 PgUp
                        ch = raw_readch ();	
                        keyvalue = CODE_HOME;
                        break;
                case '4':   // vt200 PgUp
                        ch = raw_readch ();
                        keyvalue = CODE_END;
                        break;
                case '3':   // Delete
                        ch = raw_readch();
                        if (ch == '~') {
                                keyvalue = CODE_DEL;
                        }
                        break;
                default:
                        xkey = ch;
                        break;
                }
                
        }
        DEBUG("Keyvalue = " << keyvalue << " : xkey = " << xkey );
        return keyvalue;
}

//**************************************************************************/
// CLASS: Terminal_screen
// MEMBER FUNCTION: print
//**************************************************************************/ 
//
// Prints given format string to a screen
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS: Format specifier and variables.
//
// RETURN: Number of printed characters
//**************************************************************************/
int
Terminal_screen::print( const char* fmt, ...)
{
        va_list ap;
        int rc;
        
        va_start( ap, fmt );
        
        rc = stdscr->vprint( fmt, ap );

        va_end( ap );

        return rc;
}


//**************************************************************************/
//  CLASS: Screen
//  MEMBER FUNCTION: Refresh
// 
// 
//  DESCRIPTION: Refresh screen by drawing contents of the virtual screen
//               to a physical screen.
//               If flag is REDRAW_ALL, then all is redrawed, otherwise
//               only changed part of the screen is drawed.
// 
//**************************************************************************/
void
Terminal_screen::refresh( int flag )
{
        DEBUG("Refreshing screen");
        stdscr->update();
        if (winlist.first() == true ) {
                do {
                        winlist.get()->update();
                } while (winlist.next() == true );
        }        
        if (flag == REDRAW_ALL) {
                cleared = true;
        }
        if (screen_is_active == true) {
                drawscreen();
                cleared = false;
                fflush( stdout );
        }
        DEBUG("Screen refreshed!");
}

//**************************************************************************/
// CLASS: Terminal_screen
// MEMBER FUNCTION: set_term_attr
//**************************************************************************/ 
//
// Sets terminal attribute
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: -
//**************************************************************************/
#ifdef USE_STDOUT
void
Terminal_screen::set_term_attr( int attr )
{
        static char buf[BUFSIZ];
        strcpy( buf, term_me );

        if (attr & BOLD) {
                strcat( buf, term_md);
        }
        if (attr & BLINK) {
                strcat( buf, term_mb);
        }

        // underline works only with mono displays
        if (attr & UNDERLINE) {
                if (color_term == false) {
                        strcat( buf, term_us );
                }
        }

        if (attr & INVERSE) {
                strcat( buf, term_mr );
        }

        if (color_term == true ) {
        
                if (! (attr & DEFAULT_BCOLOR) ) {
                        char tmp[10];
                        int color = GET_BCOLOR( attr );
                        sprintf( tmp, "\033[%dm", color + 40 );
                        strcat( buf, tmp );
                }
                if (! (attr & DEFAULT_FCOLOR) ) {
                        char tmp[10];
                        int color = GET_FCOLOR( attr );
                        sprintf( tmp, "\033[%dm", color + 30 );
                        strcat( buf, tmp );
                }
        }
        printf( buf );

}
#endif

//**************************************************************************/
// CLASS: Terminl_screen
// MEMBER FUNCTION: update
//**************************************************************************/ 
//
// Draws contents of given window to the virtual screen.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Terminal_screen::update( attr_t* win, int x, int y, int w, int h,
                         int xpos, int ypos)
{
        int h1, h2;
        register int yy, ww;
        register int o1;
        attr_t *buf = lastscr;

        _xpos = x + xpos;
        _ypos = y + ypos;
        DEBUG("Updating window to screen buffer");
        // return immediately, if window is not in visible area
        if ( (x + w < 0) || (y + h < 0) ||
             (x >= _xmax) || (y >= _ymax) ) {
                return;
        }

        if ( y >= 0 ) {
                h1 = 0;
                h2 = (y + h > _height) ? _height-y : h ;
                buf += y * _width;
        } else {
                h1 = -y;
                h2 = (y + h > _height) ? _height : y + h;
                
        }
        o1 = h1 * w;
        if ( x >= 0 ) {
                ww = (x + w > _width)  ? _width - x  : w ;
                buf += x;
        } else {
                o1 -= x;  // note, x is negative...
                ww = (x + w > _width) ?  _width : x + w;
        }

        ww *= sizeof( buf );
        for (yy = 0; yy < h2; yy++, buf += _width, o1 += w) {
                memcpy( buf, win+o1, ww );
        }
}

//**************************************************************************/
// CLASS: Terminal_screen
// MEMBER FUNCTION: set_cur_pos
//**************************************************************************/ 
//
// Sets cursor's location at screen to given coordinates.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Terminal_screen::set_cur_pos( int x, int y)
{
#ifdef USE_ANSI
        printf( term_cm, y+1, x+1 );
#elif defined ( USE_TERMCAP )
        char *p = tgoto( term_cm, x, y );
        tputs( p, 0, putchar );
#elif defined ( OS2 )
        VioSetCurPos( (USHORT) y, (USHORT) x, 0 );
#else
        not_implemented();
#endif
}

//**************************************************************************/
// CLASS: Terminal_screen
// MEMBER FUNCTION: clear_screen
//**************************************************************************/ 
//
// Clears screen and moves cursor to top left corner.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Terminal_screen::clear_screen()
{
#ifdef USE_ANSI
        printf( term_cl );
#elif defined USE_TERMCAP
        printf( term_cl );
        set_cur_pos( 0, 0 );
#elif defined OS2
        USHORT a = 0x700 | ' ';
        register int y;
        for( y = 0; y < _ymax; ++y ) {
                VioWrtNCell( (PBYTE) &a, (USHORT) _width, (USHORT) y,
                             (USHORT) 0, 0);
        }
        VioSetCurPos( (USHORT) 0, (USHORT) 0, 0 );
#else
        not_implemented();
#endif
}

//**************************************************************************/
// CLASS: Terminal_screen
// MEMBER FUNCTION: drawscreen
//**************************************************************************/ 
//
// Makes virtual screen visible. If flag cleared is on, then whole screen
// is cleared and updated , otherwise only changed part of the screen
// is updated.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
void
Terminal_screen::drawscreen()
{
        int i;
        int y;
        DEBUG("Drawing screen");
        if ( cleared ) {
                DEBUG("Screen is empty, drawing all");
                int size = _width * _height;
                attr_t *src;
                clear_screen();
                for ( i=0; i < size; ++i) {
                        currscr[i] = lastscr[i];
                }
                for (y=0, src = currscr; y <= _ymax; ++y, src += _width) {
                        set_cur_pos( 0, y );
                        drawline( src, _width );
                }
        } else {
                DEBUG("Screen is not empty, updating only");
                set_cur_pos( 0, 0 );
                int x = 0;
                for( y = 0; y < _height; ++y) {
                        x = 0;
                        i = y * _width;
                        while ( x < _width ) {
                                if ( currscr[i] != lastscr[i] ) {
                                        break;
                                }
                                ++i;
                                ++x;
                        }
                        if ( x == _width ) {
                                continue;
                        }

                        int sx = x;
                        
                        int num = sx;
                        while (x < _width) {
                                if ( currscr[i] != lastscr[i] ) {
                                        num = x;
                                }
                                ++i;
                                ++x;
                        }
                        num -= sx;
                        ++num;
                        i = y * _width + sx;
                        memcpy( currscr+i, lastscr+i, num * sizeof( attr_t ) );
                        
                        set_cur_pos( sx, y );
                        drawline( lastscr + i, num );
                }
        }
        DEBUG("Screen drawed, setting position and attributes");
        set_cur_pos( _xpos, _ypos);
        SET_TERM_ATTR(DEFAULT_COLORS);
        DEBUG("--Ok");
}

//**************************************************************************/
// CLASS: Terminal_screen
// MEMBER FUNCTION: drawline
//**************************************************************************/ 
//
// Draws n characters from current postion to screen, starting from
// cursor's current location at screen.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  -
//   OUT:  -
// 
// PARAMETERS:
//
// RETURN: -
//**************************************************************************/
void
Terminal_screen::drawline( attr_t *start, int n )
{
#ifndef USE_STDOUT
        // hope that nobody has wider screen than 500 characters...
        static ushort_t lbuf[500];
        register ushort_t*  ap = lbuf;
        register int cnt = n;
        USHORT x, y;
        
        // fill buffer with attributes of current line
        while (cnt) {
                *ap++ = get_pc_attr_cell( *start++ );
                --cnt;
        }
        // copy attribute cells to screen
#   ifdef OS2
        VioGetCurPos( &y, &x, 0);
        VioWrtCellStr( (PBYTE) lbuf, (USHORT) (n*2), y, x, 0 );
#   else
        not_implemented();
#   endif
#else
        attr_t ch;
        attr_t curr_attr;
        attr_t used_attr = DEFAULT_COLORS;
        SET_TERM_ATTR( used_attr );
        while (n) {
                ch = *start;
                curr_attr = ATTRIBUTES( ch );
                if ( used_attr != curr_attr ) {
                        SET_TERM_ATTR( curr_attr );
                        used_attr = curr_attr;
                }
                ch &= 0xff;
                putchar( ch );
                --n;
                ++start;
        }
#endif
}

//**************************************************************************/
// CLASS: Terminal_screen
// MEMBER FUNCTION: newwin
//**************************************************************************/ 
//
// This function creates new window and adds it to the list of windows.
// New window will be topmost of all windows.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS: x, y    Top left corner of the window
//             w, h    Width and height of the window
//             style   (optional) Style of the window. WIN_DEFAULT, WIN_BORDER,
//                     WIN_TITLEBAR. These can be combined.
//             title   Text of the titlebar. optional.
//
// RETURN: pointer to created window.
//**************************************************************************/
Window*
Terminal_screen::newwin( int x, int y, int w, int h, int style,
                         const char *title)
{
        Window *win = new Window( this, x, y, w, h, style, title);
        winlist.add( win );
        return win;
}



