// $Id: select.hh,v 1.4 1998/04/30 16:30:09 jvuokko Exp $

/*****************************************************************************
 * *
 * *      MODULE:     select.hh
 * *                  ---------
 * ***************************************************************************
 * *
 * *
 * *      COPYRIGHT (C) 1997 JUKKA VUOKKO. ALL RIGHTS RESERVED
 * ***************************************************************************
 * *
 * *      Template class for interactive selection menu.
 * *
 *****************************************************************************/


#ifndef __selection_hh__
#define __selection_hh__

#include "../utilib/my_types.h"
#include "constants.h"
#include "terminal_io.hh"
#include "jmr.hh"

template <class T> class Selection {
public:
        Selection() : ymin(0), ymax(0), ypos(0), pagesize(0), first_of_page(0),
                current_page(0), current_item(0), pages(0), maxitem(0),
                win(0) {};
        virtual ~Selection() {};
        virtual T select() = 0;
protected:
        virtual bool next();
        virtual bool prev();
        virtual bool first();
        virtual bool last();
        virtual bool up();
        virtual bool down();
        virtual bool npage();
        virtual bool ppage();
        virtual void display() = 0;
        virtual void draw_page();
        virtual void draw_item( int flag = DEFAULT_FL ) = 0;
        void    goto_current_item();

        
        int ymin;
        int ymax;
        int ypos;
        int pagesize;
        int first_of_page;
        int current_page;
        int current_item;
        int pages;
        int maxitem;
        Window *win;
        
        Infoline iline;
        List<T> itemlist;
        
};


//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: prev
//**************************************************************************/ 
//
// Moves to previous positin in itemlist.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: true if success, false if not.
//**************************************************************************/
template <class T> bool
Selection<T>::prev()
{
        bool rc = itemlist.prev();
        if (rc == true) {
                --current_item;
        }
        return rc;
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: next
//**************************************************************************/ 
//
// Moves current position in itemlist to next item.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: true if success
//**************************************************************************/
template <class T> bool
Selection<T>::next()
{
        bool rc = itemlist.next();
        if (rc == true) {
                ++current_item;
        }
        return rc;
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: first
//**************************************************************************/ 
//
// Moves to beginning of the list itemlist and updates screen view.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: true if success
//**************************************************************************/
template <class T> bool
Selection<T>::first()
{
        if (itemlist.first() == false) {
                return false;
        }
        current_item = 0;
        current_page = 0;
        first_of_page = 0;
        ypos = ymin;
        draw_page();
        return true;
}
        
//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: last
//**************************************************************************/ 
//
// Moves to end of itemlist and updates screen.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: true if success
//**************************************************************************/
template <class T> bool
Selection<T>::last()
{
        if (itemlist.last() == false) {
                return false;
        }
        current_item = itemlist.count_items()-1;

        current_page = current_item / pagesize;
        first_of_page = current_page * pagesize;
        ypos = ymin + (current_item - first_of_page );
        draw_page();
        return true;
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: up
//**************************************************************************/ 
//
// Moves one step up at selection.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: true if success
//**************************************************************************/
template <class T> bool
Selection<T>::up()
{
        if (current_item > 0) {
                if (ypos == ymin) {
                        ypos = ymax;
                        
                        first_of_page = current_item - pagesize;
                        --current_item;
                        --current_page;
                        draw_page();
                } else {
                        win->gotoxy (0, ypos);
                        draw_item();
                        win->gotoxy (0,--ypos);
                        prev();
                        draw_item(EMPHASIZED_FL);
                        win->reset_attr();
                }
                String str = "L";
                str += (ypos - ymin + first_of_page +1);
                str += "/";
                str += maxitem+1;
                win->set_sline_info( str.c_str() );
                return true;
        }
        return false;
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: down
//**************************************************************************/ 
//
// Moves one position down at selection list
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: true/false
//**************************************************************************/
template <class T> bool
Selection<T>::down()
{
        if (current_item < (itemlist.count_items()-1)) {
                if (ypos == ymax) {
                        first_of_page = ++current_item;
                        ++current_page;
                        ypos = ymin;
                        draw_page();
                } else {
                        win->gotoxy (0,ypos);
                        draw_item();
                        
                        win->gotoxy (0,++ypos);
                        next();
                        draw_item (EMPHASIZED_FL);
                        win->reset_attr();
                }
                String str = "L";
                str += (ypos - ymin + first_of_page +1);
                str += "/";
                str += maxitem+1;
                win->set_sline_info( str.c_str() );
                return true;
        }
        return false;
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
template <class T> bool
Selection<T>::ppage()
{
        int i = pagesize;
        
        if (current_item == 0) {
                return false;
        }
        if (ypos == ymin) {
                while (i && first_of_page > 0) {
                        --first_of_page;
                        --current_item;
                        --i;
                }
        } else {
                while (ypos > ymin) {
                        --ypos;
                        --current_item;
                }
        }

        goto_current_item();
        
        draw_page();

        return true;
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
template <class T> bool
Selection<T>::npage()
{
        if (current_item >= maxitem) {
                return false;
        }
        if (ypos == ymax) {
                ++current_page;
                first_of_page = ++current_item;
                for (int i=1; i < pagesize && current_item < maxitem;
                     ++i, ++current_item);
                ypos = ymin + (current_item - first_of_page);
        } else {
                while (current_item < maxitem && ypos < ymax) {
                        ++current_item;
                        ++ypos;
                }
        }
        goto_current_item();

        draw_page();

        return true;
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
template <class T> void
Selection<T>::goto_current_item()
{
        int i = 0;
        itemlist.first();
        while (i < current_item) {
                itemlist.next();
                ++i;
        }
}

//**************************************************************************/
// CLASS: Selection<T>
// MEMBER FUNCTION: 
//**************************************************************************/ 
//
// 
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  
//   OUT: 
// 
// PARAMETERS:
//
// RETURN: 
//**************************************************************************/
template <class T> void
Selection<T>::draw_page()
{
        int i;
        int pros;
        int y = ymin;
        
        assert( win );
        // find first item of the current page
        if (itemlist.first() == false ) {
                win->clear();
                win->print("Nothing to select!! (this should not happen).");
                return;
        }
        i = 0;
        while (i < first_of_page) {
                ++i;
                itemlist.next();
        }
        i = current_item;
        current_item = first_of_page;

        // display items
        do {
                win->gotoxy (0,y);
                if (current_item == i) {
                        ypos = y;
                        draw_item(EMPHASIZED_FL);
                        String str = "L";
                        str += y - ymin + first_of_page +1;
                        str += "/";
                        str += maxitem+1;
                        win->set_sline_info( str.c_str() );
                } else {
                        draw_item();
                }
                ++current_item;
                ++y;
        } while (itemlist.next() == true && (y <= ymax));

        pros = (100 * current_item) / (maxitem+1);
        win->reset_attr();

        while ( y <= ymax ) {
                win->gotoxy(0,y++);
                win->clr_line();
        }

        if (maxitem < pagesize ) {
                pros = ALL_PERCENT;
        } else if ( first_of_page == 0 ) {
                pros = TOP_PERCENT;
        }

        win->set_sline_percent( pros );
        
        // Set position in list at current Group
        current_item = i;
        goto_current_item();
}



#endif

