/* $Id: qwk.cc,v 1.28 1998/05/25 20:06:07 jvuokko Exp $ */
/****************************************************************************
 * *
 * *  MODULE : qwk.cc
 * *
 * *  Copyright (c) 1997 Jukka Vuokko
 * *  See file COPYING for more information about copyrights.
 * *
 ****************************************************************************
 * *
 * *  Functions for reading QWK-packet and for handling data of packet.
 * *
 ***************************************************************************/

#include <stdio.h>  // rename()
#include <fstream.h>
#include <stdlib.h>
#include <limits.h>
#ifdef __WATCOMC__
#   include <io.h>  // access()
#else
#   include <unistd.h> // F_OK
#endif

#include "terminal_io.hh"
#include "qwk.hh"

#define QWK_DATE_LEN     8
#define QWK_TIME_LEN     5
#define QWK_MSGNUM_LEN   7
#define QWK_PASSWD_LEN   12
#define QWK_REFNUM_LEN   8
#define QWK_BLOCKCNT_LEN 6
#define QWK_MAX_NAME_LEN 25

/************************ GLOBAL VARIABLES ***********************************/

// Screen handling
extern Terminal_screen Screen;
extern Window* Wscript;


// In this module we set up username field
extern settings_t Settings; 


//----------------------------------------------------------------------
// macro for reading one line from control.dat file
//

#ifdef __unix__
// File is in stubid MS-DOG format, every linefeed is CR/LF pair
// so first read for CR, then skip over LF...
#   define READLINE() do { \
        char ch; \
        fin.getline (line, MAX_LINE_LEN, '\r'); \
        fin.get (ch); \
        } while (0)
#else
#   define READLINE() fin.getline (line, MAX_LINE_LEN)
#endif






//**************************************************************************/
//
// CLASS: Qwk
// MEMBER FUNCTION: Qwk
//
// description: Constructor. Opens message database and then reads contents
//              of QWK mail packet.
//
//              if open_new is false, then does not read messages from
//              mail packet, but reads contents of old replypacket.
//
//**************************************************************************/
Qwk::Qwk (const char* packet_name, bool open_new) : Mail (packet_name)
{
	DEBUG("Enterting Qwk::Qwk");
        if (packet_articles != NULL) {
                delete packet_articles;
        }

        packet_articles = new List<Group>;
        if (true == open_new) {
                Wscript->enable();
                Wscript->addstr("\nReading mail packet...");
                DEBUG ("Reading qwk packet...");
                Screen.refresh();
                read_control("control.dat");
                
                read_articles();

        }

        // If previous session was killed, then do not read old replypacket
        if (ACCESS (DEADJMR_FILENAME, F_OK) != 0) {
                DEBUG ("Reading old replies...");
                read_old_replies();
                DEBUG ("OK");
        }

        // initialize message database using messages from packet.
        init();
}


//**************************************************************************/
// CLASS: Qwk
// MEMBER FUNCTION: ~Qwk
//**************************************************************************/ 
//
// Destructor for class.
// If dead_flag is not set, then writes new replies to QWK replypacket.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  Settings.zipcmd, Settings.replypath
//   OUT: 
// 
//**************************************************************************/
Qwk::~Qwk()
{
        DEBUG("Entering Qwk::~Qwk()");
        // if session is terminated abnormally, then do not pack replies
        // to QWK format.
        if (dead_flag == true) {
                return;
        }
        if (have_replies() == false) {
                return;
        }
        Wscript->enable();
        Wscript->addstr( "\nWriting replypacket..." );
        DEBUG("Writing QWK replypacket");
        Screen.refresh();
        int blocks;
        fstream file;
        Message *msg;
        String fname = bbsid + ".msg";
        String cmd;
        go_group_number (NEW_REPLIES_NUMBER);
        
        file.open (fname.get(), ios::out | ios::trunc
#ifndef __unix__
                   | ios::binary
#endif
                   );
        if (file.fail()) {
                system_error ("Cannot create replypacket!");
                return;
        }

        // Write new replies to <bbsid>.msg file
        
        write_bbsid (file);
        first_article();
        do {
                msg = get_article();

                if (msg->is_killed() == false) {
                        insert_pcb_lines(msg);
                        msg->translate (Settings.bbscharset);
                        blocks = count_record_blocks (msg);
                        write_header (msg, blocks, file);
                        write_message (msg, file);
                }

        } while (next_article() == true);
        
        file.close();

        // pack replies for uploading
        
        cmd = Settings.zipcmd + " " + Settings.replypath + bbsid + ".rep "
                + fname;
        if (jmrsystem (cmd.get())) {
                String tmp = "Packet creation failed!\n";
                tmp += "Used command was : \'" + cmd + "\'";
                handle_error( tmp.get(), HALT_FL );
        }
        Wscript->addstr("OK");
        Screen.refresh();
}

//**************************************************************************/
//
// CLASS: Qwk
// MEMBER FUNCTION: read_control
//
// description: Reads data from control.dat file
//
//
// in:  
// out: Settings.username Settings.max_subject_len, Settings.is_pcboard
//
// returns: 
//
//**************************************************************************/
void
Qwk::read_control(const char *file)
{
        DEBUG("Entering Qwk::read_control");
        char line[MAX_LINE_LEN+1];
        Group *tmp;
        int i, num;
        String tmpstr;
        
        ifstream fin (file, ios::in);
        if (fin.fail()) {
                system_error ( "Cannot access QWK-file", HALT_FL);
        }
        DEBUG( "QWK packet: Reading control.dat" );
        
        READLINE();
        bbsname = line;
        
        READLINE();
        city = line;
        
        READLINE();
        phone = line;
        
        READLINE();
        sysop = line;
        READLINE();
        door = line;
        if (door.ifind ("pcboard") != NOTHING) {
                Settings.is_pcboard = true;
                Settings.max_subject_len = JMR_MAX_SUBJECT_LEN;
        } else {
                Settings.is_pcboard = false;
                Settings.max_subject_len = MAX_SUBJECT_LEN;
        }
        i = door.findch (',');
        if (i != NOTHING) {
                i++;
                tmpstr = door.downcase();
                String oldbbsid = bbsid;
                bbsid.copy (tmpstr, i);
                bbsid.cut_tail();
                if ( bbsid.icompare( oldbbsid ) != 0 ) {
                        char foo[1024];
                        sprintf( foo, "BBSID '%s' read from the QWK packet does not match with BBSID '%s' used in the name of the QWK packet", bbsid.c_str(), oldbbsid.c_str() );
                        handle_error( foo, HALT_FL );
                }
        }
        READLINE();
        time = line;
        
        READLINE();
        username = line;
        username.convert_charset (Settings.bbscharset, Settings.syscharset);
        username.s_capitalize();
        Settings.username = username;
        // skip 2 lines
        READLINE();
        READLINE();

        // read number of messages and Groups
        READLINE();
        //articles = atoi (line);
        READLINE();
        int groups = atoi (line) +1;
        DEBUG("Groups "<<groups);
        // read numbers and names of Groups
        for ( i = 0; i < groups; ++i) {
                READLINE();
                num = atoi (line);
                READLINE();
                tmpstr = line;
                tmpstr.convert_charset (Settings.bbscharset,
                                        Settings.syscharset);
                tmp = new Group (num, tmpstr);
                packet_articles->add (tmp);
                DEBUG("Group : "<<tmpstr.get()<<" read from packet. Count: "
                      << i);
        }
        fin.close();
}


//**************************************************************************/
//
// CLASS: Qwk
// MEMBER FUNCTION: read_articles
//
// description: Reads all messages from messages.dat file to 
//              ListAllMessages-list. Characterset translation is done here.
//
// in:  Settings.username
// out: 
//
//**************************************************************************/
void Qwk::read_articles()
{
        DEBUG("Entering Qwk:read_articles");
        Message *msgdata;
        Group *grp;
        bool rc = false;
        int blocks = 0;
        byte_t header[MSG_HEADER_SIZE];
        ifstream fin ("messages.dat",
#ifndef __unix__
                      ios::binary |
#endif
                      ios::in);
        if (fin.fail()) {
                system_error ("Can't open messages.dat", HALT_FL);
        }
        //articles = 0;
        fin.seekg (MSG_HEADER_SIZE);
        Wscript->enable();
        Wscript->addstr( "\nReading articles from QWK packet..." );
        Screen.refresh();
        // read articles from QWK packet
        while (fin.eof() == 0) {
                msgdata = new Message;

                // read header
                fin.read ((char*) header, MSG_HEADER_SIZE);
                if (fin.eof()) {
                        DEBUG("EOF reached while reading header!");
                        delete msgdata;
                        break;
                }
                blocks = convert_header (msgdata, header);

                DEBUG("QWK Blocks in article: " << blocks);

                // read article
                read_msg_data (msgdata, blocks, fin);
                packet_articles->first();
                assert (packet_articles->check() == true);
                rc = false;
                DEBUG("Seeking group for article: "
                      << msgdata->get_group_number());

                    // find right group for article
                do {
                        grp = packet_articles->get();
                        if (grp->get_number()==msgdata->get_group_number()) {
                                rc = true;
                                break;
                        }
                } while (true == packet_articles->next());
                                
                if ( true == rc ) {
                            // translate characterset of article
                        msgdata->set_charset (Settings.bbscharset);
                        msgdata->translate (Settings.syscharset);

                            // add article to list
                        DEBUG("Adding article to list");
                        grp->add (msgdata);

                } else {
                        // if article belongs to nonexist group, then delete it
                        // This should not happen, but you never know with
                        // the buggy BBS softwares...
                        DEBUG("No matching group found for: " <<
                              msgdata->get_group_number() << " - Trashing...");
                        delete msgdata;
                }
        }
        Wscript->addstr("OK");
        Screen.refresh();
        fin.close();
}


//**************************************************************************/
//
// CLASS: Qwk
// MEMBER FUNCTION: convert_header
//
// description: Reads data from header of QWK message, and converts
//              it to more readable format.
//
// in:  Settings
// out: 
//
// RETURN: number of QWK-blocks in message.
//**************************************************************************/
int
Qwk::convert_header (
        // pointer to messages data structure
        Message* msgdata,
        // pointer to header of the message
        byte_t* header) 
{

        String tmp;
        int blocks;
        
        // set status of article
        if (header[0] == '*') {
                msgdata->set_status_bits (UNREAD_ST | PRIVATE_ST);
        } else {
                msgdata->set_status_bits (UNREAD_ST);
        }
        
        // set number
        msgdata->set_number (get_integer (header+1, 7));

        // set date
        tmp.put (header+8, 8);
        msgdata->set_date (convert_to_jmr_date (tmp.get()));

        // set time of article
        tmp.put (header+16, 5);
        msgdata->set_time (tmp);

        // set name of receiver
        tmp.put(header+21, 25);
        tmp.cut_tail();
        tmp.s_capitalize();
        msgdata->set_receiver (tmp);

        // set name of writer
        tmp.put(header+46, 25);
        tmp.cut_tail();
        tmp.s_capitalize();
        msgdata->set_writer (tmp);

        // set subject of article
        tmp.put(header+71, 25);
        tmp.cut_tail();
        msgdata->set_subject (tmp);

        // number of reference message
        msgdata->set_reference_num (get_integer (header+108, 8));

        // number of qwk-blocks in article
        blocks = get_integer (header+116, 6);

        //msgdata->killed = header[122] == 225 ? false : true;
        msgdata->set_group_number (get_ushort (header+123));

        //  msgdata->logical_msg_number = get_ushort (header+125);
        return blocks;
}

//**************************************************************************/
//
// CLASS: Qwk
// MEMBER FUNCTION: read_msg_data
//
// description: Reads message. Lines are stored to list.
//
//
// in:  Settings
// out: 
//
// returns: 
//
//**************************************************************************/
void
Qwk::read_msg_data (
        Message* msgdata,      // pointer to message data structure
        int qwk_blocks,        // number of qwk-blocks in article.
        ifstream& fin)         // file to read from
{
        String tmp;
        int blocksize = MSG_BLOCK_SIZE * (qwk_blocks-1);
        char *block = new char[blocksize+1];
        int i;

        DEBUG("Reading message blocks. blocksize = " << blocksize);
        
        // read raw block
        fin.read (block, blocksize);
        // strip trailing spaces out.
        --blocksize;
        while (block[blocksize] == 0 || block[blocksize] == 32
               && blocksize > 0) {
                --blocksize;
        }
        block[blocksize+1]= 0;
        // convert end of line marks to nul terminators.
        for (i=0; i <= blocksize; i++) {
                if (227 == (unsigned char) block[i]) {
                        block[i] = 0;
                }
        }
        // add lines to list
        DEBUG("Adding lines from block to current message");
        DEBUG("Blocksize is without trailing spaces : " << blocksize);
        for (i=0; i <= blocksize;) {
                DEBUG("Adding line from pos: " << i );
                i += (tmp.put (block+i) + 1);
                msgdata->add_line (tmp);
        }
        DEBUG("Deleting block");
        delete[] block;

        get_pcb_data(msgdata);
        
}

//**************************************************************************/
//  CLASS: Qwk
//  MEMBER FUNCTION: get_pcb_data
// 
// 
//  DESCRIPTION: Scans 4 first lines of message, and checks if them contains
//               "@TO", "@SUBJECT" or "@FROM" that indicates long subject
//               or name. If string found, then  receiver, subject or writer
//               will be replaced with longer string.
// 
// 
//  EXTERNAL REFERENCES:
//  IN :  Settings.is_pcboard, Settings.max_subject_len
//  OUT: 
// 
//**************************************************************************/
void
Qwk::get_pcb_data (Message *msgdata)
{
        int i = 0;
        int flag = 0;
        int n = 0;
        String str;
        String tmp;
        // if bbs-software is pcboard, then check long subjects, names etc.
        // if not, then return immediately.
        if (msgdata->first_line() == false
            || Settings.is_pcboard == false) {
                return;
        }
        for (i=0; i < 4; i++) {
                str = msgdata->getline();
                flag = DEFAULT_FL;
                if ((n = str.find("@SUBJECT")) != NOTHING) {
                        flag = SUBJECT_FL;
                } else if ((n = str.find("@FROM")) != NOTHING) {
                        flag = WRITER_FL;
                } else {
                        n = str.find("@TO");
                        flag = RECEIVER_FL;
                } 
                if (n == NOTHING || n > 2) {
                        msgdata->next_line();
                } else {
                        n = str.findch (':');
                        if (n == NOTHING) {
                                msgdata->next_line();
                        } else {
                                if (flag == SUBJECT_FL) {
                                        tmp.copy (str, n+1,
                                                  Settings.max_subject_len);
                                } else {
                                        tmp.copy(str, n+1,
                                                 Settings.max_bbs_subject_len);
                                }
                                tmp.cut_first();
                                tmp.cut_tail();
                                switch (flag) {
                                case SUBJECT_FL:
                                        msgdata->set_subject (tmp);
                                        DEBUG( "PCB-subject: '" << tmp << "'");
                                        DEBUG( "Len is : " << tmp.length() );
                                        break;
                                case RECEIVER_FL:
                                        msgdata->set_receiver (tmp);
                                        break;
                                case WRITER_FL:
                                        msgdata->set_writer (tmp);
                                        break;
                                }
                                msgdata->remove_line();
                        }
                }
        }
}


//**************************************************************************/
//
// CLASS: Qwk
// MEMBER FUNCTION: read_old_replies
//
// description: Checks if old replypacket exists. If it exists, then
//              ask if user want continue with it. Otherwise replypacket
//              will be renamed using .old -suffix
//
//              If user does not continue with these replies, then
//              replies are moved to replylog.
//
// in:  Settings.replypath, Settings.unzipcmd
//
//**************************************************************************/
void
Qwk::read_old_replies (const int flag)
{
        ifstream fin;
        Message *msg;
        Group *grp;
        byte_t header[MSG_HEADER_SIZE+1];
        String cmd;
        String str;
        int c = 0;
        int blocks = 0;

        if (flag & DEATH_FL) {
                str = DEADJMR_FILENAME;
                grp = new Group (NEW_REPLIES_NUMBER, NEW_REPLIES_TEXT);
        } else {
                cmd = Settings.replypath + bbsid + ".rep";

                if (ACCESS (cmd.get(), F_OK)) {
                        return;
                }
                Wscript->enable();
                Wscript->addstr( "\n\nOld replypacket found. Continue with it?"
                                 " (y/n) " );
                Wscript->textcolor( BOLD );
                Wscript->addch( 'y' );
                c = get_yesno('y');

                if (c == 'n') {
			Wscript->addstr( "\bNo!\n" );
                        grp = new Group (REPLYLOG_NUMBER, REPLYLOG_TEXT);
                } else {
			Wscript->addstr( "\bYes!\n" );
                        grp = new Group (NEW_REPLIES_NUMBER, NEW_REPLIES_TEXT);
                }
		Wscript->reset_attr();
		Screen.refresh();
                
                // open old packet
                str = Settings.unzipcmd + " " + cmd;
                jmrsystem (str.get());
                str = bbsid + ".msg";
        }
        
        packet_articles->add (grp);
        
        fin.open (str.get(), ios::in
#ifndef __unix__
                  | ios::binary
#endif
                  );
        if (fin.fail()) {
                handle_error ("Can't read old replies.");
                return;
        }

        fin.seekg (MSG_HEADER_SIZE);
        DEBUG( "Converting charset of replies from " << Settings.bbscharset
               << " to " << Settings.syscharset );
        while (fin.eof() == 0) {
                msg = new Message;
                fin.read ((char*) header, MSG_HEADER_SIZE);
                if (fin.eof()) {
                        break;
                }
                blocks = convert_header (msg, header);
                read_msg_data (msg, blocks, fin);
                msg->set_charset (Settings.bbscharset);
                msg->translate (Settings.syscharset);
                msg->remove_tail();
                grp->add (msg);
        }
        fin.close();
        // store old replies
        if (c == 'n' && !(flag & DEATH_FL)) {
                str = Settings.replypath + bbsid + ".old";
                remove (str.get());
                rename (cmd.get(), str.get());
        }
}


//**************************************************************************/
//  CLASS: Qwk
//  MEMBER FUNCTION: count_record_blocks
// 
// 
//  DESCRIPTION: Counts number of QWK-blocks in article
// 
// 
//**************************************************************************/
int
Qwk::count_record_blocks (Message *msg)
{
        int blocks;
        int bytes = 0;
        if (msg->first_line() == true) {
                do {
                        bytes += msg->getline().length();
                        ++bytes;
                } while (msg->next_line() == true);
        }
        blocks = bytes / MSG_BLOCK_SIZE;
        if (bytes % MSG_BLOCK_SIZE != 0) {
                ++blocks;
        }
        return ++blocks;
}


//**************************************************************************/
//  CLASS: Qwk
//  MEMBER FUNCTION: write_header
// 
// 
//  DESCRIPTION: Writes QWK-header of article to file
// 
// 
//  EXTERNAL REFERENCES:
//  IN :  Settings
//  OUT: 
// 
//  RETURNS: 
// 
//**************************************************************************/
void
Qwk::write_header (Message *msg, const int blocks, fstream& file)
{
        static int counter = 0;
        int i;
        String tmp;
        streampos pos = file.tellp();
        // tytetn otsikko spaceilla
        for (i=0; i < MSG_BLOCK_SIZE; i++) {
                file << " ";
        }
        file.seekp (pos);
        if (msg->get_status() & PRIVATE_ST) {
                file << "*";
        } else {
                file <<" ";
        }
        file << msg->get_number();
        file.seekp (pos+8);
        file << convert_to_qwk_date (msg->get_jmrdate().get());
        
        file.seekp (pos+16);
        file << msg->get_time();
        
        file.seekp (pos+21);
        tmp.copy( msg->get_receiver(), 0, QWK_MAX_NAME_LEN );
        tmp.s_upcase();
        file << tmp;
        
        file.seekp (pos+46);
        tmp.copy( msg->get_writer(), 0, QWK_MAX_NAME_LEN );
        tmp.s_upcase();
        file << tmp;
        
        file.seekp (pos+71);
        tmp.copy( msg->get_subject(), 0, MAX_SUBJECT_LEN );
        file << tmp;
        
        file.seekp (pos+108);
        file << msg->get_reference_num();
        
        file.seekp (pos+116);
        file << blocks;
        
        file.seekp (pos+122);
        file << (char) 225;
        file << (char) LOW (msg->get_group_number());
        file << (char) HIGH (msg->get_group_number());
        file << (char) LOW (counter);
        file << (char) HIGH (counter);
        file << " ";
        ++counter;
}
//**************************************************************************/
//  CLASS: Qwk
//  MEMBER FUNCTION: write_message
// 
// 
//  DESCRIPTION: Writes QWK-message to file
// 
// 
//  EXTERNAL REFERENCES:
//  IN :  Settings
//  OUT: 
// 
//**************************************************************************/
void
Qwk::write_message (Message *msg, fstream& file)
{
        int bytes = 0;
        
        if (msg->first_line() == true) {
                do {
                        bytes += msg->getline().length();
                        file << msg->getline();
                        ++bytes;
                        file << (char) 227;
                } while (msg->next_line() == true);
        }
        if ((bytes = bytes % MSG_BLOCK_SIZE) != 0) {
                while (bytes < MSG_BLOCK_SIZE) {
                        file << " ";
                        ++bytes;
                }
        }
}
//**************************************************************************/
//  CLASS: Qwk
//  MEMBER FUNCTION: write_bbsid
// 
// 
//  DESCRIPTION: Writes bbsid-block to start of qwk-file.
// 
// 
//**************************************************************************/
void
Qwk::write_bbsid (fstream& file)
{
        int i = 0;
        char ch;
        while ((ch = bbsid.nget(i)) > 0) {
                file.write ((char*) &ch, 1);
                i++;
        }
        ch = ' ';
        while (i < MSG_BLOCK_SIZE) {
                file.write ((char*) &ch, 1);
                i++;
        }
}

//**************************************************************************/
//  CLASS: Qwk
//  MEMBER FUNCTION: insert_pcb_lines
// 
// 
//  DESCRIPTION: Inserts PCBOARD-lines to start of article, if current
//               bbs is PCBOARD.
// 
// 
//  EXTERNAL REFERENCES:
//  IN :  Settings.is_pcboard
//  OUT: 
// 
//**************************************************************************/
void
Qwk::insert_pcb_lines(Message* msg)
{
        if (Settings.is_pcboard == false) {
                return;
        }
        if (msg->get_subject().length() > Settings.max_bbs_subject_len) {
                DEBUG( "Setting up PCB subject line" );
                String tmp = "\377@SUBJECT:" + msg->get_subject();
                int c;
                c = 70 - tmp.length();
                tmp.append (' ', c);
                tmp += "N";
                msg->insert_line (tmp);
        }
}



//
// this is for Gcc 2.7.2 templates....
//
#ifdef __OLD_GCC__
template class List<String>;
#endif




