// 	$Id: message_menu.cc,v 1.15 1997/07/17 15:35:03 jvuokko Exp $	

/****************************************************************************
 * *
 * *  MODULE : message_menu.cc
 * *
 * *  Copyright (c) 1997 Jukka Vuokko
 * *  See file COPYING for more information about copyrights.
 * *
 ****************************************************************************
 * *
 * *
 * *  Functions for thread selection level.
 * *  
 * *
 * *
 * *
 ***************************************************************************/

#include <fstream.h>
#include <stdlib.h>   // abs()
#ifdef __WATCOMC__
#   include <io.h>    // F_OK
#else
#   include <unistd.h>
#endif
#include "jmr.hh"
#include "terminal_io.hh"
#include "menu.hh"
#include "mail.hh"
#include "menubar.hh"

extern Terminal_screen Screen;
extern settings_t Settings;
extern Mail* mail;
extern Window* Wscript;

static const menuinfo thread_menu_array[] = {
        {"[jmr]", 0 },
        {"Exit from menu", 0},
        {"Return to group selection", QUIT_CMD},
        {"Quit jmr", FORCED_QUIT_CMD},
        {"@NEXT", 0 },
        {"Write", WRITE_CMD },
        {"@NEXT", 0 },
        {"Yank in/out", YANK_IN_OUT_CMD },
        {"@NEXT", 0 },
        {"@END", 0 }
};

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: Msgmenu
//
// description: Constructor
//
//
// in:  Screen
// out: 
//
// returns: 
//
//**************************************************************************/
Msgmenu::Msgmenu ()
{
        show_all_articles = false;
        is_status_msg = false;
        thread_mode = true;
        thread_cnt = 0;

        ymin = 6;
        ymax = Screen.get_ymax() - 2;
        lines_per_page = ymax - ymin + 1;
        pages = 0;
        first_of_page = 0;
        current = 0;
        ypos = ymin;

        menubar.init( thread_menu_array );
        menubar.set_text( MENUBAR_TEXT );

        for (int i=Screen.get_xmax(); i >= 0; i--) {
                dashline += '-';
        }
}


//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: enter
//
// description: Main procedure for class. Handles events from input.
//
// NOTE: Position in ListGroup-list must be in right place
//
// in:  mail, Screen, Quit_flag
// out: Quit_flag
//
// returns: 
//
//**************************************************************************/
void
Msgmenu::enter ()
{
        int c;
        int flag;
        bool quit = false;
        bool rc = false;
        
        show_all_articles = false;
        
        c = mail->get_group_number();
        if (mail->count_articles() > 0 && mail->count_unread() == 0) {
                show_all_articles = true;
        }

        mail->first_article();

        threads.update( show_all_articles );

        sline.init( Screen.get_xmax() + 1);

        Msgdisplay msgdisp( threads );
        update();
        display();

        do {
                c = Screen.get_ch();
                flag = DEFAULT_FL;
                iline.reset();
                if (c == ESCAPE) {
                        iline.set( MENU_MSG );
                        iline.show();
                        c = menubar.select();
                        display();
                        iline.reset();
                }
                switch (c) {
                case MESSAGEWIN_CMD:
                        Wscript->enable();
                        iline.set ( MESSAGES_MSG );
                        iline.show();
                        Screen.get_ch();
                        iline.reset();
                        Wscript->disable();
                        break;
                case NEXT_UNREAD_CMD:
                        if (go_next_article() == true) {
                                flag = msgdisp.enter(&show_all_articles,
                                                     NEXT_UNREAD_FL);
                                update_msg_counts();
                                set_selector_to_current_thread();
                                display ();
                        } else {
                                iline.set( NO_UNREAD_MSG, ERROR_FL );
                        }
                        break;
                case SELECT_CMD: 
                case CODE_RIGHT:
                        if (threads.check() == true) {
                                flag = msgdisp.enter(&show_all_articles,
                                                     DEFAULT_FL);
                                update_msg_counts();
                                set_selector_to_current_thread();
                                display ();
                        }
                        break;
                case REPLY_CMD:
                        if (threads.check() == false) {
                                break;
                        }
                        mail->reply(threads.get_article());
                        if (mail->get_group_number() == NEW_REPLIES_NUMBER) {
                                threads.update( show_all_articles );
                                update();
                        }
                        update_msg_counts();
                        display();
                        break;
                case FOLLOWUP_CMD:
                        if (threads.check() == false) {
                                break;
                        }
                        mail->followup(threads.get_article());
                        if (mail->get_group_number() == NEW_REPLIES_NUMBER) {
                                threads.update( show_all_articles );
                                update();
                        }
                        update_msg_counts();
                        display();
                        break;
                case EDIT_REPLY_CMD:
                        if (mail->get_group_number() == NEW_REPLIES_NUMBER) {
                                mail->edit(threads.get_article());
                                threads.update( show_all_articles );
                                update();
                                update_msg_counts();
                                display();
                        }
                        break;
                case 'E':
                        if (mail->get_group_number() == NEW_REPLIES_NUMBER) {
                                mail->edit_header(threads.get_article());
                                threads.update( show_all_articles );
                                update();
                                update_msg_counts();
                                display();
                        }
                        break;
                case KILL_CMD:
                        if (mail->get_group_number() > PERSONAL_NUMBER) {
                                threads.get_article()->invert_kill_flag();
                                Screen.gotoxy (0, ypos);
                                draw_article_info (EMPHASIZED_FL);
                        }
                        break;
                case WRITE_CMD:
                        if (mail->get_group_number() < BASEGROUP_NUMBER) {
                                mail->write();
                                update_msg_counts();
                                display();
                        }
                        break;
                case QUIT_CMD:
                case CODE_LEFT:
                        quit = true;
                        break;
                case CODE_DOWN:
                case NEXT_CMD:
                        if (false == down()) {
                                iline.set( AT_BOTTOM_MSG );
                        }
                        break;
                case CODE_UP:
                case PREV_CMD:
                        if (false == up()) {
                                iline.set( AT_TOP_MSG );
                        }
                        break;
                case CODE_HOME:
                        update();
                        //display();
                        draw_list();
                        break;
                case CODE_END:    
                        go_last_thread();
                        draw_list();
                        break;
                case CODE_PPAGE:  // same as PAGE_UP_CMD
                case PAGE_UP_CMD:
                        if (false ==page_up()) {
                                iline.set( AT_FIRST_PAGE_MSG );
                        }
                        break;
                case CODE_NPAGE:
                case PAGE_DOWN_CMD:
                        if (false ==page_down()) {
                                iline.set( AT_LAST_PAGE_MSG );
                        }
                        break;
                case CATCHUP_CMD:
                        quit = catchup();
                        break;
                case FORCED_CATCHUP_CMD:
                        mail->catchup();
                        quit = true;
                        break;
                case YANK_IN_OUT_CMD:
                        show_all_articles = show_all_articles == false ?
                                true : false;
                        threads.update( show_all_articles );
                        update();
                        display();
                        break;
                case SORT_CMD:
                        if (true == sort()) {
                                update();
                                display();
                                iline.set("Articles sorted!");
                        } else {
                                display();
                                iline.show( REFRESH_ALL_FL );
                        }
                        break;
                case TAG_CMD:
                        if (mail->get_sort_order() == THREAD_SORTED) {
                                iline.set(TAGGING_NOT_ALLOWED_MSG, ERROR_FL);
                                break;
                        }
                        if (mail->get_group_number() != REPLYLOG_NUMBER &&
                            mail->get_group_number() != NEW_REPLIES_NUMBER) {
                                if (!threads.check()) {
                                        iline.set("No articles to tag!",
                                                  ERROR_FL);
                                        break;
                                }
                                mail->tag_article (threads.get_article());
                                flag = TAGGED_FL;
                                Screen.gotoxy (0, ypos);
                                draw_article_info (EMPHASIZED_FL);
                        }
                        break;
                case TAG_ALL_CMD:
                        if (mail->get_sort_order() == THREAD_SORTED) {
                                iline.set( TAGGING_NOT_ALLOWED_MSG, ERROR_FL );
                                break;
                        }
                        if (mail->get_group_number() != REPLYLOG_NUMBER &&
                            mail->get_group_number() != NEW_REPLIES_NUMBER) {
                                rc = threads.tag_all();
                                flag = TAGGED_FL;
                                //update();
                                display();
                                if (rc == true) {
                                        iline.set("All articles tagged!");
                                } else {
                                        iline.set("All tags removed");
                                }
                        }
                        break;
                case FORCED_QUIT_CMD:
#ifndef NO_EXCEPTIONS
                        throw quit_exception();
#else
                        Quit_flag = true;
                        break;
#endif
                case HELP_CMD:
                        show_help();
                        display();
                        break;
                case 0:
                        break;
                default:
                        iline.set( INVALID_COMMAND_MSG );
                        break;
                }; // end of switch-case
                
                if ( mail->get_group_number() == TAGGED_ARTICLES_NUMBER &&
                     ( flag & TAGGED_FL ) ){
                        threads.update( show_all_articles );
                        update();
                        update_msg_counts();
                        display();
                }
                iline.show();
        } while (quit == false
#ifdef NO_EXCEPTIONS
                 && Quit_flag == false
#endif
                 );
        mail->unload_articles_of_group();
        
        // update read-counts in personal mail group
        mail->save_group_pos();
        mail->go_group_number (PERSONAL_NUMBER);
        mail->update_groups_counts();
        mail->restore_group_pos();

        Screen.bottom();
        Screen.clr_eol();
}

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: 
//
// description: Inits some variables.
//
//
//**************************************************************************/
void
Msgmenu::update()
{
        thread_cnt = threads.count_threads();
        pages = thread_cnt / lines_per_page;
        pages++;
        current_page = 0;
        first_of_page = 0;
        current = 0;
        ypos = ymin;
        if (mail->get_sort_order() == THREAD_SORTED) {
                thread_mode = true;
        } else {
                thread_mode = false;
        }
        if (mail->get_group_number() == NEW_REPLIES_NUMBER) {
                iline.set_default(THREAD_MENU_NEW_REPLIES_KEYS_MSG);
        } else if (mail->get_group_number() == REPLYLOG_NUMBER ||
                   mail->get_group_number() == TAGGED_ARTICLES_NUMBER) {
                iline.set_default( THREAD_MENU_BASEGROUP_KEYS_MSG );
        } else {
                iline.set_default( THREAD_MENU_KEYS_MSG );
        }
}
//**************************************************************************/
// CLASS: Msgmenu
// MEMBER FUNCTION: catchup
//**************************************************************************/ 
//
// Mark all articles of current group as read if user answers 'y' to
// question.
// 
// EXTERNAL VARIABLE REFERENCES
//   IN :  Screen, mail
//   OUT:  
// 
// RETURN: true if articles are marked as read
//**************************************************************************/
bool
Msgmenu::catchup()
{
        bool rc = false;
        Screen.reset_attr();
        Screen.bottom();
        show_option_str( "Really want mark all articles as read (y/n) @y@" );
        Screen.clr_eol();
        if (get_yesno ('y') == 'y') {
                mail->catchup();
                rc = true;
        } else {
                iline.show( REFRESH_ALL_FL );
        }
        return rc;
        
}



//**************************************************************************/
//  CLASS: Msgmenu
//  MEMBER FUNCTION: go_last_thread
// 
// 
//  DESCRIPTION: Moves position to last thread of group.
// 
// 
//  EXTERNAL REFERENCES
//  IN :  
//  OUT: 
// 
//  RETURNS: 
// 
//**************************************************************************/
void
Msgmenu::go_last_thread()
{
        threads.last();
        set_selector_to_current_thread();
}
        
//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: display
//
// description: Draws everything.
//
//  EXTERNAL REFERENCES
//  IN : Quit_flag 
//  OUT: 
//
//**************************************************************************/
void
Msgmenu::display ()
{
        String tmpstr;
        String tmpstr2;
        Screen.erase();
#ifdef NO_EXCEPTIONS
        if (Quit_flag == true) {
                return;
        }
#endif
        if (pages == 0) {
                update();
        }

        menubar.show();
        draw_header();

        tmpstr2.copy (mail->get_group_name(), 0, Screen.get_xmax() - 45 );
        tmpstr = "Grp:" + tmpstr2;        
        if (show_all_articles == true ) {
                tmpstr += " (All)" ;
        } else {
                tmpstr += " (Unread)";
        }

        if (thread_mode == true) {
                tmpstr += "--Threads:";
                tmpstr += thread_cnt;
        } else {
                tmpstr += "--Order:";
                switch ( mail->get_sort_order() ) {
                case SUBJECT_SORTED:
                        tmpstr += "Subject";
                        break;
                case DATE_SORTED:
                        tmpstr += "Date";
                        break;
                case WRITER_SORTED:
                        tmpstr += "Author";
                        break;
                case RECEIVER_SORTED:
                        tmpstr += "Receiver";
                        break;
                };
        }
        sline.set_infotext( tmpstr.get() );
        
        tmpstr = "Unread:";
        tmpstr += mail->count_unread();
        tmpstr += "/";
        tmpstr += mail->count_articles();

        sline.set_extra_info( tmpstr.get() );

        draw_list();

}

void
Msgmenu::draw_list()
{
        int i;
        int y;
        int pros;

        Screen.reset_attr();
        if (false == threads.go_index( first_of_page )) {
                Screen.gotoxy (0,ymin);
                Screen.textcolor (BOLD);
                Screen.addstr( "No unread articles in this group" );
                Screen.reset_attr();
        } else {
                i = first_of_page;
                y = ymin;

                do {
                        Screen.gotoxy (0,y);
                        if (i == current) {
                                draw_article_info (EMPHASIZED_FL);
                        } else {
                                draw_article_info();
                        }
                        y++;
                        i++;
                } while (threads.next() == true && (y <= ymax));

                while( y <= ymax) {
                        Screen.gotoxy(0,y++);
                        Screen.empty_line();
                }
                
                if (thread_cnt <= lines_per_page) {
                        sline.set_percent( ALL_PERCENT );
                } else if ( current == 0 ) {
                        sline.set_percent( TOP_PERCENT );
                } else if (i == thread_cnt) {
                        sline.set_percent( BOT_PERCENT );
                } else {
                        pros = 100*i / thread_cnt;
                        sline.set_percent( pros );
                }
                threads.go_index( current );
        }
        sline.show();
}

void
Msgmenu::draw_header()
{
        Screen.gotoxy( 0, 1 );
        Screen.reset_attr();
        Screen.print( "Group [%d] - ", mail->get_group_number() );
        Screen.textcolor( Settings.header_subject_color );
        Screen.addstr( mail->get_group_name() );
        Screen.reset_attr();
        
        Screen.gotoxy( 0, 2 );
        if (mail->get_sort_order() == THREAD_SORTED) {
                Screen.print( "Threads : %d", thread_cnt );
        } else {
                Screen.print( "Articles : %d", thread_cnt);
        }
        Screen.gotoxy( 0, 3 );
        Screen.print( "Unread articles : %d / %d", mail->count_unread(),
                      mail->count_articles());
        Screen.gotoxy( 0, 5 );
        set_dashline_color();
        Screen.addstr( dashline );
        if (mail->get_sort_order() == THREAD_SORTED ) {
                Screen.gotoxy( 9, 5 );
                Screen.addstr( " Thread: " );
                Screen.gotoxy( Screen.get_xmax() - 18, 5 );
                Screen.addstr( " Author: " );
        } else {
                Screen.gotoxy( 9, 5 );
                Screen.addstr( " Date: " );
                Screen.gotoxy( 20, 5 );
                Screen.addstr( " Subject: " );
                Screen.gotoxy( Screen.get_xmax() - 33, 5 );
                if (mail->get_group_number() == NEW_REPLIES_NUMBER ||
                    mail->get_group_number() == REPLYLOG_NUMBER) {
                        Screen.addstr( " Orig. group: " );
                } else {
                        Screen.addstr( " Author: " );
                }
                Screen.gotoxy( Screen.get_xmax() - 15, 5 );
                if (mail->get_group_number() == PERSONAL_NUMBER ||
                    mail->get_group_number() == TAGGED_ARTICLES_NUMBER) {
                        Screen.addstr( " Orig. Group: " );
                } else {
                        Screen.addstr( " Receiver: " );
                }
        }
        iline.show( REFRESH_ALL_FL );
        Screen.reset_attr();
                
}

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: draw_article_info
//
// description: Draws either thread data or detailed data from message.
//
//
// in:  
// out: 
//
// returns: 
//
//**************************************************************************/
void
Msgmenu::draw_article_info (int flag)
{
        if (thread_mode == true) {
                draw_thread (flag);
        } else {
                draw_detailed (flag);
        }
}
//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: draw_detailed
//
// description: Draws detailed data about current message.
//              If flag is EMPHASIZED_FL, then uses inverse mode.
//
// in:  Screen
// out: 
//
// returns: 
//
//**************************************************************************/
void
Msgmenu::draw_detailed (int flag)
{
        String str;
        int xmax = Screen.get_xmax();
        Message *msg = threads.get_article();

        if (flag & EMPHASIZED_FL) {
                Screen.inverse();
        }
        Screen.empty_line();
        
        Screen.print( "%5d", threads.get_index() );
        Screen.goto_x( 6 );
                
        if (threads.count_threads_unread() > 0) {
                Screen.addstr ("+");
        } 

        Screen.goto_x( 7 );
        if (msg->get_status() & TAGGED_ST) {
                if (!(flag & EMPHASIZED_FL)) {
                        Screen.textcolor (RED | BOLD);
                        Screen.addstr( "T" );
                        Screen.reset_attr();
                } else {
                        Screen.addstr( "T" );
                }
        } 

        if (msg->get_status() & REPLIED_ST) {
                if (!(flag & EMPHASIZED_FL)) {
                        Screen.textcolor (YELLOW | BOLD);
                        Screen.addch( 'R' );
                        Screen.reset_attr();
                } else {
                        Screen.addch( 'R' );
                }
        } 


        Screen.goto_x( 10 );
        Screen.addstr( msg->get_date() );

        if (!(flag & EMPHASIZED_FL)) {
                Screen.textcolor (Settings.subject_color);
        }

        Screen.goto_x( 21 );
        Screen.addstr( str.copy( msg->get_subject(), 0, xmax - 54 ) );
        if (!(flag & EMPHASIZED_FL)) {
                Screen.reset_attr();
        } 

        Screen.goto_x( xmax - 33 );
        Screen.addch ( ' ' );
        if (msg->is_killed() == true) {
                Screen.textcolor (RED | BOLD | BLINK);
                if (flag & EMPHASIZED_FL) {
                        Screen.inverse();
                }
                Screen.addstr( "!!! Killed !!!" );
                Screen.reset_attr();
        } else {
                if (mail->get_group_number() == NEW_REPLIES_NUMBER ||
                    mail->get_group_number() == REPLYLOG_NUMBER) {
                        Screen.addstr( "Grp.No: " );
                        if (! (flag & EMPHASIZED_FL)) {
                                Screen.textcolor( BOLD );
                                Screen.print ( "%d",msg->get_group_number());
                                Screen.reset_attr();
                        } else {
                                Screen.print( "%d",msg->get_group_number());
                        }
                } else {
                        Screen.addstr( str.copy( msg->get_writer(), 0, 17 ) );
                }
                Screen.goto_x( xmax - 15 );
                Screen.addch( ' ' );
                if (mail->get_group_number() == PERSONAL_NUMBER ||
                    mail->get_group_number() == TAGGED_ARTICLES_NUMBER) {
                        Screen.addstr( "Grp.No: " );
                        if (! (flag & EMPHASIZED_FL)) {
                                Screen.textcolor( BOLD );
                                Screen.print( "%d", msg->get_group_number() );
                                Screen.reset_attr();
                        } else {
                                Screen.print( "%d", msg->get_group_number() );
                        }
                } else {
                        if (msg->get_receiver() == Settings.username) {
                                Screen.textcolor (BOLD);
                                if (flag & EMPHASIZED_FL) {
                                        Screen.inverse();
                                }
                        }
                        Screen.addstr( str.copy (msg->get_receiver(), 0, 14) );
                }
        }
        Screen.goto_x( 0 );
        Screen.reset_attr();
}

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: draw_thread
//
// description: Draws data of current thread. If flag is EMPHASIZED_FL then
//              uses inverse-mode.
//
// in:  Screen
// out: 
//
// returns: 
//
//**************************************************************************/
void
Msgmenu::draw_thread (int flag)
{        
        int i;
        int y = Screen.get_y();
        String str;
        Message *msg = threads.get_article();
        
        if (flag & EMPHASIZED_FL) {
                Screen.inverse();
        }
        Screen.empty_line();

        Screen.print( "%3d ", threads.get_index() );

        if (show_all_articles == true) {
                i = threads.count_threads_articles();
        } else {
                i = threads.count_threads_unread();
        }
        if (i) {
		if (! (flag & EMPHASIZED_FL)) {
                        Screen.textcolor ( BOLD );
		}
                Screen.print( "%5d", i );
		if (! (flag & EMPHASIZED_FL)) {
			Screen.reset_attr();
		}

        }

        Screen.goto_x( 10 );
        if (!(flag & EMPHASIZED_FL)) {
                Screen.textcolor (Settings.subject_color);
        }
        Screen.addstr( str.copy( msg->get_subject(),0,Screen.get_xmax() - 27));
        if (!(flag & EMPHASIZED_FL)) {
                Screen.reset_attr();
        }

        Screen.goto_x(Screen.get_xmax() - 17);
        if (msg->is_killed() == true) {
                Screen.textcolor (RED | BOLD | BLINK);
                if (flag & EMPHASIZED_FL) {
                        Screen.inverse();
                }
                Screen.addstr( "!!! Killed !!!" );
                Screen.reset_attr();
        } else {
		str.copy (msg->get_writer(), 0, 16);
                i = str.length();
                str.append (' ', 16 - i);
		if (!(flag & EMPHASIZED_FL)) {
			Screen.textcolor (RED);
			Screen.addch ('[');
			Screen.textcolor (Settings.author_color);
			Screen.addstr( str );
			Screen.textcolor (RED);
			Screen.addch( ']' );
			Screen.reset_attr();
		} else {
                        Screen.print( "[%s]", str.c_str() );
		}
        }
        Screen.gotoxy( 0, y );
        Screen.reset_attr();
}


//**************************************************************************/
//  CLASS: Msgmenu
//  MEMBER FUNCTION: page_up
// 
//  DESCRIPTION: If current position is top of page, then switch to
//               previous page, otherwise sets current to first of page.
//
//  RETURN: false if failed
//**************************************************************************/
bool
Msgmenu::page_up()
{
        int i = lines_per_page;
        if (current == 0) {
                return false;
        }
        if (ypos == ymin) {
                while (i && first_of_page > 0) {
                        first_of_page--;
                        current--;
                        i--;
                }
        } else {
                while (ypos > ymin) {
                        ypos--;
                        current--;
                }
        }
        draw_list();
        return true;
}
//**************************************************************************/
//  CLASS: Msgmenu
//  MEMBER FUNCTION: page_down
// 
//  DESCRIPTION: If current is last of page, then switch to next page. 
//               Otherwise goto last of current page.
//
//
//  RETURN: false if failed. 
//**************************************************************************/
bool
Msgmenu::page_down()
{
        if (current >= thread_cnt-1) {
                return false;
        }
        if (ypos == ymax) {
                current_page++;
                first_of_page = ++current;
                for (int i=1; i < lines_per_page && current < thread_cnt-1;
                     i++, current++);
                ypos = ymin + (current - first_of_page);
        } else {
                while (current < thread_cnt-1 && ypos < ymax) {
                        current++;
                        ypos++;
                }
        }
        draw_list();
        return true;
}

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: down
//
// description: Moves pointer to next thread/article
//
//
// in:  Screen
// out: 
//
// returns: false if cannot move down
//
//**************************************************************************/
bool
Msgmenu::down()
{
        if (current < (thread_cnt-1)) {
                if (ypos == ymax) {
                        first_of_page = ++current;
                        current_page++;
                        ypos = ymin;
                        draw_list();
                } else {
                        Screen.gotoxy (0,ypos);
                        draw_article_info ();
                        
                        Screen.gotoxy (0,++ypos);
                        threads.next();
                        draw_article_info (EMPHASIZED_FL);
                        current++;
                }
                return true;
        }
        return false;
}

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: up
//
// description: Moves pointer to previous article/thread.
//
//
// in:  Screen
// out: 
//
// returns: false in cannot move up
//
//**************************************************************************/
bool
Msgmenu::up()
{
        if (current > 0) {
                if (ypos == ymin) {
                        ypos = ymax;
                        first_of_page = current - lines_per_page;
                        current--;
                        current_page--;
                        draw_list();
                } else {
                        Screen.gotoxy (0,ypos);
                        draw_article_info ();
                        
                        Screen.gotoxy (0,--ypos);
                        threads.prev();
                        draw_article_info (EMPHASIZED_FL);
                        current--;
                }
                return true;
        }
        return false;
}

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: set_selector_to_current_thread
//
// description: Moves pointer to current thread.
//
//
// in:  
// out: 
//
// returns: 
//
//**************************************************************************/
void
Msgmenu::set_selector_to_current_thread()
{
        current = threads.get_index();
        current_page = current / lines_per_page;
        first_of_page = current_page * lines_per_page;
        ypos = ymin + (current - first_of_page);
}

//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: update_msg_counts
//
// description: counts number of unread articles in current group.
//
//
// in:  mail
// out:  
//
// returns: 
//
//**************************************************************************/
void
Msgmenu::update_msg_counts()
{
        mail->update_groups_counts();
        threads.refresh();
}


//**************************************************************************/
//
// CLASS: Msgmenu
// MEMBER FUNCTION: go_next_article
//
// description: Moves to next unread article, or if all articles
//              is showed, then stays at current article.
//
// in:  mail
// out: 
//
// returns: true, if next unread message is available
//
//**************************************************************************/
bool
Msgmenu::go_next_article ()
{
        if (show_all_articles == false) {
                if (false == threads.next_unread()) {
                        return false;
                }
                set_selector_to_current_thread();
        }
        if (mail->count_unread() > 0) {
                return true;
        } else {
                return false;
        }
}


// **************************************************************************
//
// CLASS: Msgmenu
// MEMBER FUNCTION: sort
//
// description: sorts articles of current group to given selected order.
//
//
// in:  
// out: 
//
// returns: true if articles sorted, false if not.
//
// **************************************************************************
bool
Msgmenu::sort()
{

        Screen.bottom();
        show_option_str ( "Select sort order. [@t@]hread [@s@]ubject [@d@]ate"
                        " [@a@]uthor [@r@]eceiver @t@" );
        Screen.clr_eol();
        for (;;) {
                int c = Screen.get_ch();
                switch(c) {
                case 't':
                case SELECT_CMD:
                        threads.sort (THREAD_SORTED);
                        return true;
                case 's':
                        threads.sort (SUBJECT_SORTED);
                        return true;
                case 'd':
                        threads.sort (DATE_SORTED);
                        return true;
                case 'a':
                        threads.sort (WRITER_SORTED);
                        return true;
                case 'r':
                        threads.sort (RECEIVER_SORTED);
                        return true;
                case QUIT_CMD:
                        return false;
                }
        }
}










